"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineWlmRoutes = defineWlmRoutes;
var _configSchema = require("@osd/config-schema");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function defineWlmRoutes(router) {
  // Get WLM stats across all nodes in the cluster
  router.get({
    path: '/api/_wlm/stats',
    validate: false
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const stats = await client.transport.request({
        method: 'GET',
        path: '/_wlm/stats'
      });
      return response.ok({
        body: stats
      });
    } catch (error) {
      context.queryInsights.logger.error(`Failed to fetch WLM stats: ${error.message}`, {
        error
      });
      return response.custom({
        statusCode: error.statusCode || 500,
        body: {
          message: `Failed to fetch WLM stats: ${error.message}`
        }
      });
    }
  });

  // Get WLM stats for a specific node
  router.get({
    path: '/api/_wlm/{nodeId}/stats',
    validate: {
      params: _configSchema.schema.object({
        nodeId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const {
        nodeId
      } = request.params;
      const stats = await client.transport.request({
        method: 'GET',
        path: `/_wlm/${nodeId}/stats`
      });
      return response.ok({
        body: stats
      });
    } catch (error) {
      console.error(`Failed to fetch stats for node ${request.params.nodeId}:`, error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: {
          message: `Failed to fetch stats for node ${request.params.nodeId}: ${error.message}`
        }
      });
    }
  });

  // List all workload groups
  router.get({
    path: '/api/_wlm/workload_group',
    validate: false
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const result = await client.transport.request({
        method: 'GET',
        path: '/_wlm/workload_group'
      });
      return response.ok({
        body: result
      });
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: {
          message: `Failed to fetch workload groups: ${error.message}`
        }
      });
    }
  });

  // Get workload group by name
  router.get({
    path: '/api/_wlm/workload_group/{name}',
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const {
      name
    } = request.params;
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const result = await client.transport.request({
        method: 'GET',
        path: `/_wlm/workload_group/${name}`
      });
      return response.ok({
        body: result
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: {
          message: `Error fetching workload group: ${error.message}`
        }
      });
    }
  });

  // Create a new workload group
  router.put({
    path: '/api/_wlm/workload_group',
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        resiliency_mode: _configSchema.schema.string(),
        resource_limits: _configSchema.schema.object({
          cpu: _configSchema.schema.maybe(_configSchema.schema.number()),
          memory: _configSchema.schema.maybe(_configSchema.schema.number())
        })
      })
    }
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const body = request.body;
      const result = await client.transport.request({
        method: 'PUT',
        path: '/_wlm/workload_group',
        body
      });
      return response.ok({
        body: result
      });
    } catch (e) {
      console.error('Failed to create workload group:', e);
      return response.internalError({
        body: {
          message: `Failed to create workload group: ${e.message}`
        }
      });
    }
  });

  // Update a workload group by name
  router.put({
    path: '/api/_wlm/workload_group/{name}',
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        resiliency_mode: _configSchema.schema.string(),
        resource_limits: _configSchema.schema.object({
          cpu: _configSchema.schema.maybe(_configSchema.schema.number()),
          memory: _configSchema.schema.maybe(_configSchema.schema.number())
        })
      })
    }
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const {
        name
      } = request.params;
      const body = request.body;
      const result = await client.transport.request({
        method: 'PUT',
        path: `/_wlm/workload_group/${name}`,
        body
      });
      return response.ok({
        body: result
      });
    } catch (e) {
      console.error('Failed to update workload group:', e);
      return response.internalError({
        body: {
          message: `Failed to update workload group: ${e.message}`
        }
      });
    }
  });

  // Delete a workload group by name
  router.delete({
    path: '/api/_wlm/workload_group/{name}',
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const {
        name
      } = request.params;
      const result = await client.transport.request({
        method: 'DELETE',
        path: `/_wlm/workload_group/${name}`
      });
      return response.ok({
        body: result
      });
    } catch (e) {
      console.error(`Failed to delete workload group '${request.params.name}':`, e);
      return response.internalError({
        body: {
          message: `Failed to delete workload group '${request.params.name}': ${e.message}`
        }
      });
    }
  });

  // Get stats for a specific workload group
  router.get({
    path: '/api/_wlm/stats/{workloadGroupId}',
    validate: {
      params: _configSchema.schema.object({
        workloadGroupId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const {
        workloadGroupId
      } = request.params;
      const result = await client.transport.request({
        method: 'GET',
        path: `/_wlm/stats/${workloadGroupId}`
      });
      return response.ok({
        body: result
      });
    } catch (error) {
      console.error(`Failed to fetch WLM stats for group ${request.params.workloadGroupId}:`, error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: {
          message: `Failed to fetch WLM stats for group ${request.params.workloadGroupId}: ${error.message}`
        }
      });
    }
  });

  // Get all node IDs (used for node selection dropdown)
  router.get({
    path: '/api/_wlm_proxy/_nodes',
    validate: false
  }, async (context, request, response) => {
    const esClient = context.core.opensearch.client.asCurrentUser;
    try {
      const result = await esClient.nodes.info();
      return response.ok({
        body: result
      });
    } catch (e) {
      return response.customError({
        statusCode: e.statusCode || 500,
        body: e.message
      });
    }
  });

  // Create index rule
  router.put({
    path: '/api/_rules/workload_group',
    validate: {
      body: _configSchema.schema.object({
        description: _configSchema.schema.string(),
        index_pattern: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        workload_group: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const description = request.body.description;
      const indexPattern = request.body.index_pattern;
      const workloadGroup = request.body.workload_group;
      const result = await client.transport.request({
        method: 'PUT',
        path: '/_rules/workload_group',
        body: {
          description,
          index_pattern: indexPattern,
          workload_group: workloadGroup
        }
      });
      return response.ok({
        body: result
      });
    } catch (error) {
      console.error(`Failed to create index rule:`, error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: {
          message: `Failed to create index rule: ${error.message}`
        }
      });
    }
  });

  // Get all index rules
  router.get({
    path: '/api/_rules/workload_group',
    validate: false
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const result = await client.transport.request({
        method: 'GET',
        path: '/_rules/workload_group'
      });
      return response.ok({
        body: result
      });
    } catch (e) {
      console.error('Failed to fetch index rules:', e);
      return response.internalError({
        body: {
          message: `Failed to fetch index rules: ${e.message}`
        }
      });
    }
  });

  // Delete index rule by ID
  router.delete({
    path: '/api/_rules/workload_group/{ruleId}',
    validate: {
      params: _configSchema.schema.object({
        ruleId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const {
      ruleId
    } = request.params;
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const result = await client.transport.request({
        method: 'DELETE',
        path: `/_rules/workload_group/${ruleId}`
      });
      return response.ok({
        body: result
      });
    } catch (e) {
      console.error(`Failed to delete index rule ${ruleId}:`, e);
      return response.internalError({
        body: {
          message: `Failed to delete index rule ${ruleId}: ${e.message}`
        }
      });
    }
  });

  // Update index rule
  router.put({
    path: '/api/_rules/workload_group/{ruleId}',
    validate: {
      params: _configSchema.schema.object({
        ruleId: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        description: _configSchema.schema.string(),
        index_pattern: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        workload_group: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const {
      ruleId
    } = request.params;
    const body = request.body;
    try {
      const result = await context.core.opensearch.client.asCurrentUser.transport.request({
        method: 'PUT',
        path: `/_rules/workload_group/${ruleId}`,
        body
      });
      return response.ok({
        body: result
      });
    } catch (error) {
      console.error('Error updating rule:', error);
      return response.customError({
        body: error.message || error,
        statusCode: error.statusCode || 500
      });
    }
  });

  // Get node level cpu and memory threshold
  router.get({
    path: '/api/_wlm/thresholds',
    validate: false
  }, async (context, request, response) => {
    var _ref, _ref2, _body$transient$wlm$w, _body$transient, _body$persistent, _body$defaults, _ref3, _ref4, _body$transient$wlm$w2, _body$transient2, _body$persistent2, _body$defaults2;
    const esClient = context.core.opensearch.client.asInternalUser;
    const {
      body
    } = await esClient.cluster.getSettings({
      include_defaults: true
    });
    const cpuThreshold = (_ref = (_ref2 = (_body$transient$wlm$w = (_body$transient = body.transient) === null || _body$transient === void 0 || (_body$transient = _body$transient.wlm) === null || _body$transient === void 0 || (_body$transient = _body$transient.workload_group) === null || _body$transient === void 0 || (_body$transient = _body$transient.node) === null || _body$transient === void 0 ? void 0 : _body$transient.cpu_rejection_threshold) !== null && _body$transient$wlm$w !== void 0 ? _body$transient$wlm$w : (_body$persistent = body.persistent) === null || _body$persistent === void 0 || (_body$persistent = _body$persistent.wlm) === null || _body$persistent === void 0 || (_body$persistent = _body$persistent.workload_group) === null || _body$persistent === void 0 || (_body$persistent = _body$persistent.node) === null || _body$persistent === void 0 ? void 0 : _body$persistent.cpu_rejection_threshold) !== null && _ref2 !== void 0 ? _ref2 : (_body$defaults = body.defaults) === null || _body$defaults === void 0 || (_body$defaults = _body$defaults.wlm) === null || _body$defaults === void 0 || (_body$defaults = _body$defaults.workload_group) === null || _body$defaults === void 0 || (_body$defaults = _body$defaults.node) === null || _body$defaults === void 0 ? void 0 : _body$defaults.cpu_rejection_threshold) !== null && _ref !== void 0 ? _ref : '1';
    const memoryThreshold = (_ref3 = (_ref4 = (_body$transient$wlm$w2 = (_body$transient2 = body.transient) === null || _body$transient2 === void 0 || (_body$transient2 = _body$transient2.wlm) === null || _body$transient2 === void 0 || (_body$transient2 = _body$transient2.workload_group) === null || _body$transient2 === void 0 || (_body$transient2 = _body$transient2.node) === null || _body$transient2 === void 0 ? void 0 : _body$transient2.memory_rejection_threshold) !== null && _body$transient$wlm$w2 !== void 0 ? _body$transient$wlm$w2 : (_body$persistent2 = body.persistent) === null || _body$persistent2 === void 0 || (_body$persistent2 = _body$persistent2.wlm) === null || _body$persistent2 === void 0 || (_body$persistent2 = _body$persistent2.workload_group) === null || _body$persistent2 === void 0 || (_body$persistent2 = _body$persistent2.node) === null || _body$persistent2 === void 0 ? void 0 : _body$persistent2.memory_rejection_threshold) !== null && _ref4 !== void 0 ? _ref4 : (_body$defaults2 = body.defaults) === null || _body$defaults2 === void 0 || (_body$defaults2 = _body$defaults2.wlm) === null || _body$defaults2 === void 0 || (_body$defaults2 = _body$defaults2.workload_group) === null || _body$defaults2 === void 0 || (_body$defaults2 = _body$defaults2.node) === null || _body$defaults2 === void 0 ? void 0 : _body$defaults2.memory_rejection_threshold) !== null && _ref3 !== void 0 ? _ref3 : '1';
    return response.ok({
      body: {
        cpuRejectionThreshold: parseFloat(cpuThreshold),
        memoryRejectionThreshold: parseFloat(memoryThreshold)
      }
    });
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfY29uZmlnU2NoZW1hIiwicmVxdWlyZSIsImRlZmluZVdsbVJvdXRlcyIsInJvdXRlciIsImdldCIsInBhdGgiLCJ2YWxpZGF0ZSIsImNvbnRleHQiLCJyZXF1ZXN0IiwicmVzcG9uc2UiLCJjbGllbnQiLCJjb3JlIiwib3BlbnNlYXJjaCIsImFzQ3VycmVudFVzZXIiLCJzdGF0cyIsInRyYW5zcG9ydCIsIm1ldGhvZCIsIm9rIiwiYm9keSIsImVycm9yIiwicXVlcnlJbnNpZ2h0cyIsImxvZ2dlciIsIm1lc3NhZ2UiLCJjdXN0b20iLCJzdGF0dXNDb2RlIiwicGFyYW1zIiwic2NoZW1hIiwib2JqZWN0Iiwibm9kZUlkIiwic3RyaW5nIiwiY29uc29sZSIsInJlc3VsdCIsImN1c3RvbUVycm9yIiwibmFtZSIsInB1dCIsInJlc2lsaWVuY3lfbW9kZSIsInJlc291cmNlX2xpbWl0cyIsImNwdSIsIm1heWJlIiwibnVtYmVyIiwibWVtb3J5IiwiZSIsImludGVybmFsRXJyb3IiLCJkZWxldGUiLCJ3b3JrbG9hZEdyb3VwSWQiLCJlc0NsaWVudCIsIm5vZGVzIiwiaW5mbyIsImRlc2NyaXB0aW9uIiwiaW5kZXhfcGF0dGVybiIsImFycmF5T2YiLCJ3b3JrbG9hZF9ncm91cCIsImluZGV4UGF0dGVybiIsIndvcmtsb2FkR3JvdXAiLCJydWxlSWQiLCJfcmVmIiwiX3JlZjIiLCJfYm9keSR0cmFuc2llbnQkd2xtJHciLCJfYm9keSR0cmFuc2llbnQiLCJfYm9keSRwZXJzaXN0ZW50IiwiX2JvZHkkZGVmYXVsdHMiLCJfcmVmMyIsIl9yZWY0IiwiX2JvZHkkdHJhbnNpZW50JHdsbSR3MiIsIl9ib2R5JHRyYW5zaWVudDIiLCJfYm9keSRwZXJzaXN0ZW50MiIsIl9ib2R5JGRlZmF1bHRzMiIsImFzSW50ZXJuYWxVc2VyIiwiY2x1c3RlciIsImdldFNldHRpbmdzIiwiaW5jbHVkZV9kZWZhdWx0cyIsImNwdVRocmVzaG9sZCIsInRyYW5zaWVudCIsIndsbSIsIm5vZGUiLCJjcHVfcmVqZWN0aW9uX3RocmVzaG9sZCIsInBlcnNpc3RlbnQiLCJkZWZhdWx0cyIsIm1lbW9yeVRocmVzaG9sZCIsIm1lbW9yeV9yZWplY3Rpb25fdGhyZXNob2xkIiwiY3B1UmVqZWN0aW9uVGhyZXNob2xkIiwicGFyc2VGbG9hdCIsIm1lbW9yeVJlamVjdGlvblRocmVzaG9sZCJdLCJzb3VyY2VzIjpbIndsbVJvdXRlcy50cyJdLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuICogQ29weXJpZ2h0IE9wZW5TZWFyY2ggQ29udHJpYnV0b3JzXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuICovXG5cbmltcG9ydCB7IHNjaGVtYSB9IGZyb20gJ0Bvc2QvY29uZmlnLXNjaGVtYSc7XG5pbXBvcnQgeyBJUm91dGVyIH0gZnJvbSAnLi4vLi4vLi4vLi4vc3JjL2NvcmUvc2VydmVyJztcblxuZXhwb3J0IGZ1bmN0aW9uIGRlZmluZVdsbVJvdXRlcyhyb3V0ZXI6IElSb3V0ZXIpIHtcbiAgLy8gR2V0IFdMTSBzdGF0cyBhY3Jvc3MgYWxsIG5vZGVzIGluIHRoZSBjbHVzdGVyXG4gIHJvdXRlci5nZXQoXG4gICAge1xuICAgICAgcGF0aDogJy9hcGkvX3dsbS9zdGF0cycsXG4gICAgICB2YWxpZGF0ZTogZmFsc2UsXG4gICAgfSxcbiAgICBhc3luYyAoY29udGV4dCwgcmVxdWVzdCwgcmVzcG9uc2UpID0+IHtcbiAgICAgIHRyeSB7XG4gICAgICAgIGNvbnN0IGNsaWVudCA9IGNvbnRleHQuY29yZS5vcGVuc2VhcmNoLmNsaWVudC5hc0N1cnJlbnRVc2VyO1xuICAgICAgICBjb25zdCBzdGF0cyA9IGF3YWl0IGNsaWVudC50cmFuc3BvcnQucmVxdWVzdCh7XG4gICAgICAgICAgbWV0aG9kOiAnR0VUJyxcbiAgICAgICAgICBwYXRoOiAnL193bG0vc3RhdHMnLFxuICAgICAgICB9KTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHsgYm9keTogc3RhdHMgfSk7XG4gICAgICB9IGNhdGNoIChlcnJvcjogYW55KSB7XG4gICAgICAgIGNvbnRleHQucXVlcnlJbnNpZ2h0cy5sb2dnZXIuZXJyb3IoYEZhaWxlZCB0byBmZXRjaCBXTE0gc3RhdHM6ICR7ZXJyb3IubWVzc2FnZX1gLCB7XG4gICAgICAgICAgZXJyb3IsXG4gICAgICAgIH0pO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiBlcnJvci5zdGF0dXNDb2RlIHx8IDUwMCxcbiAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICBtZXNzYWdlOiBgRmFpbGVkIHRvIGZldGNoIFdMTSBzdGF0czogJHtlcnJvci5tZXNzYWdlfWAsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICApO1xuXG4gIC8vIEdldCBXTE0gc3RhdHMgZm9yIGEgc3BlY2lmaWMgbm9kZVxuICByb3V0ZXIuZ2V0KFxuICAgIHtcbiAgICAgIHBhdGg6ICcvYXBpL193bG0ve25vZGVJZH0vc3RhdHMnLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcGFyYW1zOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBub2RlSWQ6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKSA9PiB7XG4gICAgICB0cnkge1xuICAgICAgICBjb25zdCBjbGllbnQgPSBjb250ZXh0LmNvcmUub3BlbnNlYXJjaC5jbGllbnQuYXNDdXJyZW50VXNlcjtcbiAgICAgICAgY29uc3QgeyBub2RlSWQgfSA9IHJlcXVlc3QucGFyYW1zO1xuICAgICAgICBjb25zdCBzdGF0cyA9IGF3YWl0IGNsaWVudC50cmFuc3BvcnQucmVxdWVzdCh7XG4gICAgICAgICAgbWV0aG9kOiAnR0VUJyxcbiAgICAgICAgICBwYXRoOiBgL193bG0vJHtub2RlSWR9L3N0YXRzYCxcbiAgICAgICAgfSk7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7IGJvZHk6IHN0YXRzIH0pO1xuICAgICAgfSBjYXRjaCAoZXJyb3I6IGFueSkge1xuICAgICAgICBjb25zb2xlLmVycm9yKGBGYWlsZWQgdG8gZmV0Y2ggc3RhdHMgZm9yIG5vZGUgJHtyZXF1ZXN0LnBhcmFtcy5ub2RlSWR9OmAsIGVycm9yKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7XG4gICAgICAgICAgc3RhdHVzQ29kZTogZXJyb3Iuc3RhdHVzQ29kZSB8fCA1MDAsXG4gICAgICAgICAgYm9keToge1xuICAgICAgICAgICAgbWVzc2FnZTogYEZhaWxlZCB0byBmZXRjaCBzdGF0cyBmb3Igbm9kZSAke3JlcXVlc3QucGFyYW1zLm5vZGVJZH06ICR7ZXJyb3IubWVzc2FnZX1gLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICAvLyBMaXN0IGFsbCB3b3JrbG9hZCBncm91cHNcbiAgcm91dGVyLmdldChcbiAgICB7XG4gICAgICBwYXRoOiAnL2FwaS9fd2xtL3dvcmtsb2FkX2dyb3VwJyxcbiAgICAgIHZhbGlkYXRlOiBmYWxzZSxcbiAgICB9LFxuICAgIGFzeW5jIChjb250ZXh0LCByZXF1ZXN0LCByZXNwb25zZSkgPT4ge1xuICAgICAgdHJ5IHtcbiAgICAgICAgY29uc3QgY2xpZW50ID0gY29udGV4dC5jb3JlLm9wZW5zZWFyY2guY2xpZW50LmFzQ3VycmVudFVzZXI7XG4gICAgICAgIGNvbnN0IHJlc3VsdCA9IGF3YWl0IGNsaWVudC50cmFuc3BvcnQucmVxdWVzdCh7XG4gICAgICAgICAgbWV0aG9kOiAnR0VUJyxcbiAgICAgICAgICBwYXRoOiAnL193bG0vd29ya2xvYWRfZ3JvdXAnLFxuICAgICAgICB9KTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHsgYm9keTogcmVzdWx0IH0pO1xuICAgICAgfSBjYXRjaCAoZXJyb3I6IGFueSkge1xuICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tRXJyb3Ioe1xuICAgICAgICAgIHN0YXR1c0NvZGU6IGVycm9yLnN0YXR1c0NvZGUgfHwgNTAwLFxuICAgICAgICAgIGJvZHk6IHsgbWVzc2FnZTogYEZhaWxlZCB0byBmZXRjaCB3b3JrbG9hZCBncm91cHM6ICR7ZXJyb3IubWVzc2FnZX1gIH0sXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICAvLyBHZXQgd29ya2xvYWQgZ3JvdXAgYnkgbmFtZVxuICByb3V0ZXIuZ2V0KFxuICAgIHtcbiAgICAgIHBhdGg6ICcvYXBpL193bG0vd29ya2xvYWRfZ3JvdXAve25hbWV9JyxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIHBhcmFtczogc2NoZW1hLm9iamVjdCh7XG4gICAgICAgICAgbmFtZTogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBhc3luYyAoY29udGV4dCwgcmVxdWVzdCwgcmVzcG9uc2UpID0+IHtcbiAgICAgIGNvbnN0IHsgbmFtZSB9ID0gcmVxdWVzdC5wYXJhbXM7XG4gICAgICB0cnkge1xuICAgICAgICBjb25zdCBjbGllbnQgPSBjb250ZXh0LmNvcmUub3BlbnNlYXJjaC5jbGllbnQuYXNDdXJyZW50VXNlcjtcbiAgICAgICAgY29uc3QgcmVzdWx0ID0gYXdhaXQgY2xpZW50LnRyYW5zcG9ydC5yZXF1ZXN0KHtcbiAgICAgICAgICBtZXRob2Q6ICdHRVQnLFxuICAgICAgICAgIHBhdGg6IGAvX3dsbS93b3JrbG9hZF9ncm91cC8ke25hbWV9YCxcbiAgICAgICAgfSk7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7IGJvZHk6IHJlc3VsdCB9KTtcbiAgICAgIH0gY2F0Y2ggKGVycm9yOiBhbnkpIHtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7XG4gICAgICAgICAgc3RhdHVzQ29kZTogZXJyb3Iuc3RhdHVzQ29kZSB8fCA1MDAsXG4gICAgICAgICAgYm9keTogeyBtZXNzYWdlOiBgRXJyb3IgZmV0Y2hpbmcgd29ya2xvYWQgZ3JvdXA6ICR7ZXJyb3IubWVzc2FnZX1gIH0sXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICAvLyBDcmVhdGUgYSBuZXcgd29ya2xvYWQgZ3JvdXBcbiAgcm91dGVyLnB1dChcbiAgICB7XG4gICAgICBwYXRoOiAnL2FwaS9fd2xtL3dvcmtsb2FkX2dyb3VwJyxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIGJvZHk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIG5hbWU6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICByZXNpbGllbmN5X21vZGU6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICByZXNvdXJjZV9saW1pdHM6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgICAgY3B1OiBzY2hlbWEubWF5YmUoc2NoZW1hLm51bWJlcigpKSxcbiAgICAgICAgICAgIG1lbW9yeTogc2NoZW1hLm1heWJlKHNjaGVtYS5udW1iZXIoKSksXG4gICAgICAgICAgfSksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGFzeW5jIChjb250ZXh0LCByZXF1ZXN0LCByZXNwb25zZSkgPT4ge1xuICAgICAgdHJ5IHtcbiAgICAgICAgY29uc3QgY2xpZW50ID0gY29udGV4dC5jb3JlLm9wZW5zZWFyY2guY2xpZW50LmFzQ3VycmVudFVzZXI7XG4gICAgICAgIGNvbnN0IGJvZHkgPSByZXF1ZXN0LmJvZHk7XG5cbiAgICAgICAgY29uc3QgcmVzdWx0ID0gYXdhaXQgY2xpZW50LnRyYW5zcG9ydC5yZXF1ZXN0KHtcbiAgICAgICAgICBtZXRob2Q6ICdQVVQnLFxuICAgICAgICAgIHBhdGg6ICcvX3dsbS93b3JrbG9hZF9ncm91cCcsXG4gICAgICAgICAgYm9keSxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHsgYm9keTogcmVzdWx0IH0pO1xuICAgICAgfSBjYXRjaCAoZTogYW55KSB7XG4gICAgICAgIGNvbnNvbGUuZXJyb3IoJ0ZhaWxlZCB0byBjcmVhdGUgd29ya2xvYWQgZ3JvdXA6JywgZSk7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5pbnRlcm5hbEVycm9yKHtcbiAgICAgICAgICBib2R5OiB7IG1lc3NhZ2U6IGBGYWlsZWQgdG8gY3JlYXRlIHdvcmtsb2FkIGdyb3VwOiAke2UubWVzc2FnZX1gIH0sXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICAvLyBVcGRhdGUgYSB3b3JrbG9hZCBncm91cCBieSBuYW1lXG4gIHJvdXRlci5wdXQoXG4gICAge1xuICAgICAgcGF0aDogJy9hcGkvX3dsbS93b3JrbG9hZF9ncm91cC97bmFtZX0nLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcGFyYW1zOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBuYW1lOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgIH0pLFxuICAgICAgICBib2R5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICByZXNpbGllbmN5X21vZGU6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICByZXNvdXJjZV9saW1pdHM6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgICAgY3B1OiBzY2hlbWEubWF5YmUoc2NoZW1hLm51bWJlcigpKSxcbiAgICAgICAgICAgIG1lbW9yeTogc2NoZW1hLm1heWJlKHNjaGVtYS5udW1iZXIoKSksXG4gICAgICAgICAgfSksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGFzeW5jIChjb250ZXh0LCByZXF1ZXN0LCByZXNwb25zZSkgPT4ge1xuICAgICAgdHJ5IHtcbiAgICAgICAgY29uc3QgY2xpZW50ID0gY29udGV4dC5jb3JlLm9wZW5zZWFyY2guY2xpZW50LmFzQ3VycmVudFVzZXI7XG4gICAgICAgIGNvbnN0IHsgbmFtZSB9ID0gcmVxdWVzdC5wYXJhbXM7XG4gICAgICAgIGNvbnN0IGJvZHkgPSByZXF1ZXN0LmJvZHk7XG5cbiAgICAgICAgY29uc3QgcmVzdWx0ID0gYXdhaXQgY2xpZW50LnRyYW5zcG9ydC5yZXF1ZXN0KHtcbiAgICAgICAgICBtZXRob2Q6ICdQVVQnLFxuICAgICAgICAgIHBhdGg6IGAvX3dsbS93b3JrbG9hZF9ncm91cC8ke25hbWV9YCxcbiAgICAgICAgICBib2R5LFxuICAgICAgICB9KTtcblxuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soeyBib2R5OiByZXN1bHQgfSk7XG4gICAgICB9IGNhdGNoIChlOiBhbnkpIHtcbiAgICAgICAgY29uc29sZS5lcnJvcignRmFpbGVkIHRvIHVwZGF0ZSB3b3JrbG9hZCBncm91cDonLCBlKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmludGVybmFsRXJyb3Ioe1xuICAgICAgICAgIGJvZHk6IHsgbWVzc2FnZTogYEZhaWxlZCB0byB1cGRhdGUgd29ya2xvYWQgZ3JvdXA6ICR7ZS5tZXNzYWdlfWAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICApO1xuXG4gIC8vIERlbGV0ZSBhIHdvcmtsb2FkIGdyb3VwIGJ5IG5hbWVcbiAgcm91dGVyLmRlbGV0ZShcbiAgICB7XG4gICAgICBwYXRoOiAnL2FwaS9fd2xtL3dvcmtsb2FkX2dyb3VwL3tuYW1lfScsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBwYXJhbXM6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIG5hbWU6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKSA9PiB7XG4gICAgICB0cnkge1xuICAgICAgICBjb25zdCBjbGllbnQgPSBjb250ZXh0LmNvcmUub3BlbnNlYXJjaC5jbGllbnQuYXNDdXJyZW50VXNlcjtcbiAgICAgICAgY29uc3QgeyBuYW1lIH0gPSByZXF1ZXN0LnBhcmFtcztcblxuICAgICAgICBjb25zdCByZXN1bHQgPSBhd2FpdCBjbGllbnQudHJhbnNwb3J0LnJlcXVlc3Qoe1xuICAgICAgICAgIG1ldGhvZDogJ0RFTEVURScsXG4gICAgICAgICAgcGF0aDogYC9fd2xtL3dvcmtsb2FkX2dyb3VwLyR7bmFtZX1gLFxuICAgICAgICB9KTtcblxuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soeyBib2R5OiByZXN1bHQgfSk7XG4gICAgICB9IGNhdGNoIChlOiBhbnkpIHtcbiAgICAgICAgY29uc29sZS5lcnJvcihgRmFpbGVkIHRvIGRlbGV0ZSB3b3JrbG9hZCBncm91cCAnJHtyZXF1ZXN0LnBhcmFtcy5uYW1lfSc6YCwgZSk7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5pbnRlcm5hbEVycm9yKHtcbiAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICBtZXNzYWdlOiBgRmFpbGVkIHRvIGRlbGV0ZSB3b3JrbG9hZCBncm91cCAnJHtyZXF1ZXN0LnBhcmFtcy5uYW1lfSc6ICR7ZS5tZXNzYWdlfWAsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICApO1xuXG4gIC8vIEdldCBzdGF0cyBmb3IgYSBzcGVjaWZpYyB3b3JrbG9hZCBncm91cFxuICByb3V0ZXIuZ2V0KFxuICAgIHtcbiAgICAgIHBhdGg6ICcvYXBpL193bG0vc3RhdHMve3dvcmtsb2FkR3JvdXBJZH0nLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcGFyYW1zOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICB3b3JrbG9hZEdyb3VwSWQ6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKSA9PiB7XG4gICAgICB0cnkge1xuICAgICAgICBjb25zdCBjbGllbnQgPSBjb250ZXh0LmNvcmUub3BlbnNlYXJjaC5jbGllbnQuYXNDdXJyZW50VXNlcjtcbiAgICAgICAgY29uc3QgeyB3b3JrbG9hZEdyb3VwSWQgfSA9IHJlcXVlc3QucGFyYW1zO1xuXG4gICAgICAgIGNvbnN0IHJlc3VsdCA9IGF3YWl0IGNsaWVudC50cmFuc3BvcnQucmVxdWVzdCh7XG4gICAgICAgICAgbWV0aG9kOiAnR0VUJyxcbiAgICAgICAgICBwYXRoOiBgL193bG0vc3RhdHMvJHt3b3JrbG9hZEdyb3VwSWR9YCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHsgYm9keTogcmVzdWx0IH0pO1xuICAgICAgfSBjYXRjaCAoZXJyb3I6IGFueSkge1xuICAgICAgICBjb25zb2xlLmVycm9yKFxuICAgICAgICAgIGBGYWlsZWQgdG8gZmV0Y2ggV0xNIHN0YXRzIGZvciBncm91cCAke3JlcXVlc3QucGFyYW1zLndvcmtsb2FkR3JvdXBJZH06YCxcbiAgICAgICAgICBlcnJvclxuICAgICAgICApO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiBlcnJvci5zdGF0dXNDb2RlIHx8IDUwMCxcbiAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICBtZXNzYWdlOiBgRmFpbGVkIHRvIGZldGNoIFdMTSBzdGF0cyBmb3IgZ3JvdXAgJHtyZXF1ZXN0LnBhcmFtcy53b3JrbG9hZEdyb3VwSWR9OiAke2Vycm9yLm1lc3NhZ2V9YCxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICk7XG5cbiAgLy8gR2V0IGFsbCBub2RlIElEcyAodXNlZCBmb3Igbm9kZSBzZWxlY3Rpb24gZHJvcGRvd24pXG4gIHJvdXRlci5nZXQoXG4gICAge1xuICAgICAgcGF0aDogJy9hcGkvX3dsbV9wcm94eS9fbm9kZXMnLFxuICAgICAgdmFsaWRhdGU6IGZhbHNlLFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKSA9PiB7XG4gICAgICBjb25zdCBlc0NsaWVudCA9IGNvbnRleHQuY29yZS5vcGVuc2VhcmNoLmNsaWVudC5hc0N1cnJlbnRVc2VyO1xuICAgICAgdHJ5IHtcbiAgICAgICAgY29uc3QgcmVzdWx0ID0gYXdhaXQgZXNDbGllbnQubm9kZXMuaW5mbygpO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soeyBib2R5OiByZXN1bHQgfSk7XG4gICAgICB9IGNhdGNoIChlKSB7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b21FcnJvcih7XG4gICAgICAgICAgc3RhdHVzQ29kZTogZS5zdGF0dXNDb2RlIHx8IDUwMCxcbiAgICAgICAgICBib2R5OiBlLm1lc3NhZ2UsXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICAvLyBDcmVhdGUgaW5kZXggcnVsZVxuICByb3V0ZXIucHV0KFxuICAgIHtcbiAgICAgIHBhdGg6ICcvYXBpL19ydWxlcy93b3JrbG9hZF9ncm91cCcsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBib2R5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBkZXNjcmlwdGlvbjogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICAgIGluZGV4X3BhdHRlcm46IHNjaGVtYS5hcnJheU9mKHNjaGVtYS5zdHJpbmcoKSksXG4gICAgICAgICAgd29ya2xvYWRfZ3JvdXA6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKSA9PiB7XG4gICAgICB0cnkge1xuICAgICAgICBjb25zdCBjbGllbnQgPSBjb250ZXh0LmNvcmUub3BlbnNlYXJjaC5jbGllbnQuYXNDdXJyZW50VXNlcjtcbiAgICAgICAgY29uc3QgZGVzY3JpcHRpb24gPSByZXF1ZXN0LmJvZHkuZGVzY3JpcHRpb247XG4gICAgICAgIGNvbnN0IGluZGV4UGF0dGVybiA9IHJlcXVlc3QuYm9keS5pbmRleF9wYXR0ZXJuO1xuICAgICAgICBjb25zdCB3b3JrbG9hZEdyb3VwID0gcmVxdWVzdC5ib2R5Lndvcmtsb2FkX2dyb3VwO1xuXG4gICAgICAgIGNvbnN0IHJlc3VsdCA9IGF3YWl0IGNsaWVudC50cmFuc3BvcnQucmVxdWVzdCh7XG4gICAgICAgICAgbWV0aG9kOiAnUFVUJyxcbiAgICAgICAgICBwYXRoOiAnL19ydWxlcy93b3JrbG9hZF9ncm91cCcsXG4gICAgICAgICAgYm9keToge1xuICAgICAgICAgICAgZGVzY3JpcHRpb24sXG4gICAgICAgICAgICBpbmRleF9wYXR0ZXJuOiBpbmRleFBhdHRlcm4sXG4gICAgICAgICAgICB3b3JrbG9hZF9ncm91cDogd29ya2xvYWRHcm91cCxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcblxuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soeyBib2R5OiByZXN1bHQgfSk7XG4gICAgICB9IGNhdGNoIChlcnJvcjogYW55KSB7XG4gICAgICAgIGNvbnNvbGUuZXJyb3IoYEZhaWxlZCB0byBjcmVhdGUgaW5kZXggcnVsZTpgLCBlcnJvcik7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICAgIHN0YXR1c0NvZGU6IGVycm9yLnN0YXR1c0NvZGUgfHwgNTAwLFxuICAgICAgICAgIGJvZHk6IHsgbWVzc2FnZTogYEZhaWxlZCB0byBjcmVhdGUgaW5kZXggcnVsZTogJHtlcnJvci5tZXNzYWdlfWAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICApO1xuXG4gIC8vIEdldCBhbGwgaW5kZXggcnVsZXNcbiAgcm91dGVyLmdldChcbiAgICB7XG4gICAgICBwYXRoOiAnL2FwaS9fcnVsZXMvd29ya2xvYWRfZ3JvdXAnLFxuICAgICAgdmFsaWRhdGU6IGZhbHNlLFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKSA9PiB7XG4gICAgICB0cnkge1xuICAgICAgICBjb25zdCBjbGllbnQgPSBjb250ZXh0LmNvcmUub3BlbnNlYXJjaC5jbGllbnQuYXNDdXJyZW50VXNlcjtcblxuICAgICAgICBjb25zdCByZXN1bHQgPSBhd2FpdCBjbGllbnQudHJhbnNwb3J0LnJlcXVlc3Qoe1xuICAgICAgICAgIG1ldGhvZDogJ0dFVCcsXG4gICAgICAgICAgcGF0aDogJy9fcnVsZXMvd29ya2xvYWRfZ3JvdXAnLFxuICAgICAgICB9KTtcblxuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soeyBib2R5OiByZXN1bHQgfSk7XG4gICAgICB9IGNhdGNoIChlOiBhbnkpIHtcbiAgICAgICAgY29uc29sZS5lcnJvcignRmFpbGVkIHRvIGZldGNoIGluZGV4IHJ1bGVzOicsIGUpO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2UuaW50ZXJuYWxFcnJvcih7XG4gICAgICAgICAgYm9keTogeyBtZXNzYWdlOiBgRmFpbGVkIHRvIGZldGNoIGluZGV4IHJ1bGVzOiAke2UubWVzc2FnZX1gIH0sXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICAvLyBEZWxldGUgaW5kZXggcnVsZSBieSBJRFxuICByb3V0ZXIuZGVsZXRlKFxuICAgIHtcbiAgICAgIHBhdGg6ICcvYXBpL19ydWxlcy93b3JrbG9hZF9ncm91cC97cnVsZUlkfScsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBwYXJhbXM6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIHJ1bGVJZDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBhc3luYyAoY29udGV4dCwgcmVxdWVzdCwgcmVzcG9uc2UpID0+IHtcbiAgICAgIGNvbnN0IHsgcnVsZUlkIH0gPSByZXF1ZXN0LnBhcmFtcztcbiAgICAgIHRyeSB7XG4gICAgICAgIGNvbnN0IGNsaWVudCA9IGNvbnRleHQuY29yZS5vcGVuc2VhcmNoLmNsaWVudC5hc0N1cnJlbnRVc2VyO1xuXG4gICAgICAgIGNvbnN0IHJlc3VsdCA9IGF3YWl0IGNsaWVudC50cmFuc3BvcnQucmVxdWVzdCh7XG4gICAgICAgICAgbWV0aG9kOiAnREVMRVRFJyxcbiAgICAgICAgICBwYXRoOiBgL19ydWxlcy93b3JrbG9hZF9ncm91cC8ke3J1bGVJZH1gLFxuICAgICAgICB9KTtcblxuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soeyBib2R5OiByZXN1bHQgfSk7XG4gICAgICB9IGNhdGNoIChlOiBhbnkpIHtcbiAgICAgICAgY29uc29sZS5lcnJvcihgRmFpbGVkIHRvIGRlbGV0ZSBpbmRleCBydWxlICR7cnVsZUlkfTpgLCBlKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmludGVybmFsRXJyb3Ioe1xuICAgICAgICAgIGJvZHk6IHsgbWVzc2FnZTogYEZhaWxlZCB0byBkZWxldGUgaW5kZXggcnVsZSAke3J1bGVJZH06ICR7ZS5tZXNzYWdlfWAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICApO1xuXG4gIC8vIFVwZGF0ZSBpbmRleCBydWxlXG4gIHJvdXRlci5wdXQoXG4gICAge1xuICAgICAgcGF0aDogJy9hcGkvX3J1bGVzL3dvcmtsb2FkX2dyb3VwL3tydWxlSWR9JyxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIHBhcmFtczogc2NoZW1hLm9iamVjdCh7XG4gICAgICAgICAgcnVsZUlkOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgIH0pLFxuICAgICAgICBib2R5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBkZXNjcmlwdGlvbjogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICAgIGluZGV4X3BhdHRlcm46IHNjaGVtYS5hcnJheU9mKHNjaGVtYS5zdHJpbmcoKSksXG4gICAgICAgICAgd29ya2xvYWRfZ3JvdXA6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKSA9PiB7XG4gICAgICBjb25zdCB7IHJ1bGVJZCB9ID0gcmVxdWVzdC5wYXJhbXM7XG4gICAgICBjb25zdCBib2R5ID0gcmVxdWVzdC5ib2R5O1xuXG4gICAgICB0cnkge1xuICAgICAgICBjb25zdCByZXN1bHQgPSBhd2FpdCBjb250ZXh0LmNvcmUub3BlbnNlYXJjaC5jbGllbnQuYXNDdXJyZW50VXNlci50cmFuc3BvcnQucmVxdWVzdCh7XG4gICAgICAgICAgbWV0aG9kOiAnUFVUJyxcbiAgICAgICAgICBwYXRoOiBgL19ydWxlcy93b3JrbG9hZF9ncm91cC8ke3J1bGVJZH1gLFxuICAgICAgICAgIGJvZHksXG4gICAgICAgIH0pO1xuXG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7IGJvZHk6IHJlc3VsdCB9KTtcbiAgICAgIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgICAgIGNvbnNvbGUuZXJyb3IoJ0Vycm9yIHVwZGF0aW5nIHJ1bGU6JywgZXJyb3IpO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tRXJyb3Ioe1xuICAgICAgICAgIGJvZHk6IGVycm9yLm1lc3NhZ2UgfHwgZXJyb3IsXG4gICAgICAgICAgc3RhdHVzQ29kZTogZXJyb3Iuc3RhdHVzQ29kZSB8fCA1MDAsXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICAvLyBHZXQgbm9kZSBsZXZlbCBjcHUgYW5kIG1lbW9yeSB0aHJlc2hvbGRcbiAgcm91dGVyLmdldChcbiAgICB7XG4gICAgICBwYXRoOiAnL2FwaS9fd2xtL3RocmVzaG9sZHMnLFxuICAgICAgdmFsaWRhdGU6IGZhbHNlLFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKSA9PiB7XG4gICAgICBjb25zdCBlc0NsaWVudCA9IGNvbnRleHQuY29yZS5vcGVuc2VhcmNoLmNsaWVudC5hc0ludGVybmFsVXNlcjtcbiAgICAgIGNvbnN0IHsgYm9keSB9ID0gYXdhaXQgZXNDbGllbnQuY2x1c3Rlci5nZXRTZXR0aW5ncyh7IGluY2x1ZGVfZGVmYXVsdHM6IHRydWUgfSk7XG5cbiAgICAgIGNvbnN0IGNwdVRocmVzaG9sZCA9XG4gICAgICAgIGJvZHkudHJhbnNpZW50Py53bG0/Lndvcmtsb2FkX2dyb3VwPy5ub2RlPy5jcHVfcmVqZWN0aW9uX3RocmVzaG9sZCA/P1xuICAgICAgICBib2R5LnBlcnNpc3RlbnQ/LndsbT8ud29ya2xvYWRfZ3JvdXA/Lm5vZGU/LmNwdV9yZWplY3Rpb25fdGhyZXNob2xkID8/XG4gICAgICAgIGJvZHkuZGVmYXVsdHM/LndsbT8ud29ya2xvYWRfZ3JvdXA/Lm5vZGU/LmNwdV9yZWplY3Rpb25fdGhyZXNob2xkID8/XG4gICAgICAgICcxJztcblxuICAgICAgY29uc3QgbWVtb3J5VGhyZXNob2xkID1cbiAgICAgICAgYm9keS50cmFuc2llbnQ/LndsbT8ud29ya2xvYWRfZ3JvdXA/Lm5vZGU/Lm1lbW9yeV9yZWplY3Rpb25fdGhyZXNob2xkID8/XG4gICAgICAgIGJvZHkucGVyc2lzdGVudD8ud2xtPy53b3JrbG9hZF9ncm91cD8ubm9kZT8ubWVtb3J5X3JlamVjdGlvbl90aHJlc2hvbGQgPz9cbiAgICAgICAgYm9keS5kZWZhdWx0cz8ud2xtPy53b3JrbG9hZF9ncm91cD8ubm9kZT8ubWVtb3J5X3JlamVjdGlvbl90aHJlc2hvbGQgPz9cbiAgICAgICAgJzEnO1xuXG4gICAgICByZXR1cm4gcmVzcG9uc2Uub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgY3B1UmVqZWN0aW9uVGhyZXNob2xkOiBwYXJzZUZsb2F0KGNwdVRocmVzaG9sZCksXG4gICAgICAgICAgbWVtb3J5UmVqZWN0aW9uVGhyZXNob2xkOiBwYXJzZUZsb2F0KG1lbW9yeVRocmVzaG9sZCksXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG4gICk7XG59XG4iXSwibWFwcGluZ3MiOiI7Ozs7OztBQUtBLElBQUFBLGFBQUEsR0FBQUMsT0FBQTtBQUxBO0FBQ0E7QUFDQTtBQUNBOztBQUtPLFNBQVNDLGVBQWVBLENBQUNDLE1BQWUsRUFBRTtFQUMvQztFQUNBQSxNQUFNLENBQUNDLEdBQUcsQ0FDUjtJQUNFQyxJQUFJLEVBQUUsaUJBQWlCO0lBQ3ZCQyxRQUFRLEVBQUU7RUFDWixDQUFDLEVBQ0QsT0FBT0MsT0FBTyxFQUFFQyxPQUFPLEVBQUVDLFFBQVEsS0FBSztJQUNwQyxJQUFJO01BQ0YsTUFBTUMsTUFBTSxHQUFHSCxPQUFPLENBQUNJLElBQUksQ0FBQ0MsVUFBVSxDQUFDRixNQUFNLENBQUNHLGFBQWE7TUFDM0QsTUFBTUMsS0FBSyxHQUFHLE1BQU1KLE1BQU0sQ0FBQ0ssU0FBUyxDQUFDUCxPQUFPLENBQUM7UUFDM0NRLE1BQU0sRUFBRSxLQUFLO1FBQ2JYLElBQUksRUFBRTtNQUNSLENBQUMsQ0FBQztNQUNGLE9BQU9JLFFBQVEsQ0FBQ1EsRUFBRSxDQUFDO1FBQUVDLElBQUksRUFBRUo7TUFBTSxDQUFDLENBQUM7SUFDckMsQ0FBQyxDQUFDLE9BQU9LLEtBQVUsRUFBRTtNQUNuQlosT0FBTyxDQUFDYSxhQUFhLENBQUNDLE1BQU0sQ0FBQ0YsS0FBSyxDQUFFLDhCQUE2QkEsS0FBSyxDQUFDRyxPQUFRLEVBQUMsRUFBRTtRQUNoRkg7TUFDRixDQUFDLENBQUM7TUFDRixPQUFPVixRQUFRLENBQUNjLE1BQU0sQ0FBQztRQUNyQkMsVUFBVSxFQUFFTCxLQUFLLENBQUNLLFVBQVUsSUFBSSxHQUFHO1FBQ25DTixJQUFJLEVBQUU7VUFDSkksT0FBTyxFQUFHLDhCQUE2QkgsS0FBSyxDQUFDRyxPQUFRO1FBQ3ZEO01BQ0YsQ0FBQyxDQUFDO0lBQ0o7RUFDRixDQUNGLENBQUM7O0VBRUQ7RUFDQW5CLE1BQU0sQ0FBQ0MsR0FBRyxDQUNSO0lBQ0VDLElBQUksRUFBRSwwQkFBMEI7SUFDaENDLFFBQVEsRUFBRTtNQUNSbUIsTUFBTSxFQUFFQyxvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDcEJDLE1BQU0sRUFBRUYsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDO01BQ3hCLENBQUM7SUFDSDtFQUNGLENBQUMsRUFDRCxPQUFPdEIsT0FBTyxFQUFFQyxPQUFPLEVBQUVDLFFBQVEsS0FBSztJQUNwQyxJQUFJO01BQ0YsTUFBTUMsTUFBTSxHQUFHSCxPQUFPLENBQUNJLElBQUksQ0FBQ0MsVUFBVSxDQUFDRixNQUFNLENBQUNHLGFBQWE7TUFDM0QsTUFBTTtRQUFFZTtNQUFPLENBQUMsR0FBR3BCLE9BQU8sQ0FBQ2lCLE1BQU07TUFDakMsTUFBTVgsS0FBSyxHQUFHLE1BQU1KLE1BQU0sQ0FBQ0ssU0FBUyxDQUFDUCxPQUFPLENBQUM7UUFDM0NRLE1BQU0sRUFBRSxLQUFLO1FBQ2JYLElBQUksRUFBRyxTQUFRdUIsTUFBTztNQUN4QixDQUFDLENBQUM7TUFDRixPQUFPbkIsUUFBUSxDQUFDUSxFQUFFLENBQUM7UUFBRUMsSUFBSSxFQUFFSjtNQUFNLENBQUMsQ0FBQztJQUNyQyxDQUFDLENBQUMsT0FBT0ssS0FBVSxFQUFFO01BQ25CVyxPQUFPLENBQUNYLEtBQUssQ0FBRSxrQ0FBaUNYLE9BQU8sQ0FBQ2lCLE1BQU0sQ0FBQ0csTUFBTyxHQUFFLEVBQUVULEtBQUssQ0FBQztNQUNoRixPQUFPVixRQUFRLENBQUNjLE1BQU0sQ0FBQztRQUNyQkMsVUFBVSxFQUFFTCxLQUFLLENBQUNLLFVBQVUsSUFBSSxHQUFHO1FBQ25DTixJQUFJLEVBQUU7VUFDSkksT0FBTyxFQUFHLGtDQUFpQ2QsT0FBTyxDQUFDaUIsTUFBTSxDQUFDRyxNQUFPLEtBQUlULEtBQUssQ0FBQ0csT0FBUTtRQUNyRjtNQUNGLENBQUMsQ0FBQztJQUNKO0VBQ0YsQ0FDRixDQUFDOztFQUVEO0VBQ0FuQixNQUFNLENBQUNDLEdBQUcsQ0FDUjtJQUNFQyxJQUFJLEVBQUUsMEJBQTBCO0lBQ2hDQyxRQUFRLEVBQUU7RUFDWixDQUFDLEVBQ0QsT0FBT0MsT0FBTyxFQUFFQyxPQUFPLEVBQUVDLFFBQVEsS0FBSztJQUNwQyxJQUFJO01BQ0YsTUFBTUMsTUFBTSxHQUFHSCxPQUFPLENBQUNJLElBQUksQ0FBQ0MsVUFBVSxDQUFDRixNQUFNLENBQUNHLGFBQWE7TUFDM0QsTUFBTWtCLE1BQU0sR0FBRyxNQUFNckIsTUFBTSxDQUFDSyxTQUFTLENBQUNQLE9BQU8sQ0FBQztRQUM1Q1EsTUFBTSxFQUFFLEtBQUs7UUFDYlgsSUFBSSxFQUFFO01BQ1IsQ0FBQyxDQUFDO01BQ0YsT0FBT0ksUUFBUSxDQUFDUSxFQUFFLENBQUM7UUFBRUMsSUFBSSxFQUFFYTtNQUFPLENBQUMsQ0FBQztJQUN0QyxDQUFDLENBQUMsT0FBT1osS0FBVSxFQUFFO01BQ25CLE9BQU9WLFFBQVEsQ0FBQ3VCLFdBQVcsQ0FBQztRQUMxQlIsVUFBVSxFQUFFTCxLQUFLLENBQUNLLFVBQVUsSUFBSSxHQUFHO1FBQ25DTixJQUFJLEVBQUU7VUFBRUksT0FBTyxFQUFHLG9DQUFtQ0gsS0FBSyxDQUFDRyxPQUFRO1FBQUU7TUFDdkUsQ0FBQyxDQUFDO0lBQ0o7RUFDRixDQUNGLENBQUM7O0VBRUQ7RUFDQW5CLE1BQU0sQ0FBQ0MsR0FBRyxDQUNSO0lBQ0VDLElBQUksRUFBRSxpQ0FBaUM7SUFDdkNDLFFBQVEsRUFBRTtNQUNSbUIsTUFBTSxFQUFFQyxvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDcEJNLElBQUksRUFBRVAsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDO01BQ3RCLENBQUM7SUFDSDtFQUNGLENBQUMsRUFDRCxPQUFPdEIsT0FBTyxFQUFFQyxPQUFPLEVBQUVDLFFBQVEsS0FBSztJQUNwQyxNQUFNO01BQUV3QjtJQUFLLENBQUMsR0FBR3pCLE9BQU8sQ0FBQ2lCLE1BQU07SUFDL0IsSUFBSTtNQUNGLE1BQU1mLE1BQU0sR0FBR0gsT0FBTyxDQUFDSSxJQUFJLENBQUNDLFVBQVUsQ0FBQ0YsTUFBTSxDQUFDRyxhQUFhO01BQzNELE1BQU1rQixNQUFNLEdBQUcsTUFBTXJCLE1BQU0sQ0FBQ0ssU0FBUyxDQUFDUCxPQUFPLENBQUM7UUFDNUNRLE1BQU0sRUFBRSxLQUFLO1FBQ2JYLElBQUksRUFBRyx3QkFBdUI0QixJQUFLO01BQ3JDLENBQUMsQ0FBQztNQUNGLE9BQU94QixRQUFRLENBQUNRLEVBQUUsQ0FBQztRQUFFQyxJQUFJLEVBQUVhO01BQU8sQ0FBQyxDQUFDO0lBQ3RDLENBQUMsQ0FBQyxPQUFPWixLQUFVLEVBQUU7TUFDbkIsT0FBT1YsUUFBUSxDQUFDYyxNQUFNLENBQUM7UUFDckJDLFVBQVUsRUFBRUwsS0FBSyxDQUFDSyxVQUFVLElBQUksR0FBRztRQUNuQ04sSUFBSSxFQUFFO1VBQUVJLE9BQU8sRUFBRyxrQ0FBaUNILEtBQUssQ0FBQ0csT0FBUTtRQUFFO01BQ3JFLENBQUMsQ0FBQztJQUNKO0VBQ0YsQ0FDRixDQUFDOztFQUVEO0VBQ0FuQixNQUFNLENBQUMrQixHQUFHLENBQ1I7SUFDRTdCLElBQUksRUFBRSwwQkFBMEI7SUFDaENDLFFBQVEsRUFBRTtNQUNSWSxJQUFJLEVBQUVRLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztRQUNsQk0sSUFBSSxFQUFFUCxvQkFBTSxDQUFDRyxNQUFNLENBQUMsQ0FBQztRQUNyQk0sZUFBZSxFQUFFVCxvQkFBTSxDQUFDRyxNQUFNLENBQUMsQ0FBQztRQUNoQ08sZUFBZSxFQUFFVixvQkFBTSxDQUFDQyxNQUFNLENBQUM7VUFDN0JVLEdBQUcsRUFBRVgsb0JBQU0sQ0FBQ1ksS0FBSyxDQUFDWixvQkFBTSxDQUFDYSxNQUFNLENBQUMsQ0FBQyxDQUFDO1VBQ2xDQyxNQUFNLEVBQUVkLG9CQUFNLENBQUNZLEtBQUssQ0FBQ1osb0JBQU0sQ0FBQ2EsTUFBTSxDQUFDLENBQUM7UUFDdEMsQ0FBQztNQUNILENBQUM7SUFDSDtFQUNGLENBQUMsRUFDRCxPQUFPaEMsT0FBTyxFQUFFQyxPQUFPLEVBQUVDLFFBQVEsS0FBSztJQUNwQyxJQUFJO01BQ0YsTUFBTUMsTUFBTSxHQUFHSCxPQUFPLENBQUNJLElBQUksQ0FBQ0MsVUFBVSxDQUFDRixNQUFNLENBQUNHLGFBQWE7TUFDM0QsTUFBTUssSUFBSSxHQUFHVixPQUFPLENBQUNVLElBQUk7TUFFekIsTUFBTWEsTUFBTSxHQUFHLE1BQU1yQixNQUFNLENBQUNLLFNBQVMsQ0FBQ1AsT0FBTyxDQUFDO1FBQzVDUSxNQUFNLEVBQUUsS0FBSztRQUNiWCxJQUFJLEVBQUUsc0JBQXNCO1FBQzVCYTtNQUNGLENBQUMsQ0FBQztNQUVGLE9BQU9ULFFBQVEsQ0FBQ1EsRUFBRSxDQUFDO1FBQUVDLElBQUksRUFBRWE7TUFBTyxDQUFDLENBQUM7SUFDdEMsQ0FBQyxDQUFDLE9BQU9VLENBQU0sRUFBRTtNQUNmWCxPQUFPLENBQUNYLEtBQUssQ0FBQyxrQ0FBa0MsRUFBRXNCLENBQUMsQ0FBQztNQUNwRCxPQUFPaEMsUUFBUSxDQUFDaUMsYUFBYSxDQUFDO1FBQzVCeEIsSUFBSSxFQUFFO1VBQUVJLE9BQU8sRUFBRyxvQ0FBbUNtQixDQUFDLENBQUNuQixPQUFRO1FBQUU7TUFDbkUsQ0FBQyxDQUFDO0lBQ0o7RUFDRixDQUNGLENBQUM7O0VBRUQ7RUFDQW5CLE1BQU0sQ0FBQytCLEdBQUcsQ0FDUjtJQUNFN0IsSUFBSSxFQUFFLGlDQUFpQztJQUN2Q0MsUUFBUSxFQUFFO01BQ1JtQixNQUFNLEVBQUVDLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztRQUNwQk0sSUFBSSxFQUFFUCxvQkFBTSxDQUFDRyxNQUFNLENBQUM7TUFDdEIsQ0FBQyxDQUFDO01BQ0ZYLElBQUksRUFBRVEsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ2xCUSxlQUFlLEVBQUVULG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQ2hDTyxlQUFlLEVBQUVWLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztVQUM3QlUsR0FBRyxFQUFFWCxvQkFBTSxDQUFDWSxLQUFLLENBQUNaLG9CQUFNLENBQUNhLE1BQU0sQ0FBQyxDQUFDLENBQUM7VUFDbENDLE1BQU0sRUFBRWQsb0JBQU0sQ0FBQ1ksS0FBSyxDQUFDWixvQkFBTSxDQUFDYSxNQUFNLENBQUMsQ0FBQztRQUN0QyxDQUFDO01BQ0gsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNELE9BQU9oQyxPQUFPLEVBQUVDLE9BQU8sRUFBRUMsUUFBUSxLQUFLO0lBQ3BDLElBQUk7TUFDRixNQUFNQyxNQUFNLEdBQUdILE9BQU8sQ0FBQ0ksSUFBSSxDQUFDQyxVQUFVLENBQUNGLE1BQU0sQ0FBQ0csYUFBYTtNQUMzRCxNQUFNO1FBQUVvQjtNQUFLLENBQUMsR0FBR3pCLE9BQU8sQ0FBQ2lCLE1BQU07TUFDL0IsTUFBTVAsSUFBSSxHQUFHVixPQUFPLENBQUNVLElBQUk7TUFFekIsTUFBTWEsTUFBTSxHQUFHLE1BQU1yQixNQUFNLENBQUNLLFNBQVMsQ0FBQ1AsT0FBTyxDQUFDO1FBQzVDUSxNQUFNLEVBQUUsS0FBSztRQUNiWCxJQUFJLEVBQUcsd0JBQXVCNEIsSUFBSyxFQUFDO1FBQ3BDZjtNQUNGLENBQUMsQ0FBQztNQUVGLE9BQU9ULFFBQVEsQ0FBQ1EsRUFBRSxDQUFDO1FBQUVDLElBQUksRUFBRWE7TUFBTyxDQUFDLENBQUM7SUFDdEMsQ0FBQyxDQUFDLE9BQU9VLENBQU0sRUFBRTtNQUNmWCxPQUFPLENBQUNYLEtBQUssQ0FBQyxrQ0FBa0MsRUFBRXNCLENBQUMsQ0FBQztNQUNwRCxPQUFPaEMsUUFBUSxDQUFDaUMsYUFBYSxDQUFDO1FBQzVCeEIsSUFBSSxFQUFFO1VBQUVJLE9BQU8sRUFBRyxvQ0FBbUNtQixDQUFDLENBQUNuQixPQUFRO1FBQUU7TUFDbkUsQ0FBQyxDQUFDO0lBQ0o7RUFDRixDQUNGLENBQUM7O0VBRUQ7RUFDQW5CLE1BQU0sQ0FBQ3dDLE1BQU0sQ0FDWDtJQUNFdEMsSUFBSSxFQUFFLGlDQUFpQztJQUN2Q0MsUUFBUSxFQUFFO01BQ1JtQixNQUFNLEVBQUVDLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztRQUNwQk0sSUFBSSxFQUFFUCxvQkFBTSxDQUFDRyxNQUFNLENBQUM7TUFDdEIsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNELE9BQU90QixPQUFPLEVBQUVDLE9BQU8sRUFBRUMsUUFBUSxLQUFLO0lBQ3BDLElBQUk7TUFDRixNQUFNQyxNQUFNLEdBQUdILE9BQU8sQ0FBQ0ksSUFBSSxDQUFDQyxVQUFVLENBQUNGLE1BQU0sQ0FBQ0csYUFBYTtNQUMzRCxNQUFNO1FBQUVvQjtNQUFLLENBQUMsR0FBR3pCLE9BQU8sQ0FBQ2lCLE1BQU07TUFFL0IsTUFBTU0sTUFBTSxHQUFHLE1BQU1yQixNQUFNLENBQUNLLFNBQVMsQ0FBQ1AsT0FBTyxDQUFDO1FBQzVDUSxNQUFNLEVBQUUsUUFBUTtRQUNoQlgsSUFBSSxFQUFHLHdCQUF1QjRCLElBQUs7TUFDckMsQ0FBQyxDQUFDO01BRUYsT0FBT3hCLFFBQVEsQ0FBQ1EsRUFBRSxDQUFDO1FBQUVDLElBQUksRUFBRWE7TUFBTyxDQUFDLENBQUM7SUFDdEMsQ0FBQyxDQUFDLE9BQU9VLENBQU0sRUFBRTtNQUNmWCxPQUFPLENBQUNYLEtBQUssQ0FBRSxvQ0FBbUNYLE9BQU8sQ0FBQ2lCLE1BQU0sQ0FBQ1EsSUFBSyxJQUFHLEVBQUVRLENBQUMsQ0FBQztNQUM3RSxPQUFPaEMsUUFBUSxDQUFDaUMsYUFBYSxDQUFDO1FBQzVCeEIsSUFBSSxFQUFFO1VBQ0pJLE9BQU8sRUFBRyxvQ0FBbUNkLE9BQU8sQ0FBQ2lCLE1BQU0sQ0FBQ1EsSUFBSyxNQUFLUSxDQUFDLENBQUNuQixPQUFRO1FBQ2xGO01BQ0YsQ0FBQyxDQUFDO0lBQ0o7RUFDRixDQUNGLENBQUM7O0VBRUQ7RUFDQW5CLE1BQU0sQ0FBQ0MsR0FBRyxDQUNSO0lBQ0VDLElBQUksRUFBRSxtQ0FBbUM7SUFDekNDLFFBQVEsRUFBRTtNQUNSbUIsTUFBTSxFQUFFQyxvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDcEJpQixlQUFlLEVBQUVsQixvQkFBTSxDQUFDRyxNQUFNLENBQUM7TUFDakMsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNELE9BQU90QixPQUFPLEVBQUVDLE9BQU8sRUFBRUMsUUFBUSxLQUFLO0lBQ3BDLElBQUk7TUFDRixNQUFNQyxNQUFNLEdBQUdILE9BQU8sQ0FBQ0ksSUFBSSxDQUFDQyxVQUFVLENBQUNGLE1BQU0sQ0FBQ0csYUFBYTtNQUMzRCxNQUFNO1FBQUUrQjtNQUFnQixDQUFDLEdBQUdwQyxPQUFPLENBQUNpQixNQUFNO01BRTFDLE1BQU1NLE1BQU0sR0FBRyxNQUFNckIsTUFBTSxDQUFDSyxTQUFTLENBQUNQLE9BQU8sQ0FBQztRQUM1Q1EsTUFBTSxFQUFFLEtBQUs7UUFDYlgsSUFBSSxFQUFHLGVBQWN1QyxlQUFnQjtNQUN2QyxDQUFDLENBQUM7TUFFRixPQUFPbkMsUUFBUSxDQUFDUSxFQUFFLENBQUM7UUFBRUMsSUFBSSxFQUFFYTtNQUFPLENBQUMsQ0FBQztJQUN0QyxDQUFDLENBQUMsT0FBT1osS0FBVSxFQUFFO01BQ25CVyxPQUFPLENBQUNYLEtBQUssQ0FDVix1Q0FBc0NYLE9BQU8sQ0FBQ2lCLE1BQU0sQ0FBQ21CLGVBQWdCLEdBQUUsRUFDeEV6QixLQUNGLENBQUM7TUFDRCxPQUFPVixRQUFRLENBQUNjLE1BQU0sQ0FBQztRQUNyQkMsVUFBVSxFQUFFTCxLQUFLLENBQUNLLFVBQVUsSUFBSSxHQUFHO1FBQ25DTixJQUFJLEVBQUU7VUFDSkksT0FBTyxFQUFHLHVDQUFzQ2QsT0FBTyxDQUFDaUIsTUFBTSxDQUFDbUIsZUFBZ0IsS0FBSXpCLEtBQUssQ0FBQ0csT0FBUTtRQUNuRztNQUNGLENBQUMsQ0FBQztJQUNKO0VBQ0YsQ0FDRixDQUFDOztFQUVEO0VBQ0FuQixNQUFNLENBQUNDLEdBQUcsQ0FDUjtJQUNFQyxJQUFJLEVBQUUsd0JBQXdCO0lBQzlCQyxRQUFRLEVBQUU7RUFDWixDQUFDLEVBQ0QsT0FBT0MsT0FBTyxFQUFFQyxPQUFPLEVBQUVDLFFBQVEsS0FBSztJQUNwQyxNQUFNb0MsUUFBUSxHQUFHdEMsT0FBTyxDQUFDSSxJQUFJLENBQUNDLFVBQVUsQ0FBQ0YsTUFBTSxDQUFDRyxhQUFhO0lBQzdELElBQUk7TUFDRixNQUFNa0IsTUFBTSxHQUFHLE1BQU1jLFFBQVEsQ0FBQ0MsS0FBSyxDQUFDQyxJQUFJLENBQUMsQ0FBQztNQUMxQyxPQUFPdEMsUUFBUSxDQUFDUSxFQUFFLENBQUM7UUFBRUMsSUFBSSxFQUFFYTtNQUFPLENBQUMsQ0FBQztJQUN0QyxDQUFDLENBQUMsT0FBT1UsQ0FBQyxFQUFFO01BQ1YsT0FBT2hDLFFBQVEsQ0FBQ3VCLFdBQVcsQ0FBQztRQUMxQlIsVUFBVSxFQUFFaUIsQ0FBQyxDQUFDakIsVUFBVSxJQUFJLEdBQUc7UUFDL0JOLElBQUksRUFBRXVCLENBQUMsQ0FBQ25CO01BQ1YsQ0FBQyxDQUFDO0lBQ0o7RUFDRixDQUNGLENBQUM7O0VBRUQ7RUFDQW5CLE1BQU0sQ0FBQytCLEdBQUcsQ0FDUjtJQUNFN0IsSUFBSSxFQUFFLDRCQUE0QjtJQUNsQ0MsUUFBUSxFQUFFO01BQ1JZLElBQUksRUFBRVEsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ2xCcUIsV0FBVyxFQUFFdEIsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUM7UUFDNUJvQixhQUFhLEVBQUV2QixvQkFBTSxDQUFDd0IsT0FBTyxDQUFDeEIsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUMsQ0FBQztRQUM5Q3NCLGNBQWMsRUFBRXpCLG9CQUFNLENBQUNHLE1BQU0sQ0FBQztNQUNoQyxDQUFDO0lBQ0g7RUFDRixDQUFDLEVBQ0QsT0FBT3RCLE9BQU8sRUFBRUMsT0FBTyxFQUFFQyxRQUFRLEtBQUs7SUFDcEMsSUFBSTtNQUNGLE1BQU1DLE1BQU0sR0FBR0gsT0FBTyxDQUFDSSxJQUFJLENBQUNDLFVBQVUsQ0FBQ0YsTUFBTSxDQUFDRyxhQUFhO01BQzNELE1BQU1tQyxXQUFXLEdBQUd4QyxPQUFPLENBQUNVLElBQUksQ0FBQzhCLFdBQVc7TUFDNUMsTUFBTUksWUFBWSxHQUFHNUMsT0FBTyxDQUFDVSxJQUFJLENBQUMrQixhQUFhO01BQy9DLE1BQU1JLGFBQWEsR0FBRzdDLE9BQU8sQ0FBQ1UsSUFBSSxDQUFDaUMsY0FBYztNQUVqRCxNQUFNcEIsTUFBTSxHQUFHLE1BQU1yQixNQUFNLENBQUNLLFNBQVMsQ0FBQ1AsT0FBTyxDQUFDO1FBQzVDUSxNQUFNLEVBQUUsS0FBSztRQUNiWCxJQUFJLEVBQUUsd0JBQXdCO1FBQzlCYSxJQUFJLEVBQUU7VUFDSjhCLFdBQVc7VUFDWEMsYUFBYSxFQUFFRyxZQUFZO1VBQzNCRCxjQUFjLEVBQUVFO1FBQ2xCO01BQ0YsQ0FBQyxDQUFDO01BRUYsT0FBTzVDLFFBQVEsQ0FBQ1EsRUFBRSxDQUFDO1FBQUVDLElBQUksRUFBRWE7TUFBTyxDQUFDLENBQUM7SUFDdEMsQ0FBQyxDQUFDLE9BQU9aLEtBQVUsRUFBRTtNQUNuQlcsT0FBTyxDQUFDWCxLQUFLLENBQUUsOEJBQTZCLEVBQUVBLEtBQUssQ0FBQztNQUNwRCxPQUFPVixRQUFRLENBQUNjLE1BQU0sQ0FBQztRQUNyQkMsVUFBVSxFQUFFTCxLQUFLLENBQUNLLFVBQVUsSUFBSSxHQUFHO1FBQ25DTixJQUFJLEVBQUU7VUFBRUksT0FBTyxFQUFHLGdDQUErQkgsS0FBSyxDQUFDRyxPQUFRO1FBQUU7TUFDbkUsQ0FBQyxDQUFDO0lBQ0o7RUFDRixDQUNGLENBQUM7O0VBRUQ7RUFDQW5CLE1BQU0sQ0FBQ0MsR0FBRyxDQUNSO0lBQ0VDLElBQUksRUFBRSw0QkFBNEI7SUFDbENDLFFBQVEsRUFBRTtFQUNaLENBQUMsRUFDRCxPQUFPQyxPQUFPLEVBQUVDLE9BQU8sRUFBRUMsUUFBUSxLQUFLO0lBQ3BDLElBQUk7TUFDRixNQUFNQyxNQUFNLEdBQUdILE9BQU8sQ0FBQ0ksSUFBSSxDQUFDQyxVQUFVLENBQUNGLE1BQU0sQ0FBQ0csYUFBYTtNQUUzRCxNQUFNa0IsTUFBTSxHQUFHLE1BQU1yQixNQUFNLENBQUNLLFNBQVMsQ0FBQ1AsT0FBTyxDQUFDO1FBQzVDUSxNQUFNLEVBQUUsS0FBSztRQUNiWCxJQUFJLEVBQUU7TUFDUixDQUFDLENBQUM7TUFFRixPQUFPSSxRQUFRLENBQUNRLEVBQUUsQ0FBQztRQUFFQyxJQUFJLEVBQUVhO01BQU8sQ0FBQyxDQUFDO0lBQ3RDLENBQUMsQ0FBQyxPQUFPVSxDQUFNLEVBQUU7TUFDZlgsT0FBTyxDQUFDWCxLQUFLLENBQUMsOEJBQThCLEVBQUVzQixDQUFDLENBQUM7TUFDaEQsT0FBT2hDLFFBQVEsQ0FBQ2lDLGFBQWEsQ0FBQztRQUM1QnhCLElBQUksRUFBRTtVQUFFSSxPQUFPLEVBQUcsZ0NBQStCbUIsQ0FBQyxDQUFDbkIsT0FBUTtRQUFFO01BQy9ELENBQUMsQ0FBQztJQUNKO0VBQ0YsQ0FDRixDQUFDOztFQUVEO0VBQ0FuQixNQUFNLENBQUN3QyxNQUFNLENBQ1g7SUFDRXRDLElBQUksRUFBRSxxQ0FBcUM7SUFDM0NDLFFBQVEsRUFBRTtNQUNSbUIsTUFBTSxFQUFFQyxvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDcEIyQixNQUFNLEVBQUU1QixvQkFBTSxDQUFDRyxNQUFNLENBQUM7TUFDeEIsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNELE9BQU90QixPQUFPLEVBQUVDLE9BQU8sRUFBRUMsUUFBUSxLQUFLO0lBQ3BDLE1BQU07TUFBRTZDO0lBQU8sQ0FBQyxHQUFHOUMsT0FBTyxDQUFDaUIsTUFBTTtJQUNqQyxJQUFJO01BQ0YsTUFBTWYsTUFBTSxHQUFHSCxPQUFPLENBQUNJLElBQUksQ0FBQ0MsVUFBVSxDQUFDRixNQUFNLENBQUNHLGFBQWE7TUFFM0QsTUFBTWtCLE1BQU0sR0FBRyxNQUFNckIsTUFBTSxDQUFDSyxTQUFTLENBQUNQLE9BQU8sQ0FBQztRQUM1Q1EsTUFBTSxFQUFFLFFBQVE7UUFDaEJYLElBQUksRUFBRywwQkFBeUJpRCxNQUFPO01BQ3pDLENBQUMsQ0FBQztNQUVGLE9BQU83QyxRQUFRLENBQUNRLEVBQUUsQ0FBQztRQUFFQyxJQUFJLEVBQUVhO01BQU8sQ0FBQyxDQUFDO0lBQ3RDLENBQUMsQ0FBQyxPQUFPVSxDQUFNLEVBQUU7TUFDZlgsT0FBTyxDQUFDWCxLQUFLLENBQUUsK0JBQThCbUMsTUFBTyxHQUFFLEVBQUViLENBQUMsQ0FBQztNQUMxRCxPQUFPaEMsUUFBUSxDQUFDaUMsYUFBYSxDQUFDO1FBQzVCeEIsSUFBSSxFQUFFO1VBQUVJLE9BQU8sRUFBRywrQkFBOEJnQyxNQUFPLEtBQUliLENBQUMsQ0FBQ25CLE9BQVE7UUFBRTtNQUN6RSxDQUFDLENBQUM7SUFDSjtFQUNGLENBQ0YsQ0FBQzs7RUFFRDtFQUNBbkIsTUFBTSxDQUFDK0IsR0FBRyxDQUNSO0lBQ0U3QixJQUFJLEVBQUUscUNBQXFDO0lBQzNDQyxRQUFRLEVBQUU7TUFDUm1CLE1BQU0sRUFBRUMsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ3BCMkIsTUFBTSxFQUFFNUIsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDO01BQ3hCLENBQUMsQ0FBQztNQUNGWCxJQUFJLEVBQUVRLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztRQUNsQnFCLFdBQVcsRUFBRXRCLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQzVCb0IsYUFBYSxFQUFFdkIsb0JBQU0sQ0FBQ3dCLE9BQU8sQ0FBQ3hCLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDOUNzQixjQUFjLEVBQUV6QixvQkFBTSxDQUFDRyxNQUFNLENBQUM7TUFDaEMsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNELE9BQU90QixPQUFPLEVBQUVDLE9BQU8sRUFBRUMsUUFBUSxLQUFLO0lBQ3BDLE1BQU07TUFBRTZDO0lBQU8sQ0FBQyxHQUFHOUMsT0FBTyxDQUFDaUIsTUFBTTtJQUNqQyxNQUFNUCxJQUFJLEdBQUdWLE9BQU8sQ0FBQ1UsSUFBSTtJQUV6QixJQUFJO01BQ0YsTUFBTWEsTUFBTSxHQUFHLE1BQU14QixPQUFPLENBQUNJLElBQUksQ0FBQ0MsVUFBVSxDQUFDRixNQUFNLENBQUNHLGFBQWEsQ0FBQ0UsU0FBUyxDQUFDUCxPQUFPLENBQUM7UUFDbEZRLE1BQU0sRUFBRSxLQUFLO1FBQ2JYLElBQUksRUFBRywwQkFBeUJpRCxNQUFPLEVBQUM7UUFDeENwQztNQUNGLENBQUMsQ0FBQztNQUVGLE9BQU9ULFFBQVEsQ0FBQ1EsRUFBRSxDQUFDO1FBQUVDLElBQUksRUFBRWE7TUFBTyxDQUFDLENBQUM7SUFDdEMsQ0FBQyxDQUFDLE9BQU9aLEtBQUssRUFBRTtNQUNkVyxPQUFPLENBQUNYLEtBQUssQ0FBQyxzQkFBc0IsRUFBRUEsS0FBSyxDQUFDO01BQzVDLE9BQU9WLFFBQVEsQ0FBQ3VCLFdBQVcsQ0FBQztRQUMxQmQsSUFBSSxFQUFFQyxLQUFLLENBQUNHLE9BQU8sSUFBSUgsS0FBSztRQUM1QkssVUFBVSxFQUFFTCxLQUFLLENBQUNLLFVBQVUsSUFBSTtNQUNsQyxDQUFDLENBQUM7SUFDSjtFQUNGLENBQ0YsQ0FBQzs7RUFFRDtFQUNBckIsTUFBTSxDQUFDQyxHQUFHLENBQ1I7SUFDRUMsSUFBSSxFQUFFLHNCQUFzQjtJQUM1QkMsUUFBUSxFQUFFO0VBQ1osQ0FBQyxFQUNELE9BQU9DLE9BQU8sRUFBRUMsT0FBTyxFQUFFQyxRQUFRLEtBQUs7SUFBQSxJQUFBOEMsSUFBQSxFQUFBQyxLQUFBLEVBQUFDLHFCQUFBLEVBQUFDLGVBQUEsRUFBQUMsZ0JBQUEsRUFBQUMsY0FBQSxFQUFBQyxLQUFBLEVBQUFDLEtBQUEsRUFBQUMsc0JBQUEsRUFBQUMsZ0JBQUEsRUFBQUMsaUJBQUEsRUFBQUMsZUFBQTtJQUNwQyxNQUFNckIsUUFBUSxHQUFHdEMsT0FBTyxDQUFDSSxJQUFJLENBQUNDLFVBQVUsQ0FBQ0YsTUFBTSxDQUFDeUQsY0FBYztJQUM5RCxNQUFNO01BQUVqRDtJQUFLLENBQUMsR0FBRyxNQUFNMkIsUUFBUSxDQUFDdUIsT0FBTyxDQUFDQyxXQUFXLENBQUM7TUFBRUMsZ0JBQWdCLEVBQUU7SUFBSyxDQUFDLENBQUM7SUFFL0UsTUFBTUMsWUFBWSxJQUFBaEIsSUFBQSxJQUFBQyxLQUFBLElBQUFDLHFCQUFBLElBQUFDLGVBQUEsR0FDaEJ4QyxJQUFJLENBQUNzRCxTQUFTLGNBQUFkLGVBQUEsZ0JBQUFBLGVBQUEsR0FBZEEsZUFBQSxDQUFnQmUsR0FBRyxjQUFBZixlQUFBLGdCQUFBQSxlQUFBLEdBQW5CQSxlQUFBLENBQXFCUCxjQUFjLGNBQUFPLGVBQUEsZ0JBQUFBLGVBQUEsR0FBbkNBLGVBQUEsQ0FBcUNnQixJQUFJLGNBQUFoQixlQUFBLHVCQUF6Q0EsZUFBQSxDQUEyQ2lCLHVCQUF1QixjQUFBbEIscUJBQUEsY0FBQUEscUJBQUEsSUFBQUUsZ0JBQUEsR0FDbEV6QyxJQUFJLENBQUMwRCxVQUFVLGNBQUFqQixnQkFBQSxnQkFBQUEsZ0JBQUEsR0FBZkEsZ0JBQUEsQ0FBaUJjLEdBQUcsY0FBQWQsZ0JBQUEsZ0JBQUFBLGdCQUFBLEdBQXBCQSxnQkFBQSxDQUFzQlIsY0FBYyxjQUFBUSxnQkFBQSxnQkFBQUEsZ0JBQUEsR0FBcENBLGdCQUFBLENBQXNDZSxJQUFJLGNBQUFmLGdCQUFBLHVCQUExQ0EsZ0JBQUEsQ0FBNENnQix1QkFBdUIsY0FBQW5CLEtBQUEsY0FBQUEsS0FBQSxJQUFBSSxjQUFBLEdBQ25FMUMsSUFBSSxDQUFDMkQsUUFBUSxjQUFBakIsY0FBQSxnQkFBQUEsY0FBQSxHQUFiQSxjQUFBLENBQWVhLEdBQUcsY0FBQWIsY0FBQSxnQkFBQUEsY0FBQSxHQUFsQkEsY0FBQSxDQUFvQlQsY0FBYyxjQUFBUyxjQUFBLGdCQUFBQSxjQUFBLEdBQWxDQSxjQUFBLENBQW9DYyxJQUFJLGNBQUFkLGNBQUEsdUJBQXhDQSxjQUFBLENBQTBDZSx1QkFBdUIsY0FBQXBCLElBQUEsY0FBQUEsSUFBQSxHQUNqRSxHQUFHO0lBRUwsTUFBTXVCLGVBQWUsSUFBQWpCLEtBQUEsSUFBQUMsS0FBQSxJQUFBQyxzQkFBQSxJQUFBQyxnQkFBQSxHQUNuQjlDLElBQUksQ0FBQ3NELFNBQVMsY0FBQVIsZ0JBQUEsZ0JBQUFBLGdCQUFBLEdBQWRBLGdCQUFBLENBQWdCUyxHQUFHLGNBQUFULGdCQUFBLGdCQUFBQSxnQkFBQSxHQUFuQkEsZ0JBQUEsQ0FBcUJiLGNBQWMsY0FBQWEsZ0JBQUEsZ0JBQUFBLGdCQUFBLEdBQW5DQSxnQkFBQSxDQUFxQ1UsSUFBSSxjQUFBVixnQkFBQSx1QkFBekNBLGdCQUFBLENBQTJDZSwwQkFBMEIsY0FBQWhCLHNCQUFBLGNBQUFBLHNCQUFBLElBQUFFLGlCQUFBLEdBQ3JFL0MsSUFBSSxDQUFDMEQsVUFBVSxjQUFBWCxpQkFBQSxnQkFBQUEsaUJBQUEsR0FBZkEsaUJBQUEsQ0FBaUJRLEdBQUcsY0FBQVIsaUJBQUEsZ0JBQUFBLGlCQUFBLEdBQXBCQSxpQkFBQSxDQUFzQmQsY0FBYyxjQUFBYyxpQkFBQSxnQkFBQUEsaUJBQUEsR0FBcENBLGlCQUFBLENBQXNDUyxJQUFJLGNBQUFULGlCQUFBLHVCQUExQ0EsaUJBQUEsQ0FBNENjLDBCQUEwQixjQUFBakIsS0FBQSxjQUFBQSxLQUFBLElBQUFJLGVBQUEsR0FDdEVoRCxJQUFJLENBQUMyRCxRQUFRLGNBQUFYLGVBQUEsZ0JBQUFBLGVBQUEsR0FBYkEsZUFBQSxDQUFlTyxHQUFHLGNBQUFQLGVBQUEsZ0JBQUFBLGVBQUEsR0FBbEJBLGVBQUEsQ0FBb0JmLGNBQWMsY0FBQWUsZUFBQSxnQkFBQUEsZUFBQSxHQUFsQ0EsZUFBQSxDQUFvQ1EsSUFBSSxjQUFBUixlQUFBLHVCQUF4Q0EsZUFBQSxDQUEwQ2EsMEJBQTBCLGNBQUFsQixLQUFBLGNBQUFBLEtBQUEsR0FDcEUsR0FBRztJQUVMLE9BQU9wRCxRQUFRLENBQUNRLEVBQUUsQ0FBQztNQUNqQkMsSUFBSSxFQUFFO1FBQ0o4RCxxQkFBcUIsRUFBRUMsVUFBVSxDQUFDVixZQUFZLENBQUM7UUFDL0NXLHdCQUF3QixFQUFFRCxVQUFVLENBQUNILGVBQWU7TUFDdEQ7SUFDRixDQUFDLENBQUM7RUFDSixDQUNGLENBQUM7QUFDSCJ9