<?php

namespace {

	/**
	 * <p><b>DOMAttr</b> represents an attribute in the <code>DOMElement</code> object.</p>
	 * @link https://php.net/manual/en/class.domattr.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMAttr extends \DOMNode {

		/**
		 * @var string <p>The name of the attribute.</p>
		 * @link https://php.net/manual/en/class.domattr.php#domattr.props.name
		 */
		public string $name;

		/**
		 * @var bool <p>Not implemented yet, always is <b><code>true</code></b>.</p>
		 * @link https://php.net/manual/en/class.domattr.php#domattr.props.specified
		 */
		public bool $specified;

		/**
		 * @var string <p>The value of the attribute.</p> <p><b>Note</b>:</p><p>Note, XML entities are expanded upon setting a value. Thus the <code>&amp;</code> character has a special meaning. Setting value to itself will fail when value contains an <code>&amp;</code>. To avoid entity expansion, use <code>DOMElement::setAttribute()</code> instead.</p>
		 * @link https://php.net/manual/en/class.domattr.php#domattr.props.value
		 */
		public string $value;

		/**
		 * @var ?DOMElement <p>The element which contains the attribute or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domattr.php#domattr.props.ownerelement
		 */
		public ?\DOMElement $ownerElement;

		/**
		 * @var mixed <p>Not implemented yet, always is <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domattr.php#domattr.props.schematypeinfo
		 */
		public mixed $schemaTypeInfo;

		/**
		 * @var string <p>Returns the most accurate name for the current node type</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var ?string <p>The value of this node, depending on its type. Contrary to the W3C specification, the node value of <code>DOMElement</code> nodes is equal to DOMNode::textContent instead of <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var ?DOMNode <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentnode
		 */
		public ?\DOMNode $parentNode;

		/**
		 * @var ?DOMElement <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentelement
		 */
		public ?\DOMElement $parentElement;

		/**
		 * @var DOMNodeList <p>A <code>DOMNodeList</code> that contains all children of this node. If there are no children, this is an empty <code>DOMNodeList</code>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.childnodes
		 */
		public \DOMNodeList $childNodes;

		/**
		 * @var ?DOMNode <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.firstchild
		 */
		public ?\DOMNode $firstChild;

		/**
		 * @var ?DOMNode <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.lastchild
		 */
		public ?\DOMNode $lastChild;

		/**
		 * @var ?DOMNode <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.previoussibling
		 */
		public ?\DOMNode $previousSibling;

		/**
		 * @var ?DOMNode <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nextsibling
		 */
		public ?\DOMNode $nextSibling;

		/**
		 * @var ?DOMNamedNodeMap <p>A <code>DOMNamedNodeMap</code> containing the attributes of this node (if it is a <code>DOMElement</code>) or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.attributes
		 */
		public ?\DOMNamedNodeMap $attributes;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?DOMDocument <p>The <code>DOMDocument</code> object associated with this node, or <b><code>null</code></b> if this node is a <code>DOMDocument</code></p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.ownerdocument
		 */
		public ?\DOMDocument $ownerDocument;

		/**
		 * @var ?string <p>The namespace URI of this node, or <b><code>null</code></b> if it is unspecified.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var string <p>The namespace prefix of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.prefix
		 */
		public string $prefix;

		/**
		 * @var ?string <p>Returns the local part of the qualified name of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.localname
		 */
		public ?string $localName;

		/**
		 * @var ?string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.baseuri
		 */
		public ?string $baseURI;

		/**
		 * @var string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.textcontent
		 */
		public string $textContent;

		/**
		 * Canonicalize nodes to a string
		 * <p>Canonicalize nodes to a string</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return string|false <p>Returns canonicalized nodes as a string or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14n.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14N(bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): string|false {}

		/**
		 * Canonicalize nodes to a file
		 * <p>Canonicalize nodes to a file.</p>
		 * @param string $uri <p>Path to write the output to.</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return int|false <p>Number of bytes written or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14nfile.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14NFile(string $uri, bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): int|false {}

		/**
		 * Creates a new DOMAttr object
		 * <p>Creates a new DOMAttr object. This object is read only. It may be appended to a document, but additional nodes may not be appended to this node until the node is associated with a document. To create a writable node, use DOMDocument::createAttribute.</p>
		 * @param string $name <p>The tag name of the attribute.</p>
		 * @param string $value <p>The value of the attribute.</p>
		 * @return self
		 * @link https://php.net/manual/en/domattr.construct.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __construct(string $name, string $value = "") {}

		/**
		 * Forbids serialization unless serialization methods are implemented in a subclass
		 * <p>Forbids serialization unless serialization methods are implemented in a subclass.</p>
		 * @return array <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.sleep.php
		 * @since PHP 8 >= 8.1.25
		 */
		public function __sleep(): array {}

		/**
		 * Forbids unserialization unless unserialization methods are implemented in a subclass
		 * <p>Forbids unserialization unless unserialization methods are implemented in a subclass.</p>
		 * @return void <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.wakeup.php
		 * @since PHP 8 PHP 8 >= 8.1.25
		 */
		public function __wakeup(): void {}

		/**
		 * Adds new child at the end of the children
		 * <p>This function appends a child to an existing list of children or creates a new list of children. The child can be created with e.g. <code>DOMDocument::createElement()</code>, <code>DOMDocument::createTextNode()</code> etc. or simply by using any other node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The appended child.</p>
		 * @return DOMNode|false <p>The node added or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.appendchild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendChild(\DOMNode $node): \DOMNode|false {}

		/**
		 * Clones a node
		 * <p>Creates a copy of the node.</p>
		 * @param bool $deep <p>Indicates whether to copy all descendant nodes. This parameter is defaulted to <b><code>false</code></b>.</p>
		 * @return DOMNode|false <p>The cloned node.</p>
		 * @link https://php.net/manual/en/domnode.clonenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function cloneNode(bool $deep = false): \DOMNode|false {}

		/**
		 * Compares the position of two nodes
		 * <p>Compares the position of the other node relative to this node.</p>
		 * @param \DOMNode $other <p>The node for which the position should be compared for, relative to this node.</p>
		 * @return int <p>A bitmask of the <b><code>DOMNode::DOCUMENT_POSITION_&#42;</code></b> constants.</p>
		 * @link https://php.net/manual/en/domnode.comparedocumentposition.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function compareDocumentPosition(\DOMNode $other): int {}

		/**
		 * Checks if node contains other node
		 * <p>Checks if node contains <code>other</code> node.</p>
		 * @param \DOMNode|\DOMNameSpaceNode|null $other <p>Node to be checked.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if node contains <code>other</code> node, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.contains.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function contains(\DOMNode|\DOMNameSpaceNode|null $other): bool {}

		/**
		 * Get line number for a node
		 * <p>Gets line number for where the node was defined at parse time.</p>
		 * @return int <p>Returns the line number where the node was defined at parse time. If the node was created manually, the return value will be <code>0</code>.</p>
		 * @link https://php.net/manual/en/domnode.getlineno.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		public function getLineNo(): int {}

		/**
		 * Get an XPath for a node
		 * <p>Gets an XPath location path for the node.</p>
		 * @return ?string <p>Returns a <code>string</code> containing the XPath, or <b><code>null</code></b> in case of an error.</p>
		 * @link https://php.net/manual/en/domnode.getnodepath.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function getNodePath(): ?string {}

		/**
		 * Get root node
		 * <p>Get root node.</p>
		 * @param ?array $options <p>This parameter has no effect yet.</p>
		 * @return DOMNode <p>Returns the root node.</p>
		 * @link https://php.net/manual/en/domnode.getrootnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function getRootNode(?array $options = null): \DOMNode {}

		/**
		 * Checks if node has attributes
		 * <p>This method checks if the node has attributes. The tested node has to be an <b><code>XML_ELEMENT_NODE</code></b>.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.hasattributes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasAttributes(): bool {}

		/**
		 * Checks if node has children
		 * <p>This function checks if the node has children.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.haschildnodes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasChildNodes(): bool {}

		/**
		 * Adds a new child before a reference node
		 * <p>This function inserts a new node right before the reference node. If you plan to do further modifications on the appended child you must use the returned node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The new node.</p>
		 * @param ?\DOMNode $child <p>The reference node. If not supplied, <code>node</code> is appended to the children.</p>
		 * @return DOMNode|false <p>The inserted node or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.insertbefore.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertBefore(\DOMNode $node, ?\DOMNode $child = null): \DOMNode|false {}

		/**
		 * Checks if the specified namespaceURI is the default namespace or not
		 * <p>Tells whether <code>namespace</code> is the default namespace.</p>
		 * @param string $namespace <p>The namespace URI to look for.</p>
		 * @return bool <p>Return <b><code>true</code></b> if <code>namespace</code> is the default namespace, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isdefaultnamespace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isDefaultNamespace(string $namespace): bool {}

		/**
		 * Checks that both nodes are equal
		 * <p>Checks that both nodes are equal.</p>
		 * @param ?\DOMNode $otherNode <p>The node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if both nodes are equal, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isequalnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function isEqualNode(?\DOMNode $otherNode): bool {}

		/**
		 * Checks if attribute is a defined ID
		 * <p>This function checks if the attribute is a defined ID.</p><p>According to the DOM standard this requires a DTD which defines the attribute ID to be of type ID. You need to validate your document with DOMDocument::validate or <code>DOMDocument::$validateOnParse</code> before using this function.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if this attribute is a defined ID, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domattr.isid.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isId(): bool {}

		/**
		 * Indicates if two nodes are the same node
		 * <p>This function indicates if two nodes are the same node. The comparison is <i>not</i> based on content</p>
		 * @param \DOMNode $otherNode <p>The compared node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issamenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSameNode(\DOMNode $otherNode): bool {}

		/**
		 * Checks if feature is supported for specified version
		 * <p>Checks if the asked <code>feature</code> is supported for the specified <code>version</code>.</p>
		 * @param string $feature <p>The feature to test. See the example of <code>DOMImplementation::hasFeature()</code> for a list of features.</p>
		 * @param string $version <p>The version number of the <code>feature</code> to test.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issupported.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSupported(string $feature, string $version): bool {}

		/**
		 * Gets the namespace URI of the node based on the prefix
		 * <p>Gets the namespace URI of the node based on the <code>prefix</code>.</p>
		 * @param ?string $prefix <p>The prefix to look for. If this parameter is <b><code>null</code></b>, the method will return the default namespace URI, if any.</p>
		 * @return ?string <p>Returns the associated namespace URI or <b><code>null</code></b> if none is found.</p>
		 * @link https://php.net/manual/en/domnode.lookupnamespaceuri.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupNamespaceURI(?string $prefix): ?string {}

		/**
		 * Gets the namespace prefix of the node based on the namespace URI
		 * <p>Gets the namespace prefix of the node based on the namespace URI.</p>
		 * @param string $namespace <p>The namespace URI.</p>
		 * @return ?string <p>The prefix of the namespace or <b><code>null</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.lookupprefix.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupPrefix(string $namespace): ?string {}

		/**
		 * Normalizes the node
		 * <p>Remove empty text nodes and merge adjacent text nodes in this node and all its children.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domnode.normalize.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function normalize(): void {}

		/**
		 * Removes child from list of children
		 * <p>This functions removes a child from a list of children.</p>
		 * @param \DOMNode $child <p>The removed child.</p>
		 * @return DOMNode|false <p>If the child could be removed the function returns the old child or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.removechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function removeChild(\DOMNode $child): \DOMNode|false {}

		/**
		 * Replaces a child
		 * <p>This function replaces the child <code>child</code> with the passed new node. If the <code>node</code> is already a child it will not be added a second time. If the replacement succeeds the old node is returned.</p>
		 * @param \DOMNode $node <p>The new node. It must be a member of the target document, i.e. created by one of the DOMDocument-&gt;createXXX() methods or imported in the document by DOMDocument::importNode.</p>
		 * @param \DOMNode $child <p>The old node.</p>
		 * @return DOMNode|false <p>The old node or <b><code>false</code></b> if an error occur.</p>
		 * @link https://php.net/manual/en/domnode.replacechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceChild(\DOMNode $node, \DOMNode $child): \DOMNode|false {}
	}

	/**
	 * <p>The <b>DOMCdataSection</b> class inherits from <code>DOMText</code> for textual representation of CData constructs.</p>
	 * @link https://php.net/manual/en/class.domcdatasection.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMCdataSection extends \DOMText {

		/**
		 * @var string <p>Holds all the text of logically-adjacent (not separated by Element, Comment or Processing Instruction) Text nodes.</p>
		 * @link https://php.net/manual/en/class.domtext.php#domtext.props.wholetext
		 */
		public string $wholeText;

		/**
		 * @var string <p>The contents of the node.</p>
		 * @link https://php.net/manual/en/class.domcharacterdata.php#domcharacterdata.props.data
		 */
		public string $data;

		/**
		 * @var int <p>The length of the contents.</p>
		 * @link https://php.net/manual/en/class.domcharacterdata.php#domcharacterdata.props.length
		 */
		public int $length;

		/**
		 * @var ?DOMElement <p>The previous sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domcharacterdata.php#domcharacterdata.props.previouselementsibling
		 */
		public ?\DOMElement $previousElementSibling;

		/**
		 * @var ?DOMElement <p>The next sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domcharacterdata.php#domcharacterdata.props.nextelementsibling
		 */
		public ?\DOMElement $nextElementSibling;

		/**
		 * @var string <p>Returns the most accurate name for the current node type</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var ?string <p>The value of this node, depending on its type. Contrary to the W3C specification, the node value of <code>DOMElement</code> nodes is equal to DOMNode::textContent instead of <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var ?DOMNode <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentnode
		 */
		public ?\DOMNode $parentNode;

		/**
		 * @var ?DOMElement <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentelement
		 */
		public ?\DOMElement $parentElement;

		/**
		 * @var DOMNodeList <p>A <code>DOMNodeList</code> that contains all children of this node. If there are no children, this is an empty <code>DOMNodeList</code>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.childnodes
		 */
		public \DOMNodeList $childNodes;

		/**
		 * @var ?DOMNode <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.firstchild
		 */
		public ?\DOMNode $firstChild;

		/**
		 * @var ?DOMNode <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.lastchild
		 */
		public ?\DOMNode $lastChild;

		/**
		 * @var ?DOMNode <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.previoussibling
		 */
		public ?\DOMNode $previousSibling;

		/**
		 * @var ?DOMNode <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nextsibling
		 */
		public ?\DOMNode $nextSibling;

		/**
		 * @var ?DOMNamedNodeMap <p>A <code>DOMNamedNodeMap</code> containing the attributes of this node (if it is a <code>DOMElement</code>) or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.attributes
		 */
		public ?\DOMNamedNodeMap $attributes;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?DOMDocument <p>The <code>DOMDocument</code> object associated with this node, or <b><code>null</code></b> if this node is a <code>DOMDocument</code></p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.ownerdocument
		 */
		public ?\DOMDocument $ownerDocument;

		/**
		 * @var ?string <p>The namespace URI of this node, or <b><code>null</code></b> if it is unspecified.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var string <p>The namespace prefix of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.prefix
		 */
		public string $prefix;

		/**
		 * @var ?string <p>Returns the local part of the qualified name of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.localname
		 */
		public ?string $localName;

		/**
		 * @var ?string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.baseuri
		 */
		public ?string $baseURI;

		/**
		 * @var string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.textcontent
		 */
		public string $textContent;

		/**
		 * Canonicalize nodes to a string
		 * <p>Canonicalize nodes to a string</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return string|false <p>Returns canonicalized nodes as a string or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14n.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14N(bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): string|false {}

		/**
		 * Canonicalize nodes to a file
		 * <p>Canonicalize nodes to a file.</p>
		 * @param string $uri <p>Path to write the output to.</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return int|false <p>Number of bytes written or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14nfile.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14NFile(string $uri, bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): int|false {}

		/**
		 * Constructs a new DOMCdataSection object
		 * <p>Constructs a new CDATA node. This works like the <code>DOMText</code> class.</p>
		 * @param string $data <p>The value of the CDATA node. If not supplied, an empty CDATA node is created.</p>
		 * @return self
		 * @link https://php.net/manual/en/domcdatasection.construct.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __construct(string $data) {}

		/**
		 * Forbids serialization unless serialization methods are implemented in a subclass
		 * <p>Forbids serialization unless serialization methods are implemented in a subclass.</p>
		 * @return array <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.sleep.php
		 * @since PHP 8 >= 8.1.25
		 */
		public function __sleep(): array {}

		/**
		 * Forbids unserialization unless unserialization methods are implemented in a subclass
		 * <p>Forbids unserialization unless unserialization methods are implemented in a subclass.</p>
		 * @return void <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.wakeup.php
		 * @since PHP 8 PHP 8 >= 8.1.25
		 */
		public function __wakeup(): void {}

		/**
		 * Adds nodes after the character data
		 * <p>Adds the passed <code>nodes</code> after the character data.</p>
		 * @param \DOMNode|string $nodes <p>Nodes to be added after the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.after.php
		 * @since PHP 8
		 */
		public function after(\DOMNode|string ...$nodes): void {}

		/**
		 * Adds new child at the end of the children
		 * <p>This function appends a child to an existing list of children or creates a new list of children. The child can be created with e.g. <code>DOMDocument::createElement()</code>, <code>DOMDocument::createTextNode()</code> etc. or simply by using any other node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The appended child.</p>
		 * @return DOMNode|false <p>The node added or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.appendchild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendChild(\DOMNode $node): \DOMNode|false {}

		/**
		 * Append the string to the end of the character data of the node
		 * <p>Append the string <code>data</code> to the end of the character data of the node.</p>
		 * @param string $data <p>The string to append.</p>
		 * @return true <p>Always returns <b><code>true</code></b>.</p>
		 * @link https://php.net/manual/en/domcharacterdata.appenddata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendData(string $data): true {}

		/**
		 * Adds nodes before the character data
		 * <p>Adds the passed <code>nodes</code> before the character data.</p>
		 * @param \DOMNode|string $nodes <p>Nodes to be added before the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.before.php
		 * @since PHP 8
		 */
		public function before(\DOMNode|string ...$nodes): void {}

		/**
		 * Clones a node
		 * <p>Creates a copy of the node.</p>
		 * @param bool $deep <p>Indicates whether to copy all descendant nodes. This parameter is defaulted to <b><code>false</code></b>.</p>
		 * @return DOMNode|false <p>The cloned node.</p>
		 * @link https://php.net/manual/en/domnode.clonenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function cloneNode(bool $deep = false): \DOMNode|false {}

		/**
		 * Compares the position of two nodes
		 * <p>Compares the position of the other node relative to this node.</p>
		 * @param \DOMNode $other <p>The node for which the position should be compared for, relative to this node.</p>
		 * @return int <p>A bitmask of the <b><code>DOMNode::DOCUMENT_POSITION_&#42;</code></b> constants.</p>
		 * @link https://php.net/manual/en/domnode.comparedocumentposition.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function compareDocumentPosition(\DOMNode $other): int {}

		/**
		 * Checks if node contains other node
		 * <p>Checks if node contains <code>other</code> node.</p>
		 * @param \DOMNode|\DOMNameSpaceNode|null $other <p>Node to be checked.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if node contains <code>other</code> node, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.contains.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function contains(\DOMNode|\DOMNameSpaceNode|null $other): bool {}

		/**
		 * Remove a range of characters from the character data
		 * <p>Deletes <code>count</code> characters starting from position <code>offset</code>.</p>
		 * @param int $offset <p>The offset from which to start removing.</p>
		 * @param int $count <p>The number of characters to delete. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are deleted.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domcharacterdata.deletedata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function deleteData(int $offset, int $count): bool {}

		/**
		 * Get line number for a node
		 * <p>Gets line number for where the node was defined at parse time.</p>
		 * @return int <p>Returns the line number where the node was defined at parse time. If the node was created manually, the return value will be <code>0</code>.</p>
		 * @link https://php.net/manual/en/domnode.getlineno.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		public function getLineNo(): int {}

		/**
		 * Get an XPath for a node
		 * <p>Gets an XPath location path for the node.</p>
		 * @return ?string <p>Returns a <code>string</code> containing the XPath, or <b><code>null</code></b> in case of an error.</p>
		 * @link https://php.net/manual/en/domnode.getnodepath.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function getNodePath(): ?string {}

		/**
		 * Get root node
		 * <p>Get root node.</p>
		 * @param ?array $options <p>This parameter has no effect yet.</p>
		 * @return DOMNode <p>Returns the root node.</p>
		 * @link https://php.net/manual/en/domnode.getrootnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function getRootNode(?array $options = null): \DOMNode {}

		/**
		 * Checks if node has attributes
		 * <p>This method checks if the node has attributes. The tested node has to be an <b><code>XML_ELEMENT_NODE</code></b>.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.hasattributes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasAttributes(): bool {}

		/**
		 * Checks if node has children
		 * <p>This function checks if the node has children.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.haschildnodes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasChildNodes(): bool {}

		/**
		 * Adds a new child before a reference node
		 * <p>This function inserts a new node right before the reference node. If you plan to do further modifications on the appended child you must use the returned node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The new node.</p>
		 * @param ?\DOMNode $child <p>The reference node. If not supplied, <code>node</code> is appended to the children.</p>
		 * @return DOMNode|false <p>The inserted node or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.insertbefore.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertBefore(\DOMNode $node, ?\DOMNode $child = null): \DOMNode|false {}

		/**
		 * Insert a string at the specified UTF-8 codepoint offset
		 * <p>Inserts string <code>data</code> at position <code>offset</code>.</p>
		 * @param int $offset <p>The character offset at which to insert.</p>
		 * @param string $data <p>The string to insert.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domcharacterdata.insertdata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertData(int $offset, string $data): bool {}

		/**
		 * Checks if the specified namespaceURI is the default namespace or not
		 * <p>Tells whether <code>namespace</code> is the default namespace.</p>
		 * @param string $namespace <p>The namespace URI to look for.</p>
		 * @return bool <p>Return <b><code>true</code></b> if <code>namespace</code> is the default namespace, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isdefaultnamespace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isDefaultNamespace(string $namespace): bool {}

		/**
		 * Returns whether this text node contains whitespace in element content
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domtext.iselementcontentwhitespace.php
		 * @since No version information available, might only be in Git
		 */
		public function isElementContentWhitespace(): bool {}

		/**
		 * Checks that both nodes are equal
		 * <p>Checks that both nodes are equal.</p>
		 * @param ?\DOMNode $otherNode <p>The node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if both nodes are equal, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isequalnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function isEqualNode(?\DOMNode $otherNode): bool {}

		/**
		 * Indicates if two nodes are the same node
		 * <p>This function indicates if two nodes are the same node. The comparison is <i>not</i> based on content</p>
		 * @param \DOMNode $otherNode <p>The compared node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issamenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSameNode(\DOMNode $otherNode): bool {}

		/**
		 * Checks if feature is supported for specified version
		 * <p>Checks if the asked <code>feature</code> is supported for the specified <code>version</code>.</p>
		 * @param string $feature <p>The feature to test. See the example of <code>DOMImplementation::hasFeature()</code> for a list of features.</p>
		 * @param string $version <p>The version number of the <code>feature</code> to test.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issupported.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSupported(string $feature, string $version): bool {}

		/**
		 * Indicates whether this text node contains whitespace
		 * <p>Indicates whether this text node contains only whitespace or it is empty. The text node is determined to contain whitespace in element content during the load of the document.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if node contains zero or more whitespace characters and nothing else. Returns <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domtext.iswhitespaceinelementcontent.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isWhitespaceInElementContent(): bool {}

		/**
		 * Gets the namespace URI of the node based on the prefix
		 * <p>Gets the namespace URI of the node based on the <code>prefix</code>.</p>
		 * @param ?string $prefix <p>The prefix to look for. If this parameter is <b><code>null</code></b>, the method will return the default namespace URI, if any.</p>
		 * @return ?string <p>Returns the associated namespace URI or <b><code>null</code></b> if none is found.</p>
		 * @link https://php.net/manual/en/domnode.lookupnamespaceuri.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupNamespaceURI(?string $prefix): ?string {}

		/**
		 * Gets the namespace prefix of the node based on the namespace URI
		 * <p>Gets the namespace prefix of the node based on the namespace URI.</p>
		 * @param string $namespace <p>The namespace URI.</p>
		 * @return ?string <p>The prefix of the namespace or <b><code>null</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.lookupprefix.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupPrefix(string $namespace): ?string {}

		/**
		 * Normalizes the node
		 * <p>Remove empty text nodes and merge adjacent text nodes in this node and all its children.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domnode.normalize.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function normalize(): void {}

		/**
		 * Removes the character data node
		 * <p>Removes the character data node.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.remove.php
		 * @since PHP 8
		 */
		public function remove(): void {}

		/**
		 * Removes child from list of children
		 * <p>This functions removes a child from a list of children.</p>
		 * @param \DOMNode $child <p>The removed child.</p>
		 * @return DOMNode|false <p>If the child could be removed the function returns the old child or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.removechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function removeChild(\DOMNode $child): \DOMNode|false {}

		/**
		 * Replaces a child
		 * <p>This function replaces the child <code>child</code> with the passed new node. If the <code>node</code> is already a child it will not be added a second time. If the replacement succeeds the old node is returned.</p>
		 * @param \DOMNode $node <p>The new node. It must be a member of the target document, i.e. created by one of the DOMDocument-&gt;createXXX() methods or imported in the document by DOMDocument::importNode.</p>
		 * @param \DOMNode $child <p>The old node.</p>
		 * @return DOMNode|false <p>The old node or <b><code>false</code></b> if an error occur.</p>
		 * @link https://php.net/manual/en/domnode.replacechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceChild(\DOMNode $node, \DOMNode $child): \DOMNode|false {}

		/**
		 * Replace a substring within the character data
		 * <p>Replace <code>count</code> characters starting from position <code>offset</code> with <code>data</code>.</p>
		 * @param int $offset <p>The offset from which to start replacing.</p>
		 * @param int $count <p>The number of characters to replace. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are replaced.</p>
		 * @param string $data <p>The string with which the range must be replaced.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domcharacterdata.replacedata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceData(int $offset, int $count, string $data): bool {}

		/**
		 * Replaces the character data with new nodes
		 * <p>Replaces the character data with new <code>nodes</code>.</p>
		 * @param \DOMNode|string $nodes <p>The replacement nodes. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.replacewith.php
		 * @since PHP 8
		 */
		public function replaceWith(\DOMNode|string ...$nodes): void {}

		/**
		 * Breaks this node into two nodes at the specified offset
		 * <p>Breaks this node into two nodes at the specified <code>offset</code>, keeping both in the tree as siblings.</p><p>After being split, this node will contain all the content up to the <code>offset</code>. If the original node had a parent node, the new node is inserted as the next sibling of the original node. When the <code>offset</code> is equal to the length of this node, the new node has no data.</p>
		 * @param int $offset <p>The offset at which to split, starting from 0.</p>
		 * @return DOMText|false <p>The new node of the same type, which contains all the content at and after the <code>offset</code>.</p>
		 * @link https://php.net/manual/en/domtext.splittext.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function splitText(int $offset): \DOMText|false {}

		/**
		 * Extracts a range of data from the character data
		 * <p>Returns the specified substring.</p>
		 * @param int $offset <p>Start offset of substring to extract.</p>
		 * @param int $count <p>The number of characters to extract.</p>
		 * @return string|false <p>The specified substring. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all UTF-8 codepoints to the end of the data are returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.substringdata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function substringData(int $offset, int $count): string|false {}
	}

	/**
	 * <p>Represents nodes with character data. No nodes directly correspond to this class, but other nodes do inherit from it.</p>
	 * @link https://php.net/manual/en/class.domcharacterdata.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMCharacterData extends \DOMNode implements \DOMChildNode {

		/**
		 * @var string <p>The contents of the node.</p>
		 * @link https://php.net/manual/en/class.domcharacterdata.php#domcharacterdata.props.data
		 */
		public string $data;

		/**
		 * @var int <p>The length of the contents.</p>
		 * @link https://php.net/manual/en/class.domcharacterdata.php#domcharacterdata.props.length
		 */
		public int $length;

		/**
		 * @var ?DOMElement <p>The previous sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domcharacterdata.php#domcharacterdata.props.previouselementsibling
		 */
		public ?\DOMElement $previousElementSibling;

		/**
		 * @var ?DOMElement <p>The next sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domcharacterdata.php#domcharacterdata.props.nextelementsibling
		 */
		public ?\DOMElement $nextElementSibling;

		/**
		 * @var string <p>Returns the most accurate name for the current node type</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var ?string <p>The value of this node, depending on its type. Contrary to the W3C specification, the node value of <code>DOMElement</code> nodes is equal to DOMNode::textContent instead of <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var ?DOMNode <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentnode
		 */
		public ?\DOMNode $parentNode;

		/**
		 * @var ?DOMElement <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentelement
		 */
		public ?\DOMElement $parentElement;

		/**
		 * @var DOMNodeList <p>A <code>DOMNodeList</code> that contains all children of this node. If there are no children, this is an empty <code>DOMNodeList</code>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.childnodes
		 */
		public \DOMNodeList $childNodes;

		/**
		 * @var ?DOMNode <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.firstchild
		 */
		public ?\DOMNode $firstChild;

		/**
		 * @var ?DOMNode <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.lastchild
		 */
		public ?\DOMNode $lastChild;

		/**
		 * @var ?DOMNode <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.previoussibling
		 */
		public ?\DOMNode $previousSibling;

		/**
		 * @var ?DOMNode <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nextsibling
		 */
		public ?\DOMNode $nextSibling;

		/**
		 * @var ?DOMNamedNodeMap <p>A <code>DOMNamedNodeMap</code> containing the attributes of this node (if it is a <code>DOMElement</code>) or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.attributes
		 */
		public ?\DOMNamedNodeMap $attributes;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?DOMDocument <p>The <code>DOMDocument</code> object associated with this node, or <b><code>null</code></b> if this node is a <code>DOMDocument</code></p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.ownerdocument
		 */
		public ?\DOMDocument $ownerDocument;

		/**
		 * @var ?string <p>The namespace URI of this node, or <b><code>null</code></b> if it is unspecified.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var string <p>The namespace prefix of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.prefix
		 */
		public string $prefix;

		/**
		 * @var ?string <p>Returns the local part of the qualified name of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.localname
		 */
		public ?string $localName;

		/**
		 * @var ?string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.baseuri
		 */
		public ?string $baseURI;

		/**
		 * @var string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.textcontent
		 */
		public string $textContent;

		/**
		 * Canonicalize nodes to a string
		 * <p>Canonicalize nodes to a string</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return string|false <p>Returns canonicalized nodes as a string or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14n.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14N(bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): string|false {}

		/**
		 * Canonicalize nodes to a file
		 * <p>Canonicalize nodes to a file.</p>
		 * @param string $uri <p>Path to write the output to.</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return int|false <p>Number of bytes written or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14nfile.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14NFile(string $uri, bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): int|false {}

		/**
		 * Forbids serialization unless serialization methods are implemented in a subclass
		 * <p>Forbids serialization unless serialization methods are implemented in a subclass.</p>
		 * @return array <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.sleep.php
		 * @since PHP 8 >= 8.1.25
		 */
		public function __sleep(): array {}

		/**
		 * Forbids unserialization unless unserialization methods are implemented in a subclass
		 * <p>Forbids unserialization unless unserialization methods are implemented in a subclass.</p>
		 * @return void <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.wakeup.php
		 * @since PHP 8 PHP 8 >= 8.1.25
		 */
		public function __wakeup(): void {}

		/**
		 * Adds nodes after the character data
		 * <p>Adds the passed <code>nodes</code> after the character data.</p>
		 * @param \DOMNode|string $nodes <p>Nodes to be added after the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.after.php
		 * @since PHP 8
		 */
		public function after(\DOMNode|string ...$nodes): void {}

		/**
		 * Adds new child at the end of the children
		 * <p>This function appends a child to an existing list of children or creates a new list of children. The child can be created with e.g. <code>DOMDocument::createElement()</code>, <code>DOMDocument::createTextNode()</code> etc. or simply by using any other node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The appended child.</p>
		 * @return DOMNode|false <p>The node added or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.appendchild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendChild(\DOMNode $node): \DOMNode|false {}

		/**
		 * Append the string to the end of the character data of the node
		 * <p>Append the string <code>data</code> to the end of the character data of the node.</p>
		 * @param string $data <p>The string to append.</p>
		 * @return true <p>Always returns <b><code>true</code></b>.</p>
		 * @link https://php.net/manual/en/domcharacterdata.appenddata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendData(string $data): true {}

		/**
		 * Adds nodes before the character data
		 * <p>Adds the passed <code>nodes</code> before the character data.</p>
		 * @param \DOMNode|string $nodes <p>Nodes to be added before the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.before.php
		 * @since PHP 8
		 */
		public function before(\DOMNode|string ...$nodes): void {}

		/**
		 * Clones a node
		 * <p>Creates a copy of the node.</p>
		 * @param bool $deep <p>Indicates whether to copy all descendant nodes. This parameter is defaulted to <b><code>false</code></b>.</p>
		 * @return DOMNode|false <p>The cloned node.</p>
		 * @link https://php.net/manual/en/domnode.clonenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function cloneNode(bool $deep = false): \DOMNode|false {}

		/**
		 * Compares the position of two nodes
		 * <p>Compares the position of the other node relative to this node.</p>
		 * @param \DOMNode $other <p>The node for which the position should be compared for, relative to this node.</p>
		 * @return int <p>A bitmask of the <b><code>DOMNode::DOCUMENT_POSITION_&#42;</code></b> constants.</p>
		 * @link https://php.net/manual/en/domnode.comparedocumentposition.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function compareDocumentPosition(\DOMNode $other): int {}

		/**
		 * Checks if node contains other node
		 * <p>Checks if node contains <code>other</code> node.</p>
		 * @param \DOMNode|\DOMNameSpaceNode|null $other <p>Node to be checked.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if node contains <code>other</code> node, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.contains.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function contains(\DOMNode|\DOMNameSpaceNode|null $other): bool {}

		/**
		 * Remove a range of characters from the character data
		 * <p>Deletes <code>count</code> characters starting from position <code>offset</code>.</p>
		 * @param int $offset <p>The offset from which to start removing.</p>
		 * @param int $count <p>The number of characters to delete. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are deleted.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domcharacterdata.deletedata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function deleteData(int $offset, int $count): bool {}

		/**
		 * Get line number for a node
		 * <p>Gets line number for where the node was defined at parse time.</p>
		 * @return int <p>Returns the line number where the node was defined at parse time. If the node was created manually, the return value will be <code>0</code>.</p>
		 * @link https://php.net/manual/en/domnode.getlineno.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		public function getLineNo(): int {}

		/**
		 * Get an XPath for a node
		 * <p>Gets an XPath location path for the node.</p>
		 * @return ?string <p>Returns a <code>string</code> containing the XPath, or <b><code>null</code></b> in case of an error.</p>
		 * @link https://php.net/manual/en/domnode.getnodepath.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function getNodePath(): ?string {}

		/**
		 * Get root node
		 * <p>Get root node.</p>
		 * @param ?array $options <p>This parameter has no effect yet.</p>
		 * @return DOMNode <p>Returns the root node.</p>
		 * @link https://php.net/manual/en/domnode.getrootnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function getRootNode(?array $options = null): \DOMNode {}

		/**
		 * Checks if node has attributes
		 * <p>This method checks if the node has attributes. The tested node has to be an <b><code>XML_ELEMENT_NODE</code></b>.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.hasattributes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasAttributes(): bool {}

		/**
		 * Checks if node has children
		 * <p>This function checks if the node has children.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.haschildnodes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasChildNodes(): bool {}

		/**
		 * Adds a new child before a reference node
		 * <p>This function inserts a new node right before the reference node. If you plan to do further modifications on the appended child you must use the returned node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The new node.</p>
		 * @param ?\DOMNode $child <p>The reference node. If not supplied, <code>node</code> is appended to the children.</p>
		 * @return DOMNode|false <p>The inserted node or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.insertbefore.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertBefore(\DOMNode $node, ?\DOMNode $child = null): \DOMNode|false {}

		/**
		 * Insert a string at the specified UTF-8 codepoint offset
		 * <p>Inserts string <code>data</code> at position <code>offset</code>.</p>
		 * @param int $offset <p>The character offset at which to insert.</p>
		 * @param string $data <p>The string to insert.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domcharacterdata.insertdata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertData(int $offset, string $data): bool {}

		/**
		 * Checks if the specified namespaceURI is the default namespace or not
		 * <p>Tells whether <code>namespace</code> is the default namespace.</p>
		 * @param string $namespace <p>The namespace URI to look for.</p>
		 * @return bool <p>Return <b><code>true</code></b> if <code>namespace</code> is the default namespace, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isdefaultnamespace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isDefaultNamespace(string $namespace): bool {}

		/**
		 * Checks that both nodes are equal
		 * <p>Checks that both nodes are equal.</p>
		 * @param ?\DOMNode $otherNode <p>The node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if both nodes are equal, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isequalnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function isEqualNode(?\DOMNode $otherNode): bool {}

		/**
		 * Indicates if two nodes are the same node
		 * <p>This function indicates if two nodes are the same node. The comparison is <i>not</i> based on content</p>
		 * @param \DOMNode $otherNode <p>The compared node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issamenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSameNode(\DOMNode $otherNode): bool {}

		/**
		 * Checks if feature is supported for specified version
		 * <p>Checks if the asked <code>feature</code> is supported for the specified <code>version</code>.</p>
		 * @param string $feature <p>The feature to test. See the example of <code>DOMImplementation::hasFeature()</code> for a list of features.</p>
		 * @param string $version <p>The version number of the <code>feature</code> to test.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issupported.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSupported(string $feature, string $version): bool {}

		/**
		 * Gets the namespace URI of the node based on the prefix
		 * <p>Gets the namespace URI of the node based on the <code>prefix</code>.</p>
		 * @param ?string $prefix <p>The prefix to look for. If this parameter is <b><code>null</code></b>, the method will return the default namespace URI, if any.</p>
		 * @return ?string <p>Returns the associated namespace URI or <b><code>null</code></b> if none is found.</p>
		 * @link https://php.net/manual/en/domnode.lookupnamespaceuri.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupNamespaceURI(?string $prefix): ?string {}

		/**
		 * Gets the namespace prefix of the node based on the namespace URI
		 * <p>Gets the namespace prefix of the node based on the namespace URI.</p>
		 * @param string $namespace <p>The namespace URI.</p>
		 * @return ?string <p>The prefix of the namespace or <b><code>null</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.lookupprefix.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupPrefix(string $namespace): ?string {}

		/**
		 * Normalizes the node
		 * <p>Remove empty text nodes and merge adjacent text nodes in this node and all its children.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domnode.normalize.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function normalize(): void {}

		/**
		 * Removes the character data node
		 * <p>Removes the character data node.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.remove.php
		 * @since PHP 8
		 */
		public function remove(): void {}

		/**
		 * Removes child from list of children
		 * <p>This functions removes a child from a list of children.</p>
		 * @param \DOMNode $child <p>The removed child.</p>
		 * @return DOMNode|false <p>If the child could be removed the function returns the old child or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.removechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function removeChild(\DOMNode $child): \DOMNode|false {}

		/**
		 * Replaces a child
		 * <p>This function replaces the child <code>child</code> with the passed new node. If the <code>node</code> is already a child it will not be added a second time. If the replacement succeeds the old node is returned.</p>
		 * @param \DOMNode $node <p>The new node. It must be a member of the target document, i.e. created by one of the DOMDocument-&gt;createXXX() methods or imported in the document by DOMDocument::importNode.</p>
		 * @param \DOMNode $child <p>The old node.</p>
		 * @return DOMNode|false <p>The old node or <b><code>false</code></b> if an error occur.</p>
		 * @link https://php.net/manual/en/domnode.replacechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceChild(\DOMNode $node, \DOMNode $child): \DOMNode|false {}

		/**
		 * Replace a substring within the character data
		 * <p>Replace <code>count</code> characters starting from position <code>offset</code> with <code>data</code>.</p>
		 * @param int $offset <p>The offset from which to start replacing.</p>
		 * @param int $count <p>The number of characters to replace. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are replaced.</p>
		 * @param string $data <p>The string with which the range must be replaced.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domcharacterdata.replacedata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceData(int $offset, int $count, string $data): bool {}

		/**
		 * Replaces the character data with new nodes
		 * <p>Replaces the character data with new <code>nodes</code>.</p>
		 * @param \DOMNode|string $nodes <p>The replacement nodes. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.replacewith.php
		 * @since PHP 8
		 */
		public function replaceWith(\DOMNode|string ...$nodes): void {}

		/**
		 * Extracts a range of data from the character data
		 * <p>Returns the specified substring.</p>
		 * @param int $offset <p>Start offset of substring to extract.</p>
		 * @param int $count <p>The number of characters to extract.</p>
		 * @return string|false <p>The specified substring. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all UTF-8 codepoints to the end of the data are returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.substringdata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function substringData(int $offset, int $count): string|false {}
	}

	/**
	 * @link https://php.net/manual/en/class.domchildnode.php
	 * @since PHP 8
	 */
	interface DOMChildNode {

		/**
		 * Adds nodes after the node
		 * <p>Adds the passed <code>nodes</code> after the node.</p>
		 * @param \DOMNode|string $nodes <p>Nodes to be added after the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domchildnode.after.php
		 * @since PHP 8
		 */
		public function after(\DOMNode|string ...$nodes): void;

		/**
		 * Adds nodes before the node
		 * <p>Adds the passed <code>nodes</code> before the node.</p>
		 * @param \DOMNode|string $nodes <p>Nodes to be added before the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domchildnode.before.php
		 * @since PHP 8
		 */
		public function before(\DOMNode|string ...$nodes): void;

		/**
		 * Removes the node
		 * <p>Removes the node.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domchildnode.remove.php
		 * @since PHP 8
		 */
		public function remove(): void;

		/**
		 * Replaces the node with new nodes
		 * <p>Replaces the node with new <code>nodes</code>.</p>
		 * @param \DOMNode|string $nodes <p>The replacement nodes. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domchildnode.replacewith.php
		 * @since PHP 8
		 */
		public function replaceWith(\DOMNode|string ...$nodes): void;
	}

	/**
	 * <p>Represents comment nodes, characters delimited by <code>&lt;!--</code> and <code>--&gt;</code>.</p>
	 * @link https://php.net/manual/en/class.domcomment.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMComment extends \DOMCharacterData {

		/**
		 * @var string <p>The contents of the node.</p>
		 * @link https://php.net/manual/en/class.domcharacterdata.php#domcharacterdata.props.data
		 */
		public string $data;

		/**
		 * @var int <p>The length of the contents.</p>
		 * @link https://php.net/manual/en/class.domcharacterdata.php#domcharacterdata.props.length
		 */
		public int $length;

		/**
		 * @var ?DOMElement <p>The previous sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domcharacterdata.php#domcharacterdata.props.previouselementsibling
		 */
		public ?\DOMElement $previousElementSibling;

		/**
		 * @var ?DOMElement <p>The next sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domcharacterdata.php#domcharacterdata.props.nextelementsibling
		 */
		public ?\DOMElement $nextElementSibling;

		/**
		 * @var string <p>Returns the most accurate name for the current node type</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var ?string <p>The value of this node, depending on its type. Contrary to the W3C specification, the node value of <code>DOMElement</code> nodes is equal to DOMNode::textContent instead of <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var ?DOMNode <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentnode
		 */
		public ?\DOMNode $parentNode;

		/**
		 * @var ?DOMElement <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentelement
		 */
		public ?\DOMElement $parentElement;

		/**
		 * @var DOMNodeList <p>A <code>DOMNodeList</code> that contains all children of this node. If there are no children, this is an empty <code>DOMNodeList</code>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.childnodes
		 */
		public \DOMNodeList $childNodes;

		/**
		 * @var ?DOMNode <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.firstchild
		 */
		public ?\DOMNode $firstChild;

		/**
		 * @var ?DOMNode <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.lastchild
		 */
		public ?\DOMNode $lastChild;

		/**
		 * @var ?DOMNode <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.previoussibling
		 */
		public ?\DOMNode $previousSibling;

		/**
		 * @var ?DOMNode <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nextsibling
		 */
		public ?\DOMNode $nextSibling;

		/**
		 * @var ?DOMNamedNodeMap <p>A <code>DOMNamedNodeMap</code> containing the attributes of this node (if it is a <code>DOMElement</code>) or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.attributes
		 */
		public ?\DOMNamedNodeMap $attributes;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?DOMDocument <p>The <code>DOMDocument</code> object associated with this node, or <b><code>null</code></b> if this node is a <code>DOMDocument</code></p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.ownerdocument
		 */
		public ?\DOMDocument $ownerDocument;

		/**
		 * @var ?string <p>The namespace URI of this node, or <b><code>null</code></b> if it is unspecified.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var string <p>The namespace prefix of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.prefix
		 */
		public string $prefix;

		/**
		 * @var ?string <p>Returns the local part of the qualified name of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.localname
		 */
		public ?string $localName;

		/**
		 * @var ?string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.baseuri
		 */
		public ?string $baseURI;

		/**
		 * @var string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.textcontent
		 */
		public string $textContent;

		/**
		 * Canonicalize nodes to a string
		 * <p>Canonicalize nodes to a string</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return string|false <p>Returns canonicalized nodes as a string or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14n.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14N(bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): string|false {}

		/**
		 * Canonicalize nodes to a file
		 * <p>Canonicalize nodes to a file.</p>
		 * @param string $uri <p>Path to write the output to.</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return int|false <p>Number of bytes written or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14nfile.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14NFile(string $uri, bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): int|false {}

		/**
		 * Creates a new DOMComment object
		 * <p>Creates a new <code>DOMComment</code> object. This object is read only. It may be appended to a document, but additional nodes may not be appended to this node until the node is associated with a document. To create a writeable node, use DOMDocument::createComment.</p>
		 * @param string $data <p>The value of the comment.</p>
		 * @return self
		 * @link https://php.net/manual/en/domcomment.construct.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __construct(string $data = "") {}

		/**
		 * Forbids serialization unless serialization methods are implemented in a subclass
		 * <p>Forbids serialization unless serialization methods are implemented in a subclass.</p>
		 * @return array <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.sleep.php
		 * @since PHP 8 >= 8.1.25
		 */
		public function __sleep(): array {}

		/**
		 * Forbids unserialization unless unserialization methods are implemented in a subclass
		 * <p>Forbids unserialization unless unserialization methods are implemented in a subclass.</p>
		 * @return void <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.wakeup.php
		 * @since PHP 8 PHP 8 >= 8.1.25
		 */
		public function __wakeup(): void {}

		/**
		 * Adds nodes after the character data
		 * <p>Adds the passed <code>nodes</code> after the character data.</p>
		 * @param \DOMNode|string $nodes <p>Nodes to be added after the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.after.php
		 * @since PHP 8
		 */
		public function after(\DOMNode|string ...$nodes): void {}

		/**
		 * Adds new child at the end of the children
		 * <p>This function appends a child to an existing list of children or creates a new list of children. The child can be created with e.g. <code>DOMDocument::createElement()</code>, <code>DOMDocument::createTextNode()</code> etc. or simply by using any other node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The appended child.</p>
		 * @return DOMNode|false <p>The node added or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.appendchild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendChild(\DOMNode $node): \DOMNode|false {}

		/**
		 * Append the string to the end of the character data of the node
		 * <p>Append the string <code>data</code> to the end of the character data of the node.</p>
		 * @param string $data <p>The string to append.</p>
		 * @return true <p>Always returns <b><code>true</code></b>.</p>
		 * @link https://php.net/manual/en/domcharacterdata.appenddata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendData(string $data): true {}

		/**
		 * Adds nodes before the character data
		 * <p>Adds the passed <code>nodes</code> before the character data.</p>
		 * @param \DOMNode|string $nodes <p>Nodes to be added before the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.before.php
		 * @since PHP 8
		 */
		public function before(\DOMNode|string ...$nodes): void {}

		/**
		 * Clones a node
		 * <p>Creates a copy of the node.</p>
		 * @param bool $deep <p>Indicates whether to copy all descendant nodes. This parameter is defaulted to <b><code>false</code></b>.</p>
		 * @return DOMNode|false <p>The cloned node.</p>
		 * @link https://php.net/manual/en/domnode.clonenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function cloneNode(bool $deep = false): \DOMNode|false {}

		/**
		 * Compares the position of two nodes
		 * <p>Compares the position of the other node relative to this node.</p>
		 * @param \DOMNode $other <p>The node for which the position should be compared for, relative to this node.</p>
		 * @return int <p>A bitmask of the <b><code>DOMNode::DOCUMENT_POSITION_&#42;</code></b> constants.</p>
		 * @link https://php.net/manual/en/domnode.comparedocumentposition.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function compareDocumentPosition(\DOMNode $other): int {}

		/**
		 * Checks if node contains other node
		 * <p>Checks if node contains <code>other</code> node.</p>
		 * @param \DOMNode|\DOMNameSpaceNode|null $other <p>Node to be checked.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if node contains <code>other</code> node, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.contains.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function contains(\DOMNode|\DOMNameSpaceNode|null $other): bool {}

		/**
		 * Remove a range of characters from the character data
		 * <p>Deletes <code>count</code> characters starting from position <code>offset</code>.</p>
		 * @param int $offset <p>The offset from which to start removing.</p>
		 * @param int $count <p>The number of characters to delete. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are deleted.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domcharacterdata.deletedata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function deleteData(int $offset, int $count): bool {}

		/**
		 * Get line number for a node
		 * <p>Gets line number for where the node was defined at parse time.</p>
		 * @return int <p>Returns the line number where the node was defined at parse time. If the node was created manually, the return value will be <code>0</code>.</p>
		 * @link https://php.net/manual/en/domnode.getlineno.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		public function getLineNo(): int {}

		/**
		 * Get an XPath for a node
		 * <p>Gets an XPath location path for the node.</p>
		 * @return ?string <p>Returns a <code>string</code> containing the XPath, or <b><code>null</code></b> in case of an error.</p>
		 * @link https://php.net/manual/en/domnode.getnodepath.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function getNodePath(): ?string {}

		/**
		 * Get root node
		 * <p>Get root node.</p>
		 * @param ?array $options <p>This parameter has no effect yet.</p>
		 * @return DOMNode <p>Returns the root node.</p>
		 * @link https://php.net/manual/en/domnode.getrootnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function getRootNode(?array $options = null): \DOMNode {}

		/**
		 * Checks if node has attributes
		 * <p>This method checks if the node has attributes. The tested node has to be an <b><code>XML_ELEMENT_NODE</code></b>.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.hasattributes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasAttributes(): bool {}

		/**
		 * Checks if node has children
		 * <p>This function checks if the node has children.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.haschildnodes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasChildNodes(): bool {}

		/**
		 * Adds a new child before a reference node
		 * <p>This function inserts a new node right before the reference node. If you plan to do further modifications on the appended child you must use the returned node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The new node.</p>
		 * @param ?\DOMNode $child <p>The reference node. If not supplied, <code>node</code> is appended to the children.</p>
		 * @return DOMNode|false <p>The inserted node or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.insertbefore.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertBefore(\DOMNode $node, ?\DOMNode $child = null): \DOMNode|false {}

		/**
		 * Insert a string at the specified UTF-8 codepoint offset
		 * <p>Inserts string <code>data</code> at position <code>offset</code>.</p>
		 * @param int $offset <p>The character offset at which to insert.</p>
		 * @param string $data <p>The string to insert.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domcharacterdata.insertdata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertData(int $offset, string $data): bool {}

		/**
		 * Checks if the specified namespaceURI is the default namespace or not
		 * <p>Tells whether <code>namespace</code> is the default namespace.</p>
		 * @param string $namespace <p>The namespace URI to look for.</p>
		 * @return bool <p>Return <b><code>true</code></b> if <code>namespace</code> is the default namespace, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isdefaultnamespace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isDefaultNamespace(string $namespace): bool {}

		/**
		 * Checks that both nodes are equal
		 * <p>Checks that both nodes are equal.</p>
		 * @param ?\DOMNode $otherNode <p>The node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if both nodes are equal, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isequalnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function isEqualNode(?\DOMNode $otherNode): bool {}

		/**
		 * Indicates if two nodes are the same node
		 * <p>This function indicates if two nodes are the same node. The comparison is <i>not</i> based on content</p>
		 * @param \DOMNode $otherNode <p>The compared node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issamenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSameNode(\DOMNode $otherNode): bool {}

		/**
		 * Checks if feature is supported for specified version
		 * <p>Checks if the asked <code>feature</code> is supported for the specified <code>version</code>.</p>
		 * @param string $feature <p>The feature to test. See the example of <code>DOMImplementation::hasFeature()</code> for a list of features.</p>
		 * @param string $version <p>The version number of the <code>feature</code> to test.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issupported.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSupported(string $feature, string $version): bool {}

		/**
		 * Gets the namespace URI of the node based on the prefix
		 * <p>Gets the namespace URI of the node based on the <code>prefix</code>.</p>
		 * @param ?string $prefix <p>The prefix to look for. If this parameter is <b><code>null</code></b>, the method will return the default namespace URI, if any.</p>
		 * @return ?string <p>Returns the associated namespace URI or <b><code>null</code></b> if none is found.</p>
		 * @link https://php.net/manual/en/domnode.lookupnamespaceuri.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupNamespaceURI(?string $prefix): ?string {}

		/**
		 * Gets the namespace prefix of the node based on the namespace URI
		 * <p>Gets the namespace prefix of the node based on the namespace URI.</p>
		 * @param string $namespace <p>The namespace URI.</p>
		 * @return ?string <p>The prefix of the namespace or <b><code>null</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.lookupprefix.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupPrefix(string $namespace): ?string {}

		/**
		 * Normalizes the node
		 * <p>Remove empty text nodes and merge adjacent text nodes in this node and all its children.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domnode.normalize.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function normalize(): void {}

		/**
		 * Removes the character data node
		 * <p>Removes the character data node.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.remove.php
		 * @since PHP 8
		 */
		public function remove(): void {}

		/**
		 * Removes child from list of children
		 * <p>This functions removes a child from a list of children.</p>
		 * @param \DOMNode $child <p>The removed child.</p>
		 * @return DOMNode|false <p>If the child could be removed the function returns the old child or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.removechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function removeChild(\DOMNode $child): \DOMNode|false {}

		/**
		 * Replaces a child
		 * <p>This function replaces the child <code>child</code> with the passed new node. If the <code>node</code> is already a child it will not be added a second time. If the replacement succeeds the old node is returned.</p>
		 * @param \DOMNode $node <p>The new node. It must be a member of the target document, i.e. created by one of the DOMDocument-&gt;createXXX() methods or imported in the document by DOMDocument::importNode.</p>
		 * @param \DOMNode $child <p>The old node.</p>
		 * @return DOMNode|false <p>The old node or <b><code>false</code></b> if an error occur.</p>
		 * @link https://php.net/manual/en/domnode.replacechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceChild(\DOMNode $node, \DOMNode $child): \DOMNode|false {}

		/**
		 * Replace a substring within the character data
		 * <p>Replace <code>count</code> characters starting from position <code>offset</code> with <code>data</code>.</p>
		 * @param int $offset <p>The offset from which to start replacing.</p>
		 * @param int $count <p>The number of characters to replace. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are replaced.</p>
		 * @param string $data <p>The string with which the range must be replaced.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domcharacterdata.replacedata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceData(int $offset, int $count, string $data): bool {}

		/**
		 * Replaces the character data with new nodes
		 * <p>Replaces the character data with new <code>nodes</code>.</p>
		 * @param \DOMNode|string $nodes <p>The replacement nodes. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.replacewith.php
		 * @since PHP 8
		 */
		public function replaceWith(\DOMNode|string ...$nodes): void {}

		/**
		 * Extracts a range of data from the character data
		 * <p>Returns the specified substring.</p>
		 * @param int $offset <p>Start offset of substring to extract.</p>
		 * @param int $count <p>The number of characters to extract.</p>
		 * @return string|false <p>The specified substring. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all UTF-8 codepoints to the end of the data are returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.substringdata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function substringData(int $offset, int $count): string|false {}
	}

	/**
	 * <p>Represents an entire HTML or XML document; serves as the root of the document tree.</p>
	 * @link https://php.net/manual/en/class.domdocument.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMDocument extends \DOMNode implements \DOMParentNode {

		/**
		 * @var ?DOMDocumentType <p>The Document Type Declaration associated with this document.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.doctype
		 */
		public ?\DOMDocumentType $doctype;

		/**
		 * @var DOMImplementation <p>The <code>DOMImplementation</code> object that handles this document.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.implementation
		 */
		public \DOMImplementation $implementation;

		/**
		 * @var ?DOMElement <p>The <code>DOMElement</code> object that is the first document element. If not found, this evaluates to <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.documentelement
		 */
		public ?\DOMElement $documentElement;

		/**
		 * @var ?string <p><i>Deprecated as of PHP 8.4.0</i>. Actual encoding of the document, is a readonly equivalent to encoding.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.actualencoding
		 */
		public ?string $actualEncoding;

		/**
		 * @var ?string <p>Encoding of the document, as specified by the XML declaration. This attribute is not present in the final DOM Level 3 specification, but is the only way of manipulating XML document encoding in this implementation.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.encoding
		 */
		public ?string $encoding;

		/**
		 * @var ?string <p>An attribute specifying, as part of the XML declaration, the encoding of this document. This is <b><code>null</code></b> when unspecified or when it is not known, such as when the Document was created in memory.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.xmlencoding
		 */
		public ?string $xmlEncoding;

		/**
		 * @var bool <p><i>Deprecated</i>. Whether or not the document is standalone, as specified by the XML declaration, corresponds to xmlStandalone.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.standalone
		 */
		public bool $standalone;

		/**
		 * @var bool <p>An attribute specifying, as part of the XML declaration, whether this document is standalone. This is <b><code>false</code></b> when unspecified. A standalone document is one where there are no external markup declarations. An example of such a markup declaration is when the DTD declares an attribute with a default value.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.xmlstandalone
		 */
		public bool $xmlStandalone;

		/**
		 * @var ?string <p><i>Deprecated</i>. Version of XML, corresponds to xmlVersion.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.version
		 */
		public ?string $version;

		/**
		 * @var ?string <p>An attribute specifying, as part of the XML declaration, the version number of this document. If there is no declaration and if this document supports the "XML" feature, the value is "1.0".</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.xmlversion
		 */
		public ?string $xmlVersion;

		/**
		 * @var bool <p>Throws <code>DOMException</code> on errors. Default to <b><code>true</code></b>.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.stricterrorchecking
		 */
		public bool $strictErrorChecking;

		/**
		 * @var ?string <p>The location of the document or <b><code>null</code></b> if undefined.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.documenturi
		 */
		public ?string $documentURI;

		/**
		 * @var mixed <p><i>Deprecated as of PHP 8.4.0</i>. Configuration used when <code>DOMDocument::normalizeDocument()</code> is invoked.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.config
		 */
		public mixed $config;

		/**
		 * @var bool <p>Nicely formats output with indentation and extra space. This has no effect if the document was loaded with preserveWhitespace enabled.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.formatoutput
		 */
		public bool $formatOutput;

		/**
		 * @var bool <p>Loads and validates against the DTD. Default to <b><code>false</code></b>.</p> <b>Caution</b> <p>Enabling validating the DTD may facilitate XML External Entity (XXE) attacks.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.validateonparse
		 */
		public bool $validateOnParse;

		/**
		 * @var bool <p>Set it to <b><code>true</code></b> to load external entities from a doctype declaration. This is useful for including character entities in your XML document.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.resolveexternals
		 */
		public bool $resolveExternals;

		/**
		 * @var bool <p>Do not remove redundant white space. Default to <b><code>true</code></b>. Setting this to <b><code>false</code></b> has the same effect as passing <b><code>LIBXML_NOBLANKS</code></b> as <code>option</code> to <code>DOMDocument::load()</code> etc.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.preservewhitespace
		 */
		public bool $preserveWhiteSpace;

		/**
		 * @var bool <p><i>Proprietary</i>. Enables recovery mode, i.e. trying to parse non-well formed documents. This attribute is not part of the DOM specification and is specific to libxml.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.recover
		 */
		public bool $recover;

		/**
		 * @var bool <p><i>Proprietary</i>. Whether or not to substitute entities. This attribute is not part of the DOM specification and is specific to libxml. Default to <b><code>false</code></b>.</p> <b>Caution</b> <p>Enabling entity substitution may facilitate XML External Entity (XXE) attacks.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.substituteentities
		 */
		public bool $substituteEntities;

		/**
		 * @var ?DOMElement <p>First child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.firstelementchild
		 */
		public ?\DOMElement $firstElementChild;

		/**
		 * @var ?DOMElement <p>Last child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.lastelementchild
		 */
		public ?\DOMElement $lastElementChild;

		/**
		 * @var int <p>The number of child elements.</p>
		 * @link https://php.net/manual/en/class.domdocument.php#domdocument.props.childelementcount
		 */
		public int $childElementCount;

		/**
		 * @var string <p>Returns the most accurate name for the current node type</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var ?string <p>The value of this node, depending on its type. Contrary to the W3C specification, the node value of <code>DOMElement</code> nodes is equal to DOMNode::textContent instead of <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var ?DOMNode <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentnode
		 */
		public ?\DOMNode $parentNode;

		/**
		 * @var ?DOMElement <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentelement
		 */
		public ?\DOMElement $parentElement;

		/**
		 * @var DOMNodeList <p>A <code>DOMNodeList</code> that contains all children of this node. If there are no children, this is an empty <code>DOMNodeList</code>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.childnodes
		 */
		public \DOMNodeList $childNodes;

		/**
		 * @var ?DOMNode <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.firstchild
		 */
		public ?\DOMNode $firstChild;

		/**
		 * @var ?DOMNode <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.lastchild
		 */
		public ?\DOMNode $lastChild;

		/**
		 * @var ?DOMNode <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.previoussibling
		 */
		public ?\DOMNode $previousSibling;

		/**
		 * @var ?DOMNode <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nextsibling
		 */
		public ?\DOMNode $nextSibling;

		/**
		 * @var ?DOMNamedNodeMap <p>A <code>DOMNamedNodeMap</code> containing the attributes of this node (if it is a <code>DOMElement</code>) or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.attributes
		 */
		public ?\DOMNamedNodeMap $attributes;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?DOMDocument <p>The <code>DOMDocument</code> object associated with this node, or <b><code>null</code></b> if this node is a <code>DOMDocument</code></p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.ownerdocument
		 */
		public ?\DOMDocument $ownerDocument;

		/**
		 * @var ?string <p>The namespace URI of this node, or <b><code>null</code></b> if it is unspecified.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var string <p>The namespace prefix of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.prefix
		 */
		public string $prefix;

		/**
		 * @var ?string <p>Returns the local part of the qualified name of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.localname
		 */
		public ?string $localName;

		/**
		 * @var ?string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.baseuri
		 */
		public ?string $baseURI;

		/**
		 * @var string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.textcontent
		 */
		public string $textContent;

		/**
		 * Canonicalize nodes to a string
		 * <p>Canonicalize nodes to a string</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return string|false <p>Returns canonicalized nodes as a string or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14n.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14N(bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): string|false {}

		/**
		 * Canonicalize nodes to a file
		 * <p>Canonicalize nodes to a file.</p>
		 * @param string $uri <p>Path to write the output to.</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return int|false <p>Number of bytes written or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14nfile.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14NFile(string $uri, bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): int|false {}

		/**
		 * Creates a new DOMDocument object
		 * <p>Creates a new <code>DOMDocument</code> object.</p>
		 * @param string $version <p>The version number of the document as part of the XML declaration.</p>
		 * @param string $encoding <p>The encoding of the document as part of the XML declaration.</p>
		 * @return self
		 * @link https://php.net/manual/en/domdocument.construct.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __construct(string $version = "1.0", string $encoding = "") {}

		/**
		 * Forbids serialization unless serialization methods are implemented in a subclass
		 * <p>Forbids serialization unless serialization methods are implemented in a subclass.</p>
		 * @return array <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.sleep.php
		 * @since PHP 8 >= 8.1.25
		 */
		public function __sleep(): array {}

		/**
		 * Forbids unserialization unless unserialization methods are implemented in a subclass
		 * <p>Forbids unserialization unless unserialization methods are implemented in a subclass.</p>
		 * @return void <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.wakeup.php
		 * @since PHP 8 PHP 8 >= 8.1.25
		 */
		public function __wakeup(): void {}

		/**
		 * Transfer a node from another document
		 * <p>Transfer a node from another document into the current document.</p>
		 * @param \DOMNode $node <p>The node to transfer.</p>
		 * @return DOMNode|false <p>The node that was transfered, or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domdocument.adoptnode.php
		 * @since PHP >= 8.3
		 */
		public function adoptNode(\DOMNode $node): \DOMNode|false {}

		/**
		 * Appends nodes after the last child node
		 * <p>Appends one or many <code>nodes</code> to the list of children after the last child node.</p>
		 * @param \DOMNode|string $nodes <p>The nodes to append. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domdocument.append.php
		 * @since PHP 8
		 */
		public function append(\DOMNode|string ...$nodes): void {}

		/**
		 * Adds new child at the end of the children
		 * <p>This function appends a child to an existing list of children or creates a new list of children. The child can be created with e.g. <code>DOMDocument::createElement()</code>, <code>DOMDocument::createTextNode()</code> etc. or simply by using any other node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The appended child.</p>
		 * @return DOMNode|false <p>The node added or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.appendchild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendChild(\DOMNode $node): \DOMNode|false {}

		/**
		 * Clones a node
		 * <p>Creates a copy of the node.</p>
		 * @param bool $deep <p>Indicates whether to copy all descendant nodes. This parameter is defaulted to <b><code>false</code></b>.</p>
		 * @return DOMNode|false <p>The cloned node.</p>
		 * @link https://php.net/manual/en/domnode.clonenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function cloneNode(bool $deep = false): \DOMNode|false {}

		/**
		 * Compares the position of two nodes
		 * <p>Compares the position of the other node relative to this node.</p>
		 * @param \DOMNode $other <p>The node for which the position should be compared for, relative to this node.</p>
		 * @return int <p>A bitmask of the <b><code>DOMNode::DOCUMENT_POSITION_&#42;</code></b> constants.</p>
		 * @link https://php.net/manual/en/domnode.comparedocumentposition.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function compareDocumentPosition(\DOMNode $other): int {}

		/**
		 * Checks if node contains other node
		 * <p>Checks if node contains <code>other</code> node.</p>
		 * @param \DOMNode|\DOMNameSpaceNode|null $other <p>Node to be checked.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if node contains <code>other</code> node, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.contains.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function contains(\DOMNode|\DOMNameSpaceNode|null $other): bool {}

		/**
		 * Create new attribute
		 * <p>This function creates a new instance of class <code>DOMAttr</code>. This node will not show up in the document unless it is inserted with (e.g.) <code>DOMNode::appendChild()</code>.</p>
		 * @param string $localName <p>The name of the attribute.</p>
		 * @return DOMAttr|false <p>The new <code>DOMAttr</code> or <b><code>false</code></b> if an error occurred.</p>
		 * @link https://php.net/manual/en/domdocument.createattribute.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function createAttribute(string $localName): \DOMAttr|false {}

		/**
		 * Create new attribute node with an associated namespace
		 * <p>This function creates a new instance of class <code>DOMAttr</code>. This node will not show up in the document unless it is inserted with (e.g.) <code>DOMNode::appendChild()</code>.</p>
		 * @param ?string $namespace <p>The URI of the namespace.</p>
		 * @param string $qualifiedName <p>The tag name and prefix of the attribute, as <code>prefix:tagname</code>.</p>
		 * @return DOMAttr|false <p>The new <code>DOMAttr</code> or <b><code>false</code></b> if an error occurred.</p>
		 * @link https://php.net/manual/en/domdocument.createattributens.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function createAttributeNS(?string $namespace, string $qualifiedName): \DOMAttr|false {}

		/**
		 * Create new cdata node
		 * <p>This function creates a new instance of class <code>DOMCDATASection</code>. This node will not show up in the document unless it is inserted with (e.g.) <code>DOMNode::appendChild()</code>.</p>
		 * @param string $data <p>The content of the cdata.</p>
		 * @return DOMCdataSection|false <p>The new <code>DOMCDATASection</code> or <b><code>false</code></b> if an error occurred.</p>
		 * @link https://php.net/manual/en/domdocument.createcdatasection.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function createCDATASection(string $data): \DOMCdataSection|false {}

		/**
		 * Create new comment node
		 * <p>This function creates a new instance of class <code>DOMComment</code>. This node will not show up in the document unless it is inserted with (e.g.) <code>DOMNode::appendChild()</code>.</p>
		 * @param string $data <p>The content of the comment.</p>
		 * @return DOMComment <p>The new <code>DOMComment</code>.</p>
		 * @link https://php.net/manual/en/domdocument.createcomment.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function createComment(string $data): \DOMComment {}

		/**
		 * Create new document fragment
		 * <p>This function creates a new instance of class <code>DOMDocumentFragment</code>. This node will not show up in the document unless it is inserted with (e.g.) <code>DOMNode::appendChild()</code>.</p>
		 * @return DOMDocumentFragment <p>The new <code>DOMDocumentFragment</code>.</p>
		 * @link https://php.net/manual/en/domdocument.createdocumentfragment.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function createDocumentFragment(): \DOMDocumentFragment {}

		/**
		 * Create new element node
		 * <p>This function creates a new instance of class <code>DOMElement</code>. This node will not show up in the document unless it is inserted with (e.g.) <code>DOMNode::appendChild()</code>.</p>
		 * @param string $localName <p>The tag name of the element.</p>
		 * @param string $value <p>The value of the element. By default, an empty element will be created. The value can also be set later with DOMElement::$nodeValue.</p> <p>The value is used verbatim except that the &lt; and &gt; entity references will escaped. Note that &amp; has to be manually escaped; otherwise it is regarded as starting an entity reference. Also " won't be escaped.</p>
		 * @return DOMElement|false <p>Returns a new instance of class <code>DOMElement</code> or <b><code>false</code></b> if an error occurred.</p>
		 * @link https://php.net/manual/en/domdocument.createelement.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function createElement(string $localName, string $value = ""): \DOMElement|false {}

		/**
		 * Create new element node with an associated namespace
		 * <p>This function creates a new element node with an associated namespace. This node will not show up in the document unless it is inserted with (e.g.) <code>DOMNode::appendChild()</code>.</p>
		 * @param ?string $namespace <p>The URI of the namespace.</p>
		 * @param string $qualifiedName <p>The qualified name of the element, as <code>prefix:tagname</code>.</p>
		 * @param string $value <p>The value of the element. By default, an empty element will be created. You can also set the value later with DOMElement::$nodeValue.</p>
		 * @return DOMElement|false <p>The new <code>DOMElement</code> or <b><code>false</code></b> if an error occurred.</p>
		 * @link https://php.net/manual/en/domdocument.createelementns.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function createElementNS(?string $namespace, string $qualifiedName, string $value = ""): \DOMElement|false {}

		/**
		 * Create new entity reference node
		 * <p>This function creates a new instance of class <code>DOMEntityReference</code>. This node will not show up in the document unless it is inserted with (e.g.) <code>DOMNode::appendChild()</code>.</p>
		 * @param string $name <p>The content of the entity reference, e.g. the entity reference minus the leading <code>&amp;</code> and the trailing <code>;</code> characters.</p>
		 * @return DOMEntityReference|false <p>The new <code>DOMEntityReference</code> or <b><code>false</code></b> if an error occurred.</p>
		 * @link https://php.net/manual/en/domdocument.createentityreference.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function createEntityReference(string $name): \DOMEntityReference|false {}

		/**
		 * Creates new PI node
		 * <p>This function creates a new instance of class <code>DOMProcessingInstruction</code>. This node will not show up in the document unless it is inserted with (e.g.) <code>DOMNode::appendChild()</code>.</p>
		 * @param string $target <p>The target of the processing instruction.</p>
		 * @param string $data <p>The content of the processing instruction.</p>
		 * @return DOMProcessingInstruction|false <p>The new <code>DOMProcessingInstruction</code> or <b><code>false</code></b> if an error occurred.</p>
		 * @link https://php.net/manual/en/domdocument.createprocessinginstruction.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function createProcessingInstruction(string $target, string $data = ""): \DOMProcessingInstruction|false {}

		/**
		 * Create new text node
		 * <p>This function creates a new instance of class <code>DOMText</code>. This node will not show up in the document unless it is inserted with (e.g.) <code>DOMNode::appendChild()</code>.</p>
		 * @param string $data <p>The content of the text.</p>
		 * @return DOMText <p>The new <code>DOMText</code>.</p>
		 * @link https://php.net/manual/en/domdocument.createtextnode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function createTextNode(string $data): \DOMText {}

		/**
		 * Searches for an element with a certain id
		 * <p>This function is similar to DOMDocument::getElementsByTagName but searches for an element with a given id.</p><p>For this function to work, you will need either to set some ID attributes with DOMElement::setIdAttribute or a DTD which defines an attribute to be of type ID. In the later case, you will need to validate your document with DOMDocument::validate or DOMDocument::$validateOnParse before using this function.</p>
		 * @param string $elementId <p>The unique id value for an element.</p>
		 * @return ?DOMElement <p>Returns the <code>DOMElement</code> or <b><code>null</code></b> if the element is not found.</p>
		 * @link https://php.net/manual/en/domdocument.getelementbyid.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function getElementById(string $elementId): ?\DOMElement {}

		/**
		 * Searches for all elements with given local tag name
		 * <p>This function returns a new instance of class <code>DOMNodeList</code> containing all the elements with a given local tag name.</p>
		 * @param string $qualifiedName <p>The local name (without namespace) of the tag to match on. The special value <code>&#42;</code> matches all tags.</p>
		 * @return DOMNodeList <p>A new <code>DOMNodeList</code> object containing all the matched elements.</p>
		 * @link https://php.net/manual/en/domdocument.getelementsbytagname.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function getElementsByTagName(string $qualifiedName): \DOMNodeList {}

		/**
		 * Searches for all elements with given tag name in specified namespace
		 * <p>Returns a <code>DOMNodeList</code> of all elements with a given local name and a namespace URI.</p>
		 * @param ?string $namespace <p>The namespace URI of the elements to match on. The special value <code>"&#42;"</code> matches all namespaces. Passing <b><code>null</code></b> matches the empty namespace.</p>
		 * @param string $localName <p>The local name of the elements to match on. The special value <code>"&#42;"</code> matches all local names.</p>
		 * @return DOMNodeList <p>A new <code>DOMNodeList</code> object containing all the matched elements.</p>
		 * @link https://php.net/manual/en/domdocument.getelementsbytagnamens.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function getElementsByTagNameNS(?string $namespace, string $localName): \DOMNodeList {}

		/**
		 * Get line number for a node
		 * <p>Gets line number for where the node was defined at parse time.</p>
		 * @return int <p>Returns the line number where the node was defined at parse time. If the node was created manually, the return value will be <code>0</code>.</p>
		 * @link https://php.net/manual/en/domnode.getlineno.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		public function getLineNo(): int {}

		/**
		 * Get an XPath for a node
		 * <p>Gets an XPath location path for the node.</p>
		 * @return ?string <p>Returns a <code>string</code> containing the XPath, or <b><code>null</code></b> in case of an error.</p>
		 * @link https://php.net/manual/en/domnode.getnodepath.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function getNodePath(): ?string {}

		/**
		 * Get root node
		 * <p>Get root node.</p>
		 * @param ?array $options <p>This parameter has no effect yet.</p>
		 * @return DOMNode <p>Returns the root node.</p>
		 * @link https://php.net/manual/en/domnode.getrootnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function getRootNode(?array $options = null): \DOMNode {}

		/**
		 * Checks if node has attributes
		 * <p>This method checks if the node has attributes. The tested node has to be an <b><code>XML_ELEMENT_NODE</code></b>.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.hasattributes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasAttributes(): bool {}

		/**
		 * Checks if node has children
		 * <p>This function checks if the node has children.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.haschildnodes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasChildNodes(): bool {}

		/**
		 * Import node into current document
		 * <p>This function returns a copy of the node to import and associates it with the current document.</p>
		 * @param \DOMNode $node <p>The node to import.</p>
		 * @param bool $deep <p>If set to <b><code>true</code></b>, this method will recursively import the subtree under the <code>node</code>.</p> <p><b>Note</b>:</p><p>To copy the nodes attributes <code>deep</code> needs to be set to <b><code>true</code></b></p>
		 * @return DOMNode|false <p>The copied node or <b><code>false</code></b>, if it cannot be copied.</p>
		 * @link https://php.net/manual/en/domdocument.importnode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function importNode(\DOMNode $node, bool $deep = false): \DOMNode|false {}

		/**
		 * Adds a new child before a reference node
		 * <p>This function inserts a new node right before the reference node. If you plan to do further modifications on the appended child you must use the returned node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The new node.</p>
		 * @param ?\DOMNode $child <p>The reference node. If not supplied, <code>node</code> is appended to the children.</p>
		 * @return DOMNode|false <p>The inserted node or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.insertbefore.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertBefore(\DOMNode $node, ?\DOMNode $child = null): \DOMNode|false {}

		/**
		 * Checks if the specified namespaceURI is the default namespace or not
		 * <p>Tells whether <code>namespace</code> is the default namespace.</p>
		 * @param string $namespace <p>The namespace URI to look for.</p>
		 * @return bool <p>Return <b><code>true</code></b> if <code>namespace</code> is the default namespace, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isdefaultnamespace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isDefaultNamespace(string $namespace): bool {}

		/**
		 * Checks that both nodes are equal
		 * <p>Checks that both nodes are equal.</p>
		 * @param ?\DOMNode $otherNode <p>The node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if both nodes are equal, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isequalnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function isEqualNode(?\DOMNode $otherNode): bool {}

		/**
		 * Indicates if two nodes are the same node
		 * <p>This function indicates if two nodes are the same node. The comparison is <i>not</i> based on content</p>
		 * @param \DOMNode $otherNode <p>The compared node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issamenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSameNode(\DOMNode $otherNode): bool {}

		/**
		 * Checks if feature is supported for specified version
		 * <p>Checks if the asked <code>feature</code> is supported for the specified <code>version</code>.</p>
		 * @param string $feature <p>The feature to test. See the example of <code>DOMImplementation::hasFeature()</code> for a list of features.</p>
		 * @param string $version <p>The version number of the <code>feature</code> to test.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issupported.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSupported(string $feature, string $version): bool {}

		/**
		 * Load XML from a file
		 * <p>Loads an XML document from a file.</p><p>Unix style paths with forward slashes can cause significant performance degradation on Windows systems; be sure to call <code>realpath()</code> in such a case.</p>
		 * @param string $filename <p>The path to the XML document.</p>
		 * @param int $options <p>Bitwise <code>OR</code> of the libxml option constants.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domdocument.load.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function load(string $filename, int $options = 0): bool {}

		/**
		 * Load HTML from a string
		 * <p>The function parses the HTML contained in the string <code>source</code>. Unlike loading XML, HTML does not have to be well-formed to load.</p><p>Use <code>Dom\HTMLDocument</code> to parse and process modern HTML instead of <code>DOMDocument</code>.</p><p>This function parses the input using an HTML 4 parser. The parsing rules of HTML 5, which is what modern web browsers use, are different. Depending on the input this might result in a different DOM structure. Therefore this function cannot be safely used for sanitizing HTML.</p><p>The behavior when parsing HTML can depend on the version of <code>libxml</code> that is being used, particularly with regards to edge conditions and error handling. For parsing that conforms to the HTML5 specification, use <b>Dom\HTMLDocument::createFromString()</b> or <b>Dom\HTMLDocument::createFromFile()</b>, added in PHP 8.4.</p><p>As an example, some HTML elements will implicitly close a parent element when encountered. The rules for automatically closing parent elements differ between HTML 4 and HTML 5 and thus the resulting DOM structure that <code>DOMDocument</code> sees might be different from the DOM structure a web browser sees, possibly allowing an attacker to break the resulting HTML.</p>
		 * @param string $source <p>The HTML string.</p>
		 * @param int $options <p>Bitwise <code>OR</code> of the libxml option constants.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domdocument.loadhtml.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function loadHTML(string $source, int $options = 0): bool {}

		/**
		 * Load HTML from a file
		 * <p>The function parses the HTML document in the file named <code>filename</code>. Unlike loading XML, HTML does not have to be well-formed to load.</p><p>Use <code>Dom\HTMLDocument</code> to parse and process modern HTML instead of <code>DOMDocument</code>.</p><p>This function parses the input using an HTML 4 parser. The parsing rules of HTML 5, which is what modern web browsers use, are different. Depending on the input this might result in a different DOM structure. Therefore this function cannot be safely used for sanitizing HTML.</p><p>The behavior when parsing HTML can depend on the version of <code>libxml</code> that is being used, particularly with regards to edge conditions and error handling. For parsing that conforms to the HTML5 specification, use <b>Dom\HTMLDocument::createFromString()</b> or <b>Dom\HTMLDocument::createFromFile()</b>, added in PHP 8.4.</p><p>As an example, some HTML elements will implicitly close a parent element when encountered. The rules for automatically closing parent elements differ between HTML 4 and HTML 5 and thus the resulting DOM structure that <code>DOMDocument</code> sees might be different from the DOM structure a web browser sees, possibly allowing an attacker to break the resulting HTML.</p>
		 * @param string $filename <p>The path to the HTML file.</p>
		 * @param int $options <p>Bitwise <code>OR</code> of the libxml option constants.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domdocument.loadhtmlfile.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function loadHTMLFile(string $filename, int $options = 0): bool {}

		/**
		 * Load XML from a string
		 * <p>Loads an XML document from a string.</p>
		 * @param string $source <p>The string containing the XML.</p>
		 * @param int $options <p>Bitwise <code>OR</code> of the libxml option constants.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domdocument.loadxml.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function loadXML(string $source, int $options = 0): bool {}

		/**
		 * Gets the namespace URI of the node based on the prefix
		 * <p>Gets the namespace URI of the node based on the <code>prefix</code>.</p>
		 * @param ?string $prefix <p>The prefix to look for. If this parameter is <b><code>null</code></b>, the method will return the default namespace URI, if any.</p>
		 * @return ?string <p>Returns the associated namespace URI or <b><code>null</code></b> if none is found.</p>
		 * @link https://php.net/manual/en/domnode.lookupnamespaceuri.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupNamespaceURI(?string $prefix): ?string {}

		/**
		 * Gets the namespace prefix of the node based on the namespace URI
		 * <p>Gets the namespace prefix of the node based on the namespace URI.</p>
		 * @param string $namespace <p>The namespace URI.</p>
		 * @return ?string <p>The prefix of the namespace or <b><code>null</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.lookupprefix.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupPrefix(string $namespace): ?string {}

		/**
		 * Normalizes the node
		 * <p>Remove empty text nodes and merge adjacent text nodes in this node and all its children.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domnode.normalize.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function normalize(): void {}

		/**
		 * Normalizes the document
		 * <p>This method acts as if you saved and then loaded the document, putting the document in a "normal" form.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domdocument.normalizedocument.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function normalizeDocument(): void {}

		/**
		 * Prepends nodes before the first child node
		 * <p>Prepends one or many <code>nodes</code> to the list of children before the first child node.</p>
		 * @param \DOMNode|string $nodes <p>The nodes to prepend. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domdocument.prepend.php
		 * @since PHP 8
		 */
		public function prepend(\DOMNode|string ...$nodes): void {}

		/**
		 * Register extended class used to create base node type
		 * <p>This method allows you to register your own extended DOM class to be used afterward by the PHP DOM extension.</p><p>This method is not part of the DOM standard.</p><p>The constructor of the objects of the registered node classes is not called.</p>
		 * @param string $baseClass <p>The DOM class that you want to extend. You can find a list of these classes in the chapter introduction.</p>
		 * @param ?string $extendedClass <p>Your extended class name. If <b><code>null</code></b> is provided, any previously registered class extending <code>baseClass</code> will be removed.</p>
		 * @return true <p>Always returns <b><code>true</code></b>.</p>
		 * @link https://php.net/manual/en/domdocument.registernodeclass.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function registerNodeClass(string $baseClass, ?string $extendedClass): true {}

		/**
		 * Performs relaxNG validation on the document
		 * <p>Performs &#xBB;&#xA0;relaxNG validation on the document based on the given RNG schema.</p>
		 * @param string $filename <p>The RNG file.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domdocument.relaxngvalidate.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function relaxNGValidate(string $filename): bool {}

		/**
		 * Performs relaxNG validation on the document
		 * <p>Performs &#xBB;&#xA0;relaxNG validation on the document based on the given RNG source.</p>
		 * @param string $source <p>A string containing the RNG schema.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domdocument.relaxngvalidatesource.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function relaxNGValidateSource(string $source): bool {}

		/**
		 * Removes child from list of children
		 * <p>This functions removes a child from a list of children.</p>
		 * @param \DOMNode $child <p>The removed child.</p>
		 * @return DOMNode|false <p>If the child could be removed the function returns the old child or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.removechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function removeChild(\DOMNode $child): \DOMNode|false {}

		/**
		 * Replaces a child
		 * <p>This function replaces the child <code>child</code> with the passed new node. If the <code>node</code> is already a child it will not be added a second time. If the replacement succeeds the old node is returned.</p>
		 * @param \DOMNode $node <p>The new node. It must be a member of the target document, i.e. created by one of the DOMDocument-&gt;createXXX() methods or imported in the document by DOMDocument::importNode.</p>
		 * @param \DOMNode $child <p>The old node.</p>
		 * @return DOMNode|false <p>The old node or <b><code>false</code></b> if an error occur.</p>
		 * @link https://php.net/manual/en/domnode.replacechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceChild(\DOMNode $node, \DOMNode $child): \DOMNode|false {}

		/**
		 * Replace children in document
		 * <p>Replaces the children in the document with new <code>nodes</code>.</p>
		 * @param \DOMNode|string $nodes <p>The nodes replacing the children. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domdocument.replacechildren.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function replaceChildren(\DOMNode|string ...$nodes): void {}

		/**
		 * Dumps the internal XML tree back into a file
		 * <p>Creates an XML document from the DOM representation. This function is usually called after building a new dom document from scratch as in the example below.</p>
		 * @param string $filename <p>The path to the saved XML document.</p>
		 * @param int $options <p>Additional Options. Currently only LIBXML_NOEMPTYTAG is supported.</p>
		 * @return int|false <p>Returns the number of bytes written or <b><code>false</code></b> if an error occurred.</p>
		 * @link https://php.net/manual/en/domdocument.save.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function save(string $filename, int $options = 0): int|false {}

		/**
		 * Dumps the internal document into a string using HTML formatting
		 * <p>Creates an HTML document from the DOM representation. This function is usually called after building a new dom document from scratch as in the example below.</p>
		 * @param ?\DOMNode $node <p>Optional parameter to output a subset of the document.</p>
		 * @return string|false <p>Returns the HTML, or <b><code>false</code></b> if an error occurred.</p>
		 * @link https://php.net/manual/en/domdocument.savehtml.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function saveHTML(?\DOMNode $node = null): string|false {}

		/**
		 * Dumps the internal document into a file using HTML formatting
		 * <p>Creates an HTML document from the DOM representation. This function is usually called after building a new dom document from scratch as in the example below.</p>
		 * @param string $filename <p>The path to the saved HTML document.</p>
		 * @return int|false <p>Returns the number of bytes written or <b><code>false</code></b> if an error occurred.</p>
		 * @link https://php.net/manual/en/domdocument.savehtmlfile.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function saveHTMLFile(string $filename): int|false {}

		/**
		 * Dumps the internal XML tree back into a string
		 * <p>Creates an XML document from the DOM representation. This function is usually called after building a new dom document from scratch as in the example below.</p>
		 * @param ?\DOMNode $node <p>Use this parameter to output only a specific node without XML declaration rather than the entire document.</p>
		 * @param int $options <p>Additional Options. The LIBXML_NOEMPTYTAG and LIBXML_NOXMLDECL options are supported. Prior to PHP 8.3.0, only the LIBXML_NOEMPTYTAG option is supported.</p>
		 * @return string|false <p>Returns the XML, or <b><code>false</code></b> if an error occurred.</p>
		 * @link https://php.net/manual/en/domdocument.savexml.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function saveXML(?\DOMNode $node = null, int $options = 0): string|false {}

		/**
		 * Validates a document based on a schema. Only XML Schema 1.0 is supported.
		 * <p>Validates a document based on the given schema file.</p>
		 * @param string $filename <p>The path to the schema.</p>
		 * @param int $flags <p>A bitmask of Libxml schema validation flags. Currently the only supported value is LIBXML_SCHEMA_CREATE. Available since Libxml 2.6.14.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domdocument.schemavalidate.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function schemaValidate(string $filename, int $flags = 0): bool {}

		/**
		 * Validates a document based on a schema
		 * <p>Validates a document based on a schema defined in the given string.</p>
		 * @param string $source <p>A string containing the schema.</p>
		 * @param int $flags <p>A bitmask of Libxml schema validation flags. Currently the only supported value is LIBXML_SCHEMA_CREATE. Available since Libxml 2.6.14.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domdocument.schemavalidatesource.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function schemaValidateSource(string $source, int $flags = 0): bool {}

		/**
		 * Validates the document based on its DTD
		 * <p>Validates the document based on its DTD.</p><p>You can also use the <code>validateOnParse</code> property of <code>DOMDocument</code> to make a DTD validation.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure. If the document has no DTD attached, this method will return <b><code>false</code></b>.</p>
		 * @link https://php.net/manual/en/domdocument.validate.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function validate(): bool {}

		/**
		 * Substitutes XIncludes in a DOMDocument Object
		 * <p>This method substitutes &#xBB;&#xA0;XIncludes in a DOMDocument object.</p><p><b>Note</b>:</p><p>Due to libxml2 automatically resolving entities, this method will produce unexpected results if the included XML file have an attached DTD.</p>
		 * @param int $options <p>Bitwise <code>OR</code> of the libxml option constants.</p>
		 * @return int|false <p>Returns the number of XIncludes in the document, -1 if some processing failed, or <b><code>false</code></b> if there were no substitutions.</p>
		 * @link https://php.net/manual/en/domdocument.xinclude.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function xinclude(int $options = 0): int|false {}
	}

	/**
	 * @link https://php.net/manual/en/class.domdocumentfragment.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMDocumentFragment extends \DOMNode implements \DOMParentNode {

		/**
		 * @var ?DOMElement <p>First child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domdocumentfragment.php#domdocumentfragment.props.firstelementchild
		 */
		public ?\DOMElement $firstElementChild;

		/**
		 * @var ?DOMElement <p>Last child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domdocumentfragment.php#domdocumentfragment.props.lastelementchild
		 */
		public ?\DOMElement $lastElementChild;

		/**
		 * @var int <p>The number of child elements.</p>
		 * @link https://php.net/manual/en/class.domdocumentfragment.php#domdocumentfragment.props.childelementcount
		 */
		public int $childElementCount;

		/**
		 * @var string <p>Returns the most accurate name for the current node type</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var ?string <p>The value of this node, depending on its type. Contrary to the W3C specification, the node value of <code>DOMElement</code> nodes is equal to DOMNode::textContent instead of <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var ?DOMNode <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentnode
		 */
		public ?\DOMNode $parentNode;

		/**
		 * @var ?DOMElement <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentelement
		 */
		public ?\DOMElement $parentElement;

		/**
		 * @var DOMNodeList <p>A <code>DOMNodeList</code> that contains all children of this node. If there are no children, this is an empty <code>DOMNodeList</code>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.childnodes
		 */
		public \DOMNodeList $childNodes;

		/**
		 * @var ?DOMNode <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.firstchild
		 */
		public ?\DOMNode $firstChild;

		/**
		 * @var ?DOMNode <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.lastchild
		 */
		public ?\DOMNode $lastChild;

		/**
		 * @var ?DOMNode <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.previoussibling
		 */
		public ?\DOMNode $previousSibling;

		/**
		 * @var ?DOMNode <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nextsibling
		 */
		public ?\DOMNode $nextSibling;

		/**
		 * @var ?DOMNamedNodeMap <p>A <code>DOMNamedNodeMap</code> containing the attributes of this node (if it is a <code>DOMElement</code>) or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.attributes
		 */
		public ?\DOMNamedNodeMap $attributes;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?DOMDocument <p>The <code>DOMDocument</code> object associated with this node, or <b><code>null</code></b> if this node is a <code>DOMDocument</code></p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.ownerdocument
		 */
		public ?\DOMDocument $ownerDocument;

		/**
		 * @var ?string <p>The namespace URI of this node, or <b><code>null</code></b> if it is unspecified.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var string <p>The namespace prefix of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.prefix
		 */
		public string $prefix;

		/**
		 * @var ?string <p>Returns the local part of the qualified name of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.localname
		 */
		public ?string $localName;

		/**
		 * @var ?string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.baseuri
		 */
		public ?string $baseURI;

		/**
		 * @var string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.textcontent
		 */
		public string $textContent;

		/**
		 * Canonicalize nodes to a string
		 * <p>Canonicalize nodes to a string</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return string|false <p>Returns canonicalized nodes as a string or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14n.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14N(bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): string|false {}

		/**
		 * Canonicalize nodes to a file
		 * <p>Canonicalize nodes to a file.</p>
		 * @param string $uri <p>Path to write the output to.</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return int|false <p>Number of bytes written or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14nfile.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14NFile(string $uri, bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): int|false {}

		/**
		 * Constructs a DOMDocumentFragment object
		 * @return self
		 * @link https://php.net/manual/en/domdocumentfragment.construct.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __construct() {}

		/**
		 * Forbids serialization unless serialization methods are implemented in a subclass
		 * <p>Forbids serialization unless serialization methods are implemented in a subclass.</p>
		 * @return array <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.sleep.php
		 * @since PHP 8 >= 8.1.25
		 */
		public function __sleep(): array {}

		/**
		 * Forbids unserialization unless unserialization methods are implemented in a subclass
		 * <p>Forbids unserialization unless unserialization methods are implemented in a subclass.</p>
		 * @return void <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.wakeup.php
		 * @since PHP 8 PHP 8 >= 8.1.25
		 */
		public function __wakeup(): void {}

		/**
		 * Appends nodes after the last child node
		 * <p>Appends one or many <code>nodes</code> to the list of children after the last child node.</p>
		 * @param \DOMNode|string $nodes <p>The nodes to append. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domdocumentfragment.append.php
		 * @since PHP 8
		 */
		public function append(\DOMNode|string ...$nodes): void {}

		/**
		 * Adds new child at the end of the children
		 * <p>This function appends a child to an existing list of children or creates a new list of children. The child can be created with e.g. <code>DOMDocument::createElement()</code>, <code>DOMDocument::createTextNode()</code> etc. or simply by using any other node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The appended child.</p>
		 * @return DOMNode|false <p>The node added or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.appendchild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendChild(\DOMNode $node): \DOMNode|false {}

		/**
		 * Append raw XML data
		 * <p>Appends raw XML data to a DOMDocumentFragment.</p><p>This method is not part of the DOM standard. It was created as a simpler approach for appending an XML DocumentFragment in a DOMDocument.</p><p>If you want to stick to the standards, you will have to create a temporary DOMDocument with a dummy root and then loop through the child nodes of the root of your XML data to append them.</p>
		 * @param string $data <p>XML to append.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domdocumentfragment.appendxml.php
		 * @since PHP 5 >= 5.1.0, PHP 7, PHP 8
		 */
		public function appendXML(string $data): bool {}

		/**
		 * Clones a node
		 * <p>Creates a copy of the node.</p>
		 * @param bool $deep <p>Indicates whether to copy all descendant nodes. This parameter is defaulted to <b><code>false</code></b>.</p>
		 * @return DOMNode|false <p>The cloned node.</p>
		 * @link https://php.net/manual/en/domnode.clonenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function cloneNode(bool $deep = false): \DOMNode|false {}

		/**
		 * Compares the position of two nodes
		 * <p>Compares the position of the other node relative to this node.</p>
		 * @param \DOMNode $other <p>The node for which the position should be compared for, relative to this node.</p>
		 * @return int <p>A bitmask of the <b><code>DOMNode::DOCUMENT_POSITION_&#42;</code></b> constants.</p>
		 * @link https://php.net/manual/en/domnode.comparedocumentposition.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function compareDocumentPosition(\DOMNode $other): int {}

		/**
		 * Checks if node contains other node
		 * <p>Checks if node contains <code>other</code> node.</p>
		 * @param \DOMNode|\DOMNameSpaceNode|null $other <p>Node to be checked.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if node contains <code>other</code> node, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.contains.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function contains(\DOMNode|\DOMNameSpaceNode|null $other): bool {}

		/**
		 * Get line number for a node
		 * <p>Gets line number for where the node was defined at parse time.</p>
		 * @return int <p>Returns the line number where the node was defined at parse time. If the node was created manually, the return value will be <code>0</code>.</p>
		 * @link https://php.net/manual/en/domnode.getlineno.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		public function getLineNo(): int {}

		/**
		 * Get an XPath for a node
		 * <p>Gets an XPath location path for the node.</p>
		 * @return ?string <p>Returns a <code>string</code> containing the XPath, or <b><code>null</code></b> in case of an error.</p>
		 * @link https://php.net/manual/en/domnode.getnodepath.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function getNodePath(): ?string {}

		/**
		 * Get root node
		 * <p>Get root node.</p>
		 * @param ?array $options <p>This parameter has no effect yet.</p>
		 * @return DOMNode <p>Returns the root node.</p>
		 * @link https://php.net/manual/en/domnode.getrootnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function getRootNode(?array $options = null): \DOMNode {}

		/**
		 * Checks if node has attributes
		 * <p>This method checks if the node has attributes. The tested node has to be an <b><code>XML_ELEMENT_NODE</code></b>.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.hasattributes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasAttributes(): bool {}

		/**
		 * Checks if node has children
		 * <p>This function checks if the node has children.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.haschildnodes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasChildNodes(): bool {}

		/**
		 * Adds a new child before a reference node
		 * <p>This function inserts a new node right before the reference node. If you plan to do further modifications on the appended child you must use the returned node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The new node.</p>
		 * @param ?\DOMNode $child <p>The reference node. If not supplied, <code>node</code> is appended to the children.</p>
		 * @return DOMNode|false <p>The inserted node or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.insertbefore.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertBefore(\DOMNode $node, ?\DOMNode $child = null): \DOMNode|false {}

		/**
		 * Checks if the specified namespaceURI is the default namespace or not
		 * <p>Tells whether <code>namespace</code> is the default namespace.</p>
		 * @param string $namespace <p>The namespace URI to look for.</p>
		 * @return bool <p>Return <b><code>true</code></b> if <code>namespace</code> is the default namespace, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isdefaultnamespace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isDefaultNamespace(string $namespace): bool {}

		/**
		 * Checks that both nodes are equal
		 * <p>Checks that both nodes are equal.</p>
		 * @param ?\DOMNode $otherNode <p>The node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if both nodes are equal, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isequalnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function isEqualNode(?\DOMNode $otherNode): bool {}

		/**
		 * Indicates if two nodes are the same node
		 * <p>This function indicates if two nodes are the same node. The comparison is <i>not</i> based on content</p>
		 * @param \DOMNode $otherNode <p>The compared node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issamenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSameNode(\DOMNode $otherNode): bool {}

		/**
		 * Checks if feature is supported for specified version
		 * <p>Checks if the asked <code>feature</code> is supported for the specified <code>version</code>.</p>
		 * @param string $feature <p>The feature to test. See the example of <code>DOMImplementation::hasFeature()</code> for a list of features.</p>
		 * @param string $version <p>The version number of the <code>feature</code> to test.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issupported.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSupported(string $feature, string $version): bool {}

		/**
		 * Gets the namespace URI of the node based on the prefix
		 * <p>Gets the namespace URI of the node based on the <code>prefix</code>.</p>
		 * @param ?string $prefix <p>The prefix to look for. If this parameter is <b><code>null</code></b>, the method will return the default namespace URI, if any.</p>
		 * @return ?string <p>Returns the associated namespace URI or <b><code>null</code></b> if none is found.</p>
		 * @link https://php.net/manual/en/domnode.lookupnamespaceuri.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupNamespaceURI(?string $prefix): ?string {}

		/**
		 * Gets the namespace prefix of the node based on the namespace URI
		 * <p>Gets the namespace prefix of the node based on the namespace URI.</p>
		 * @param string $namespace <p>The namespace URI.</p>
		 * @return ?string <p>The prefix of the namespace or <b><code>null</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.lookupprefix.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupPrefix(string $namespace): ?string {}

		/**
		 * Normalizes the node
		 * <p>Remove empty text nodes and merge adjacent text nodes in this node and all its children.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domnode.normalize.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function normalize(): void {}

		/**
		 * Prepends nodes before the first child node
		 * <p>Prepends one or many <code>nodes</code> to the list of children before the first child node.</p>
		 * @param \DOMNode|string $nodes <p>The nodes to prepend. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domdocumentfragment.prepend.php
		 * @since PHP 8
		 */
		public function prepend(\DOMNode|string ...$nodes): void {}

		/**
		 * Removes child from list of children
		 * <p>This functions removes a child from a list of children.</p>
		 * @param \DOMNode $child <p>The removed child.</p>
		 * @return DOMNode|false <p>If the child could be removed the function returns the old child or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.removechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function removeChild(\DOMNode $child): \DOMNode|false {}

		/**
		 * Replaces a child
		 * <p>This function replaces the child <code>child</code> with the passed new node. If the <code>node</code> is already a child it will not be added a second time. If the replacement succeeds the old node is returned.</p>
		 * @param \DOMNode $node <p>The new node. It must be a member of the target document, i.e. created by one of the DOMDocument-&gt;createXXX() methods or imported in the document by DOMDocument::importNode.</p>
		 * @param \DOMNode $child <p>The old node.</p>
		 * @return DOMNode|false <p>The old node or <b><code>false</code></b> if an error occur.</p>
		 * @link https://php.net/manual/en/domnode.replacechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceChild(\DOMNode $node, \DOMNode $child): \DOMNode|false {}

		/**
		 * Replace children in fragment
		 * <p>Replaces the children in the document fragment with new <code>nodes</code>.</p>
		 * @param \DOMNode|string $nodes <p>The nodes replacing the children. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domdocumentfragment.replacechildren.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function replaceChildren(\DOMNode|string ...$nodes): void {}
	}

	/**
	 * <p>Each <code>DOMDocument</code> has a <code>doctype</code> attribute whose value is either <b><code>null</code></b> or a <b>DOMDocumentType</b> object.</p>
	 * @link https://php.net/manual/en/class.domdocumenttype.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMDocumentType extends \DOMNode {

		/**
		 * @var string <p>The name of DTD; i.e., the name immediately following the <code>DOCTYPE</code> keyword.</p>
		 * @link https://php.net/manual/en/class.domdocumenttype.php#domdocumenttype.props.name
		 */
		public string $name;

		/**
		 * @var DOMNamedNodeMap <p>A <code>DOMNamedNodeMap</code> containing the general entities, both external and internal, declared in the DTD.</p>
		 * @link https://php.net/manual/en/class.domdocumenttype.php#domdocumenttype.props.entities
		 */
		public \DOMNamedNodeMap $entities;

		/**
		 * @var DOMNamedNodeMap <p>A <code>DOMNamedNodeMap</code> containing the notations declared in the DTD.</p>
		 * @link https://php.net/manual/en/class.domdocumenttype.php#domdocumenttype.props.notations
		 */
		public \DOMNamedNodeMap $notations;

		/**
		 * @var string <p>The public identifier of the external subset.</p>
		 * @link https://php.net/manual/en/class.domdocumenttype.php#domdocumenttype.props.publicid
		 */
		public string $publicId;

		/**
		 * @var string <p>The system identifier of the external subset. This may be an absolute URI or not.</p>
		 * @link https://php.net/manual/en/class.domdocumenttype.php#domdocumenttype.props.systemid
		 */
		public string $systemId;

		/**
		 * @var ?string <p>The internal subset as a string, or <b><code>null</code></b> if there is none. This does not contain the delimiting square brackets.</p>
		 * @link https://php.net/manual/en/class.domdocumenttype.php#domdocumenttype.props.internalsubset
		 */
		public ?string $internalSubset;

		/**
		 * @var string <p>Returns the most accurate name for the current node type</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var ?string <p>The value of this node, depending on its type. Contrary to the W3C specification, the node value of <code>DOMElement</code> nodes is equal to DOMNode::textContent instead of <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var ?DOMNode <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentnode
		 */
		public ?\DOMNode $parentNode;

		/**
		 * @var ?DOMElement <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentelement
		 */
		public ?\DOMElement $parentElement;

		/**
		 * @var DOMNodeList <p>A <code>DOMNodeList</code> that contains all children of this node. If there are no children, this is an empty <code>DOMNodeList</code>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.childnodes
		 */
		public \DOMNodeList $childNodes;

		/**
		 * @var ?DOMNode <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.firstchild
		 */
		public ?\DOMNode $firstChild;

		/**
		 * @var ?DOMNode <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.lastchild
		 */
		public ?\DOMNode $lastChild;

		/**
		 * @var ?DOMNode <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.previoussibling
		 */
		public ?\DOMNode $previousSibling;

		/**
		 * @var ?DOMNode <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nextsibling
		 */
		public ?\DOMNode $nextSibling;

		/**
		 * @var ?DOMNamedNodeMap <p>A <code>DOMNamedNodeMap</code> containing the attributes of this node (if it is a <code>DOMElement</code>) or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.attributes
		 */
		public ?\DOMNamedNodeMap $attributes;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?DOMDocument <p>The <code>DOMDocument</code> object associated with this node, or <b><code>null</code></b> if this node is a <code>DOMDocument</code></p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.ownerdocument
		 */
		public ?\DOMDocument $ownerDocument;

		/**
		 * @var ?string <p>The namespace URI of this node, or <b><code>null</code></b> if it is unspecified.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var string <p>The namespace prefix of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.prefix
		 */
		public string $prefix;

		/**
		 * @var ?string <p>Returns the local part of the qualified name of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.localname
		 */
		public ?string $localName;

		/**
		 * @var ?string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.baseuri
		 */
		public ?string $baseURI;

		/**
		 * @var string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.textcontent
		 */
		public string $textContent;

		/**
		 * Canonicalize nodes to a string
		 * <p>Canonicalize nodes to a string</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return string|false <p>Returns canonicalized nodes as a string or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14n.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14N(bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): string|false {}

		/**
		 * Canonicalize nodes to a file
		 * <p>Canonicalize nodes to a file.</p>
		 * @param string $uri <p>Path to write the output to.</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return int|false <p>Number of bytes written or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14nfile.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14NFile(string $uri, bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): int|false {}

		/**
		 * Forbids serialization unless serialization methods are implemented in a subclass
		 * <p>Forbids serialization unless serialization methods are implemented in a subclass.</p>
		 * @return array <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.sleep.php
		 * @since PHP 8 >= 8.1.25
		 */
		public function __sleep(): array {}

		/**
		 * Forbids unserialization unless unserialization methods are implemented in a subclass
		 * <p>Forbids unserialization unless unserialization methods are implemented in a subclass.</p>
		 * @return void <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.wakeup.php
		 * @since PHP 8 PHP 8 >= 8.1.25
		 */
		public function __wakeup(): void {}

		/**
		 * Adds new child at the end of the children
		 * <p>This function appends a child to an existing list of children or creates a new list of children. The child can be created with e.g. <code>DOMDocument::createElement()</code>, <code>DOMDocument::createTextNode()</code> etc. or simply by using any other node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The appended child.</p>
		 * @return DOMNode|false <p>The node added or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.appendchild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendChild(\DOMNode $node): \DOMNode|false {}

		/**
		 * Clones a node
		 * <p>Creates a copy of the node.</p>
		 * @param bool $deep <p>Indicates whether to copy all descendant nodes. This parameter is defaulted to <b><code>false</code></b>.</p>
		 * @return DOMNode|false <p>The cloned node.</p>
		 * @link https://php.net/manual/en/domnode.clonenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function cloneNode(bool $deep = false): \DOMNode|false {}

		/**
		 * Compares the position of two nodes
		 * <p>Compares the position of the other node relative to this node.</p>
		 * @param \DOMNode $other <p>The node for which the position should be compared for, relative to this node.</p>
		 * @return int <p>A bitmask of the <b><code>DOMNode::DOCUMENT_POSITION_&#42;</code></b> constants.</p>
		 * @link https://php.net/manual/en/domnode.comparedocumentposition.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function compareDocumentPosition(\DOMNode $other): int {}

		/**
		 * Checks if node contains other node
		 * <p>Checks if node contains <code>other</code> node.</p>
		 * @param \DOMNode|\DOMNameSpaceNode|null $other <p>Node to be checked.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if node contains <code>other</code> node, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.contains.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function contains(\DOMNode|\DOMNameSpaceNode|null $other): bool {}

		/**
		 * Get line number for a node
		 * <p>Gets line number for where the node was defined at parse time.</p>
		 * @return int <p>Returns the line number where the node was defined at parse time. If the node was created manually, the return value will be <code>0</code>.</p>
		 * @link https://php.net/manual/en/domnode.getlineno.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		public function getLineNo(): int {}

		/**
		 * Get an XPath for a node
		 * <p>Gets an XPath location path for the node.</p>
		 * @return ?string <p>Returns a <code>string</code> containing the XPath, or <b><code>null</code></b> in case of an error.</p>
		 * @link https://php.net/manual/en/domnode.getnodepath.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function getNodePath(): ?string {}

		/**
		 * Get root node
		 * <p>Get root node.</p>
		 * @param ?array $options <p>This parameter has no effect yet.</p>
		 * @return DOMNode <p>Returns the root node.</p>
		 * @link https://php.net/manual/en/domnode.getrootnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function getRootNode(?array $options = null): \DOMNode {}

		/**
		 * Checks if node has attributes
		 * <p>This method checks if the node has attributes. The tested node has to be an <b><code>XML_ELEMENT_NODE</code></b>.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.hasattributes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasAttributes(): bool {}

		/**
		 * Checks if node has children
		 * <p>This function checks if the node has children.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.haschildnodes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasChildNodes(): bool {}

		/**
		 * Adds a new child before a reference node
		 * <p>This function inserts a new node right before the reference node. If you plan to do further modifications on the appended child you must use the returned node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The new node.</p>
		 * @param ?\DOMNode $child <p>The reference node. If not supplied, <code>node</code> is appended to the children.</p>
		 * @return DOMNode|false <p>The inserted node or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.insertbefore.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertBefore(\DOMNode $node, ?\DOMNode $child = null): \DOMNode|false {}

		/**
		 * Checks if the specified namespaceURI is the default namespace or not
		 * <p>Tells whether <code>namespace</code> is the default namespace.</p>
		 * @param string $namespace <p>The namespace URI to look for.</p>
		 * @return bool <p>Return <b><code>true</code></b> if <code>namespace</code> is the default namespace, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isdefaultnamespace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isDefaultNamespace(string $namespace): bool {}

		/**
		 * Checks that both nodes are equal
		 * <p>Checks that both nodes are equal.</p>
		 * @param ?\DOMNode $otherNode <p>The node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if both nodes are equal, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isequalnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function isEqualNode(?\DOMNode $otherNode): bool {}

		/**
		 * Indicates if two nodes are the same node
		 * <p>This function indicates if two nodes are the same node. The comparison is <i>not</i> based on content</p>
		 * @param \DOMNode $otherNode <p>The compared node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issamenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSameNode(\DOMNode $otherNode): bool {}

		/**
		 * Checks if feature is supported for specified version
		 * <p>Checks if the asked <code>feature</code> is supported for the specified <code>version</code>.</p>
		 * @param string $feature <p>The feature to test. See the example of <code>DOMImplementation::hasFeature()</code> for a list of features.</p>
		 * @param string $version <p>The version number of the <code>feature</code> to test.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issupported.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSupported(string $feature, string $version): bool {}

		/**
		 * Gets the namespace URI of the node based on the prefix
		 * <p>Gets the namespace URI of the node based on the <code>prefix</code>.</p>
		 * @param ?string $prefix <p>The prefix to look for. If this parameter is <b><code>null</code></b>, the method will return the default namespace URI, if any.</p>
		 * @return ?string <p>Returns the associated namespace URI or <b><code>null</code></b> if none is found.</p>
		 * @link https://php.net/manual/en/domnode.lookupnamespaceuri.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupNamespaceURI(?string $prefix): ?string {}

		/**
		 * Gets the namespace prefix of the node based on the namespace URI
		 * <p>Gets the namespace prefix of the node based on the namespace URI.</p>
		 * @param string $namespace <p>The namespace URI.</p>
		 * @return ?string <p>The prefix of the namespace or <b><code>null</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.lookupprefix.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupPrefix(string $namespace): ?string {}

		/**
		 * Normalizes the node
		 * <p>Remove empty text nodes and merge adjacent text nodes in this node and all its children.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domnode.normalize.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function normalize(): void {}

		/**
		 * Removes child from list of children
		 * <p>This functions removes a child from a list of children.</p>
		 * @param \DOMNode $child <p>The removed child.</p>
		 * @return DOMNode|false <p>If the child could be removed the function returns the old child or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.removechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function removeChild(\DOMNode $child): \DOMNode|false {}

		/**
		 * Replaces a child
		 * <p>This function replaces the child <code>child</code> with the passed new node. If the <code>node</code> is already a child it will not be added a second time. If the replacement succeeds the old node is returned.</p>
		 * @param \DOMNode $node <p>The new node. It must be a member of the target document, i.e. created by one of the DOMDocument-&gt;createXXX() methods or imported in the document by DOMDocument::importNode.</p>
		 * @param \DOMNode $child <p>The old node.</p>
		 * @return DOMNode|false <p>The old node or <b><code>false</code></b> if an error occur.</p>
		 * @link https://php.net/manual/en/domnode.replacechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceChild(\DOMNode $node, \DOMNode $child): \DOMNode|false {}
	}

	/**
	 * @link https://php.net/manual/en/class.domelement.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMElement extends \DOMNode implements \DOMParentNode, \DOMChildNode {

		/**
		 * @var string <p>The element name</p>
		 * @link https://php.net/manual/en/class.domelement.php#domelement.props.tagname
		 */
		public string $tagName;

		/**
		 * @var string <p>A string representing the classes of the element separated by spaces.</p>
		 * @link https://php.net/manual/en/class.domelement.php#domelement.props.classname
		 */
		public string $className;

		/**
		 * @var string <p>Reflects the element ID through the <code>"id"</code> attribute.</p>
		 * @link https://php.net/manual/en/class.domelement.php#domelement.props.id
		 */
		public string $id;

		/**
		 * @var mixed <p>Not implemented yet, always return <b><code>null</code></b></p>
		 * @link https://php.net/manual/en/class.domelement.php#domelement.props.schematypeinfo
		 */
		public mixed $schemaTypeInfo;

		/**
		 * @var ?DOMElement <p>First child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domelement.php#domelement.props.firstelementchild
		 */
		public ?\DOMElement $firstElementChild;

		/**
		 * @var ?DOMElement <p>Last child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domelement.php#domelement.props.lastelementchild
		 */
		public ?\DOMElement $lastElementChild;

		/**
		 * @var int <p>The number of child elements.</p>
		 * @link https://php.net/manual/en/class.domelement.php#domelement.props.childelementcount
		 */
		public int $childElementCount;

		/**
		 * @var ?DOMElement <p>The previous sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domelement.php#domelement.props.previouselementsibling
		 */
		public ?\DOMElement $previousElementSibling;

		/**
		 * @var ?DOMElement <p>The next sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domelement.php#domelement.props.nextelementsibling
		 */
		public ?\DOMElement $nextElementSibling;

		/**
		 * @var string <p>Returns the most accurate name for the current node type</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var ?string <p>The value of this node, depending on its type. Contrary to the W3C specification, the node value of <code>DOMElement</code> nodes is equal to DOMNode::textContent instead of <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var ?DOMNode <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentnode
		 */
		public ?\DOMNode $parentNode;

		/**
		 * @var ?DOMElement <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentelement
		 */
		public ?\DOMElement $parentElement;

		/**
		 * @var DOMNodeList <p>A <code>DOMNodeList</code> that contains all children of this node. If there are no children, this is an empty <code>DOMNodeList</code>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.childnodes
		 */
		public \DOMNodeList $childNodes;

		/**
		 * @var ?DOMNode <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.firstchild
		 */
		public ?\DOMNode $firstChild;

		/**
		 * @var ?DOMNode <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.lastchild
		 */
		public ?\DOMNode $lastChild;

		/**
		 * @var ?DOMNode <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.previoussibling
		 */
		public ?\DOMNode $previousSibling;

		/**
		 * @var ?DOMNode <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nextsibling
		 */
		public ?\DOMNode $nextSibling;

		/**
		 * @var ?DOMNamedNodeMap <p>A <code>DOMNamedNodeMap</code> containing the attributes of this node (if it is a <code>DOMElement</code>) or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.attributes
		 */
		public ?\DOMNamedNodeMap $attributes;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?DOMDocument <p>The <code>DOMDocument</code> object associated with this node, or <b><code>null</code></b> if this node is a <code>DOMDocument</code></p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.ownerdocument
		 */
		public ?\DOMDocument $ownerDocument;

		/**
		 * @var ?string <p>The namespace URI of this node, or <b><code>null</code></b> if it is unspecified.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var string <p>The namespace prefix of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.prefix
		 */
		public string $prefix;

		/**
		 * @var ?string <p>Returns the local part of the qualified name of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.localname
		 */
		public ?string $localName;

		/**
		 * @var ?string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.baseuri
		 */
		public ?string $baseURI;

		/**
		 * @var string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.textcontent
		 */
		public string $textContent;

		/**
		 * Canonicalize nodes to a string
		 * <p>Canonicalize nodes to a string</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return string|false <p>Returns canonicalized nodes as a string or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14n.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14N(bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): string|false {}

		/**
		 * Canonicalize nodes to a file
		 * <p>Canonicalize nodes to a file.</p>
		 * @param string $uri <p>Path to write the output to.</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return int|false <p>Number of bytes written or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14nfile.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14NFile(string $uri, bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): int|false {}

		/**
		 * Creates a new DOMElement object
		 * <p>Creates a new <code>DOMElement</code> object. This object is read only. It may be appended to a document, but additional nodes may not be appended to this node until the node is associated with a document. To create a writeable node, use DOMDocument::createElement or DOMDocument::createElementNS.</p>
		 * @param string $qualifiedName <p>The tag name of the element. When also passing in namespaceURI, the element name may take a prefix to be associated with the URI.</p>
		 * @param ?string $value <p>The value of the element.</p>
		 * @param string $namespace <p>A namespace URI to create the element within a specific namespace.</p>
		 * @return self
		 * @link https://php.net/manual/en/domelement.construct.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __construct(string $qualifiedName, ?string $value = null, string $namespace = "") {}

		/**
		 * Forbids serialization unless serialization methods are implemented in a subclass
		 * <p>Forbids serialization unless serialization methods are implemented in a subclass.</p>
		 * @return array <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.sleep.php
		 * @since PHP 8 >= 8.1.25
		 */
		public function __sleep(): array {}

		/**
		 * Forbids unserialization unless unserialization methods are implemented in a subclass
		 * <p>Forbids unserialization unless unserialization methods are implemented in a subclass.</p>
		 * @return void <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.wakeup.php
		 * @since PHP 8 PHP 8 >= 8.1.25
		 */
		public function __wakeup(): void {}

		/**
		 * Adds nodes after the element
		 * <p>Adds the passed <code>nodes</code> after the element.</p>
		 * @param \DOMNode|string $nodes <p>Nodes to be added after the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domelement.after.php
		 * @since PHP 8
		 */
		public function after(\DOMNode|string ...$nodes): void {}

		/**
		 * Appends nodes after the last child node
		 * <p>Appends one or many <code>nodes</code> to the list of children after the last child node.</p>
		 * @param \DOMNode|string $nodes <p>The nodes to append. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domelement.append.php
		 * @since PHP 8
		 */
		public function append(\DOMNode|string ...$nodes): void {}

		/**
		 * Adds new child at the end of the children
		 * <p>This function appends a child to an existing list of children or creates a new list of children. The child can be created with e.g. <code>DOMDocument::createElement()</code>, <code>DOMDocument::createTextNode()</code> etc. or simply by using any other node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The appended child.</p>
		 * @return DOMNode|false <p>The node added or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.appendchild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendChild(\DOMNode $node): \DOMNode|false {}

		/**
		 * Adds nodes before the element
		 * <p>Adds the passed <code>nodes</code> before the element.</p>
		 * @param \DOMNode|string $nodes <p>Nodes to be added before the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domelement.before.php
		 * @since PHP 8
		 */
		public function before(\DOMNode|string ...$nodes): void {}

		/**
		 * Clones a node
		 * <p>Creates a copy of the node.</p>
		 * @param bool $deep <p>Indicates whether to copy all descendant nodes. This parameter is defaulted to <b><code>false</code></b>.</p>
		 * @return DOMNode|false <p>The cloned node.</p>
		 * @link https://php.net/manual/en/domnode.clonenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function cloneNode(bool $deep = false): \DOMNode|false {}

		/**
		 * Compares the position of two nodes
		 * <p>Compares the position of the other node relative to this node.</p>
		 * @param \DOMNode $other <p>The node for which the position should be compared for, relative to this node.</p>
		 * @return int <p>A bitmask of the <b><code>DOMNode::DOCUMENT_POSITION_&#42;</code></b> constants.</p>
		 * @link https://php.net/manual/en/domnode.comparedocumentposition.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function compareDocumentPosition(\DOMNode $other): int {}

		/**
		 * Checks if node contains other node
		 * <p>Checks if node contains <code>other</code> node.</p>
		 * @param \DOMNode|\DOMNameSpaceNode|null $other <p>Node to be checked.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if node contains <code>other</code> node, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.contains.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function contains(\DOMNode|\DOMNameSpaceNode|null $other): bool {}

		/**
		 * Returns value of attribute
		 * <p>Gets the value of the attribute with name <code>qualifiedName</code> for the current node.</p>
		 * @param string $qualifiedName <p>The name of the attribute.</p>
		 * @return string <p>The value of the attribute, or an empty string if no attribute with the given <code>qualifiedName</code> is found.</p>
		 * @link https://php.net/manual/en/domelement.getattribute.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function getAttribute(string $qualifiedName): string {}

		/**
		 * Returns value of attribute
		 * <p>Gets the value of the attribute in namespace <code>namespace</code> with local name <code>localName</code> for the current node.</p>
		 * @param ?string $namespace <p>The namespace URI.</p>
		 * @param string $localName <p>The local name.</p>
		 * @return string <p>The value of the attribute, or an empty string if no attribute with the given <code>localName</code> and <code>namespace</code> is found.</p>
		 * @link https://php.net/manual/en/domelement.getattributens.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function getAttributeNS(?string $namespace, string $localName): string {}

		/**
		 * Get attribute names
		 * <p>Get attribute names.</p>
		 * @return array <p>Return attribute names.</p>
		 * @link https://php.net/manual/en/domelement.getattributenames.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function getAttributeNames(): array {}

		/**
		 * Returns attribute node
		 * <p>Returns the attribute node with name <code>qualifiedName</code> for the current element.</p>
		 * @param string $qualifiedName <p>The name of the attribute.</p>
		 * @return DOMAttr|DOMNameSpaceNode|false <p>The attribute node. Note that for XML namespace declarations (<code>xmlns</code> and <code>xmlns:&#42;</code> attributes) an instance of <code>DOMNameSpaceNode</code> is returned instead of a <code>DOMAttr</code>.</p>
		 * @link https://php.net/manual/en/domelement.getattributenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function getAttributeNode(string $qualifiedName): \DOMAttr|\DOMNameSpaceNode|false {}

		/**
		 * Returns attribute node
		 * <p>Returns the attribute node in namespace <code>namespace</code> with local name <code>localName</code> for the current node.</p>
		 * @param ?string $namespace <p>The namespace URI.</p>
		 * @param string $localName <p>The local name.</p>
		 * @return DOMAttr|DOMNameSpaceNode|null <p>The attribute node. Note that for XML namespace declarations (<code>xmlns</code> and <code>xmlns:&#42;</code> attributes) an instance of <code>DOMNameSpaceNode</code> is returned instead of a <code>DOMAttr</code> object.</p>
		 * @link https://php.net/manual/en/domelement.getattributenodens.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function getAttributeNodeNS(?string $namespace, string $localName): \DOMAttr|\DOMNameSpaceNode|null {}

		/**
		 * Gets elements by tagname
		 * <p>This function returns a new instance of the class <code>DOMNodeList</code> of all descendant elements with a given tag <code>qualifiedName</code>, in the order in which they are encountered in a preorder traversal of this element tree.</p>
		 * @param string $qualifiedName <p>The tag name. Use <code>&#42;</code> to return all elements within the element tree.</p>
		 * @return DOMNodeList <p>This function returns a new instance of the class <code>DOMNodeList</code> of all matched elements.</p>
		 * @link https://php.net/manual/en/domelement.getelementsbytagname.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function getElementsByTagName(string $qualifiedName): \DOMNodeList {}

		/**
		 * Get elements by namespaceURI and localName
		 * <p>This function fetch all the descendant elements with a given <code>localName</code> and <code>namespace</code>.</p>
		 * @param ?string $namespace <p>The namespace URI of the elements to match on. The special value <code>"&#42;"</code> matches all namespaces. Passing <b><code>null</code></b> matches the empty namespace.</p>
		 * @param string $localName <p>The local name of the elements to match on. The special value <code>"&#42;"</code> matches all local names.</p>
		 * @return DOMNodeList <p>This function returns a new instance of the class <code>DOMNodeList</code> of all matched elements in the order in which they are encountered in a preorder traversal of this element tree.</p>
		 * @link https://php.net/manual/en/domelement.getelementsbytagnamens.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function getElementsByTagNameNS(?string $namespace, string $localName): \DOMNodeList {}

		/**
		 * Get line number for a node
		 * <p>Gets line number for where the node was defined at parse time.</p>
		 * @return int <p>Returns the line number where the node was defined at parse time. If the node was created manually, the return value will be <code>0</code>.</p>
		 * @link https://php.net/manual/en/domnode.getlineno.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		public function getLineNo(): int {}

		/**
		 * Get an XPath for a node
		 * <p>Gets an XPath location path for the node.</p>
		 * @return ?string <p>Returns a <code>string</code> containing the XPath, or <b><code>null</code></b> in case of an error.</p>
		 * @link https://php.net/manual/en/domnode.getnodepath.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function getNodePath(): ?string {}

		/**
		 * Get root node
		 * <p>Get root node.</p>
		 * @param ?array $options <p>This parameter has no effect yet.</p>
		 * @return DOMNode <p>Returns the root node.</p>
		 * @link https://php.net/manual/en/domnode.getrootnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function getRootNode(?array $options = null): \DOMNode {}

		/**
		 * Checks to see if attribute exists
		 * <p>Indicates whether attribute named <code>qualifiedName</code> exists as a member of the element.</p>
		 * @param string $qualifiedName <p>The attribute name.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domelement.hasattribute.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasAttribute(string $qualifiedName): bool {}

		/**
		 * Checks to see if attribute exists
		 * <p>Indicates whether attribute in namespace <code>namespace</code> named <code>localName</code> exists as a member of the element.</p>
		 * @param ?string $namespace <p>The namespace URI.</p>
		 * @param string $localName <p>The local name.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domelement.hasattributens.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasAttributeNS(?string $namespace, string $localName): bool {}

		/**
		 * Checks if node has attributes
		 * <p>This method checks if the node has attributes. The tested node has to be an <b><code>XML_ELEMENT_NODE</code></b>.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.hasattributes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasAttributes(): bool {}

		/**
		 * Checks if node has children
		 * <p>This function checks if the node has children.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.haschildnodes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasChildNodes(): bool {}

		/**
		 * Insert adjacent element
		 * <p>Inserts an element at a relative position given by <code>where</code>.</p>
		 * @param string $where <p></p><ul> <li><code>beforebegin</code> - Insert before the target element.</li> <li><code>afterbegin</code> - Insert as the first child of the target element.</li> <li><code>beforeend</code> - Insert as the last child of the target element.</li> <li><code>afterend</code> - Insert after the target element.</li> </ul>
		 * @param \DOMElement $element <p>The element to insert.</p>
		 * @return ?DOMElement <p>Return <code>DOMElement</code> or <b><code>null</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domelement.insertadjacentelement.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function insertAdjacentElement(string $where, \DOMElement $element): ?\DOMElement {}

		/**
		 * Insert adjacent text
		 * <p>Inserts text at a relative position given by <code>where</code>.</p>
		 * @param string $where <p></p><ul> <li><code>beforebegin</code> - Insert before the target element.</li> <li><code>afterbegin</code> - Insert as the first child of the target element.</li> <li><code>beforeend</code> - Insert as the last child of the target element.</li> <li><code>afterend</code> - Insert after the target element.</li> </ul>
		 * @param string $data <p>The string to insert.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domelement.insertadjacenttext.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function insertAdjacentText(string $where, string $data): void {}

		/**
		 * Adds a new child before a reference node
		 * <p>This function inserts a new node right before the reference node. If you plan to do further modifications on the appended child you must use the returned node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The new node.</p>
		 * @param ?\DOMNode $child <p>The reference node. If not supplied, <code>node</code> is appended to the children.</p>
		 * @return DOMNode|false <p>The inserted node or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.insertbefore.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertBefore(\DOMNode $node, ?\DOMNode $child = null): \DOMNode|false {}

		/**
		 * Checks if the specified namespaceURI is the default namespace or not
		 * <p>Tells whether <code>namespace</code> is the default namespace.</p>
		 * @param string $namespace <p>The namespace URI to look for.</p>
		 * @return bool <p>Return <b><code>true</code></b> if <code>namespace</code> is the default namespace, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isdefaultnamespace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isDefaultNamespace(string $namespace): bool {}

		/**
		 * Checks that both nodes are equal
		 * <p>Checks that both nodes are equal.</p>
		 * @param ?\DOMNode $otherNode <p>The node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if both nodes are equal, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isequalnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function isEqualNode(?\DOMNode $otherNode): bool {}

		/**
		 * Indicates if two nodes are the same node
		 * <p>This function indicates if two nodes are the same node. The comparison is <i>not</i> based on content</p>
		 * @param \DOMNode $otherNode <p>The compared node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issamenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSameNode(\DOMNode $otherNode): bool {}

		/**
		 * Checks if feature is supported for specified version
		 * <p>Checks if the asked <code>feature</code> is supported for the specified <code>version</code>.</p>
		 * @param string $feature <p>The feature to test. See the example of <code>DOMImplementation::hasFeature()</code> for a list of features.</p>
		 * @param string $version <p>The version number of the <code>feature</code> to test.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issupported.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSupported(string $feature, string $version): bool {}

		/**
		 * Gets the namespace URI of the node based on the prefix
		 * <p>Gets the namespace URI of the node based on the <code>prefix</code>.</p>
		 * @param ?string $prefix <p>The prefix to look for. If this parameter is <b><code>null</code></b>, the method will return the default namespace URI, if any.</p>
		 * @return ?string <p>Returns the associated namespace URI or <b><code>null</code></b> if none is found.</p>
		 * @link https://php.net/manual/en/domnode.lookupnamespaceuri.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupNamespaceURI(?string $prefix): ?string {}

		/**
		 * Gets the namespace prefix of the node based on the namespace URI
		 * <p>Gets the namespace prefix of the node based on the namespace URI.</p>
		 * @param string $namespace <p>The namespace URI.</p>
		 * @return ?string <p>The prefix of the namespace or <b><code>null</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.lookupprefix.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupPrefix(string $namespace): ?string {}

		/**
		 * Normalizes the node
		 * <p>Remove empty text nodes and merge adjacent text nodes in this node and all its children.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domnode.normalize.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function normalize(): void {}

		/**
		 * Prepends nodes before the first child node
		 * <p>Prepends one or many <code>nodes</code> to the list of children before the first child node.</p>
		 * @param \DOMNode|string $nodes <p>The nodes to prepend. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domelement.prepend.php
		 * @since PHP 8
		 */
		public function prepend(\DOMNode|string ...$nodes): void {}

		/**
		 * Removes the element
		 * <p>Removes the element.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domelement.remove.php
		 * @since PHP 8
		 */
		public function remove(): void {}

		/**
		 * Removes attribute
		 * <p>Removes attribute named <code>qualifiedName</code> from the element.</p>
		 * @param string $qualifiedName <p>The name of the attribute.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domelement.removeattribute.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function removeAttribute(string $qualifiedName): bool {}

		/**
		 * Removes attribute
		 * <p>Removes attribute <code>localName</code> in namespace <code>namespace</code> from the element.</p>
		 * @param ?string $namespace <p>The namespace URI.</p>
		 * @param string $localName <p>The local name.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domelement.removeattributens.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function removeAttributeNS(?string $namespace, string $localName): void {}

		/**
		 * Removes attribute
		 * <p>Removes attribute <code>attr</code> from the element.</p>
		 * @param \DOMAttr $attr <p>The attribute node.</p>
		 * @return DOMAttr|false <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domelement.removeattributenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function removeAttributeNode(\DOMAttr $attr): \DOMAttr|false {}

		/**
		 * Removes child from list of children
		 * <p>This functions removes a child from a list of children.</p>
		 * @param \DOMNode $child <p>The removed child.</p>
		 * @return DOMNode|false <p>If the child could be removed the function returns the old child or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.removechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function removeChild(\DOMNode $child): \DOMNode|false {}

		/**
		 * Replaces a child
		 * <p>This function replaces the child <code>child</code> with the passed new node. If the <code>node</code> is already a child it will not be added a second time. If the replacement succeeds the old node is returned.</p>
		 * @param \DOMNode $node <p>The new node. It must be a member of the target document, i.e. created by one of the DOMDocument-&gt;createXXX() methods or imported in the document by DOMDocument::importNode.</p>
		 * @param \DOMNode $child <p>The old node.</p>
		 * @return DOMNode|false <p>The old node or <b><code>false</code></b> if an error occur.</p>
		 * @link https://php.net/manual/en/domnode.replacechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceChild(\DOMNode $node, \DOMNode $child): \DOMNode|false {}

		/**
		 * Replace children in element
		 * <p>Replaces the children in the element with new <code>nodes</code>.</p>
		 * @param \DOMNode|string $nodes <p>The nodes replacing the children. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domelement.replacechildren.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function replaceChildren(\DOMNode|string ...$nodes): void {}

		/**
		 * Replaces the element with new nodes
		 * <p>Replaces the element with new <code>nodes</code>.</p>
		 * @param \DOMNode|string $nodes <p>The replacement nodes. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domelement.replacewith.php
		 * @since PHP 8
		 */
		public function replaceWith(\DOMNode|string ...$nodes): void {}

		/**
		 * Adds new or modifies existing attribute
		 * <p>Sets an attribute with name <code>qualifiedName</code> to the given value. If the attribute does not exist, it will be created.</p>
		 * @param string $qualifiedName <p>The name of the attribute.</p>
		 * @param string $value <p>The value of the attribute.</p>
		 * @return DOMAttr|bool <p>The created or modified <code>DOMAttr</code> or <b><code>false</code></b> if an error occurred.</p>
		 * @link https://php.net/manual/en/domelement.setattribute.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function setAttribute(string $qualifiedName, string $value): \DOMAttr|bool {}

		/**
		 * Adds new attribute
		 * <p>Sets an attribute with namespace <code>namespace</code> and name <code>qualifiedName</code> to the given value. If the attribute does not exist, it will be created.</p>
		 * @param ?string $namespace <p>The namespace URI.</p>
		 * @param string $qualifiedName <p>The qualified name of the attribute, as <code>prefix:tagname</code>.</p>
		 * @param string $value <p>The value of the attribute.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domelement.setattributens.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function setAttributeNS(?string $namespace, string $qualifiedName, string $value): void {}

		/**
		 * Adds new attribute node to element
		 * <p>Adds new attribute node <code>attr</code> to element. If an attribute with the same name already exists on the element, that attribute is replaced by <code>attr</code>.</p>
		 * @param \DOMAttr $attr <p>The attribute node.</p>
		 * @return DOMAttr|null|false <p>Returns the old attribute if it has been replaced or <b><code>null</code></b> if there was no old attribute. If a <b><code>DOM_WRONG_DOCUMENT_ERR</code></b> error is raised, and strictErrorChecking is <b><code>false</code></b>, <b><code>false</code></b> is returned.</p>
		 * @link https://php.net/manual/en/domelement.setattributenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function setAttributeNode(\DOMAttr $attr): \DOMAttr|null|false {}

		/**
		 * Adds new attribute node to element
		 * <p>Adds new attribute node <code>attr</code> to element, keeping the namespace into account. If an attribute with the same name already exists on the element, that attribute is replaced by <code>attr</code>.</p>
		 * @param \DOMAttr $attr <p>The attribute node.</p>
		 * @return DOMAttr|null|false <p>Returns the old attribute if it has been replaced or <b><code>null</code></b> if there was no old attribute. If a <b><code>DOM_WRONG_DOCUMENT_ERR</code></b> error is raised, and strictErrorChecking is <b><code>false</code></b>, <b><code>false</code></b> is returned.</p>
		 * @link https://php.net/manual/en/domelement.setattributenodens.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function setAttributeNodeNS(\DOMAttr $attr): \DOMAttr|null|false {}

		/**
		 * Declares the attribute specified by name to be of type ID
		 * <p>Declares the attribute <code>qualifiedName</code> to be of type ID.</p>
		 * @param string $qualifiedName <p>The name of the attribute.</p>
		 * @param bool $isId <p>Set it to <b><code>true</code></b> if you want <code>qualifiedName</code> to be of type ID, <b><code>false</code></b> otherwise.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domelement.setidattribute.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function setIdAttribute(string $qualifiedName, bool $isId): void {}

		/**
		 * Declares the attribute specified by local name and namespace URI to be of type ID
		 * <p>Declares the attribute specified by <code>qualifiedName</code> and <code>namespace</code> to be of type ID.</p>
		 * @param string $namespace <p>The namespace URI of the attribute.</p>
		 * @param string $qualifiedName <p>The local name of the attribute, as <code>prefix:tagname</code>.</p>
		 * @param bool $isId <p>Set it to <b><code>true</code></b> if you want <code>name</code> to be of type ID, <b><code>false</code></b> otherwise.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domelement.setidattributens.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function setIdAttributeNS(string $namespace, string $qualifiedName, bool $isId): void {}

		/**
		 * Declares the attribute specified by node to be of type ID
		 * <p>Declares the attribute specified by <code>attr</code> to be of type ID.</p>
		 * @param \DOMAttr $attr <p>The attribute node.</p>
		 * @param bool $isId <p>Set it to <b><code>true</code></b> if you want <code>name</code> to be of type ID, <b><code>false</code></b> otherwise.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domelement.setidattributenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function setIdAttributeNode(\DOMAttr $attr, bool $isId): void {}

		/**
		 * Toggle attribute
		 * <p>Toggle the attribute.</p>
		 * @param string $qualifiedName <p>The qualified name of the attribute.</p>
		 * @param ?bool $force <p></p><ul> <li>if <b><code>null</code></b>, the function will toggle the attribute.</li> <li>if <b><code>true</code></b>, the function adds the attribute.</li> <li>if <b><code>false</code></b>, the function removes the attribute.</li> </ul>
		 * @return bool <p>Returns <b><code>true</code></b> if the attribute is present after finishing the call, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domelement.toggleattribute.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function toggleAttribute(string $qualifiedName, ?bool $force = null): bool {}
	}

	/**
	 * <p>This interface represents a known entity, either parsed or unparsed, in an XML document.</p>
	 * @link https://php.net/manual/en/class.domentity.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMEntity extends \DOMNode {

		/**
		 * @var ?string <p>The public identifier associated with the entity if specified, and <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/class.domentity.php#domentity.props.publicid
		 */
		public ?string $publicId;

		/**
		 * @var ?string <p>The system identifier associated with the entity if specified, and <b><code>null</code></b> otherwise. This may be an absolute URI or not.</p>
		 * @link https://php.net/manual/en/class.domentity.php#domentity.props.systemid
		 */
		public ?string $systemId;

		/**
		 * @var ?string <p>For unparsed entities, the name of the notation for the entity. For parsed entities, this is <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domentity.php#domentity.props.notationname
		 */
		public ?string $notationName;

		/**
		 * @var ?string <p><i>Deprecated as of PHP 8.4.0</i>. This has always been equal to <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domentity.php#domentity.props.actualencoding
		 */
		public ?string $actualEncoding;

		/**
		 * @var ?string <p><i>Deprecated as of PHP 8.4.0</i>. This has always been equal to <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domentity.php#domentity.props.encoding
		 */
		public ?string $encoding;

		/**
		 * @var ?string <p><i>Deprecated as of PHP 8.4.0</i>. This has always been equal to <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domentity.php#domentity.props.version
		 */
		public ?string $version;

		/**
		 * @var string <p>Returns the most accurate name for the current node type</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var ?string <p>The value of this node, depending on its type. Contrary to the W3C specification, the node value of <code>DOMElement</code> nodes is equal to DOMNode::textContent instead of <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var ?DOMNode <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentnode
		 */
		public ?\DOMNode $parentNode;

		/**
		 * @var ?DOMElement <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentelement
		 */
		public ?\DOMElement $parentElement;

		/**
		 * @var DOMNodeList <p>A <code>DOMNodeList</code> that contains all children of this node. If there are no children, this is an empty <code>DOMNodeList</code>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.childnodes
		 */
		public \DOMNodeList $childNodes;

		/**
		 * @var ?DOMNode <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.firstchild
		 */
		public ?\DOMNode $firstChild;

		/**
		 * @var ?DOMNode <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.lastchild
		 */
		public ?\DOMNode $lastChild;

		/**
		 * @var ?DOMNode <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.previoussibling
		 */
		public ?\DOMNode $previousSibling;

		/**
		 * @var ?DOMNode <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nextsibling
		 */
		public ?\DOMNode $nextSibling;

		/**
		 * @var ?DOMNamedNodeMap <p>A <code>DOMNamedNodeMap</code> containing the attributes of this node (if it is a <code>DOMElement</code>) or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.attributes
		 */
		public ?\DOMNamedNodeMap $attributes;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?DOMDocument <p>The <code>DOMDocument</code> object associated with this node, or <b><code>null</code></b> if this node is a <code>DOMDocument</code></p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.ownerdocument
		 */
		public ?\DOMDocument $ownerDocument;

		/**
		 * @var ?string <p>The namespace URI of this node, or <b><code>null</code></b> if it is unspecified.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var string <p>The namespace prefix of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.prefix
		 */
		public string $prefix;

		/**
		 * @var ?string <p>Returns the local part of the qualified name of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.localname
		 */
		public ?string $localName;

		/**
		 * @var ?string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.baseuri
		 */
		public ?string $baseURI;

		/**
		 * @var string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.textcontent
		 */
		public string $textContent;

		/**
		 * Canonicalize nodes to a string
		 * <p>Canonicalize nodes to a string</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return string|false <p>Returns canonicalized nodes as a string or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14n.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14N(bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): string|false {}

		/**
		 * Canonicalize nodes to a file
		 * <p>Canonicalize nodes to a file.</p>
		 * @param string $uri <p>Path to write the output to.</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return int|false <p>Number of bytes written or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14nfile.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14NFile(string $uri, bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): int|false {}

		/**
		 * Forbids serialization unless serialization methods are implemented in a subclass
		 * <p>Forbids serialization unless serialization methods are implemented in a subclass.</p>
		 * @return array <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.sleep.php
		 * @since PHP 8 >= 8.1.25
		 */
		public function __sleep(): array {}

		/**
		 * Forbids unserialization unless unserialization methods are implemented in a subclass
		 * <p>Forbids unserialization unless unserialization methods are implemented in a subclass.</p>
		 * @return void <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.wakeup.php
		 * @since PHP 8 PHP 8 >= 8.1.25
		 */
		public function __wakeup(): void {}

		/**
		 * Adds new child at the end of the children
		 * <p>This function appends a child to an existing list of children or creates a new list of children. The child can be created with e.g. <code>DOMDocument::createElement()</code>, <code>DOMDocument::createTextNode()</code> etc. or simply by using any other node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The appended child.</p>
		 * @return DOMNode|false <p>The node added or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.appendchild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendChild(\DOMNode $node): \DOMNode|false {}

		/**
		 * Clones a node
		 * <p>Creates a copy of the node.</p>
		 * @param bool $deep <p>Indicates whether to copy all descendant nodes. This parameter is defaulted to <b><code>false</code></b>.</p>
		 * @return DOMNode|false <p>The cloned node.</p>
		 * @link https://php.net/manual/en/domnode.clonenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function cloneNode(bool $deep = false): \DOMNode|false {}

		/**
		 * Compares the position of two nodes
		 * <p>Compares the position of the other node relative to this node.</p>
		 * @param \DOMNode $other <p>The node for which the position should be compared for, relative to this node.</p>
		 * @return int <p>A bitmask of the <b><code>DOMNode::DOCUMENT_POSITION_&#42;</code></b> constants.</p>
		 * @link https://php.net/manual/en/domnode.comparedocumentposition.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function compareDocumentPosition(\DOMNode $other): int {}

		/**
		 * Checks if node contains other node
		 * <p>Checks if node contains <code>other</code> node.</p>
		 * @param \DOMNode|\DOMNameSpaceNode|null $other <p>Node to be checked.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if node contains <code>other</code> node, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.contains.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function contains(\DOMNode|\DOMNameSpaceNode|null $other): bool {}

		/**
		 * Get line number for a node
		 * <p>Gets line number for where the node was defined at parse time.</p>
		 * @return int <p>Returns the line number where the node was defined at parse time. If the node was created manually, the return value will be <code>0</code>.</p>
		 * @link https://php.net/manual/en/domnode.getlineno.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		public function getLineNo(): int {}

		/**
		 * Get an XPath for a node
		 * <p>Gets an XPath location path for the node.</p>
		 * @return ?string <p>Returns a <code>string</code> containing the XPath, or <b><code>null</code></b> in case of an error.</p>
		 * @link https://php.net/manual/en/domnode.getnodepath.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function getNodePath(): ?string {}

		/**
		 * Get root node
		 * <p>Get root node.</p>
		 * @param ?array $options <p>This parameter has no effect yet.</p>
		 * @return DOMNode <p>Returns the root node.</p>
		 * @link https://php.net/manual/en/domnode.getrootnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function getRootNode(?array $options = null): \DOMNode {}

		/**
		 * Checks if node has attributes
		 * <p>This method checks if the node has attributes. The tested node has to be an <b><code>XML_ELEMENT_NODE</code></b>.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.hasattributes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasAttributes(): bool {}

		/**
		 * Checks if node has children
		 * <p>This function checks if the node has children.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.haschildnodes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasChildNodes(): bool {}

		/**
		 * Adds a new child before a reference node
		 * <p>This function inserts a new node right before the reference node. If you plan to do further modifications on the appended child you must use the returned node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The new node.</p>
		 * @param ?\DOMNode $child <p>The reference node. If not supplied, <code>node</code> is appended to the children.</p>
		 * @return DOMNode|false <p>The inserted node or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.insertbefore.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertBefore(\DOMNode $node, ?\DOMNode $child = null): \DOMNode|false {}

		/**
		 * Checks if the specified namespaceURI is the default namespace or not
		 * <p>Tells whether <code>namespace</code> is the default namespace.</p>
		 * @param string $namespace <p>The namespace URI to look for.</p>
		 * @return bool <p>Return <b><code>true</code></b> if <code>namespace</code> is the default namespace, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isdefaultnamespace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isDefaultNamespace(string $namespace): bool {}

		/**
		 * Checks that both nodes are equal
		 * <p>Checks that both nodes are equal.</p>
		 * @param ?\DOMNode $otherNode <p>The node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if both nodes are equal, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isequalnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function isEqualNode(?\DOMNode $otherNode): bool {}

		/**
		 * Indicates if two nodes are the same node
		 * <p>This function indicates if two nodes are the same node. The comparison is <i>not</i> based on content</p>
		 * @param \DOMNode $otherNode <p>The compared node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issamenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSameNode(\DOMNode $otherNode): bool {}

		/**
		 * Checks if feature is supported for specified version
		 * <p>Checks if the asked <code>feature</code> is supported for the specified <code>version</code>.</p>
		 * @param string $feature <p>The feature to test. See the example of <code>DOMImplementation::hasFeature()</code> for a list of features.</p>
		 * @param string $version <p>The version number of the <code>feature</code> to test.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issupported.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSupported(string $feature, string $version): bool {}

		/**
		 * Gets the namespace URI of the node based on the prefix
		 * <p>Gets the namespace URI of the node based on the <code>prefix</code>.</p>
		 * @param ?string $prefix <p>The prefix to look for. If this parameter is <b><code>null</code></b>, the method will return the default namespace URI, if any.</p>
		 * @return ?string <p>Returns the associated namespace URI or <b><code>null</code></b> if none is found.</p>
		 * @link https://php.net/manual/en/domnode.lookupnamespaceuri.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupNamespaceURI(?string $prefix): ?string {}

		/**
		 * Gets the namespace prefix of the node based on the namespace URI
		 * <p>Gets the namespace prefix of the node based on the namespace URI.</p>
		 * @param string $namespace <p>The namespace URI.</p>
		 * @return ?string <p>The prefix of the namespace or <b><code>null</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.lookupprefix.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupPrefix(string $namespace): ?string {}

		/**
		 * Normalizes the node
		 * <p>Remove empty text nodes and merge adjacent text nodes in this node and all its children.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domnode.normalize.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function normalize(): void {}

		/**
		 * Removes child from list of children
		 * <p>This functions removes a child from a list of children.</p>
		 * @param \DOMNode $child <p>The removed child.</p>
		 * @return DOMNode|false <p>If the child could be removed the function returns the old child or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.removechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function removeChild(\DOMNode $child): \DOMNode|false {}

		/**
		 * Replaces a child
		 * <p>This function replaces the child <code>child</code> with the passed new node. If the <code>node</code> is already a child it will not be added a second time. If the replacement succeeds the old node is returned.</p>
		 * @param \DOMNode $node <p>The new node. It must be a member of the target document, i.e. created by one of the DOMDocument-&gt;createXXX() methods or imported in the document by DOMDocument::importNode.</p>
		 * @param \DOMNode $child <p>The old node.</p>
		 * @return DOMNode|false <p>The old node or <b><code>false</code></b> if an error occur.</p>
		 * @link https://php.net/manual/en/domnode.replacechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceChild(\DOMNode $node, \DOMNode $child): \DOMNode|false {}
	}

	/**
	 * @link https://php.net/manual/en/class.domentityreference.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMEntityReference extends \DOMNode {

		/**
		 * @var string <p>Returns the most accurate name for the current node type</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var ?string <p>The value of this node, depending on its type. Contrary to the W3C specification, the node value of <code>DOMElement</code> nodes is equal to DOMNode::textContent instead of <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var ?DOMNode <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentnode
		 */
		public ?\DOMNode $parentNode;

		/**
		 * @var ?DOMElement <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentelement
		 */
		public ?\DOMElement $parentElement;

		/**
		 * @var DOMNodeList <p>A <code>DOMNodeList</code> that contains all children of this node. If there are no children, this is an empty <code>DOMNodeList</code>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.childnodes
		 */
		public \DOMNodeList $childNodes;

		/**
		 * @var ?DOMNode <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.firstchild
		 */
		public ?\DOMNode $firstChild;

		/**
		 * @var ?DOMNode <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.lastchild
		 */
		public ?\DOMNode $lastChild;

		/**
		 * @var ?DOMNode <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.previoussibling
		 */
		public ?\DOMNode $previousSibling;

		/**
		 * @var ?DOMNode <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nextsibling
		 */
		public ?\DOMNode $nextSibling;

		/**
		 * @var ?DOMNamedNodeMap <p>A <code>DOMNamedNodeMap</code> containing the attributes of this node (if it is a <code>DOMElement</code>) or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.attributes
		 */
		public ?\DOMNamedNodeMap $attributes;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?DOMDocument <p>The <code>DOMDocument</code> object associated with this node, or <b><code>null</code></b> if this node is a <code>DOMDocument</code></p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.ownerdocument
		 */
		public ?\DOMDocument $ownerDocument;

		/**
		 * @var ?string <p>The namespace URI of this node, or <b><code>null</code></b> if it is unspecified.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var string <p>The namespace prefix of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.prefix
		 */
		public string $prefix;

		/**
		 * @var ?string <p>Returns the local part of the qualified name of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.localname
		 */
		public ?string $localName;

		/**
		 * @var ?string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.baseuri
		 */
		public ?string $baseURI;

		/**
		 * @var string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.textcontent
		 */
		public string $textContent;

		/**
		 * Canonicalize nodes to a string
		 * <p>Canonicalize nodes to a string</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return string|false <p>Returns canonicalized nodes as a string or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14n.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14N(bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): string|false {}

		/**
		 * Canonicalize nodes to a file
		 * <p>Canonicalize nodes to a file.</p>
		 * @param string $uri <p>Path to write the output to.</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return int|false <p>Number of bytes written or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14nfile.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14NFile(string $uri, bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): int|false {}

		/**
		 * Creates a new DOMEntityReference object
		 * <p>Creates a new <code>DOMEntityReference</code> object.</p>
		 * @param string $name <p>The name of the entity reference.</p>
		 * @return self
		 * @link https://php.net/manual/en/domentityreference.construct.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __construct(string $name) {}

		/**
		 * Forbids serialization unless serialization methods are implemented in a subclass
		 * <p>Forbids serialization unless serialization methods are implemented in a subclass.</p>
		 * @return array <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.sleep.php
		 * @since PHP 8 >= 8.1.25
		 */
		public function __sleep(): array {}

		/**
		 * Forbids unserialization unless unserialization methods are implemented in a subclass
		 * <p>Forbids unserialization unless unserialization methods are implemented in a subclass.</p>
		 * @return void <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.wakeup.php
		 * @since PHP 8 PHP 8 >= 8.1.25
		 */
		public function __wakeup(): void {}

		/**
		 * Adds new child at the end of the children
		 * <p>This function appends a child to an existing list of children or creates a new list of children. The child can be created with e.g. <code>DOMDocument::createElement()</code>, <code>DOMDocument::createTextNode()</code> etc. or simply by using any other node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The appended child.</p>
		 * @return DOMNode|false <p>The node added or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.appendchild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendChild(\DOMNode $node): \DOMNode|false {}

		/**
		 * Clones a node
		 * <p>Creates a copy of the node.</p>
		 * @param bool $deep <p>Indicates whether to copy all descendant nodes. This parameter is defaulted to <b><code>false</code></b>.</p>
		 * @return DOMNode|false <p>The cloned node.</p>
		 * @link https://php.net/manual/en/domnode.clonenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function cloneNode(bool $deep = false): \DOMNode|false {}

		/**
		 * Compares the position of two nodes
		 * <p>Compares the position of the other node relative to this node.</p>
		 * @param \DOMNode $other <p>The node for which the position should be compared for, relative to this node.</p>
		 * @return int <p>A bitmask of the <b><code>DOMNode::DOCUMENT_POSITION_&#42;</code></b> constants.</p>
		 * @link https://php.net/manual/en/domnode.comparedocumentposition.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function compareDocumentPosition(\DOMNode $other): int {}

		/**
		 * Checks if node contains other node
		 * <p>Checks if node contains <code>other</code> node.</p>
		 * @param \DOMNode|\DOMNameSpaceNode|null $other <p>Node to be checked.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if node contains <code>other</code> node, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.contains.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function contains(\DOMNode|\DOMNameSpaceNode|null $other): bool {}

		/**
		 * Get line number for a node
		 * <p>Gets line number for where the node was defined at parse time.</p>
		 * @return int <p>Returns the line number where the node was defined at parse time. If the node was created manually, the return value will be <code>0</code>.</p>
		 * @link https://php.net/manual/en/domnode.getlineno.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		public function getLineNo(): int {}

		/**
		 * Get an XPath for a node
		 * <p>Gets an XPath location path for the node.</p>
		 * @return ?string <p>Returns a <code>string</code> containing the XPath, or <b><code>null</code></b> in case of an error.</p>
		 * @link https://php.net/manual/en/domnode.getnodepath.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function getNodePath(): ?string {}

		/**
		 * Get root node
		 * <p>Get root node.</p>
		 * @param ?array $options <p>This parameter has no effect yet.</p>
		 * @return DOMNode <p>Returns the root node.</p>
		 * @link https://php.net/manual/en/domnode.getrootnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function getRootNode(?array $options = null): \DOMNode {}

		/**
		 * Checks if node has attributes
		 * <p>This method checks if the node has attributes. The tested node has to be an <b><code>XML_ELEMENT_NODE</code></b>.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.hasattributes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasAttributes(): bool {}

		/**
		 * Checks if node has children
		 * <p>This function checks if the node has children.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.haschildnodes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasChildNodes(): bool {}

		/**
		 * Adds a new child before a reference node
		 * <p>This function inserts a new node right before the reference node. If you plan to do further modifications on the appended child you must use the returned node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The new node.</p>
		 * @param ?\DOMNode $child <p>The reference node. If not supplied, <code>node</code> is appended to the children.</p>
		 * @return DOMNode|false <p>The inserted node or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.insertbefore.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertBefore(\DOMNode $node, ?\DOMNode $child = null): \DOMNode|false {}

		/**
		 * Checks if the specified namespaceURI is the default namespace or not
		 * <p>Tells whether <code>namespace</code> is the default namespace.</p>
		 * @param string $namespace <p>The namespace URI to look for.</p>
		 * @return bool <p>Return <b><code>true</code></b> if <code>namespace</code> is the default namespace, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isdefaultnamespace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isDefaultNamespace(string $namespace): bool {}

		/**
		 * Checks that both nodes are equal
		 * <p>Checks that both nodes are equal.</p>
		 * @param ?\DOMNode $otherNode <p>The node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if both nodes are equal, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isequalnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function isEqualNode(?\DOMNode $otherNode): bool {}

		/**
		 * Indicates if two nodes are the same node
		 * <p>This function indicates if two nodes are the same node. The comparison is <i>not</i> based on content</p>
		 * @param \DOMNode $otherNode <p>The compared node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issamenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSameNode(\DOMNode $otherNode): bool {}

		/**
		 * Checks if feature is supported for specified version
		 * <p>Checks if the asked <code>feature</code> is supported for the specified <code>version</code>.</p>
		 * @param string $feature <p>The feature to test. See the example of <code>DOMImplementation::hasFeature()</code> for a list of features.</p>
		 * @param string $version <p>The version number of the <code>feature</code> to test.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issupported.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSupported(string $feature, string $version): bool {}

		/**
		 * Gets the namespace URI of the node based on the prefix
		 * <p>Gets the namespace URI of the node based on the <code>prefix</code>.</p>
		 * @param ?string $prefix <p>The prefix to look for. If this parameter is <b><code>null</code></b>, the method will return the default namespace URI, if any.</p>
		 * @return ?string <p>Returns the associated namespace URI or <b><code>null</code></b> if none is found.</p>
		 * @link https://php.net/manual/en/domnode.lookupnamespaceuri.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupNamespaceURI(?string $prefix): ?string {}

		/**
		 * Gets the namespace prefix of the node based on the namespace URI
		 * <p>Gets the namespace prefix of the node based on the namespace URI.</p>
		 * @param string $namespace <p>The namespace URI.</p>
		 * @return ?string <p>The prefix of the namespace or <b><code>null</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.lookupprefix.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupPrefix(string $namespace): ?string {}

		/**
		 * Normalizes the node
		 * <p>Remove empty text nodes and merge adjacent text nodes in this node and all its children.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domnode.normalize.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function normalize(): void {}

		/**
		 * Removes child from list of children
		 * <p>This functions removes a child from a list of children.</p>
		 * @param \DOMNode $child <p>The removed child.</p>
		 * @return DOMNode|false <p>If the child could be removed the function returns the old child or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.removechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function removeChild(\DOMNode $child): \DOMNode|false {}

		/**
		 * Replaces a child
		 * <p>This function replaces the child <code>child</code> with the passed new node. If the <code>node</code> is already a child it will not be added a second time. If the replacement succeeds the old node is returned.</p>
		 * @param \DOMNode $node <p>The new node. It must be a member of the target document, i.e. created by one of the DOMDocument-&gt;createXXX() methods or imported in the document by DOMDocument::importNode.</p>
		 * @param \DOMNode $child <p>The old node.</p>
		 * @return DOMNode|false <p>The old node or <b><code>false</code></b> if an error occur.</p>
		 * @link https://php.net/manual/en/domnode.replacechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceChild(\DOMNode $node, \DOMNode $child): \DOMNode|false {}
	}

	/**
	 * <p>DOM operations raise exceptions under particular circumstances, i.e., when an operation is impossible to perform for logical reasons.</p>
	 * <p>This class is aliased as <b>Dom\Exception</b> in the Dom namespace.</p>
	 * <p>See also Exceptions.</p>
	 * @link https://php.net/manual/en/class.domexception.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	final class DOMException extends \Exception {

		/**
		 * @var int <p>The exception code</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.code
		 */
		protected int $code;

		/**
		 * @var string <p>The exception message</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.message
		 */
		protected string $message = "";

		/**
		 * @var string <p>The string representation of the stack trace</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.string
		 */
		private string $string = "";

		/**
		 * @var string <p>The filename where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.file
		 */
		protected string $file = "";

		/**
		 * @var int <p>The line where the exception was created</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.line
		 */
		protected int $line;

		/**
		 * @var array <p>The stack trace as an array</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.trace
		 */
		private array $trace = [];

		/**
		 * @var ?Throwable <p>The previously thrown exception</p>
		 * @link https://php.net/manual/en/class.exception.php#exception.props.previous
		 */
		private ?\Throwable $previous = null;

		/**
		 * Clone the exception
		 * <p><code>Exception</code>s cannot be cloned, and attempting to do so will throw an <code>Error</code>.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/exception.clone.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		private function __clone() {}

		/**
		 * Construct the exception
		 * <p>Constructs the Exception.</p>
		 * @param string $message <p>The Exception message to throw.</p>
		 * @param int $code <p>The Exception code.</p>
		 * @param ?\Throwable $previous <p>The previous exception used for the exception chaining.</p>
		 * @return self
		 * @link https://php.net/manual/en/exception.construct.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __construct(string $message = "", int $code = 0, ?\Throwable $previous = null) {}

		/**
		 * String representation of the exception
		 * <p>Returns the <code>string</code> representation of the exception.</p>
		 * @return string <p>Returns the <code>string</code> representation of the exception.</p>
		 * @link https://php.net/manual/en/exception.tostring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __toString(): string {}

		/**
		 * Gets the Exception code
		 * <p>Returns the Exception code.</p>
		 * @return int <p>Returns the exception code as <code>int</code> in <code>Exception</code> but possibly as other type in <code>Exception</code> descendants (for example as <code>string</code> in <code>PDOException</code>).</p>
		 * @link https://php.net/manual/en/exception.getcode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getCode(): int {}

		/**
		 * Gets the file in which the exception was created
		 * <p>Get the name of the file in which the exception was created.</p>
		 * @return string <p>Returns the filename in which the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getfile.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getFile(): string {}

		/**
		 * Gets the line in which the exception was created
		 * <p>Get line number where the exception was created.</p>
		 * @return int <p>Returns the line number where the exception was created.</p>
		 * @link https://php.net/manual/en/exception.getline.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getLine(): int {}

		/**
		 * Gets the Exception message
		 * <p>Returns the Exception message.</p>
		 * @return string <p>Returns the Exception message as a string.</p>
		 * @link https://php.net/manual/en/exception.getmessage.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getMessage(): string {}

		/**
		 * Returns previous Throwable
		 * <p>Returns previous <code>Throwable</code> (which had been passed as the third parameter of <code>Exception::__construct()</code>).</p>
		 * @return ?Throwable <p>Returns the previous <code>Throwable</code> if available or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/exception.getprevious.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		final public function getPrevious(): ?\Throwable {}

		/**
		 * Gets the stack trace
		 * <p>Returns the Exception stack trace.</p>
		 * @return array <p>Returns the Exception stack trace as an <code>array</code>.</p>
		 * @link https://php.net/manual/en/exception.gettrace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTrace(): array {}

		/**
		 * Gets the stack trace as a string
		 * <p>Returns the Exception stack trace as a string.</p>
		 * @return string <p>Returns the Exception stack trace as a string.</p>
		 * @link https://php.net/manual/en/exception.gettraceasstring.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		final public function getTraceAsString(): string {}
	}

	/**
	 * <p>This class provides a number of methods for performing operations that are independent of any particular instance of the document object model.</p>
	 * @link https://php.net/manual/en/class.domimplementation.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMImplementation {

		/**
		 * Creates a DOMDocument object of the specified type with its document element
		 * <p>Creates a <code>DOMDocument</code> object of the specified type with its document element.</p>
		 * @param ?string $namespace <p>The namespace URI of the document element to create.</p>
		 * @param string $qualifiedName <p>The qualified name of the document element to create.</p>
		 * @param ?\DOMDocumentType $doctype <p>The type of document to create or <b><code>null</code></b>.</p>
		 * @return DOMDocument <p>A new <code>DOMDocument</code> object. If <code>namespace</code>, <code>qualifiedName</code>, and <code>doctype</code> are null, the returned <code>DOMDocument</code> is empty with no document element.</p>
		 * @link https://php.net/manual/en/domimplementation.createdocument.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function createDocument(?string $namespace = null, string $qualifiedName = "", ?\DOMDocumentType $doctype = null): \DOMDocument {}

		/**
		 * Creates an empty DOMDocumentType object
		 * <p>Creates an empty <code>DOMDocumentType</code> object. Entity declarations and notations are not made available. Entity reference expansions and default attribute additions do not occur.</p>
		 * @param string $qualifiedName <p>The qualified name of the document type to create.</p>
		 * @param string $publicId <p>The external subset public identifier.</p>
		 * @param string $systemId <p>The external subset system identifier.</p>
		 * @return DOMDocumentType|false <p>A new <code>DOMDocumentType</code> node with its <code>ownerDocument</code> set to <b><code>null</code></b> or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domimplementation.createdocumenttype.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function createDocumentType(string $qualifiedName, string $publicId = "", string $systemId = ""): \DOMDocumentType|false {}

		/**
		 * Test if the DOM implementation implements a specific feature
		 * <p>Test if the DOM implementation implements a specific <code>feature</code>.</p><p>You can find a list of all features in the &#xBB;&#xA0;Conformance section of the DOM specification.</p>
		 * @param string $feature <p>The feature to test.</p>
		 * @param string $version <p>The version number of the <code>feature</code> to test. In level 2, this can be either <code>2.0</code> or <code>1.0</code>.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domimplementation.hasfeature.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasFeature(string $feature, string $version): bool {}
	}

	/**
	 * @link https://php.net/manual/en/class.domnamednodemap.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMNamedNodeMap implements \IteratorAggregate, \Countable {

		/**
		 * @var int <p>The number of nodes in the map. The range of valid child node indices is <code>0</code> to <code>length - 1</code> inclusive.</p>
		 * @link https://php.net/manual/en/class.domnamednodemap.php#domnamednodemap.props.length
		 */
		public int $length;

		/**
		 * Get number of nodes in the map
		 * <p>Gets the number of nodes in the map.</p>
		 * @return int <p>Returns the number of nodes in the map, which is identical to the length property.</p>
		 * @link https://php.net/manual/en/domnamednodemap.count.php
		 * @since PHP 7 >= 7.2.0, PHP 8
		 */
		public function count(): int {}

		/**
		 * Retrieve an external iterator
		 * <p>Returns an external iterator for the named node map.</p>
		 * @return Iterator <p>An instance of an object implementing <code>Iterator</code> or <code>Traversable</code></p>
		 * @link https://php.net/manual/en/domnamednodemap.getiterator.php
		 * @since PHP 8
		 */
		public function getIterator(): \Iterator {}

		/**
		 * Retrieves a node specified by name
		 * <p>Retrieves a node specified by its <code>nodeName</code>.</p>
		 * @param string $qualifiedName <p>The <code>nodeName</code> of the node to retrieve.</p>
		 * @return ?DOMNode <p>A node (of any type) with the specified <code>nodeName</code>, or <b><code>null</code></b> if no node is found.</p>
		 * @link https://php.net/manual/en/domnamednodemap.getnameditem.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function getNamedItem(string $qualifiedName): ?\DOMNode {}

		/**
		 * Retrieves a node specified by local name and namespace URI
		 * <p>Retrieves a node specified by <code>localName</code> and <code>namespace</code>.</p>
		 * @param ?string $namespace <p>The namespace URI of the node to retrieve.</p>
		 * @param string $localName <p>The local name of the node to retrieve.</p>
		 * @return ?DOMNode <p>A node (of any type) with the specified local name and namespace URI, or <b><code>null</code></b> if no node is found.</p>
		 * @link https://php.net/manual/en/domnamednodemap.getnameditemns.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function getNamedItemNS(?string $namespace, string $localName): ?\DOMNode {}

		/**
		 * Retrieves a node specified by index
		 * <p>Retrieves a node specified by <code>index</code> within the <code>DOMNamedNodeMap</code> object.</p>
		 * @param int $index <p>Index into this map.</p>
		 * @return ?DOMNode <p>The node at the <code>index</code>th position in the map, or <b><code>null</code></b> if that is not a valid index (greater than or equal to the number of nodes in this map).</p>
		 * @link https://php.net/manual/en/domnamednodemap.item.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function item(int $index): ?\DOMNode {}
	}

	/**
	 * @link https://php.net/manual/en/class.domnamespacenode.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMNameSpaceNode {

		/**
		 * @var string <p>The qualified name of this node.</p>
		 * @link https://php.net/manual/en/class.domnamespacenode.php#domnamespacenode.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var ?string <p>The namespace URI declared by this node, or <b><code>null</code></b> if the empty namespace.</p>
		 * @link https://php.net/manual/en/class.domnamespacenode.php#domnamespacenode.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var int <p>The type of the node. In this case it returns  <b><code>XML_NAMESPACE_DECL_NODE</code></b> .</p>
		 * @link https://php.net/manual/en/class.domnamespacenode.php#domnamespacenode.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string <p>The namespace prefix declared by this node.</p>
		 * @link https://php.net/manual/en/class.domnamespacenode.php#domnamespacenode.props.prefix
		 */
		public string $prefix;

		/**
		 * @var ?string <p>The local part of the qualified name of this node.</p>
		 * @link https://php.net/manual/en/class.domnamespacenode.php#domnamespacenode.props.localname
		 */
		public ?string $localName;

		/**
		 * @var ?string <p>The namespace URI declared by this node, or <b><code>null</code></b> if it is unspecified.</p>
		 * @link https://php.net/manual/en/class.domnamespacenode.php#domnamespacenode.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var bool <p>Whether the node is connected to a document.</p>
		 * @link https://php.net/manual/en/class.domnamespacenode.php#domnamespacenode.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?DOMDocument <p>The <code>DOMDocument</code> object associated with this node, or <b><code>null</code></b> if this node is a <code>DOMDocument</code></p>
		 * @link https://php.net/manual/en/class.domnamespacenode.php#domnamespacenode.props.ownerdocument
		 */
		public ?\DOMDocument $ownerDocument;

		/**
		 * @var ?DOMNode <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnamespacenode.php#domnamespacenode.props.parentnode
		 */
		public ?\DOMNode $parentNode;

		/**
		 * @var ?DOMElement <p>The parent element of this node. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnamespacenode.php#domnamespacenode.props.parentelement
		 */
		public ?\DOMElement $parentElement;

		/**
		 * Forbids serialization unless serialization methods are implemented in a subclass
		 * <p>Forbids serialization unless serialization methods are implemented in a subclass.</p>
		 * @return array <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnamespacenode.sleep.php
		 * @since PHP 8 >= 8.1.25
		 */
		public function __sleep(): array {}

		/**
		 * Forbids unserialization unless unserialization methods are implemented in a subclass
		 * <p>Forbids unserialization unless unserialization methods are implemented in a subclass.</p>
		 * @return void <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnamespacenode.wakeup.php
		 * @since PHP 8 PHP 8 >= 8.1.25
		 */
		public function __wakeup(): void {}
	}

	/**
	 * @link https://php.net/manual/en/class.domnode.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMNode {

		/**
		 * @var int Set when the other node and reference node are not in the same tree.
		 * @link https://php.net/manual/en/class.domnode.php
		 */
		public const int DOCUMENT_POSITION_DISCONNECTED = 0x1;

		/**
		 * @var int Set when the other node precedes the reference node.
		 * @link https://php.net/manual/en/class.domnode.php
		 */
		public const int DOCUMENT_POSITION_PRECEDING = 0x2;

		/**
		 * @var int Set when the other node follows the reference node.
		 * @link https://php.net/manual/en/class.domnode.php
		 */
		public const int DOCUMENT_POSITION_FOLLOWING = 0x4;

		/**
		 * @var int Set when the other node is an ancestor of the reference node.
		 * @link https://php.net/manual/en/class.domnode.php
		 */
		public const int DOCUMENT_POSITION_CONTAINS = 0x8;

		/**
		 * @var int Set when the other node is a descendant of the reference node.
		 * @link https://php.net/manual/en/class.domnode.php
		 */
		public const int DOCUMENT_POSITION_CONTAINED_BY = 0x10;

		/**
		 * @var int Set when the result depends on implementation-specific behaviour and may not be portable. This may happen with disconnected nodes or with attribute nodes.
		 * @link https://php.net/manual/en/class.domnode.php
		 */
		public const int DOCUMENT_POSITION_IMPLEMENTATION_SPECIFIC = 0x20;

		/**
		 * @var string <p>Returns the most accurate name for the current node type</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var ?string <p>The value of this node, depending on its type. Contrary to the W3C specification, the node value of <code>DOMElement</code> nodes is equal to DOMNode::textContent instead of <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var ?DOMNode <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentnode
		 */
		public ?\DOMNode $parentNode;

		/**
		 * @var ?DOMElement <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentelement
		 */
		public ?\DOMElement $parentElement;

		/**
		 * @var DOMNodeList <p>A <code>DOMNodeList</code> that contains all children of this node. If there are no children, this is an empty <code>DOMNodeList</code>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.childnodes
		 */
		public \DOMNodeList $childNodes;

		/**
		 * @var ?DOMNode <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.firstchild
		 */
		public ?\DOMNode $firstChild;

		/**
		 * @var ?DOMNode <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.lastchild
		 */
		public ?\DOMNode $lastChild;

		/**
		 * @var ?DOMNode <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.previoussibling
		 */
		public ?\DOMNode $previousSibling;

		/**
		 * @var ?DOMNode <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nextsibling
		 */
		public ?\DOMNode $nextSibling;

		/**
		 * @var ?DOMNamedNodeMap <p>A <code>DOMNamedNodeMap</code> containing the attributes of this node (if it is a <code>DOMElement</code>) or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.attributes
		 */
		public ?\DOMNamedNodeMap $attributes;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?DOMDocument <p>The <code>DOMDocument</code> object associated with this node, or <b><code>null</code></b> if this node is a <code>DOMDocument</code></p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.ownerdocument
		 */
		public ?\DOMDocument $ownerDocument;

		/**
		 * @var ?string <p>The namespace URI of this node, or <b><code>null</code></b> if it is unspecified.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var string <p>The namespace prefix of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.prefix
		 */
		public string $prefix;

		/**
		 * @var ?string <p>Returns the local part of the qualified name of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.localname
		 */
		public ?string $localName;

		/**
		 * @var ?string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.baseuri
		 */
		public ?string $baseURI;

		/**
		 * @var string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.textcontent
		 */
		public string $textContent;

		/**
		 * Canonicalize nodes to a string
		 * <p>Canonicalize nodes to a string</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return string|false <p>Returns canonicalized nodes as a string or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14n.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14N(bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): string|false {}

		/**
		 * Canonicalize nodes to a file
		 * <p>Canonicalize nodes to a file.</p>
		 * @param string $uri <p>Path to write the output to.</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return int|false <p>Number of bytes written or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14nfile.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14NFile(string $uri, bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): int|false {}

		/**
		 * Forbids serialization unless serialization methods are implemented in a subclass
		 * <p>Forbids serialization unless serialization methods are implemented in a subclass.</p>
		 * @return array <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.sleep.php
		 * @since PHP 8 >= 8.1.25
		 */
		public function __sleep(): array {}

		/**
		 * Forbids unserialization unless unserialization methods are implemented in a subclass
		 * <p>Forbids unserialization unless unserialization methods are implemented in a subclass.</p>
		 * @return void <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.wakeup.php
		 * @since PHP 8 PHP 8 >= 8.1.25
		 */
		public function __wakeup(): void {}

		/**
		 * Adds new child at the end of the children
		 * <p>This function appends a child to an existing list of children or creates a new list of children. The child can be created with e.g. <code>DOMDocument::createElement()</code>, <code>DOMDocument::createTextNode()</code> etc. or simply by using any other node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The appended child.</p>
		 * @return DOMNode|false <p>The node added or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.appendchild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendChild(\DOMNode $node): \DOMNode|false {}

		/**
		 * Clones a node
		 * <p>Creates a copy of the node.</p>
		 * @param bool $deep <p>Indicates whether to copy all descendant nodes. This parameter is defaulted to <b><code>false</code></b>.</p>
		 * @return DOMNode|false <p>The cloned node.</p>
		 * @link https://php.net/manual/en/domnode.clonenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function cloneNode(bool $deep = false): \DOMNode|false {}

		/**
		 * Compares the position of two nodes
		 * <p>Compares the position of the other node relative to this node.</p>
		 * @param \DOMNode $other <p>The node for which the position should be compared for, relative to this node.</p>
		 * @return int <p>A bitmask of the <b><code>DOMNode::DOCUMENT_POSITION_&#42;</code></b> constants.</p>
		 * @link https://php.net/manual/en/domnode.comparedocumentposition.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function compareDocumentPosition(\DOMNode $other): int {}

		/**
		 * Checks if node contains other node
		 * <p>Checks if node contains <code>other</code> node.</p>
		 * @param \DOMNode|\DOMNameSpaceNode|null $other <p>Node to be checked.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if node contains <code>other</code> node, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.contains.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function contains(\DOMNode|\DOMNameSpaceNode|null $other): bool {}

		/**
		 * Get line number for a node
		 * <p>Gets line number for where the node was defined at parse time.</p>
		 * @return int <p>Returns the line number where the node was defined at parse time. If the node was created manually, the return value will be <code>0</code>.</p>
		 * @link https://php.net/manual/en/domnode.getlineno.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		public function getLineNo(): int {}

		/**
		 * Get an XPath for a node
		 * <p>Gets an XPath location path for the node.</p>
		 * @return ?string <p>Returns a <code>string</code> containing the XPath, or <b><code>null</code></b> in case of an error.</p>
		 * @link https://php.net/manual/en/domnode.getnodepath.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function getNodePath(): ?string {}

		/**
		 * Get root node
		 * <p>Get root node.</p>
		 * @param ?array $options <p>This parameter has no effect yet.</p>
		 * @return DOMNode <p>Returns the root node.</p>
		 * @link https://php.net/manual/en/domnode.getrootnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function getRootNode(?array $options = null): \DOMNode {}

		/**
		 * Checks if node has attributes
		 * <p>This method checks if the node has attributes. The tested node has to be an <b><code>XML_ELEMENT_NODE</code></b>.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.hasattributes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasAttributes(): bool {}

		/**
		 * Checks if node has children
		 * <p>This function checks if the node has children.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.haschildnodes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasChildNodes(): bool {}

		/**
		 * Adds a new child before a reference node
		 * <p>This function inserts a new node right before the reference node. If you plan to do further modifications on the appended child you must use the returned node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The new node.</p>
		 * @param ?\DOMNode $child <p>The reference node. If not supplied, <code>node</code> is appended to the children.</p>
		 * @return DOMNode|false <p>The inserted node or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.insertbefore.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertBefore(\DOMNode $node, ?\DOMNode $child = null): \DOMNode|false {}

		/**
		 * Checks if the specified namespaceURI is the default namespace or not
		 * <p>Tells whether <code>namespace</code> is the default namespace.</p>
		 * @param string $namespace <p>The namespace URI to look for.</p>
		 * @return bool <p>Return <b><code>true</code></b> if <code>namespace</code> is the default namespace, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isdefaultnamespace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isDefaultNamespace(string $namespace): bool {}

		/**
		 * Checks that both nodes are equal
		 * <p>Checks that both nodes are equal.</p>
		 * @param ?\DOMNode $otherNode <p>The node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if both nodes are equal, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isequalnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function isEqualNode(?\DOMNode $otherNode): bool {}

		/**
		 * Indicates if two nodes are the same node
		 * <p>This function indicates if two nodes are the same node. The comparison is <i>not</i> based on content</p>
		 * @param \DOMNode $otherNode <p>The compared node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issamenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSameNode(\DOMNode $otherNode): bool {}

		/**
		 * Checks if feature is supported for specified version
		 * <p>Checks if the asked <code>feature</code> is supported for the specified <code>version</code>.</p>
		 * @param string $feature <p>The feature to test. See the example of <code>DOMImplementation::hasFeature()</code> for a list of features.</p>
		 * @param string $version <p>The version number of the <code>feature</code> to test.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issupported.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSupported(string $feature, string $version): bool {}

		/**
		 * Gets the namespace URI of the node based on the prefix
		 * <p>Gets the namespace URI of the node based on the <code>prefix</code>.</p>
		 * @param ?string $prefix <p>The prefix to look for. If this parameter is <b><code>null</code></b>, the method will return the default namespace URI, if any.</p>
		 * @return ?string <p>Returns the associated namespace URI or <b><code>null</code></b> if none is found.</p>
		 * @link https://php.net/manual/en/domnode.lookupnamespaceuri.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupNamespaceURI(?string $prefix): ?string {}

		/**
		 * Gets the namespace prefix of the node based on the namespace URI
		 * <p>Gets the namespace prefix of the node based on the namespace URI.</p>
		 * @param string $namespace <p>The namespace URI.</p>
		 * @return ?string <p>The prefix of the namespace or <b><code>null</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.lookupprefix.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupPrefix(string $namespace): ?string {}

		/**
		 * Normalizes the node
		 * <p>Remove empty text nodes and merge adjacent text nodes in this node and all its children.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domnode.normalize.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function normalize(): void {}

		/**
		 * Removes child from list of children
		 * <p>This functions removes a child from a list of children.</p>
		 * @param \DOMNode $child <p>The removed child.</p>
		 * @return DOMNode|false <p>If the child could be removed the function returns the old child or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.removechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function removeChild(\DOMNode $child): \DOMNode|false {}

		/**
		 * Replaces a child
		 * <p>This function replaces the child <code>child</code> with the passed new node. If the <code>node</code> is already a child it will not be added a second time. If the replacement succeeds the old node is returned.</p>
		 * @param \DOMNode $node <p>The new node. It must be a member of the target document, i.e. created by one of the DOMDocument-&gt;createXXX() methods or imported in the document by DOMDocument::importNode.</p>
		 * @param \DOMNode $child <p>The old node.</p>
		 * @return DOMNode|false <p>The old node or <b><code>false</code></b> if an error occur.</p>
		 * @link https://php.net/manual/en/domnode.replacechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceChild(\DOMNode $node, \DOMNode $child): \DOMNode|false {}
	}

	/**
	 * <p>Represents a live list of nodes.</p>
	 * @link https://php.net/manual/en/class.domnodelist.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMNodeList implements \IteratorAggregate, \Countable {

		/**
		 * @var int <p>The number of nodes in the list. The range of valid child node indices is 0 to <code>length - 1</code> inclusive.</p>
		 * @link https://php.net/manual/en/class.domnodelist.php#domnodelist.props.length
		 */
		public int $length;

		/**
		 * Get number of nodes in the list
		 * <p>Gets the number of nodes in the list.</p>
		 * @return int <p>Returns the number of nodes in the list, which is identical to the length property.</p>
		 * @link https://php.net/manual/en/domnodelist.count.php
		 * @since PHP 7 >= 7.2.0, PHP 8
		 */
		public function count(): int {}

		/**
		 * Retrieve an external iterator
		 * <p>Returns an external iterator for the node list.</p>
		 * @return Iterator <p>An instance of an object implementing <code>Iterator</code> or <code>Traversable</code></p>
		 * @link https://php.net/manual/en/domnodelist.getiterator.php
		 * @since PHP 8
		 */
		public function getIterator(): \Iterator {}

		/**
		 * Retrieves a node specified by index
		 * <p>Retrieves a node specified by <code>index</code> within the <code>DOMNodeList</code> object.</p><p>If you need to know the number of nodes in the collection, use the <code>length</code> property of the <code>DOMNodeList</code> object.</p>
		 * @param int $index <p>Index of the node into the collection.</p>
		 * @return DOMElement|DOMNode|DOMNameSpaceNode|null <p>The node at the <code>index</code>th position in the <code>DOMNodeList</code>, or <b><code>null</code></b> if that is not a valid index.</p>
		 * @link https://php.net/manual/en/domnodelist.item.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function item(int $index): \DOMElement|\DOMNode|\DOMNameSpaceNode|null {}
	}

	/**
	 * @link https://php.net/manual/en/class.domnotation.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMNotation extends \DOMNode {

		/**
		 * @var string The public identifier associated with the notation.
		 * @link https://php.net/manual/en/class.domnotation.php#domnotation.props.publicid
		 */
		public string $publicId;

		/**
		 * @var string The system identifier associated with the notation.
		 * @link https://php.net/manual/en/class.domnotation.php#domnotation.props.systemid
		 */
		public string $systemId;

		/**
		 * @var string <p>Returns the most accurate name for the current node type</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var ?string <p>The value of this node, depending on its type. Contrary to the W3C specification, the node value of <code>DOMElement</code> nodes is equal to DOMNode::textContent instead of <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var ?DOMNode <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentnode
		 */
		public ?\DOMNode $parentNode;

		/**
		 * @var ?DOMElement <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentelement
		 */
		public ?\DOMElement $parentElement;

		/**
		 * @var DOMNodeList <p>A <code>DOMNodeList</code> that contains all children of this node. If there are no children, this is an empty <code>DOMNodeList</code>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.childnodes
		 */
		public \DOMNodeList $childNodes;

		/**
		 * @var ?DOMNode <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.firstchild
		 */
		public ?\DOMNode $firstChild;

		/**
		 * @var ?DOMNode <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.lastchild
		 */
		public ?\DOMNode $lastChild;

		/**
		 * @var ?DOMNode <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.previoussibling
		 */
		public ?\DOMNode $previousSibling;

		/**
		 * @var ?DOMNode <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nextsibling
		 */
		public ?\DOMNode $nextSibling;

		/**
		 * @var ?DOMNamedNodeMap <p>A <code>DOMNamedNodeMap</code> containing the attributes of this node (if it is a <code>DOMElement</code>) or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.attributes
		 */
		public ?\DOMNamedNodeMap $attributes;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?DOMDocument <p>The <code>DOMDocument</code> object associated with this node, or <b><code>null</code></b> if this node is a <code>DOMDocument</code></p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.ownerdocument
		 */
		public ?\DOMDocument $ownerDocument;

		/**
		 * @var ?string <p>The namespace URI of this node, or <b><code>null</code></b> if it is unspecified.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var string <p>The namespace prefix of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.prefix
		 */
		public string $prefix;

		/**
		 * @var ?string <p>Returns the local part of the qualified name of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.localname
		 */
		public ?string $localName;

		/**
		 * @var ?string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.baseuri
		 */
		public ?string $baseURI;

		/**
		 * @var string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.textcontent
		 */
		public string $textContent;

		/**
		 * Canonicalize nodes to a string
		 * <p>Canonicalize nodes to a string</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return string|false <p>Returns canonicalized nodes as a string or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14n.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14N(bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): string|false {}

		/**
		 * Canonicalize nodes to a file
		 * <p>Canonicalize nodes to a file.</p>
		 * @param string $uri <p>Path to write the output to.</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return int|false <p>Number of bytes written or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14nfile.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14NFile(string $uri, bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): int|false {}

		/**
		 * Forbids serialization unless serialization methods are implemented in a subclass
		 * <p>Forbids serialization unless serialization methods are implemented in a subclass.</p>
		 * @return array <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.sleep.php
		 * @since PHP 8 >= 8.1.25
		 */
		public function __sleep(): array {}

		/**
		 * Forbids unserialization unless unserialization methods are implemented in a subclass
		 * <p>Forbids unserialization unless unserialization methods are implemented in a subclass.</p>
		 * @return void <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.wakeup.php
		 * @since PHP 8 PHP 8 >= 8.1.25
		 */
		public function __wakeup(): void {}

		/**
		 * Adds new child at the end of the children
		 * <p>This function appends a child to an existing list of children or creates a new list of children. The child can be created with e.g. <code>DOMDocument::createElement()</code>, <code>DOMDocument::createTextNode()</code> etc. or simply by using any other node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The appended child.</p>
		 * @return DOMNode|false <p>The node added or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.appendchild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendChild(\DOMNode $node): \DOMNode|false {}

		/**
		 * Clones a node
		 * <p>Creates a copy of the node.</p>
		 * @param bool $deep <p>Indicates whether to copy all descendant nodes. This parameter is defaulted to <b><code>false</code></b>.</p>
		 * @return DOMNode|false <p>The cloned node.</p>
		 * @link https://php.net/manual/en/domnode.clonenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function cloneNode(bool $deep = false): \DOMNode|false {}

		/**
		 * Compares the position of two nodes
		 * <p>Compares the position of the other node relative to this node.</p>
		 * @param \DOMNode $other <p>The node for which the position should be compared for, relative to this node.</p>
		 * @return int <p>A bitmask of the <b><code>DOMNode::DOCUMENT_POSITION_&#42;</code></b> constants.</p>
		 * @link https://php.net/manual/en/domnode.comparedocumentposition.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function compareDocumentPosition(\DOMNode $other): int {}

		/**
		 * Checks if node contains other node
		 * <p>Checks if node contains <code>other</code> node.</p>
		 * @param \DOMNode|\DOMNameSpaceNode|null $other <p>Node to be checked.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if node contains <code>other</code> node, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.contains.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function contains(\DOMNode|\DOMNameSpaceNode|null $other): bool {}

		/**
		 * Get line number for a node
		 * <p>Gets line number for where the node was defined at parse time.</p>
		 * @return int <p>Returns the line number where the node was defined at parse time. If the node was created manually, the return value will be <code>0</code>.</p>
		 * @link https://php.net/manual/en/domnode.getlineno.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		public function getLineNo(): int {}

		/**
		 * Get an XPath for a node
		 * <p>Gets an XPath location path for the node.</p>
		 * @return ?string <p>Returns a <code>string</code> containing the XPath, or <b><code>null</code></b> in case of an error.</p>
		 * @link https://php.net/manual/en/domnode.getnodepath.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function getNodePath(): ?string {}

		/**
		 * Get root node
		 * <p>Get root node.</p>
		 * @param ?array $options <p>This parameter has no effect yet.</p>
		 * @return DOMNode <p>Returns the root node.</p>
		 * @link https://php.net/manual/en/domnode.getrootnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function getRootNode(?array $options = null): \DOMNode {}

		/**
		 * Checks if node has attributes
		 * <p>This method checks if the node has attributes. The tested node has to be an <b><code>XML_ELEMENT_NODE</code></b>.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.hasattributes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasAttributes(): bool {}

		/**
		 * Checks if node has children
		 * <p>This function checks if the node has children.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.haschildnodes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasChildNodes(): bool {}

		/**
		 * Adds a new child before a reference node
		 * <p>This function inserts a new node right before the reference node. If you plan to do further modifications on the appended child you must use the returned node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The new node.</p>
		 * @param ?\DOMNode $child <p>The reference node. If not supplied, <code>node</code> is appended to the children.</p>
		 * @return DOMNode|false <p>The inserted node or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.insertbefore.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertBefore(\DOMNode $node, ?\DOMNode $child = null): \DOMNode|false {}

		/**
		 * Checks if the specified namespaceURI is the default namespace or not
		 * <p>Tells whether <code>namespace</code> is the default namespace.</p>
		 * @param string $namespace <p>The namespace URI to look for.</p>
		 * @return bool <p>Return <b><code>true</code></b> if <code>namespace</code> is the default namespace, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isdefaultnamespace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isDefaultNamespace(string $namespace): bool {}

		/**
		 * Checks that both nodes are equal
		 * <p>Checks that both nodes are equal.</p>
		 * @param ?\DOMNode $otherNode <p>The node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if both nodes are equal, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isequalnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function isEqualNode(?\DOMNode $otherNode): bool {}

		/**
		 * Indicates if two nodes are the same node
		 * <p>This function indicates if two nodes are the same node. The comparison is <i>not</i> based on content</p>
		 * @param \DOMNode $otherNode <p>The compared node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issamenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSameNode(\DOMNode $otherNode): bool {}

		/**
		 * Checks if feature is supported for specified version
		 * <p>Checks if the asked <code>feature</code> is supported for the specified <code>version</code>.</p>
		 * @param string $feature <p>The feature to test. See the example of <code>DOMImplementation::hasFeature()</code> for a list of features.</p>
		 * @param string $version <p>The version number of the <code>feature</code> to test.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issupported.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSupported(string $feature, string $version): bool {}

		/**
		 * Gets the namespace URI of the node based on the prefix
		 * <p>Gets the namespace URI of the node based on the <code>prefix</code>.</p>
		 * @param ?string $prefix <p>The prefix to look for. If this parameter is <b><code>null</code></b>, the method will return the default namespace URI, if any.</p>
		 * @return ?string <p>Returns the associated namespace URI or <b><code>null</code></b> if none is found.</p>
		 * @link https://php.net/manual/en/domnode.lookupnamespaceuri.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupNamespaceURI(?string $prefix): ?string {}

		/**
		 * Gets the namespace prefix of the node based on the namespace URI
		 * <p>Gets the namespace prefix of the node based on the namespace URI.</p>
		 * @param string $namespace <p>The namespace URI.</p>
		 * @return ?string <p>The prefix of the namespace or <b><code>null</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.lookupprefix.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupPrefix(string $namespace): ?string {}

		/**
		 * Normalizes the node
		 * <p>Remove empty text nodes and merge adjacent text nodes in this node and all its children.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domnode.normalize.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function normalize(): void {}

		/**
		 * Removes child from list of children
		 * <p>This functions removes a child from a list of children.</p>
		 * @param \DOMNode $child <p>The removed child.</p>
		 * @return DOMNode|false <p>If the child could be removed the function returns the old child or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.removechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function removeChild(\DOMNode $child): \DOMNode|false {}

		/**
		 * Replaces a child
		 * <p>This function replaces the child <code>child</code> with the passed new node. If the <code>node</code> is already a child it will not be added a second time. If the replacement succeeds the old node is returned.</p>
		 * @param \DOMNode $node <p>The new node. It must be a member of the target document, i.e. created by one of the DOMDocument-&gt;createXXX() methods or imported in the document by DOMDocument::importNode.</p>
		 * @param \DOMNode $child <p>The old node.</p>
		 * @return DOMNode|false <p>The old node or <b><code>false</code></b> if an error occur.</p>
		 * @link https://php.net/manual/en/domnode.replacechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceChild(\DOMNode $node, \DOMNode $child): \DOMNode|false {}
	}

	/**
	 * @link https://php.net/manual/en/class.domparentnode.php
	 * @since PHP 8
	 */
	interface DOMParentNode {

		/**
		 * Appends nodes after the last child node
		 * <p>Appends one or many <code>nodes</code> to the list of children after the last child node.</p>
		 * @param \DOMNode|string $nodes <p>The nodes to append. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domparentnode.append.php
		 * @since PHP 8
		 */
		public function append(\DOMNode|string ...$nodes): void;

		/**
		 * Prepends nodes before the first child node
		 * <p>Prepends one or many <code>nodes</code> to the list of children before the first child node.</p>
		 * @param \DOMNode|string $nodes <p>The nodes to prepend. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domparentnode.prepend.php
		 * @since PHP 8
		 */
		public function prepend(\DOMNode|string ...$nodes): void;

		/**
		 * Replace children in node
		 * <p>Replace children in node.</p>
		 * @param \DOMNode|string $nodes <p>The nodes replacing the children. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domparentnode.replacechildren.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function replaceChildren(\DOMNode|string ...$nodes): void;
	}

	/**
	 * <p>This represents a processing instruction (PI) node. These are meant to indicate data areas meant for processing by specific applications.</p>
	 * @link https://php.net/manual/en/class.domprocessinginstruction.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMProcessingInstruction extends \DOMNode {

		/**
		 * @var string A string representing to what application the data is intended for.
		 * @link https://php.net/manual/en/class.domprocessinginstruction.php#domprocessinginstruction.props.target
		 */
		public string $target;

		/**
		 * @var string Application-specific data.
		 * @link https://php.net/manual/en/class.domprocessinginstruction.php#domprocessinginstruction.props.data
		 */
		public string $data;

		/**
		 * @var string <p>Returns the most accurate name for the current node type</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var ?string <p>The value of this node, depending on its type. Contrary to the W3C specification, the node value of <code>DOMElement</code> nodes is equal to DOMNode::textContent instead of <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var ?DOMNode <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentnode
		 */
		public ?\DOMNode $parentNode;

		/**
		 * @var ?DOMElement <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentelement
		 */
		public ?\DOMElement $parentElement;

		/**
		 * @var DOMNodeList <p>A <code>DOMNodeList</code> that contains all children of this node. If there are no children, this is an empty <code>DOMNodeList</code>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.childnodes
		 */
		public \DOMNodeList $childNodes;

		/**
		 * @var ?DOMNode <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.firstchild
		 */
		public ?\DOMNode $firstChild;

		/**
		 * @var ?DOMNode <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.lastchild
		 */
		public ?\DOMNode $lastChild;

		/**
		 * @var ?DOMNode <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.previoussibling
		 */
		public ?\DOMNode $previousSibling;

		/**
		 * @var ?DOMNode <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nextsibling
		 */
		public ?\DOMNode $nextSibling;

		/**
		 * @var ?DOMNamedNodeMap <p>A <code>DOMNamedNodeMap</code> containing the attributes of this node (if it is a <code>DOMElement</code>) or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.attributes
		 */
		public ?\DOMNamedNodeMap $attributes;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?DOMDocument <p>The <code>DOMDocument</code> object associated with this node, or <b><code>null</code></b> if this node is a <code>DOMDocument</code></p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.ownerdocument
		 */
		public ?\DOMDocument $ownerDocument;

		/**
		 * @var ?string <p>The namespace URI of this node, or <b><code>null</code></b> if it is unspecified.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var string <p>The namespace prefix of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.prefix
		 */
		public string $prefix;

		/**
		 * @var ?string <p>Returns the local part of the qualified name of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.localname
		 */
		public ?string $localName;

		/**
		 * @var ?string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.baseuri
		 */
		public ?string $baseURI;

		/**
		 * @var string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.textcontent
		 */
		public string $textContent;

		/**
		 * Canonicalize nodes to a string
		 * <p>Canonicalize nodes to a string</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return string|false <p>Returns canonicalized nodes as a string or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14n.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14N(bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): string|false {}

		/**
		 * Canonicalize nodes to a file
		 * <p>Canonicalize nodes to a file.</p>
		 * @param string $uri <p>Path to write the output to.</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return int|false <p>Number of bytes written or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14nfile.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14NFile(string $uri, bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): int|false {}

		/**
		 * Creates a new DOMProcessingInstruction object
		 * <p>Creates a new <code>DOMProcessingInstruction</code> object. This object is read only. It may be appended to a document, but additional nodes may not be appended to this node until the node is associated with a document. To create a writeable node, use DOMDocument::createProcessingInstruction.</p>
		 * @param string $name <p>The tag name of the processing instruction.</p>
		 * @param string $value <p>The value of the processing instruction.</p>
		 * @return self
		 * @link https://php.net/manual/en/domprocessinginstruction.construct.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __construct(string $name, string $value = "") {}

		/**
		 * Forbids serialization unless serialization methods are implemented in a subclass
		 * <p>Forbids serialization unless serialization methods are implemented in a subclass.</p>
		 * @return array <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.sleep.php
		 * @since PHP 8 >= 8.1.25
		 */
		public function __sleep(): array {}

		/**
		 * Forbids unserialization unless unserialization methods are implemented in a subclass
		 * <p>Forbids unserialization unless unserialization methods are implemented in a subclass.</p>
		 * @return void <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.wakeup.php
		 * @since PHP 8 PHP 8 >= 8.1.25
		 */
		public function __wakeup(): void {}

		/**
		 * Adds new child at the end of the children
		 * <p>This function appends a child to an existing list of children or creates a new list of children. The child can be created with e.g. <code>DOMDocument::createElement()</code>, <code>DOMDocument::createTextNode()</code> etc. or simply by using any other node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The appended child.</p>
		 * @return DOMNode|false <p>The node added or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.appendchild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendChild(\DOMNode $node): \DOMNode|false {}

		/**
		 * Clones a node
		 * <p>Creates a copy of the node.</p>
		 * @param bool $deep <p>Indicates whether to copy all descendant nodes. This parameter is defaulted to <b><code>false</code></b>.</p>
		 * @return DOMNode|false <p>The cloned node.</p>
		 * @link https://php.net/manual/en/domnode.clonenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function cloneNode(bool $deep = false): \DOMNode|false {}

		/**
		 * Compares the position of two nodes
		 * <p>Compares the position of the other node relative to this node.</p>
		 * @param \DOMNode $other <p>The node for which the position should be compared for, relative to this node.</p>
		 * @return int <p>A bitmask of the <b><code>DOMNode::DOCUMENT_POSITION_&#42;</code></b> constants.</p>
		 * @link https://php.net/manual/en/domnode.comparedocumentposition.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function compareDocumentPosition(\DOMNode $other): int {}

		/**
		 * Checks if node contains other node
		 * <p>Checks if node contains <code>other</code> node.</p>
		 * @param \DOMNode|\DOMNameSpaceNode|null $other <p>Node to be checked.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if node contains <code>other</code> node, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.contains.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function contains(\DOMNode|\DOMNameSpaceNode|null $other): bool {}

		/**
		 * Get line number for a node
		 * <p>Gets line number for where the node was defined at parse time.</p>
		 * @return int <p>Returns the line number where the node was defined at parse time. If the node was created manually, the return value will be <code>0</code>.</p>
		 * @link https://php.net/manual/en/domnode.getlineno.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		public function getLineNo(): int {}

		/**
		 * Get an XPath for a node
		 * <p>Gets an XPath location path for the node.</p>
		 * @return ?string <p>Returns a <code>string</code> containing the XPath, or <b><code>null</code></b> in case of an error.</p>
		 * @link https://php.net/manual/en/domnode.getnodepath.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function getNodePath(): ?string {}

		/**
		 * Get root node
		 * <p>Get root node.</p>
		 * @param ?array $options <p>This parameter has no effect yet.</p>
		 * @return DOMNode <p>Returns the root node.</p>
		 * @link https://php.net/manual/en/domnode.getrootnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function getRootNode(?array $options = null): \DOMNode {}

		/**
		 * Checks if node has attributes
		 * <p>This method checks if the node has attributes. The tested node has to be an <b><code>XML_ELEMENT_NODE</code></b>.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.hasattributes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasAttributes(): bool {}

		/**
		 * Checks if node has children
		 * <p>This function checks if the node has children.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.haschildnodes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasChildNodes(): bool {}

		/**
		 * Adds a new child before a reference node
		 * <p>This function inserts a new node right before the reference node. If you plan to do further modifications on the appended child you must use the returned node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The new node.</p>
		 * @param ?\DOMNode $child <p>The reference node. If not supplied, <code>node</code> is appended to the children.</p>
		 * @return DOMNode|false <p>The inserted node or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.insertbefore.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertBefore(\DOMNode $node, ?\DOMNode $child = null): \DOMNode|false {}

		/**
		 * Checks if the specified namespaceURI is the default namespace or not
		 * <p>Tells whether <code>namespace</code> is the default namespace.</p>
		 * @param string $namespace <p>The namespace URI to look for.</p>
		 * @return bool <p>Return <b><code>true</code></b> if <code>namespace</code> is the default namespace, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isdefaultnamespace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isDefaultNamespace(string $namespace): bool {}

		/**
		 * Checks that both nodes are equal
		 * <p>Checks that both nodes are equal.</p>
		 * @param ?\DOMNode $otherNode <p>The node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if both nodes are equal, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isequalnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function isEqualNode(?\DOMNode $otherNode): bool {}

		/**
		 * Indicates if two nodes are the same node
		 * <p>This function indicates if two nodes are the same node. The comparison is <i>not</i> based on content</p>
		 * @param \DOMNode $otherNode <p>The compared node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issamenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSameNode(\DOMNode $otherNode): bool {}

		/**
		 * Checks if feature is supported for specified version
		 * <p>Checks if the asked <code>feature</code> is supported for the specified <code>version</code>.</p>
		 * @param string $feature <p>The feature to test. See the example of <code>DOMImplementation::hasFeature()</code> for a list of features.</p>
		 * @param string $version <p>The version number of the <code>feature</code> to test.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issupported.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSupported(string $feature, string $version): bool {}

		/**
		 * Gets the namespace URI of the node based on the prefix
		 * <p>Gets the namespace URI of the node based on the <code>prefix</code>.</p>
		 * @param ?string $prefix <p>The prefix to look for. If this parameter is <b><code>null</code></b>, the method will return the default namespace URI, if any.</p>
		 * @return ?string <p>Returns the associated namespace URI or <b><code>null</code></b> if none is found.</p>
		 * @link https://php.net/manual/en/domnode.lookupnamespaceuri.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupNamespaceURI(?string $prefix): ?string {}

		/**
		 * Gets the namespace prefix of the node based on the namespace URI
		 * <p>Gets the namespace prefix of the node based on the namespace URI.</p>
		 * @param string $namespace <p>The namespace URI.</p>
		 * @return ?string <p>The prefix of the namespace or <b><code>null</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.lookupprefix.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupPrefix(string $namespace): ?string {}

		/**
		 * Normalizes the node
		 * <p>Remove empty text nodes and merge adjacent text nodes in this node and all its children.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domnode.normalize.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function normalize(): void {}

		/**
		 * Removes child from list of children
		 * <p>This functions removes a child from a list of children.</p>
		 * @param \DOMNode $child <p>The removed child.</p>
		 * @return DOMNode|false <p>If the child could be removed the function returns the old child or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.removechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function removeChild(\DOMNode $child): \DOMNode|false {}

		/**
		 * Replaces a child
		 * <p>This function replaces the child <code>child</code> with the passed new node. If the <code>node</code> is already a child it will not be added a second time. If the replacement succeeds the old node is returned.</p>
		 * @param \DOMNode $node <p>The new node. It must be a member of the target document, i.e. created by one of the DOMDocument-&gt;createXXX() methods or imported in the document by DOMDocument::importNode.</p>
		 * @param \DOMNode $child <p>The old node.</p>
		 * @return DOMNode|false <p>The old node or <b><code>false</code></b> if an error occur.</p>
		 * @link https://php.net/manual/en/domnode.replacechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceChild(\DOMNode $node, \DOMNode $child): \DOMNode|false {}
	}

	/**
	 * <p>The <b>DOMText</b> class inherits from <code>DOMCharacterData</code> and represents the textual content of a <code>DOMElement</code> or <code>DOMAttr</code>.</p>
	 * @link https://php.net/manual/en/class.domtext.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMText extends \DOMCharacterData {

		/**
		 * @var string <p>Holds all the text of logically-adjacent (not separated by Element, Comment or Processing Instruction) Text nodes.</p>
		 * @link https://php.net/manual/en/class.domtext.php#domtext.props.wholetext
		 */
		public string $wholeText;

		/**
		 * @var string <p>The contents of the node.</p>
		 * @link https://php.net/manual/en/class.domcharacterdata.php#domcharacterdata.props.data
		 */
		public string $data;

		/**
		 * @var int <p>The length of the contents.</p>
		 * @link https://php.net/manual/en/class.domcharacterdata.php#domcharacterdata.props.length
		 */
		public int $length;

		/**
		 * @var ?DOMElement <p>The previous sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domcharacterdata.php#domcharacterdata.props.previouselementsibling
		 */
		public ?\DOMElement $previousElementSibling;

		/**
		 * @var ?DOMElement <p>The next sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domcharacterdata.php#domcharacterdata.props.nextelementsibling
		 */
		public ?\DOMElement $nextElementSibling;

		/**
		 * @var string <p>Returns the most accurate name for the current node type</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var ?string <p>The value of this node, depending on its type. Contrary to the W3C specification, the node value of <code>DOMElement</code> nodes is equal to DOMNode::textContent instead of <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var ?DOMNode <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentnode
		 */
		public ?\DOMNode $parentNode;

		/**
		 * @var ?DOMElement <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.parentelement
		 */
		public ?\DOMElement $parentElement;

		/**
		 * @var DOMNodeList <p>A <code>DOMNodeList</code> that contains all children of this node. If there are no children, this is an empty <code>DOMNodeList</code>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.childnodes
		 */
		public \DOMNodeList $childNodes;

		/**
		 * @var ?DOMNode <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.firstchild
		 */
		public ?\DOMNode $firstChild;

		/**
		 * @var ?DOMNode <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.lastchild
		 */
		public ?\DOMNode $lastChild;

		/**
		 * @var ?DOMNode <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.previoussibling
		 */
		public ?\DOMNode $previousSibling;

		/**
		 * @var ?DOMNode <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.nextsibling
		 */
		public ?\DOMNode $nextSibling;

		/**
		 * @var ?DOMNamedNodeMap <p>A <code>DOMNamedNodeMap</code> containing the attributes of this node (if it is a <code>DOMElement</code>) or <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.attributes
		 */
		public ?\DOMNamedNodeMap $attributes;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?DOMDocument <p>The <code>DOMDocument</code> object associated with this node, or <b><code>null</code></b> if this node is a <code>DOMDocument</code></p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.ownerdocument
		 */
		public ?\DOMDocument $ownerDocument;

		/**
		 * @var ?string <p>The namespace URI of this node, or <b><code>null</code></b> if it is unspecified.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var string <p>The namespace prefix of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.prefix
		 */
		public string $prefix;

		/**
		 * @var ?string <p>Returns the local part of the qualified name of this node.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.localname
		 */
		public ?string $localName;

		/**
		 * @var ?string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.baseuri
		 */
		public ?string $baseURI;

		/**
		 * @var string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.domnode.php#domnode.props.textcontent
		 */
		public string $textContent;

		/**
		 * Canonicalize nodes to a string
		 * <p>Canonicalize nodes to a string</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return string|false <p>Returns canonicalized nodes as a string or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14n.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14N(bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): string|false {}

		/**
		 * Canonicalize nodes to a file
		 * <p>Canonicalize nodes to a file.</p>
		 * @param string $uri <p>Path to write the output to.</p>
		 * @param bool $exclusive <p>Enable exclusive parsing of only the nodes matched by the provided xpath or namespace prefixes.</p>
		 * @param bool $withComments <p>Retain comments in output.</p>
		 * @param ?array $xpath <p>An array of XPaths to filter the nodes by. Each entry in this array is an associative array with:</p><ul> <li>  A required <code>query</code> key containing the XPath expression as a string.  </li> <li>  An optional <code>namespaces</code> key containing an array that maps namespace prefixes (keys) to namespace URIs (values).  </li> </ul>
		 * @param ?array $nsPrefixes <p>An array of namespace prefixes to filter the nodes by.</p>
		 * @return int|false <p>Number of bytes written or <b><code>false</code></b> on failure</p>
		 * @link https://php.net/manual/en/domnode.c14nfile.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function C14NFile(string $uri, bool $exclusive = false, bool $withComments = false, ?array $xpath = null, ?array $nsPrefixes = null): int|false {}

		/**
		 * Creates a new DOMText object
		 * <p>Creates a new <code>DOMText</code> object.</p>
		 * @param string $data <p>The value of the text node. If not supplied an empty text node is created.</p>
		 * @return self
		 * @link https://php.net/manual/en/domtext.construct.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __construct(string $data = "") {}

		/**
		 * Forbids serialization unless serialization methods are implemented in a subclass
		 * <p>Forbids serialization unless serialization methods are implemented in a subclass.</p>
		 * @return array <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.sleep.php
		 * @since PHP 8 >= 8.1.25
		 */
		public function __sleep(): array {}

		/**
		 * Forbids unserialization unless unserialization methods are implemented in a subclass
		 * <p>Forbids unserialization unless unserialization methods are implemented in a subclass.</p>
		 * @return void <p>The method always throws an exception.</p>
		 * @link https://php.net/manual/en/domnode.wakeup.php
		 * @since PHP 8 PHP 8 >= 8.1.25
		 */
		public function __wakeup(): void {}

		/**
		 * Adds nodes after the character data
		 * <p>Adds the passed <code>nodes</code> after the character data.</p>
		 * @param \DOMNode|string $nodes <p>Nodes to be added after the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.after.php
		 * @since PHP 8
		 */
		public function after(\DOMNode|string ...$nodes): void {}

		/**
		 * Adds new child at the end of the children
		 * <p>This function appends a child to an existing list of children or creates a new list of children. The child can be created with e.g. <code>DOMDocument::createElement()</code>, <code>DOMDocument::createTextNode()</code> etc. or simply by using any other node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The appended child.</p>
		 * @return DOMNode|false <p>The node added or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.appendchild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendChild(\DOMNode $node): \DOMNode|false {}

		/**
		 * Append the string to the end of the character data of the node
		 * <p>Append the string <code>data</code> to the end of the character data of the node.</p>
		 * @param string $data <p>The string to append.</p>
		 * @return true <p>Always returns <b><code>true</code></b>.</p>
		 * @link https://php.net/manual/en/domcharacterdata.appenddata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function appendData(string $data): true {}

		/**
		 * Adds nodes before the character data
		 * <p>Adds the passed <code>nodes</code> before the character data.</p>
		 * @param \DOMNode|string $nodes <p>Nodes to be added before the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.before.php
		 * @since PHP 8
		 */
		public function before(\DOMNode|string ...$nodes): void {}

		/**
		 * Clones a node
		 * <p>Creates a copy of the node.</p>
		 * @param bool $deep <p>Indicates whether to copy all descendant nodes. This parameter is defaulted to <b><code>false</code></b>.</p>
		 * @return DOMNode|false <p>The cloned node.</p>
		 * @link https://php.net/manual/en/domnode.clonenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function cloneNode(bool $deep = false): \DOMNode|false {}

		/**
		 * Compares the position of two nodes
		 * <p>Compares the position of the other node relative to this node.</p>
		 * @param \DOMNode $other <p>The node for which the position should be compared for, relative to this node.</p>
		 * @return int <p>A bitmask of the <b><code>DOMNode::DOCUMENT_POSITION_&#42;</code></b> constants.</p>
		 * @link https://php.net/manual/en/domnode.comparedocumentposition.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function compareDocumentPosition(\DOMNode $other): int {}

		/**
		 * Checks if node contains other node
		 * <p>Checks if node contains <code>other</code> node.</p>
		 * @param \DOMNode|\DOMNameSpaceNode|null $other <p>Node to be checked.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if node contains <code>other</code> node, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.contains.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function contains(\DOMNode|\DOMNameSpaceNode|null $other): bool {}

		/**
		 * Remove a range of characters from the character data
		 * <p>Deletes <code>count</code> characters starting from position <code>offset</code>.</p>
		 * @param int $offset <p>The offset from which to start removing.</p>
		 * @param int $count <p>The number of characters to delete. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are deleted.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domcharacterdata.deletedata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function deleteData(int $offset, int $count): bool {}

		/**
		 * Get line number for a node
		 * <p>Gets line number for where the node was defined at parse time.</p>
		 * @return int <p>Returns the line number where the node was defined at parse time. If the node was created manually, the return value will be <code>0</code>.</p>
		 * @link https://php.net/manual/en/domnode.getlineno.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		public function getLineNo(): int {}

		/**
		 * Get an XPath for a node
		 * <p>Gets an XPath location path for the node.</p>
		 * @return ?string <p>Returns a <code>string</code> containing the XPath, or <b><code>null</code></b> in case of an error.</p>
		 * @link https://php.net/manual/en/domnode.getnodepath.php
		 * @since PHP 5 >= 5.2.0, PHP 7, PHP 8
		 */
		public function getNodePath(): ?string {}

		/**
		 * Get root node
		 * <p>Get root node.</p>
		 * @param ?array $options <p>This parameter has no effect yet.</p>
		 * @return DOMNode <p>Returns the root node.</p>
		 * @link https://php.net/manual/en/domnode.getrootnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function getRootNode(?array $options = null): \DOMNode {}

		/**
		 * Checks if node has attributes
		 * <p>This method checks if the node has attributes. The tested node has to be an <b><code>XML_ELEMENT_NODE</code></b>.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.hasattributes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasAttributes(): bool {}

		/**
		 * Checks if node has children
		 * <p>This function checks if the node has children.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.haschildnodes.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function hasChildNodes(): bool {}

		/**
		 * Adds a new child before a reference node
		 * <p>This function inserts a new node right before the reference node. If you plan to do further modifications on the appended child you must use the returned node.</p><p>When using an existing node it will be moved.</p>
		 * @param \DOMNode $node <p>The new node.</p>
		 * @param ?\DOMNode $child <p>The reference node. If not supplied, <code>node</code> is appended to the children.</p>
		 * @return DOMNode|false <p>The inserted node or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.insertbefore.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertBefore(\DOMNode $node, ?\DOMNode $child = null): \DOMNode|false {}

		/**
		 * Insert a string at the specified UTF-8 codepoint offset
		 * <p>Inserts string <code>data</code> at position <code>offset</code>.</p>
		 * @param int $offset <p>The character offset at which to insert.</p>
		 * @param string $data <p>The string to insert.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domcharacterdata.insertdata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function insertData(int $offset, string $data): bool {}

		/**
		 * Checks if the specified namespaceURI is the default namespace or not
		 * <p>Tells whether <code>namespace</code> is the default namespace.</p>
		 * @param string $namespace <p>The namespace URI to look for.</p>
		 * @return bool <p>Return <b><code>true</code></b> if <code>namespace</code> is the default namespace, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isdefaultnamespace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isDefaultNamespace(string $namespace): bool {}

		/**
		 * Returns whether this text node contains whitespace in element content
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domtext.iselementcontentwhitespace.php
		 * @since No version information available, might only be in Git
		 */
		public function isElementContentWhitespace(): bool {}

		/**
		 * Checks that both nodes are equal
		 * <p>Checks that both nodes are equal.</p>
		 * @param ?\DOMNode $otherNode <p>The node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if both nodes are equal, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domnode.isequalnode.php
		 * @since PHP 8 >= 8.3.0
		 */
		public function isEqualNode(?\DOMNode $otherNode): bool {}

		/**
		 * Indicates if two nodes are the same node
		 * <p>This function indicates if two nodes are the same node. The comparison is <i>not</i> based on content</p>
		 * @param \DOMNode $otherNode <p>The compared node.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issamenode.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSameNode(\DOMNode $otherNode): bool {}

		/**
		 * Checks if feature is supported for specified version
		 * <p>Checks if the asked <code>feature</code> is supported for the specified <code>version</code>.</p>
		 * @param string $feature <p>The feature to test. See the example of <code>DOMImplementation::hasFeature()</code> for a list of features.</p>
		 * @param string $version <p>The version number of the <code>feature</code> to test.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domnode.issupported.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isSupported(string $feature, string $version): bool {}

		/**
		 * Indicates whether this text node contains whitespace
		 * <p>Indicates whether this text node contains only whitespace or it is empty. The text node is determined to contain whitespace in element content during the load of the document.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if node contains zero or more whitespace characters and nothing else. Returns <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/domtext.iswhitespaceinelementcontent.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function isWhitespaceInElementContent(): bool {}

		/**
		 * Gets the namespace URI of the node based on the prefix
		 * <p>Gets the namespace URI of the node based on the <code>prefix</code>.</p>
		 * @param ?string $prefix <p>The prefix to look for. If this parameter is <b><code>null</code></b>, the method will return the default namespace URI, if any.</p>
		 * @return ?string <p>Returns the associated namespace URI or <b><code>null</code></b> if none is found.</p>
		 * @link https://php.net/manual/en/domnode.lookupnamespaceuri.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupNamespaceURI(?string $prefix): ?string {}

		/**
		 * Gets the namespace prefix of the node based on the namespace URI
		 * <p>Gets the namespace prefix of the node based on the namespace URI.</p>
		 * @param string $namespace <p>The namespace URI.</p>
		 * @return ?string <p>The prefix of the namespace or <b><code>null</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.lookupprefix.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function lookupPrefix(string $namespace): ?string {}

		/**
		 * Normalizes the node
		 * <p>Remove empty text nodes and merge adjacent text nodes in this node and all its children.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domnode.normalize.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function normalize(): void {}

		/**
		 * Removes the character data node
		 * <p>Removes the character data node.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.remove.php
		 * @since PHP 8
		 */
		public function remove(): void {}

		/**
		 * Removes child from list of children
		 * <p>This functions removes a child from a list of children.</p>
		 * @param \DOMNode $child <p>The removed child.</p>
		 * @return DOMNode|false <p>If the child could be removed the function returns the old child or <b><code>false</code></b> on error.</p>
		 * @link https://php.net/manual/en/domnode.removechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function removeChild(\DOMNode $child): \DOMNode|false {}

		/**
		 * Replaces a child
		 * <p>This function replaces the child <code>child</code> with the passed new node. If the <code>node</code> is already a child it will not be added a second time. If the replacement succeeds the old node is returned.</p>
		 * @param \DOMNode $node <p>The new node. It must be a member of the target document, i.e. created by one of the DOMDocument-&gt;createXXX() methods or imported in the document by DOMDocument::importNode.</p>
		 * @param \DOMNode $child <p>The old node.</p>
		 * @return DOMNode|false <p>The old node or <b><code>false</code></b> if an error occur.</p>
		 * @link https://php.net/manual/en/domnode.replacechild.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceChild(\DOMNode $node, \DOMNode $child): \DOMNode|false {}

		/**
		 * Replace a substring within the character data
		 * <p>Replace <code>count</code> characters starting from position <code>offset</code> with <code>data</code>.</p>
		 * @param int $offset <p>The offset from which to start replacing.</p>
		 * @param int $count <p>The number of characters to replace. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are replaced.</p>
		 * @param string $data <p>The string with which the range must be replaced.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domcharacterdata.replacedata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function replaceData(int $offset, int $count, string $data): bool {}

		/**
		 * Replaces the character data with new nodes
		 * <p>Replaces the character data with new <code>nodes</code>.</p>
		 * @param \DOMNode|string $nodes <p>The replacement nodes. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.replacewith.php
		 * @since PHP 8
		 */
		public function replaceWith(\DOMNode|string ...$nodes): void {}

		/**
		 * Breaks this node into two nodes at the specified offset
		 * <p>Breaks this node into two nodes at the specified <code>offset</code>, keeping both in the tree as siblings.</p><p>After being split, this node will contain all the content up to the <code>offset</code>. If the original node had a parent node, the new node is inserted as the next sibling of the original node. When the <code>offset</code> is equal to the length of this node, the new node has no data.</p>
		 * @param int $offset <p>The offset at which to split, starting from 0.</p>
		 * @return DOMText|false <p>The new node of the same type, which contains all the content at and after the <code>offset</code>.</p>
		 * @link https://php.net/manual/en/domtext.splittext.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function splitText(int $offset): \DOMText|false {}

		/**
		 * Extracts a range of data from the character data
		 * <p>Returns the specified substring.</p>
		 * @param int $offset <p>Start offset of substring to extract.</p>
		 * @param int $count <p>The number of characters to extract.</p>
		 * @return string|false <p>The specified substring. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all UTF-8 codepoints to the end of the data are returned.</p>
		 * @link https://php.net/manual/en/domcharacterdata.substringdata.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function substringData(int $offset, int $count): string|false {}
	}

	/**
	 * <p>Allows to use XPath 1.0 queries on HTML or XML documents.</p>
	 * @link https://php.net/manual/en/class.domxpath.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	class DOMXPath {

		/**
		 * @var DOMDocument The document that is linked to this object.
		 * @link https://php.net/manual/en/class.domxpath.php#domxpath.props.document
		 */
		public \DOMDocument $document;

		/**
		 * @var bool When set to <b><code>true</code></b>, namespaces in the node are registered.
		 * @link https://php.net/manual/en/class.domxpath.php#domxpath.props.registernodenamespaces
		 */
		public bool $registerNodeNamespaces;

		/**
		 * Creates a new DOMXPath object
		 * <p>Creates a new <code>DOMXPath</code> object.</p>
		 * @param \DOMDocument $document <p>The <code>DOMDocument</code> associated with the <code>DOMXPath</code>.</p>
		 * @param bool $registerNodeNS <p>Whether to automatically register the in-scope namespace prefixes of the context node to the <code>DOMXPath</code> object. This can be used to avoid needing to call <code>DOMXPath::registerNamespace()</code> manually for each in-scope namespaces. When a namespace prefix conflict exists, only the nearest descendant namespace prefix is registered.</p>
		 * @return self
		 * @link https://php.net/manual/en/domxpath.construct.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function __construct(\DOMDocument $document, bool $registerNodeNS = true) {}

		/**
		 * Evaluates the given XPath expression and returns a typed result if possible
		 * <p>Executes the given XPath <code>expression</code> and returns a typed result if possible.</p>
		 * @param string $expression <p>The XPath expression to execute.</p>
		 * @param ?\DOMNode $contextNode <p>The optional <code>contextNode</code> can be specified for doing relative XPath queries. By default, the queries are relative to the root element.</p>
		 * @param bool $registerNodeNS <p>Whether to automatically register the in-scope namespace prefixes of the context node to the <code>DOMXPath</code> object. This can be used to avoid needing to call <code>DOMXPath::registerNamespace()</code> manually for each in-scope namespaces. When a namespace prefix conflict exists, only the nearest descendant namespace prefix is registered.</p>
		 * @return mixed <p>Returns a typed result if possible or a <code>DOMNodeList</code> containing all nodes matching the given XPath <code>expression</code>.</p><p>If the <code>expression</code> is malformed or the <code>contextNode</code> is invalid, <b>DOMXPath::evaluate()</b> returns <b><code>false</code></b>.</p>
		 * @link https://php.net/manual/en/domxpath.evaluate.php
		 * @since PHP 5 >= 5.1.0, PHP 7, PHP 8
		 */
		public function evaluate(string $expression, ?\DOMNode $contextNode = null, bool $registerNodeNS = true): mixed {}

		/**
		 * Evaluates the given XPath expression
		 * <p>Executes the given XPath <code>expression</code>.</p>
		 * @param string $expression <p>The XPath expression to execute.</p>
		 * @param ?\DOMNode $contextNode <p>The optional <code>contextNode</code> can be specified for doing relative XPath queries. By default, the queries are relative to the root element.</p>
		 * @param bool $registerNodeNS <p>Whether to automatically register the in-scope namespace prefixes of the context node to the <code>DOMXPath</code> object. This can be used to avoid needing to call <code>DOMXPath::registerNamespace()</code> manually for each in-scope namespaces. When a namespace prefix conflict exists, only the nearest descendant namespace prefix is registered.</p>
		 * @return mixed <p>Returns a <code>DOMNodeList</code> containing all nodes matching the given XPath <code>expression</code>. Any expression which does not return nodes will return an empty <code>DOMNodeList</code>.</p><p>If the <code>expression</code> is malformed or the <code>contextNode</code> is invalid, <b>DOMXPath::query()</b> returns <b><code>false</code></b>.</p>
		 * @link https://php.net/manual/en/domxpath.query.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function query(string $expression, ?\DOMNode $contextNode = null, bool $registerNodeNS = true): mixed {}

		/**
		 * Quotes a string for use in an XPath expression
		 * <p>Quotes <code>str</code> for use in an XPath expression.</p>
		 * @param string $str The string to quote.
		 * @return string <p>Returns a quoted string to be used in an XPath expression.</p>
		 * @link https://php.net/manual/en/domxpath.quote.php
		 * @since PHP 8 >= 8.4.0
		 */
		public static function quote(string $str): string {}

		/**
		 * Registers the namespace with the DOMXPath object
		 * <p>Registers the <code>namespace</code> and <code>prefix</code> with the DOMXPath object.</p>
		 * @param string $prefix <p>The prefix.</p>
		 * @param string $namespace <p>The URI of the namespace.</p>
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/domxpath.registernamespace.php
		 * @since PHP 5, PHP 7, PHP 8
		 */
		public function registerNamespace(string $prefix, string $namespace): bool {}

		/**
		 * Register PHP functions as XPath functions
		 * <p>This method enables the ability to use PHP functions within XPath expressions.</p>
		 * @param string|array|null $restrict <p>Use this parameter to only allow certain functions to be called from XPath.</p> <p>This parameter can be one of the following: a <code>string</code> (a function name), an indexed <code>array</code> of function names, or an associative <code>array</code> with keys being the function name and the associated value being the <code>callable</code>.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/domxpath.registerphpfunctions.php
		 * @since PHP 5 >= 5.3.0, PHP 7, PHP 8
		 */
		public function registerPhpFunctions(string|array|null $restrict = null): void {}
	}

	/**
	 * Gets a DOMAttr or DOMElement object from a SimpleXMLElement object
	 * <p>This function takes the given attribute or element <code>node</code> (a <code>SimpleXMLElement</code> instance) and creates a <code>DOMAttr</code> or <code>DOMElement</code> node, repectively. The new <code>DOMNode</code> refers to the same underlying XML node as the <code>SimpleXMLElement</code>.</p>
	 * @param object $node <p>The attribute or element node to import (a <code>SimpleXMLElement</code> instance).</p>
	 * @return DOMAttr|DOMElement <p>The <code>DOMAttr</code> or <code>DOMElement</code>.</p>
	 * @link https://php.net/manual/en/function.dom-import-simplexml.php
	 * @see simplexml_import_dom()
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function dom_import_simplexml(object $node): \DOMAttr|\DOMElement {}

	/**
	 * If any node is inserted somewhere it doesn't belong
	 */
	define('DOM_HIERARCHY_REQUEST_ERR', 3);

	/**
	 * If index or size is negative, or greater than the allowed value.
	 */
	define('DOM_INDEX_SIZE_ERR', 1);

	/**
	 * If an attempt is made to add an attribute that is already in use elsewhere.
	 */
	define('DOM_INUSE_ATTRIBUTE_ERR', 10);

	/**
	 * If a parameter or an operation is not supported by the underlying object.
	 */
	define('DOM_INVALID_ACCESS_ERR', 15);

	/**
	 * If an invalid or illegal character is specified, such as in a name.
	 */
	define('DOM_INVALID_CHARACTER_ERR', 5);

	/**
	 * If an attempt is made to modify the type of the underlying object.
	 */
	define('DOM_INVALID_MODIFICATION_ERR', 13);

	/**
	 * If an attempt is made to use an object that is not, or is no longer, usable.
	 */
	define('DOM_INVALID_STATE_ERR', 11);

	/**
	 * If an attempt is made to create or change an object in a way which is incorrect with regard to namespaces.
	 */
	define('DOM_NAMESPACE_ERR', 14);

	/**
	 * If data is specified for a node which does not support data.
	 */
	define('DOM_NO_DATA_ALLOWED_ERR', 6);

	/**
	 * If an attempt is made to modify an object where modifications are not allowed.
	 */
	define('DOM_NO_MODIFICATION_ALLOWED_ERR', 7);

	/**
	 * If an attempt is made to reference a node in a context where it does not exist.
	 */
	define('DOM_NOT_FOUND_ERR', 8);

	/**
	 * If the implementation does not support the requested type of object or operation.
	 */
	define('DOM_NOT_SUPPORTED_ERR', 9);

	/**
	 * Error code not part of the DOM specification. Meant for PHP errors. Deprecated as of PHP 8.4.0 because this is no longer used. Prior to PHP 8.4.0, this was inconsistently used to indicate out-of-memory situations.
	 */
	define('DOM_PHP_ERR', 0);

	/**
	 * If an invalid or illegal string is specified.
	 */
	define('DOM_SYNTAX_ERR', 12);

	/**
	 * If a call to a method such as insertBefore or removeChild would make the Node invalid with respect to "partial validity", this exception would be raised and the operation would not be done.
	 */
	define('DOM_VALIDATION_ERR', 16);

	/**
	 * If a node is used in a different document than the one that created it.
	 */
	define('DOM_WRONG_DOCUMENT_ERR', 4);

	/**
	 * If the specified range of text does not fit into a <code>string</code>.
	 */
	define('DOMSTRING_SIZE_ERR', 2);

	/**
	 * 1
	 */
	define('XML_ATTRIBUTE_CDATA', 1);

	/**
	 * 16
	 */
	define('XML_ATTRIBUTE_DECL_NODE', 16);

	/**
	 * 5
	 */
	define('XML_ATTRIBUTE_ENTITY', 6);

	/**
	 * 9
	 */
	define('XML_ATTRIBUTE_ENUMERATION', 9);

	/**
	 * 2
	 */
	define('XML_ATTRIBUTE_ID', 2);

	/**
	 * 3
	 */
	define('XML_ATTRIBUTE_IDREF', 3);

	/**
	 * 4
	 */
	define('XML_ATTRIBUTE_IDREFS', 4);

	/**
	 * 7
	 */
	define('XML_ATTRIBUTE_NMTOKEN', 7);

	/**
	 * 8
	 */
	define('XML_ATTRIBUTE_NMTOKENS', 8);

	/**
	 * Node is a <code>DOMAttr</code> / <code>Dom\Attr</code>
	 */
	define('XML_ATTRIBUTE_NODE', 2);

	/**
	 * 10
	 */
	define('XML_ATTRIBUTE_NOTATION', 10);

	/**
	 * Node is a <code>DOMCharacterData</code> / <code>Dom\CharacterData</code>
	 */
	define('XML_CDATA_SECTION_NODE', 4);

	/**
	 * Node is a <code>DOMComment</code> / <code>Dom\Comment</code>
	 */
	define('XML_COMMENT_NODE', 8);

	/**
	 * Node is a <code>DOMDocumentFragment</code> / <code>Dom\DocumentFragment</code>
	 */
	define('XML_DOCUMENT_FRAG_NODE', 11);

	/**
	 * Node is a <code>DOMDocument</code> / <code>Dom\Document</code>
	 */
	define('XML_DOCUMENT_NODE', 9);

	/**
	 * Node is a <code>DOMDocumentType</code> / <code>Dom\DocumentType</code>
	 */
	define('XML_DOCUMENT_TYPE_NODE', 10);

	/**
	 * 14
	 */
	define('XML_DTD_NODE', 14);

	/**
	 * 15
	 */
	define('XML_ELEMENT_DECL_NODE', 15);

	/**
	 * Node is a <code>DOMElement</code> / <code>Dom\Element</code>
	 */
	define('XML_ELEMENT_NODE', 1);

	/**
	 * 17
	 */
	define('XML_ENTITY_DECL_NODE', 17);

	/**
	 * Node is a <code>DOMEntity</code> / <code>Dom\Entity</code>
	 */
	define('XML_ENTITY_NODE', 6);

	/**
	 * Node is a <code>DOMEntityReference</code> / <code>Dom\EntityReference</code>
	 */
	define('XML_ENTITY_REF_NODE', 5);

	/**
	 * 13
	 */
	define('XML_HTML_DOCUMENT_NODE', 13);

	/**
	 * A namespace declaration node.
	 */
	define('XML_LOCAL_NAMESPACE', 18);

	/**
	 * 18
	 */
	define('XML_NAMESPACE_DECL_NODE', 18);

	/**
	 * Node is a <code>DOMNotation</code> / <code>Dom\Notation</code>
	 */
	define('XML_NOTATION_NODE', 12);

	/**
	 * Node is a <code>DOMProcessingInstruction</code> / <code>Dom\ProcessingInstruction</code>
	 */
	define('XML_PI_NODE', 7);

	/**
	 * Node is a <code>DOMText</code> / <code>Dom\Text</code>
	 */
	define('XML_TEXT_NODE', 3);

}

namespace Dom {

	/**
	 * <p><b>Dom\Attr</b> represents an attribute in the <code>Dom\Element</code> object.</p>
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMAttr</code>.</p>
	 * @link https://php.net/manual/en/class.dom-attr.php
	 * @since PHP 8 >= 8.4.0
	 */
	class Attr extends \Dom\Node {

		/**
		 * @var ?string The namespace URI of the attribute.
		 * @link https://php.net/manual/en/class.dom-attr.php#dom-attr.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var ?string The namespace prefix of the attribute.
		 * @link https://php.net/manual/en/class.dom-attr.php#dom-attr.props.prefix
		 */
		public ?string $prefix;

		/**
		 * @var string The local name of the attribute.
		 * @link https://php.net/manual/en/class.dom-attr.php#dom-attr.props.localname
		 */
		public string $localName;

		/**
		 * @var string The qualified name of the attribute.
		 * @link https://php.net/manual/en/class.dom-attr.php#dom-attr.props.name
		 */
		public string $name;

		/**
		 * @var string The value of the attribute. <p><b>Note</b>:  Unlike the equivalent property in <code>DOMAttr</code>, this does not substitute entities. </p>
		 * @link https://php.net/manual/en/class.dom-attr.php#dom-attr.props.value
		 */
		public string $value;

		/**
		 * @var ?Dom\Element The element that contains the attribute or <b><code>null</code></b>.
		 * @link https://php.net/manual/en/class.dom-attr.php#dom-attr.props.ownerelement
		 */
		public ?\Dom\Element $ownerElement;

		/**
		 * @var bool Legacy option, always is <b><code>true</code></b>.
		 * @link https://php.net/manual/en/class.dom-attr.php#dom-attr.props.specified
		 */
		public bool $specified;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string Returns the most accurate name for the current node type. <ul> <li>For elements, this is the HTML-uppercased qualified name.</li> <li>For attributes, this is the qualified name.</li> <li>For processing instructions, this is the target.</li> <li>For document type nodes, this is the name.</li> </ul>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.baseuri
		 */
		public string $baseURI;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?Dom\Document The <code>Dom\Document</code> object associated with this node, or <b><code>null</code></b> if this node is a document.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.ownerdocument
		 */
		public ?\Dom\Document $ownerDocument;

		/**
		 * @var ?Dom\Node <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentnode
		 */
		public ?\Dom\Node $parentNode;

		/**
		 * @var ?Dom\Element <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentelement
		 */
		public ?\Dom\Element $parentElement;

		/**
		 * @var Dom\NodeList A <code>Dom\NodeList</code> that contains all children of this node. If there are no children, this is an empty <code>Dom\NodeList</code>.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.childnodes
		 */
		public \Dom\NodeList $childNodes;

		/**
		 * @var ?Dom\Node <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.firstchild
		 */
		public ?\Dom\Node $firstChild;

		/**
		 * @var ?Dom\Node <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.lastchild
		 */
		public ?\Dom\Node $lastChild;

		/**
		 * @var ?Dom\Node <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.previoussibling
		 */
		public ?\Dom\Node $previousSibling;

		/**
		 * @var ?Dom\Node <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nextsibling
		 */
		public ?\Dom\Node $nextSibling;

		/**
		 * @var ?string The value of this node, depending on its type.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var ?string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.textcontent
		 */
		public ?string $textContent;

		/**
		 * Checks if attribute is a defined ID
		 * <p>This function checks if the attribute is a defined ID.</p><p>According to the DOM standard this requires a DTD which defines the attribute ID to be of type ID. To utilise this method the document must be validated at parse time by passing <b><code>LIBXML_DTDVALID</code></b> as an option.</p>
		 * @return bool <p>Returns <b><code>true</code></b> if this attribute is a defined ID, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/dom-attr.isid.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function isId(): bool {}

		/**
		 * Changes the qualified name or namespace of an attribute
		 * <p>This method changes the qualified name or namespace of an attribute.</p>
		 * @param ?string $namespaceURI The new namespace URI of the attribute.
		 * @param string $qualifiedName The new qualified name of the attribute.
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-attr.rename.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function rename(?string $namespaceURI, string $qualifiedName): void {}
	}

	/**
	 * <p>The <b>Dom\CDATASection</b> class inherits from <code>Dom\Text</code> for textual representation of CData constructs.</p>
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMCdataSection</code>.</p>
	 * @link https://php.net/manual/en/class.dom-cdatasection.php
	 * @since PHP 8 >= 8.4.0
	 */
	class CDATASection extends \Dom\Text {

		/**
		 * @var string <p>Holds all the text of logically-adjacent (not separated by Element, Comment or Processing Instruction) Text nodes.</p>
		 * @link https://php.net/manual/en/class.dom-text.php#dom-text.props.wholetext
		 */
		public string $wholeText;

		/**
		 * @var ?Dom\Element <p>The previous sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.previouselementsibling
		 */
		public ?\Dom\Element $previousElementSibling;

		/**
		 * @var ?Dom\Element <p>The next sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.nextelementsibling
		 */
		public ?\Dom\Element $nextElementSibling;

		/**
		 * @var string <p>The contents of the node.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.data
		 */
		public string $data;

		/**
		 * @var int <p>The length of the contents.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.length
		 */
		public int $length;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string Returns the most accurate name for the current node type. <ul> <li>For elements, this is the HTML-uppercased qualified name.</li> <li>For attributes, this is the qualified name.</li> <li>For processing instructions, this is the target.</li> <li>For document type nodes, this is the name.</li> </ul>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.baseuri
		 */
		public string $baseURI;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?Dom\Document The <code>Dom\Document</code> object associated with this node, or <b><code>null</code></b> if this node is a document.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.ownerdocument
		 */
		public ?\Dom\Document $ownerDocument;

		/**
		 * @var ?Dom\Node <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentnode
		 */
		public ?\Dom\Node $parentNode;

		/**
		 * @var ?Dom\Element <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentelement
		 */
		public ?\Dom\Element $parentElement;

		/**
		 * @var Dom\NodeList A <code>Dom\NodeList</code> that contains all children of this node. If there are no children, this is an empty <code>Dom\NodeList</code>.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.childnodes
		 */
		public \Dom\NodeList $childNodes;

		/**
		 * @var ?Dom\Node <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.firstchild
		 */
		public ?\Dom\Node $firstChild;

		/**
		 * @var ?Dom\Node <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.lastchild
		 */
		public ?\Dom\Node $lastChild;

		/**
		 * @var ?Dom\Node <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.previoussibling
		 */
		public ?\Dom\Node $previousSibling;

		/**
		 * @var ?Dom\Node <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nextsibling
		 */
		public ?\Dom\Node $nextSibling;

		/**
		 * @var ?string The value of this node, depending on its type.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var ?string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.textcontent
		 */
		public ?string $textContent;

		/**
		 * Adds nodes after the character data
		 * <p>Adds the passed <code>nodes</code> after the character data.</p>
		 * @param \Dom\Node|string $nodes <p>Nodes to be added after the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.after.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function after(\Dom\Node|string ...$nodes): void {}

		/**
		 * Append the string to the end of the character data of the node
		 * <p>Append the string <code>data</code> to the end of the character data of the node.</p>
		 * @param string $data <p>The string to append.</p>
		 * @return void <p>Always returns <b><code>true</code></b>.</p>
		 * @link https://php.net/manual/en/dom-characterdata.appenddata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function appendData(string $data): void {}

		/**
		 * Adds nodes before the character data
		 * <p>Adds the passed <code>nodes</code> before the character data.</p>
		 * @param \Dom\Node|string $nodes <p>Nodes to be added before the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.before.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function before(\Dom\Node|string ...$nodes): void {}

		/**
		 * Remove a range of characters from the character data
		 * <p>Deletes <code>count</code> characters starting from position <code>offset</code>.</p>
		 * @param int $offset <p>The offset from which to start removing.</p>
		 * @param int $count <p>The number of characters to delete. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are deleted.</p>
		 * @return void <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/dom-characterdata.deletedata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function deleteData(int $offset, int $count): void {}

		/**
		 * Insert a string at the specified UTF-8 codepoint offset
		 * <p>Inserts string <code>data</code> at position <code>offset</code>.</p>
		 * @param int $offset <p>The character offset at which to insert.</p>
		 * @param string $data <p>The string to insert.</p>
		 * @return void <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/dom-characterdata.insertdata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function insertData(int $offset, string $data): void {}

		/**
		 * Removes the character data node
		 * <p>Removes the character data node.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.remove.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function remove(): void {}

		/**
		 * Replace a substring within the character data
		 * <p>Replace <code>count</code> characters starting from position <code>offset</code> with <code>data</code>.</p>
		 * @param int $offset <p>The offset from which to start replacing.</p>
		 * @param int $count <p>The number of characters to replace. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are replaced.</p>
		 * @param string $data <p>The string with which the range must be replaced.</p>
		 * @return void <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/dom-characterdata.replacedata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function replaceData(int $offset, int $count, string $data): void {}

		/**
		 * Replaces the character data with new nodes
		 * <p>Replaces the character data with new <code>nodes</code>.</p>
		 * @param \Dom\Node|string $nodes <p>The replacement nodes. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.replacewith.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function replaceWith(\Dom\Node|string ...$nodes): void {}

		/**
		 * Breaks this node into two nodes at the specified offset
		 * <p>Breaks this node into two nodes at the specified <code>offset</code>, keeping both in the tree as siblings.</p>
		 * @param int $offset <p>The offset at which to split, starting from 0.</p>
		 * @return Dom\Text <p>The new node of the same type, which contains all the content at and after the <code>offset</code>.</p>
		 * @link https://php.net/manual/en/dom-text.splittext.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function splitText(int $offset): \Dom\Text {}

		/**
		 * Extracts a range of data from the character data
		 * <p>Returns the specified substring.</p>
		 * @param int $offset <p>Start offset of substring to extract.</p>
		 * @param int $count <p>The number of characters to extract.</p>
		 * @return string <p>The specified substring. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all UTF-8 codepoints to the end of the data are returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.substringdata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function substringData(int $offset, int $count): string {}
	}

	/**
	 * <p>Represents nodes with character data. No nodes directly correspond to this class, but other nodes do inherit from it.</p>
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMCharacterData</code>.</p>
	 * @link https://php.net/manual/en/class.dom-characterdata.php
	 * @since PHP 8 >= 8.4.0
	 */
	class CharacterData extends \Dom\Node implements \Dom\ChildNode {

		/**
		 * @var ?Dom\Element <p>The previous sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.previouselementsibling
		 */
		public ?\Dom\Element $previousElementSibling;

		/**
		 * @var ?Dom\Element <p>The next sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.nextelementsibling
		 */
		public ?\Dom\Element $nextElementSibling;

		/**
		 * @var string <p>The contents of the node.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.data
		 */
		public string $data;

		/**
		 * @var int <p>The length of the contents.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.length
		 */
		public int $length;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string Returns the most accurate name for the current node type. <ul> <li>For elements, this is the HTML-uppercased qualified name.</li> <li>For attributes, this is the qualified name.</li> <li>For processing instructions, this is the target.</li> <li>For document type nodes, this is the name.</li> </ul>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.baseuri
		 */
		public string $baseURI;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?Dom\Document The <code>Dom\Document</code> object associated with this node, or <b><code>null</code></b> if this node is a document.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.ownerdocument
		 */
		public ?\Dom\Document $ownerDocument;

		/**
		 * @var ?Dom\Node <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentnode
		 */
		public ?\Dom\Node $parentNode;

		/**
		 * @var ?Dom\Element <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentelement
		 */
		public ?\Dom\Element $parentElement;

		/**
		 * @var Dom\NodeList A <code>Dom\NodeList</code> that contains all children of this node. If there are no children, this is an empty <code>Dom\NodeList</code>.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.childnodes
		 */
		public \Dom\NodeList $childNodes;

		/**
		 * @var ?Dom\Node <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.firstchild
		 */
		public ?\Dom\Node $firstChild;

		/**
		 * @var ?Dom\Node <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.lastchild
		 */
		public ?\Dom\Node $lastChild;

		/**
		 * @var ?Dom\Node <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.previoussibling
		 */
		public ?\Dom\Node $previousSibling;

		/**
		 * @var ?Dom\Node <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nextsibling
		 */
		public ?\Dom\Node $nextSibling;

		/**
		 * @var ?string The value of this node, depending on its type.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var ?string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.textcontent
		 */
		public ?string $textContent;

		/**
		 * Adds nodes after the character data
		 * <p>Adds the passed <code>nodes</code> after the character data.</p>
		 * @param \Dom\Node|string $nodes <p>Nodes to be added after the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.after.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function after(\Dom\Node|string ...$nodes): void {}

		/**
		 * Append the string to the end of the character data of the node
		 * <p>Append the string <code>data</code> to the end of the character data of the node.</p>
		 * @param string $data <p>The string to append.</p>
		 * @return void <p>Always returns <b><code>true</code></b>.</p>
		 * @link https://php.net/manual/en/dom-characterdata.appenddata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function appendData(string $data): void {}

		/**
		 * Adds nodes before the character data
		 * <p>Adds the passed <code>nodes</code> before the character data.</p>
		 * @param \Dom\Node|string $nodes <p>Nodes to be added before the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.before.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function before(\Dom\Node|string ...$nodes): void {}

		/**
		 * Remove a range of characters from the character data
		 * <p>Deletes <code>count</code> characters starting from position <code>offset</code>.</p>
		 * @param int $offset <p>The offset from which to start removing.</p>
		 * @param int $count <p>The number of characters to delete. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are deleted.</p>
		 * @return void <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/dom-characterdata.deletedata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function deleteData(int $offset, int $count): void {}

		/**
		 * Insert a string at the specified UTF-8 codepoint offset
		 * <p>Inserts string <code>data</code> at position <code>offset</code>.</p>
		 * @param int $offset <p>The character offset at which to insert.</p>
		 * @param string $data <p>The string to insert.</p>
		 * @return void <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/dom-characterdata.insertdata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function insertData(int $offset, string $data): void {}

		/**
		 * Removes the character data node
		 * <p>Removes the character data node.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.remove.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function remove(): void {}

		/**
		 * Replace a substring within the character data
		 * <p>Replace <code>count</code> characters starting from position <code>offset</code> with <code>data</code>.</p>
		 * @param int $offset <p>The offset from which to start replacing.</p>
		 * @param int $count <p>The number of characters to replace. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are replaced.</p>
		 * @param string $data <p>The string with which the range must be replaced.</p>
		 * @return void <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/dom-characterdata.replacedata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function replaceData(int $offset, int $count, string $data): void {}

		/**
		 * Replaces the character data with new nodes
		 * <p>Replaces the character data with new <code>nodes</code>.</p>
		 * @param \Dom\Node|string $nodes <p>The replacement nodes. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.replacewith.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function replaceWith(\Dom\Node|string ...$nodes): void {}

		/**
		 * Extracts a range of data from the character data
		 * <p>Returns the specified substring.</p>
		 * @param int $offset <p>Start offset of substring to extract.</p>
		 * @param int $count <p>The number of characters to extract.</p>
		 * @return string <p>The specified substring. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all UTF-8 codepoints to the end of the data are returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.substringdata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function substringData(int $offset, int $count): string {}
	}

	/**
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMChildNode</code>.</p>
	 * @link https://php.net/manual/en/class.dom-childnode.php
	 * @since PHP 8 >= 8.4.0
	 */
	interface ChildNode {

		/**
		 * Adds nodes after the node
		 * <p>Adds the passed <code>nodes</code> after the node.</p>
		 * @param \Dom\Node|string $nodes <p>Nodes to be added after the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-childnode.after.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function after(\Dom\Node|string ...$nodes): void;

		/**
		 * Adds nodes before the node
		 * <p>Adds the passed <code>nodes</code> before the node.</p>
		 * @param \Dom\Node|string $nodes <p>Nodes to be added before the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-childnode.before.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function before(\Dom\Node|string ...$nodes): void;

		/**
		 * Removes the node
		 * <p>Removes the node.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-childnode.remove.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function remove(): void;

		/**
		 * Replaces the node with new nodes
		 * <p>Replaces the node with new <code>nodes</code>.</p>
		 * @param \Dom\Node|string $nodes <p>The replacement nodes. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-childnode.replacewith.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function replaceWith(\Dom\Node|string ...$nodes): void;
	}

	/**
	 * <p>Represents comment nodes, characters delimited by <code>&lt;!--</code> and <code>--&gt;</code>.</p>
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMComment</code>.</p>
	 * @link https://php.net/manual/en/class.dom-comment.php
	 * @since PHP 8 >= 8.4.0
	 */
	class Comment extends \Dom\CharacterData {

		/**
		 * @var ?Dom\Element <p>The previous sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.previouselementsibling
		 */
		public ?\Dom\Element $previousElementSibling;

		/**
		 * @var ?Dom\Element <p>The next sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.nextelementsibling
		 */
		public ?\Dom\Element $nextElementSibling;

		/**
		 * @var string <p>The contents of the node.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.data
		 */
		public string $data;

		/**
		 * @var int <p>The length of the contents.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.length
		 */
		public int $length;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string Returns the most accurate name for the current node type. <ul> <li>For elements, this is the HTML-uppercased qualified name.</li> <li>For attributes, this is the qualified name.</li> <li>For processing instructions, this is the target.</li> <li>For document type nodes, this is the name.</li> </ul>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.baseuri
		 */
		public string $baseURI;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?Dom\Document The <code>Dom\Document</code> object associated with this node, or <b><code>null</code></b> if this node is a document.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.ownerdocument
		 */
		public ?\Dom\Document $ownerDocument;

		/**
		 * @var ?Dom\Node <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentnode
		 */
		public ?\Dom\Node $parentNode;

		/**
		 * @var ?Dom\Element <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentelement
		 */
		public ?\Dom\Element $parentElement;

		/**
		 * @var Dom\NodeList A <code>Dom\NodeList</code> that contains all children of this node. If there are no children, this is an empty <code>Dom\NodeList</code>.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.childnodes
		 */
		public \Dom\NodeList $childNodes;

		/**
		 * @var ?Dom\Node <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.firstchild
		 */
		public ?\Dom\Node $firstChild;

		/**
		 * @var ?Dom\Node <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.lastchild
		 */
		public ?\Dom\Node $lastChild;

		/**
		 * @var ?Dom\Node <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.previoussibling
		 */
		public ?\Dom\Node $previousSibling;

		/**
		 * @var ?Dom\Node <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nextsibling
		 */
		public ?\Dom\Node $nextSibling;

		/**
		 * @var ?string The value of this node, depending on its type.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var ?string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.textcontent
		 */
		public ?string $textContent;

		/**
		 * Adds nodes after the character data
		 * <p>Adds the passed <code>nodes</code> after the character data.</p>
		 * @param \Dom\Node|string $nodes <p>Nodes to be added after the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.after.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function after(\Dom\Node|string ...$nodes): void {}

		/**
		 * Append the string to the end of the character data of the node
		 * <p>Append the string <code>data</code> to the end of the character data of the node.</p>
		 * @param string $data <p>The string to append.</p>
		 * @return void <p>Always returns <b><code>true</code></b>.</p>
		 * @link https://php.net/manual/en/dom-characterdata.appenddata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function appendData(string $data): void {}

		/**
		 * Adds nodes before the character data
		 * <p>Adds the passed <code>nodes</code> before the character data.</p>
		 * @param \Dom\Node|string $nodes <p>Nodes to be added before the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.before.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function before(\Dom\Node|string ...$nodes): void {}

		/**
		 * Remove a range of characters from the character data
		 * <p>Deletes <code>count</code> characters starting from position <code>offset</code>.</p>
		 * @param int $offset <p>The offset from which to start removing.</p>
		 * @param int $count <p>The number of characters to delete. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are deleted.</p>
		 * @return void <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/dom-characterdata.deletedata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function deleteData(int $offset, int $count): void {}

		/**
		 * Insert a string at the specified UTF-8 codepoint offset
		 * <p>Inserts string <code>data</code> at position <code>offset</code>.</p>
		 * @param int $offset <p>The character offset at which to insert.</p>
		 * @param string $data <p>The string to insert.</p>
		 * @return void <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/dom-characterdata.insertdata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function insertData(int $offset, string $data): void {}

		/**
		 * Removes the character data node
		 * <p>Removes the character data node.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.remove.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function remove(): void {}

		/**
		 * Replace a substring within the character data
		 * <p>Replace <code>count</code> characters starting from position <code>offset</code> with <code>data</code>.</p>
		 * @param int $offset <p>The offset from which to start replacing.</p>
		 * @param int $count <p>The number of characters to replace. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are replaced.</p>
		 * @param string $data <p>The string with which the range must be replaced.</p>
		 * @return void <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/dom-characterdata.replacedata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function replaceData(int $offset, int $count, string $data): void {}

		/**
		 * Replaces the character data with new nodes
		 * <p>Replaces the character data with new <code>nodes</code>.</p>
		 * @param \Dom\Node|string $nodes <p>The replacement nodes. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.replacewith.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function replaceWith(\Dom\Node|string ...$nodes): void {}

		/**
		 * Extracts a range of data from the character data
		 * <p>Returns the specified substring.</p>
		 * @param int $offset <p>Start offset of substring to extract.</p>
		 * @param int $count <p>The number of characters to extract.</p>
		 * @return string <p>The specified substring. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all UTF-8 codepoints to the end of the data are returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.substringdata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function substringData(int $offset, int $count): string {}
	}

	/**
	 * <p>Represents an entire HTML or XML document; serves as the root of the document tree.</p>
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMDocument</code>. It is the base class for <code>Dom\XMLDocument</code> and <code>Dom\HTMLDocument</code>.</p>
	 * @link https://php.net/manual/en/class.dom-document.php
	 * @since PHP 8 >= 8.4.0
	 */
	abstract class Document extends \Dom\Node implements \Dom\ParentNode {

		/**
		 * @var Dom\Implementation <p>The <code>DOMImplementation</code> object that handles this document.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.implementation
		 */
		public \Dom\Implementation $implementation;

		/**
		 * @var string Equivalent to documentURI.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.url
		 */
		public string $URL;

		/**
		 * @var string <p>The location of the document or <b><code>null</code></b> if undefined.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.documenturi
		 */
		public string $documentURI;

		/**
		 * @var string The encoding of the document used for serialization. Upon parsing a document, this is set to the input encoding of that document.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.characterset
		 */
		public string $characterSet;

		/**
		 * @var string Legacy alias for characterSet.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.charset
		 */
		public string $charset;

		/**
		 * @var string Legacy alias for characterSet.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.inputencoding
		 */
		public string $inputEncoding;

		/**
		 * @var ?Dom\DocumentType <p>The Document Type Declaration associated with this document.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.doctype
		 */
		public ?\Dom\DocumentType $doctype;

		/**
		 * @var ?Dom\Element The <code>Dom\Element</code> that is the document element. This evaluates to <b><code>null</code></b> for document without elements.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.documentelement
		 */
		public ?\Dom\Element $documentElement;

		/**
		 * @var ?Dom\Element <p>First child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.firstelementchild
		 */
		public ?\Dom\Element $firstElementChild;

		/**
		 * @var ?Dom\Element <p>Last child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.lastelementchild
		 */
		public ?\Dom\Element $lastElementChild;

		/**
		 * @var int <p>The number of child elements.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.childelementcount
		 */
		public int $childElementCount;

		/**
		 * @var ?Dom\HTMLElement The first child of the <code>html</code> element that is either a <code>body</code> tag or a <code>frameset</code> tag. These need to be in the HTML namespace. If no element matches, this evaluates to <b><code>null</code></b>.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.body
		 */
		public ?\Dom\HTMLElement $body;

		/**
		 * @var ?Dom\HTMLElement The first <code>head</code> element that is a child of the <code>html</code> element. These need to be in the HTML namespace. If no element matches, this evaluates to <b><code>null</code></b>.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.head
		 */
		public ?\Dom\HTMLElement $head;

		/**
		 * @var string The title of the document as set by the <code>title</code> element for HTML or the SVG <code>title</code> element for SVG. If there is no title, this evaluates to the empty string.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.title
		 */
		public string $title;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string Returns the most accurate name for the current node type. <ul> <li>For elements, this is the HTML-uppercased qualified name.</li> <li>For attributes, this is the qualified name.</li> <li>For processing instructions, this is the target.</li> <li>For document type nodes, this is the name.</li> </ul>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.baseuri
		 */
		public string $baseURI;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?Dom\Document The <code>Dom\Document</code> object associated with this node, or <b><code>null</code></b> if this node is a document.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.ownerdocument
		 */
		public ?\Dom\Document $ownerDocument;

		/**
		 * @var ?Dom\Node <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentnode
		 */
		public ?\Dom\Node $parentNode;

		/**
		 * @var ?Dom\Element <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentelement
		 */
		public ?\Dom\Element $parentElement;

		/**
		 * @var Dom\NodeList A <code>Dom\NodeList</code> that contains all children of this node. If there are no children, this is an empty <code>Dom\NodeList</code>.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.childnodes
		 */
		public \Dom\NodeList $childNodes;

		/**
		 * @var ?Dom\Node <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.firstchild
		 */
		public ?\Dom\Node $firstChild;

		/**
		 * @var ?Dom\Node <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.lastchild
		 */
		public ?\Dom\Node $lastChild;

		/**
		 * @var ?Dom\Node <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.previoussibling
		 */
		public ?\Dom\Node $previousSibling;

		/**
		 * @var ?Dom\Node <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nextsibling
		 */
		public ?\Dom\Node $nextSibling;

		/**
		 * @var ?string The value of this node, depending on its type.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var ?string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.textcontent
		 */
		public ?string $textContent;
	}

	/**
	 * <p>This represents a document fragment, which can be used as a container for other nodes.</p>
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMDocumentFragment</code>.</p>
	 * @link https://php.net/manual/en/class.dom-documentfragment.php
	 * @since PHP 8 >= 8.4.0
	 */
	class DocumentFragment extends \Dom\Node implements \Dom\ParentNode {

		/**
		 * @var ?Dom\Element <p>First child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-documentfragment.php#dom-documentfragment.props.firstelementchild
		 */
		public ?\Dom\Element $firstElementChild;

		/**
		 * @var ?Dom\Element <p>Last child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-documentfragment.php#dom-documentfragment.props.lastelementchild
		 */
		public ?\Dom\Element $lastElementChild;

		/**
		 * @var int <p>The number of child elements.</p>
		 * @link https://php.net/manual/en/class.dom-documentfragment.php#dom-documentfragment.props.childelementcount
		 */
		public int $childElementCount;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string Returns the most accurate name for the current node type. <ul> <li>For elements, this is the HTML-uppercased qualified name.</li> <li>For attributes, this is the qualified name.</li> <li>For processing instructions, this is the target.</li> <li>For document type nodes, this is the name.</li> </ul>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.baseuri
		 */
		public string $baseURI;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?Dom\Document The <code>Dom\Document</code> object associated with this node, or <b><code>null</code></b> if this node is a document.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.ownerdocument
		 */
		public ?\Dom\Document $ownerDocument;

		/**
		 * @var ?Dom\Node <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentnode
		 */
		public ?\Dom\Node $parentNode;

		/**
		 * @var ?Dom\Element <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentelement
		 */
		public ?\Dom\Element $parentElement;

		/**
		 * @var Dom\NodeList A <code>Dom\NodeList</code> that contains all children of this node. If there are no children, this is an empty <code>Dom\NodeList</code>.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.childnodes
		 */
		public \Dom\NodeList $childNodes;

		/**
		 * @var ?Dom\Node <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.firstchild
		 */
		public ?\Dom\Node $firstChild;

		/**
		 * @var ?Dom\Node <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.lastchild
		 */
		public ?\Dom\Node $lastChild;

		/**
		 * @var ?Dom\Node <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.previoussibling
		 */
		public ?\Dom\Node $previousSibling;

		/**
		 * @var ?Dom\Node <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nextsibling
		 */
		public ?\Dom\Node $nextSibling;

		/**
		 * @var ?string The value of this node, depending on its type.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var ?string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.textcontent
		 */
		public ?string $textContent;
	}

	/**
	 * <p>Each <code>Dom\Document</code> has a <code>doctype</code> attribute whose value is either <b><code>null</code></b> or a <b>Dom\DocumentType</b> object.</p>
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMImplementation</code>.</p>
	 * @link https://php.net/manual/en/class.dom-documenttype.php
	 * @since PHP 8 >= 8.4.0
	 */
	class DocumentType extends \Dom\Node implements \Dom\ChildNode {

		/**
		 * @var string <p>The name of DTD; i.e., the name immediately following the <code>DOCTYPE</code> keyword.</p>
		 * @link https://php.net/manual/en/class.dom-documenttype.php#dom-documenttype.props.name
		 */
		public string $name;

		/**
		 * @var Dom\DtdNamedNodeMap A <code>Dom\DtdNamedNodeMap</code> containing the general entities, both external and internal, declared in the DTD.
		 * @link https://php.net/manual/en/class.dom-documenttype.php#dom-documenttype.props.entities
		 */
		public \Dom\DtdNamedNodeMap $entities;

		/**
		 * @var Dom\DtdNamedNodeMap A <code>Dom\DtdNamedNodeMap</code> containing the notations declared in the DTD.
		 * @link https://php.net/manual/en/class.dom-documenttype.php#dom-documenttype.props.notations
		 */
		public \Dom\DtdNamedNodeMap $notations;

		/**
		 * @var string <p>The public identifier of the external subset.</p>
		 * @link https://php.net/manual/en/class.dom-documenttype.php#dom-documenttype.props.publicid
		 */
		public string $publicId;

		/**
		 * @var string <p>The system identifier of the external subset. This may be an absolute URI or not.</p>
		 * @link https://php.net/manual/en/class.dom-documenttype.php#dom-documenttype.props.systemid
		 */
		public string $systemId;

		/**
		 * @var ?string <p>The internal subset as a string, or <b><code>null</code></b> if there is none. This does not contain the delimiting square brackets.</p>
		 * @link https://php.net/manual/en/class.dom-documenttype.php#dom-documenttype.props.internalsubset
		 */
		public ?string $internalSubset;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string Returns the most accurate name for the current node type. <ul> <li>For elements, this is the HTML-uppercased qualified name.</li> <li>For attributes, this is the qualified name.</li> <li>For processing instructions, this is the target.</li> <li>For document type nodes, this is the name.</li> </ul>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.baseuri
		 */
		public string $baseURI;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?Dom\Document The <code>Dom\Document</code> object associated with this node, or <b><code>null</code></b> if this node is a document.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.ownerdocument
		 */
		public ?\Dom\Document $ownerDocument;

		/**
		 * @var ?Dom\Node <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentnode
		 */
		public ?\Dom\Node $parentNode;

		/**
		 * @var ?Dom\Element <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentelement
		 */
		public ?\Dom\Element $parentElement;

		/**
		 * @var Dom\NodeList A <code>Dom\NodeList</code> that contains all children of this node. If there are no children, this is an empty <code>Dom\NodeList</code>.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.childnodes
		 */
		public \Dom\NodeList $childNodes;

		/**
		 * @var ?Dom\Node <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.firstchild
		 */
		public ?\Dom\Node $firstChild;

		/**
		 * @var ?Dom\Node <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.lastchild
		 */
		public ?\Dom\Node $lastChild;

		/**
		 * @var ?Dom\Node <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.previoussibling
		 */
		public ?\Dom\Node $previousSibling;

		/**
		 * @var ?Dom\Node <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nextsibling
		 */
		public ?\Dom\Node $nextSibling;

		/**
		 * @var ?string The value of this node, depending on its type.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var ?string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.textcontent
		 */
		public ?string $textContent;
	}

	/**
	 * <p>Represents a named node map for entities and notation nodes of the DTD.</p>
	 * @link https://php.net/manual/en/class.dom-dtdnamednodemap.php
	 * @since PHP 8 >= 8.4.0
	 */
	class DtdNamedNodeMap implements \IteratorAggregate, \Countable {

		/**
		 * @var int The total number of entities and notation nodes.
		 * @link https://php.net/manual/en/class.dom-dtdnamednodemap.php#dom-dtdnamednodemap.props.length
		 */
		public int $length;
	}

	/**
	 * <p>Represents an element.</p>
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMElement</code>.</p>
	 * @link https://php.net/manual/en/class.dom-element.php
	 * @since PHP 8 >= 8.4.0
	 */
	class Element extends \Dom\Node implements \Dom\ParentNode, \Dom\ChildNode {

		/**
		 * @var ?string The namespace URI of the element.
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var ?string The namespace prefix of the element.
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.prefix
		 */
		public ?string $prefix;

		/**
		 * @var string The local name of the element.
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.localname
		 */
		public string $localName;

		/**
		 * @var string The HTML-uppercased qualified name of the element.
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.tagname
		 */
		public string $tagName;

		/**
		 * @var string <p>Reflects the element ID through the <code>"id"</code> attribute.</p>
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.id
		 */
		public string $id;

		/**
		 * @var string <p>A string representing the classes of the element separated by spaces.</p>
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.classname
		 */
		public string $className;

		/**
		 * @var Dom\TokenList Returns an instance of <code>Dom\TokenList</code> to easily manage the classes on this element.
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.classlist
		 */
		public \Dom\TokenList $classList;

		/**
		 * @var Dom\NamedNodeMap Returns an instance of <code>Dom\NamedNodeMap</code> that represents the attributes of this element.
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.attributes
		 */
		public \Dom\NamedNodeMap $attributes;

		/**
		 * @var ?Dom\Element <p>First child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.firstelementchild
		 */
		public ?\Dom\Element $firstElementChild;

		/**
		 * @var ?Dom\Element <p>Last child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.lastelementchild
		 */
		public ?\Dom\Element $lastElementChild;

		/**
		 * @var int <p>The number of child elements.</p>
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.childelementcount
		 */
		public int $childElementCount;

		/**
		 * @var ?Dom\Element <p>The previous sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.previouselementsibling
		 */
		public ?\Dom\Element $previousElementSibling;

		/**
		 * @var ?Dom\Element <p>The next sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.nextelementsibling
		 */
		public ?\Dom\Element $nextElementSibling;

		/**
		 * @var string The inner HTML (or XML for XML documents) of the element.
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.innerhtml
		 */
		public string $innerHTML;

		/**
		 * @var string The node value with entity substitution enabled.
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.substitutednodevalue
		 */
		public string $substitutedNodeValue;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string Returns the most accurate name for the current node type. <ul> <li>For elements, this is the HTML-uppercased qualified name.</li> <li>For attributes, this is the qualified name.</li> <li>For processing instructions, this is the target.</li> <li>For document type nodes, this is the name.</li> </ul>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.baseuri
		 */
		public string $baseURI;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?Dom\Document The <code>Dom\Document</code> object associated with this node, or <b><code>null</code></b> if this node is a document.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.ownerdocument
		 */
		public ?\Dom\Document $ownerDocument;

		/**
		 * @var ?Dom\Node <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentnode
		 */
		public ?\Dom\Node $parentNode;

		/**
		 * @var ?Dom\Element <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentelement
		 */
		public ?\Dom\Element $parentElement;

		/**
		 * @var Dom\NodeList A <code>Dom\NodeList</code> that contains all children of this node. If there are no children, this is an empty <code>Dom\NodeList</code>.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.childnodes
		 */
		public \Dom\NodeList $childNodes;

		/**
		 * @var ?Dom\Node <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.firstchild
		 */
		public ?\Dom\Node $firstChild;

		/**
		 * @var ?Dom\Node <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.lastchild
		 */
		public ?\Dom\Node $lastChild;

		/**
		 * @var ?Dom\Node <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.previoussibling
		 */
		public ?\Dom\Node $previousSibling;

		/**
		 * @var ?Dom\Node <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nextsibling
		 */
		public ?\Dom\Node $nextSibling;

		/**
		 * @var ?string The value of this node, depending on its type.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var ?string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.textcontent
		 */
		public ?string $textContent;
	}

	/**
	 * <p>This interface represents a known entity, either parsed or unparsed, in an XML document.</p>
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMEntity</code>.</p>
	 * @link https://php.net/manual/en/class.dom-entity.php
	 * @since PHP 8 >= 8.4.0
	 */
	class Entity extends \Dom\Node {

		/**
		 * @var ?string <p>The public identifier associated with the entity if specified, and <b><code>null</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/class.dom-entity.php#dom-entity.props.publicid
		 */
		public ?string $publicId;

		/**
		 * @var ?string <p>The system identifier associated with the entity if specified, and <b><code>null</code></b> otherwise. This may be an absolute URI or not.</p>
		 * @link https://php.net/manual/en/class.dom-entity.php#dom-entity.props.systemid
		 */
		public ?string $systemId;

		/**
		 * @var ?string <p>For unparsed entities, the name of the notation for the entity. For parsed entities, this is <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-entity.php#dom-entity.props.notationname
		 */
		public ?string $notationName;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string Returns the most accurate name for the current node type. <ul> <li>For elements, this is the HTML-uppercased qualified name.</li> <li>For attributes, this is the qualified name.</li> <li>For processing instructions, this is the target.</li> <li>For document type nodes, this is the name.</li> </ul>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.baseuri
		 */
		public string $baseURI;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?Dom\Document The <code>Dom\Document</code> object associated with this node, or <b><code>null</code></b> if this node is a document.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.ownerdocument
		 */
		public ?\Dom\Document $ownerDocument;

		/**
		 * @var ?Dom\Node <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentnode
		 */
		public ?\Dom\Node $parentNode;

		/**
		 * @var ?Dom\Element <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentelement
		 */
		public ?\Dom\Element $parentElement;

		/**
		 * @var Dom\NodeList A <code>Dom\NodeList</code> that contains all children of this node. If there are no children, this is an empty <code>Dom\NodeList</code>.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.childnodes
		 */
		public \Dom\NodeList $childNodes;

		/**
		 * @var ?Dom\Node <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.firstchild
		 */
		public ?\Dom\Node $firstChild;

		/**
		 * @var ?Dom\Node <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.lastchild
		 */
		public ?\Dom\Node $lastChild;

		/**
		 * @var ?Dom\Node <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.previoussibling
		 */
		public ?\Dom\Node $previousSibling;

		/**
		 * @var ?Dom\Node <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nextsibling
		 */
		public ?\Dom\Node $nextSibling;

		/**
		 * @var ?string The value of this node, depending on its type.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var ?string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.textcontent
		 */
		public ?string $textContent;
	}

	/**
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMEntityReference</code>.</p>
	 * @link https://php.net/manual/en/class.dom-entityreference.php
	 * @since PHP 8 >= 8.4.0
	 */
	class EntityReference extends \Dom\Node {

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string Returns the most accurate name for the current node type. <ul> <li>For elements, this is the HTML-uppercased qualified name.</li> <li>For attributes, this is the qualified name.</li> <li>For processing instructions, this is the target.</li> <li>For document type nodes, this is the name.</li> </ul>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.baseuri
		 */
		public string $baseURI;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?Dom\Document The <code>Dom\Document</code> object associated with this node, or <b><code>null</code></b> if this node is a document.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.ownerdocument
		 */
		public ?\Dom\Document $ownerDocument;

		/**
		 * @var ?Dom\Node <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentnode
		 */
		public ?\Dom\Node $parentNode;

		/**
		 * @var ?Dom\Element <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentelement
		 */
		public ?\Dom\Element $parentElement;

		/**
		 * @var Dom\NodeList A <code>Dom\NodeList</code> that contains all children of this node. If there are no children, this is an empty <code>Dom\NodeList</code>.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.childnodes
		 */
		public \Dom\NodeList $childNodes;

		/**
		 * @var ?Dom\Node <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.firstchild
		 */
		public ?\Dom\Node $firstChild;

		/**
		 * @var ?Dom\Node <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.lastchild
		 */
		public ?\Dom\Node $lastChild;

		/**
		 * @var ?Dom\Node <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.previoussibling
		 */
		public ?\Dom\Node $previousSibling;

		/**
		 * @var ?Dom\Node <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nextsibling
		 */
		public ?\Dom\Node $nextSibling;

		/**
		 * @var ?string The value of this node, depending on its type.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var ?string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.textcontent
		 */
		public ?string $textContent;
	}

	/**
	 * <p>Represents a static set of elements.</p>
	 * @link https://php.net/manual/en/class.dom-htmlcollection.php
	 * @since PHP 8 >= 8.4.0
	 */
	class HTMLCollection implements \IteratorAggregate, \Countable {

		/**
		 * @var int The number of elements.
		 * @link https://php.net/manual/en/class.dom-htmlcollection.php#dom-htmlcollection.props.length
		 */
		public int $length;
	}

	/**
	 * <p>Represents an HTML document.</p>
	 * @link https://php.net/manual/en/class.dom-htmldocument.php
	 * @since PHP 8 >= 8.4.0
	 */
	final class HTMLDocument extends \Dom\Document {

		/**
		 * @var Dom\Implementation <p>The <code>DOMImplementation</code> object that handles this document.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.implementation
		 */
		public \Dom\Implementation $implementation;

		/**
		 * @var string Equivalent to documentURI.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.url
		 */
		public string $URL;

		/**
		 * @var string <p>The location of the document or <b><code>null</code></b> if undefined.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.documenturi
		 */
		public string $documentURI;

		/**
		 * @var string The encoding of the document used for serialization. Upon parsing a document, this is set to the input encoding of that document.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.characterset
		 */
		public string $characterSet;

		/**
		 * @var string Legacy alias for characterSet.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.charset
		 */
		public string $charset;

		/**
		 * @var string Legacy alias for characterSet.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.inputencoding
		 */
		public string $inputEncoding;

		/**
		 * @var ?Dom\DocumentType <p>The Document Type Declaration associated with this document.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.doctype
		 */
		public ?\Dom\DocumentType $doctype;

		/**
		 * @var ?Dom\Element The <code>Dom\Element</code> that is the document element. This evaluates to <b><code>null</code></b> for document without elements.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.documentelement
		 */
		public ?\Dom\Element $documentElement;

		/**
		 * @var ?Dom\Element <p>First child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.firstelementchild
		 */
		public ?\Dom\Element $firstElementChild;

		/**
		 * @var ?Dom\Element <p>Last child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.lastelementchild
		 */
		public ?\Dom\Element $lastElementChild;

		/**
		 * @var int <p>The number of child elements.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.childelementcount
		 */
		public int $childElementCount;

		/**
		 * @var ?Dom\HTMLElement The first child of the <code>html</code> element that is either a <code>body</code> tag or a <code>frameset</code> tag. These need to be in the HTML namespace. If no element matches, this evaluates to <b><code>null</code></b>.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.body
		 */
		public ?\Dom\HTMLElement $body;

		/**
		 * @var ?Dom\HTMLElement The first <code>head</code> element that is a child of the <code>html</code> element. These need to be in the HTML namespace. If no element matches, this evaluates to <b><code>null</code></b>.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.head
		 */
		public ?\Dom\HTMLElement $head;

		/**
		 * @var string The title of the document as set by the <code>title</code> element for HTML or the SVG <code>title</code> element for SVG. If there is no title, this evaluates to the empty string.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.title
		 */
		public string $title;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string Returns the most accurate name for the current node type. <ul> <li>For elements, this is the HTML-uppercased qualified name.</li> <li>For attributes, this is the qualified name.</li> <li>For processing instructions, this is the target.</li> <li>For document type nodes, this is the name.</li> </ul>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.baseuri
		 */
		public string $baseURI;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?Dom\Document The <code>Dom\Document</code> object associated with this node, or <b><code>null</code></b> if this node is a document.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.ownerdocument
		 */
		public ?\Dom\Document $ownerDocument;

		/**
		 * @var ?Dom\Node <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentnode
		 */
		public ?\Dom\Node $parentNode;

		/**
		 * @var ?Dom\Element <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentelement
		 */
		public ?\Dom\Element $parentElement;

		/**
		 * @var Dom\NodeList A <code>Dom\NodeList</code> that contains all children of this node. If there are no children, this is an empty <code>Dom\NodeList</code>.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.childnodes
		 */
		public \Dom\NodeList $childNodes;

		/**
		 * @var ?Dom\Node <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.firstchild
		 */
		public ?\Dom\Node $firstChild;

		/**
		 * @var ?Dom\Node <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.lastchild
		 */
		public ?\Dom\Node $lastChild;

		/**
		 * @var ?Dom\Node <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.previoussibling
		 */
		public ?\Dom\Node $previousSibling;

		/**
		 * @var ?Dom\Node <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nextsibling
		 */
		public ?\Dom\Node $nextSibling;

		/**
		 * @var ?string The value of this node, depending on its type.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var ?string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.textcontent
		 */
		public ?string $textContent;
	}

	/**
	 * <p>Represents an element in the HTML namespace.</p>
	 * @link https://php.net/manual/en/class.dom-htmlelement.php
	 * @since PHP 8 >= 8.4.0
	 */
	class HTMLElement extends \Dom\Element {

		/**
		 * @var ?string The namespace URI of the element.
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var ?string The namespace prefix of the element.
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.prefix
		 */
		public ?string $prefix;

		/**
		 * @var string The local name of the element.
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.localname
		 */
		public string $localName;

		/**
		 * @var string The HTML-uppercased qualified name of the element.
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.tagname
		 */
		public string $tagName;

		/**
		 * @var string <p>Reflects the element ID through the <code>"id"</code> attribute.</p>
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.id
		 */
		public string $id;

		/**
		 * @var string <p>A string representing the classes of the element separated by spaces.</p>
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.classname
		 */
		public string $className;

		/**
		 * @var Dom\TokenList Returns an instance of <code>Dom\TokenList</code> to easily manage the classes on this element.
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.classlist
		 */
		public \Dom\TokenList $classList;

		/**
		 * @var Dom\NamedNodeMap Returns an instance of <code>Dom\NamedNodeMap</code> that represents the attributes of this element.
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.attributes
		 */
		public \Dom\NamedNodeMap $attributes;

		/**
		 * @var ?Dom\Element <p>First child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.firstelementchild
		 */
		public ?\Dom\Element $firstElementChild;

		/**
		 * @var ?Dom\Element <p>Last child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.lastelementchild
		 */
		public ?\Dom\Element $lastElementChild;

		/**
		 * @var int <p>The number of child elements.</p>
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.childelementcount
		 */
		public int $childElementCount;

		/**
		 * @var ?Dom\Element <p>The previous sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.previouselementsibling
		 */
		public ?\Dom\Element $previousElementSibling;

		/**
		 * @var ?Dom\Element <p>The next sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.nextelementsibling
		 */
		public ?\Dom\Element $nextElementSibling;

		/**
		 * @var string The inner HTML (or XML for XML documents) of the element.
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.innerhtml
		 */
		public string $innerHTML;

		/**
		 * @var string The node value with entity substitution enabled.
		 * @link https://php.net/manual/en/class.dom-element.php#dom-element.props.substitutednodevalue
		 */
		public string $substitutedNodeValue;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string Returns the most accurate name for the current node type. <ul> <li>For elements, this is the HTML-uppercased qualified name.</li> <li>For attributes, this is the qualified name.</li> <li>For processing instructions, this is the target.</li> <li>For document type nodes, this is the name.</li> </ul>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.baseuri
		 */
		public string $baseURI;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?Dom\Document The <code>Dom\Document</code> object associated with this node, or <b><code>null</code></b> if this node is a document.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.ownerdocument
		 */
		public ?\Dom\Document $ownerDocument;

		/**
		 * @var ?Dom\Node <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentnode
		 */
		public ?\Dom\Node $parentNode;

		/**
		 * @var ?Dom\Element <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentelement
		 */
		public ?\Dom\Element $parentElement;

		/**
		 * @var Dom\NodeList A <code>Dom\NodeList</code> that contains all children of this node. If there are no children, this is an empty <code>Dom\NodeList</code>.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.childnodes
		 */
		public \Dom\NodeList $childNodes;

		/**
		 * @var ?Dom\Node <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.firstchild
		 */
		public ?\Dom\Node $firstChild;

		/**
		 * @var ?Dom\Node <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.lastchild
		 */
		public ?\Dom\Node $lastChild;

		/**
		 * @var ?Dom\Node <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.previoussibling
		 */
		public ?\Dom\Node $previousSibling;

		/**
		 * @var ?Dom\Node <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nextsibling
		 */
		public ?\Dom\Node $nextSibling;

		/**
		 * @var ?string The value of this node, depending on its type.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var ?string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.textcontent
		 */
		public ?string $textContent;
	}

	/**
	 * <p>This class provides a number of methods for performing operations that are independent of any particular instance of the document object model.</p>
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMImplementation</code>.</p>
	 * @link https://php.net/manual/en/class.dom-implementation.php
	 * @since PHP 8 >= 8.4.0
	 */
	class Implementation {
	}

	/**
	 * <p>Represents the set of attributes on an element.</p>
	 * @link https://php.net/manual/en/class.dom-namednodemap.php
	 * @since PHP 8 >= 8.4.0
	 */
	class NamedNodeMap implements \IteratorAggregate, \Countable {

		/**
		 * @var int The number of attributes.
		 * @link https://php.net/manual/en/class.dom-namednodemap.php#dom-namednodemap.props.length
		 */
		public int $length;
	}

	/**
	 * <p>This represents immutable information about namespaces of an element. This decouples namespaces from attributes, which was incorrectly intertwined for the old DOM classes.</p>
	 * @link https://php.net/manual/en/class.dom-namespaceinfo.php
	 * @since PHP 8 >= 8.4.0
	 */
	final readonly class NamespaceInfo {

		/**
		 * @var ?string The namespace prefix of the attribute.
		 * @link https://php.net/manual/en/class.dom-namespaceinfo.php#dom-namespaceinfo.props.prefix
		 */
		public ?string $prefix;

		/**
		 * @var ?string The namespace URI of the attribute.
		 * @link https://php.net/manual/en/class.dom-namespaceinfo.php#dom-namespaceinfo.props.namespaceuri
		 */
		public ?string $namespaceURI;

		/**
		 * @var Dom\Element The element that this namespace information is about.
		 * @link https://php.net/manual/en/class.dom-namespaceinfo.php#dom-namespaceinfo.props.element
		 */
		public \Dom\Element $element;
	}

	/**
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMNode</code>.</p>
	 * @link https://php.net/manual/en/class.dom-node.php
	 * @since PHP 8 >= 8.4.0
	 */
	class Node {

		/**
		 * @var int Set when the other node and reference node are not in the same tree.
		 * @link https://php.net/manual/en/class.dom-node.php
		 */
		public const int DOCUMENT_POSITION_DISCONNECTED = 0x1;

		/**
		 * @var int Set when the other node precedes the reference node.
		 * @link https://php.net/manual/en/class.dom-node.php
		 */
		public const int DOCUMENT_POSITION_PRECEDING = 0x2;

		/**
		 * @var int Set when the other node follows the reference node.
		 * @link https://php.net/manual/en/class.dom-node.php
		 */
		public const int DOCUMENT_POSITION_FOLLOWING = 0x4;

		/**
		 * @var int Set when the other node is an ancestor of the reference node.
		 * @link https://php.net/manual/en/class.dom-node.php
		 */
		public const int DOCUMENT_POSITION_CONTAINS = 0x8;

		/**
		 * @var int Set when the other node is a descendant of the reference node.
		 * @link https://php.net/manual/en/class.dom-node.php
		 */
		public const int DOCUMENT_POSITION_CONTAINED_BY = 0x10;

		/**
		 * @var int Set when the result depends on implementation-specific behaviour and may not be portable. This may happen with disconnected nodes or with attribute nodes.
		 * @link https://php.net/manual/en/class.dom-node.php
		 */
		public const int DOCUMENT_POSITION_IMPLEMENTATION_SPECIFIC = 0x20;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string Returns the most accurate name for the current node type. <ul> <li>For elements, this is the HTML-uppercased qualified name.</li> <li>For attributes, this is the qualified name.</li> <li>For processing instructions, this is the target.</li> <li>For document type nodes, this is the name.</li> </ul>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.baseuri
		 */
		public string $baseURI;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?Dom\Document The <code>Dom\Document</code> object associated with this node, or <b><code>null</code></b> if this node is a document.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.ownerdocument
		 */
		public ?\Dom\Document $ownerDocument;

		/**
		 * @var ?Dom\Node <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentnode
		 */
		public ?\Dom\Node $parentNode;

		/**
		 * @var ?Dom\Element <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentelement
		 */
		public ?\Dom\Element $parentElement;

		/**
		 * @var Dom\NodeList A <code>Dom\NodeList</code> that contains all children of this node. If there are no children, this is an empty <code>Dom\NodeList</code>.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.childnodes
		 */
		public \Dom\NodeList $childNodes;

		/**
		 * @var ?Dom\Node <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.firstchild
		 */
		public ?\Dom\Node $firstChild;

		/**
		 * @var ?Dom\Node <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.lastchild
		 */
		public ?\Dom\Node $lastChild;

		/**
		 * @var ?Dom\Node <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.previoussibling
		 */
		public ?\Dom\Node $previousSibling;

		/**
		 * @var ?Dom\Node <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nextsibling
		 */
		public ?\Dom\Node $nextSibling;

		/**
		 * @var ?string The value of this node, depending on its type.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var ?string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.textcontent
		 */
		public ?string $textContent;
	}

	/**
	 * <p>Represents a live list of nodes.</p>
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMNodeList</code>.</p>
	 * @link https://php.net/manual/en/class.dom-nodelist.php
	 * @since PHP 8 >= 8.4.0
	 */
	class NodeList implements \IteratorAggregate, \Countable {

		/**
		 * @var int <p>The number of nodes in the list. The range of valid child node indices is 0 to <code>length - 1</code> inclusive.</p>
		 * @link https://php.net/manual/en/class.dom-nodelist.php#dom-nodelist.props.length
		 */
		public int $length;
	}

	/**
	 * @link https://php.net/manual/en/class.dom-notation.php
	 * @since PHP 8 >= 8.4.0
	 */
	class Notation extends \Dom\Node {

		/**
		 * @var string The public identifier associated with the notation.
		 * @link https://php.net/manual/en/class.dom-notation.php#dom-notation.props.publicid
		 */
		public string $publicId;

		/**
		 * @var string The system identifier associated with the notation.
		 * @link https://php.net/manual/en/class.dom-notation.php#dom-notation.props.systemid
		 */
		public string $systemId;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string Returns the most accurate name for the current node type. <ul> <li>For elements, this is the HTML-uppercased qualified name.</li> <li>For attributes, this is the qualified name.</li> <li>For processing instructions, this is the target.</li> <li>For document type nodes, this is the name.</li> </ul>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.baseuri
		 */
		public string $baseURI;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?Dom\Document The <code>Dom\Document</code> object associated with this node, or <b><code>null</code></b> if this node is a document.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.ownerdocument
		 */
		public ?\Dom\Document $ownerDocument;

		/**
		 * @var ?Dom\Node <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentnode
		 */
		public ?\Dom\Node $parentNode;

		/**
		 * @var ?Dom\Element <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentelement
		 */
		public ?\Dom\Element $parentElement;

		/**
		 * @var Dom\NodeList A <code>Dom\NodeList</code> that contains all children of this node. If there are no children, this is an empty <code>Dom\NodeList</code>.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.childnodes
		 */
		public \Dom\NodeList $childNodes;

		/**
		 * @var ?Dom\Node <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.firstchild
		 */
		public ?\Dom\Node $firstChild;

		/**
		 * @var ?Dom\Node <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.lastchild
		 */
		public ?\Dom\Node $lastChild;

		/**
		 * @var ?Dom\Node <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.previoussibling
		 */
		public ?\Dom\Node $previousSibling;

		/**
		 * @var ?Dom\Node <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nextsibling
		 */
		public ?\Dom\Node $nextSibling;

		/**
		 * @var ?string The value of this node, depending on its type.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var ?string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.textcontent
		 */
		public ?string $textContent;
	}

	/**
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMParentNode</code>.</p>
	 * @link https://php.net/manual/en/class.dom-parentnode.php
	 * @since PHP 8 >= 8.4.0
	 */
	interface ParentNode {

		/**
		 * Appends nodes after the last child node
		 * <p>Appends one or many <code>nodes</code> to the list of children after the last child node.</p>
		 * @param \Dom\Node|string $nodes <p>The nodes to append. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-parentnode.append.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function append(\Dom\Node|string ...$nodes): void;

		/**
		 * Prepends nodes before the first child node
		 * <p>Prepends one or many <code>nodes</code> to the list of children before the first child node.</p>
		 * @param \Dom\Node|string $nodes <p>The nodes to prepend. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-parentnode.prepend.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function prepend(\Dom\Node|string ...$nodes): void;

		/**
		 * Returns the first element that matches the CSS selectors
		 * <p>Returns the first element that matches the CSS selectors specified in <code>selectors</code>.</p>
		 * @param string $selectors A string containing one or more CSS selectors.
		 * @return ?Dom\Element <p>Returns the first <code>Dom\Element</code> that matches <code>selectors</code>. Returns <b><code>null</code></b> if no element matches.</p>
		 * @link https://php.net/manual/en/dom-parentnode.queryselector.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function querySelector(string $selectors): ?\Dom\Element;

		/**
		 * Returns a collection of elements that match the CSS selectors
		 * <p>Returns a collection of elements that match the CSS selectors specified in <code>selectors</code>.</p>
		 * @param string $selectors A string containing one or more CSS selectors.
		 * @return Dom\NodeList <p>Returns a static collection of elements that match the CSS selectors specified in <code>selectors</code>.</p>
		 * @link https://php.net/manual/en/dom-parentnode.queryselectorall.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function querySelectorAll(string $selectors): \Dom\NodeList;

		/**
		 * Replace children in node
		 * <p>Replace children in node.</p>
		 * @param \Dom\Node|string $nodes <p>The nodes replacing the children. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-parentnode.replacechildren.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function replaceChildren(\Dom\Node|string ...$nodes): void;
	}

	/**
	 * <p>This represents a processing instruction (PI) node. These are meant to indicate data areas meant for processing by specific applications.</p>
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMProcessingInstruction</code>.</p>
	 * @link https://php.net/manual/en/class.dom-processinginstruction.php
	 * @since PHP 8 >= 8.4.0
	 */
	class ProcessingInstruction extends \Dom\CharacterData {

		/**
		 * @var string A string representing to what application the data is intended for.
		 * @link https://php.net/manual/en/class.dom-processinginstruction.php#dom-processinginstruction.props.target
		 */
		public string $target;

		/**
		 * @var ?Dom\Element <p>The previous sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.previouselementsibling
		 */
		public ?\Dom\Element $previousElementSibling;

		/**
		 * @var ?Dom\Element <p>The next sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.nextelementsibling
		 */
		public ?\Dom\Element $nextElementSibling;

		/**
		 * @var string <p>The contents of the node.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.data
		 */
		public string $data;

		/**
		 * @var int <p>The length of the contents.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.length
		 */
		public int $length;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string Returns the most accurate name for the current node type. <ul> <li>For elements, this is the HTML-uppercased qualified name.</li> <li>For attributes, this is the qualified name.</li> <li>For processing instructions, this is the target.</li> <li>For document type nodes, this is the name.</li> </ul>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.baseuri
		 */
		public string $baseURI;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?Dom\Document The <code>Dom\Document</code> object associated with this node, or <b><code>null</code></b> if this node is a document.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.ownerdocument
		 */
		public ?\Dom\Document $ownerDocument;

		/**
		 * @var ?Dom\Node <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentnode
		 */
		public ?\Dom\Node $parentNode;

		/**
		 * @var ?Dom\Element <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentelement
		 */
		public ?\Dom\Element $parentElement;

		/**
		 * @var Dom\NodeList A <code>Dom\NodeList</code> that contains all children of this node. If there are no children, this is an empty <code>Dom\NodeList</code>.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.childnodes
		 */
		public \Dom\NodeList $childNodes;

		/**
		 * @var ?Dom\Node <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.firstchild
		 */
		public ?\Dom\Node $firstChild;

		/**
		 * @var ?Dom\Node <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.lastchild
		 */
		public ?\Dom\Node $lastChild;

		/**
		 * @var ?Dom\Node <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.previoussibling
		 */
		public ?\Dom\Node $previousSibling;

		/**
		 * @var ?Dom\Node <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nextsibling
		 */
		public ?\Dom\Node $nextSibling;

		/**
		 * @var ?string The value of this node, depending on its type.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var ?string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.textcontent
		 */
		public ?string $textContent;

		/**
		 * Adds nodes after the character data
		 * <p>Adds the passed <code>nodes</code> after the character data.</p>
		 * @param \Dom\Node|string $nodes <p>Nodes to be added after the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.after.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function after(\Dom\Node|string ...$nodes): void {}

		/**
		 * Append the string to the end of the character data of the node
		 * <p>Append the string <code>data</code> to the end of the character data of the node.</p>
		 * @param string $data <p>The string to append.</p>
		 * @return void <p>Always returns <b><code>true</code></b>.</p>
		 * @link https://php.net/manual/en/dom-characterdata.appenddata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function appendData(string $data): void {}

		/**
		 * Adds nodes before the character data
		 * <p>Adds the passed <code>nodes</code> before the character data.</p>
		 * @param \Dom\Node|string $nodes <p>Nodes to be added before the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.before.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function before(\Dom\Node|string ...$nodes): void {}

		/**
		 * Remove a range of characters from the character data
		 * <p>Deletes <code>count</code> characters starting from position <code>offset</code>.</p>
		 * @param int $offset <p>The offset from which to start removing.</p>
		 * @param int $count <p>The number of characters to delete. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are deleted.</p>
		 * @return void <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/dom-characterdata.deletedata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function deleteData(int $offset, int $count): void {}

		/**
		 * Insert a string at the specified UTF-8 codepoint offset
		 * <p>Inserts string <code>data</code> at position <code>offset</code>.</p>
		 * @param int $offset <p>The character offset at which to insert.</p>
		 * @param string $data <p>The string to insert.</p>
		 * @return void <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/dom-characterdata.insertdata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function insertData(int $offset, string $data): void {}

		/**
		 * Removes the character data node
		 * <p>Removes the character data node.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.remove.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function remove(): void {}

		/**
		 * Replace a substring within the character data
		 * <p>Replace <code>count</code> characters starting from position <code>offset</code> with <code>data</code>.</p>
		 * @param int $offset <p>The offset from which to start replacing.</p>
		 * @param int $count <p>The number of characters to replace. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are replaced.</p>
		 * @param string $data <p>The string with which the range must be replaced.</p>
		 * @return void <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/dom-characterdata.replacedata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function replaceData(int $offset, int $count, string $data): void {}

		/**
		 * Replaces the character data with new nodes
		 * <p>Replaces the character data with new <code>nodes</code>.</p>
		 * @param \Dom\Node|string $nodes <p>The replacement nodes. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.replacewith.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function replaceWith(\Dom\Node|string ...$nodes): void {}

		/**
		 * Extracts a range of data from the character data
		 * <p>Returns the specified substring.</p>
		 * @param int $offset <p>Start offset of substring to extract.</p>
		 * @param int $count <p>The number of characters to extract.</p>
		 * @return string <p>The specified substring. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all UTF-8 codepoints to the end of the data are returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.substringdata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function substringData(int $offset, int $count): string {}
	}

	/**
	 * <p>The <b>Dom\Text</b> class inherits from <code>Dom\CharacterData</code> and represents a text node.</p>
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMText</code>.</p>
	 * @link https://php.net/manual/en/class.dom-text.php
	 * @since PHP 8 >= 8.4.0
	 */
	class Text extends \Dom\CharacterData {

		/**
		 * @var string <p>Holds all the text of logically-adjacent (not separated by Element, Comment or Processing Instruction) Text nodes.</p>
		 * @link https://php.net/manual/en/class.dom-text.php#dom-text.props.wholetext
		 */
		public string $wholeText;

		/**
		 * @var ?Dom\Element <p>The previous sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.previouselementsibling
		 */
		public ?\Dom\Element $previousElementSibling;

		/**
		 * @var ?Dom\Element <p>The next sibling element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.nextelementsibling
		 */
		public ?\Dom\Element $nextElementSibling;

		/**
		 * @var string <p>The contents of the node.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.data
		 */
		public string $data;

		/**
		 * @var int <p>The length of the contents.</p>
		 * @link https://php.net/manual/en/class.dom-characterdata.php#dom-characterdata.props.length
		 */
		public int $length;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string Returns the most accurate name for the current node type. <ul> <li>For elements, this is the HTML-uppercased qualified name.</li> <li>For attributes, this is the qualified name.</li> <li>For processing instructions, this is the target.</li> <li>For document type nodes, this is the name.</li> </ul>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.baseuri
		 */
		public string $baseURI;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?Dom\Document The <code>Dom\Document</code> object associated with this node, or <b><code>null</code></b> if this node is a document.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.ownerdocument
		 */
		public ?\Dom\Document $ownerDocument;

		/**
		 * @var ?Dom\Node <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentnode
		 */
		public ?\Dom\Node $parentNode;

		/**
		 * @var ?Dom\Element <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentelement
		 */
		public ?\Dom\Element $parentElement;

		/**
		 * @var Dom\NodeList A <code>Dom\NodeList</code> that contains all children of this node. If there are no children, this is an empty <code>Dom\NodeList</code>.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.childnodes
		 */
		public \Dom\NodeList $childNodes;

		/**
		 * @var ?Dom\Node <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.firstchild
		 */
		public ?\Dom\Node $firstChild;

		/**
		 * @var ?Dom\Node <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.lastchild
		 */
		public ?\Dom\Node $lastChild;

		/**
		 * @var ?Dom\Node <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.previoussibling
		 */
		public ?\Dom\Node $previousSibling;

		/**
		 * @var ?Dom\Node <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nextsibling
		 */
		public ?\Dom\Node $nextSibling;

		/**
		 * @var ?string The value of this node, depending on its type.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var ?string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.textcontent
		 */
		public ?string $textContent;

		/**
		 * Adds nodes after the character data
		 * <p>Adds the passed <code>nodes</code> after the character data.</p>
		 * @param \Dom\Node|string $nodes <p>Nodes to be added after the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.after.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function after(\Dom\Node|string ...$nodes): void {}

		/**
		 * Append the string to the end of the character data of the node
		 * <p>Append the string <code>data</code> to the end of the character data of the node.</p>
		 * @param string $data <p>The string to append.</p>
		 * @return void <p>Always returns <b><code>true</code></b>.</p>
		 * @link https://php.net/manual/en/dom-characterdata.appenddata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function appendData(string $data): void {}

		/**
		 * Adds nodes before the character data
		 * <p>Adds the passed <code>nodes</code> before the character data.</p>
		 * @param \Dom\Node|string $nodes <p>Nodes to be added before the node. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.before.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function before(\Dom\Node|string ...$nodes): void {}

		/**
		 * Remove a range of characters from the character data
		 * <p>Deletes <code>count</code> characters starting from position <code>offset</code>.</p>
		 * @param int $offset <p>The offset from which to start removing.</p>
		 * @param int $count <p>The number of characters to delete. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are deleted.</p>
		 * @return void <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/dom-characterdata.deletedata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function deleteData(int $offset, int $count): void {}

		/**
		 * Insert a string at the specified UTF-8 codepoint offset
		 * <p>Inserts string <code>data</code> at position <code>offset</code>.</p>
		 * @param int $offset <p>The character offset at which to insert.</p>
		 * @param string $data <p>The string to insert.</p>
		 * @return void <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/dom-characterdata.insertdata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function insertData(int $offset, string $data): void {}

		/**
		 * Removes the character data node
		 * <p>Removes the character data node.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.remove.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function remove(): void {}

		/**
		 * Replace a substring within the character data
		 * <p>Replace <code>count</code> characters starting from position <code>offset</code> with <code>data</code>.</p>
		 * @param int $offset <p>The offset from which to start replacing.</p>
		 * @param int $count <p>The number of characters to replace. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all characters to the end of the data are replaced.</p>
		 * @param string $data <p>The string with which the range must be replaced.</p>
		 * @return void <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/dom-characterdata.replacedata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function replaceData(int $offset, int $count, string $data): void {}

		/**
		 * Replaces the character data with new nodes
		 * <p>Replaces the character data with new <code>nodes</code>.</p>
		 * @param \Dom\Node|string $nodes <p>The replacement nodes. Strings are automatically converted to text nodes.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.replacewith.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function replaceWith(\Dom\Node|string ...$nodes): void {}

		/**
		 * Breaks this node into two nodes at the specified offset
		 * <p>Breaks this node into two nodes at the specified <code>offset</code>, keeping both in the tree as siblings.</p>
		 * @param int $offset <p>The offset at which to split, starting from 0.</p>
		 * @return Dom\Text <p>The new node of the same type, which contains all the content at and after the <code>offset</code>.</p>
		 * @link https://php.net/manual/en/dom-text.splittext.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function splitText(int $offset): \Dom\Text {}

		/**
		 * Extracts a range of data from the character data
		 * <p>Returns the specified substring.</p>
		 * @param int $offset <p>Start offset of substring to extract.</p>
		 * @param int $count <p>The number of characters to extract.</p>
		 * @return string <p>The specified substring. If the sum of <code>offset</code> and <code>count</code> exceeds the length, then all UTF-8 codepoints to the end of the data are returned.</p>
		 * @link https://php.net/manual/en/dom-characterdata.substringdata.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function substringData(int $offset, int $count): string {}
	}

	/**
	 * <p>Represents a set of tokens in an attribute (e.g. class names).</p>
	 * @link https://php.net/manual/en/class.dom-tokenlist.php
	 * @since PHP 8 >= 8.4.0
	 */
	final class TokenList implements \IteratorAggregate, \Countable {

		/**
		 * @var int The number of tokens.
		 * @link https://php.net/manual/en/class.dom-tokenlist.php#dom-tokenlist.props.length
		 */
		public int $length;

		/**
		 * @var string The value of the attribute linked to this object.
		 * @link https://php.net/manual/en/class.dom-tokenlist.php#dom-tokenlist.props.value
		 */
		public string $value;

		/**
		 * Adds the given tokens to the list
		 * <p>Adds the given <code>tokens</code> to the list, but not any that were already present.</p>
		 * @param string $tokens The tokens to add.
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-tokenlist.add.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function add(string ...$tokens): void {}

		/**
		 * Returns whether the list contains a given token
		 * <p>Returns whether the list contains <code>token</code>.</p>
		 * @param string $token The token.
		 * @return bool <p>Returns <b><code>true</code></b> if the list contains <code>token</code>, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/dom-tokenlist.contains.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function contains(string $token): bool {}

		/**
		 * Returns the number of tokens in the list
		 * <p>Returns the number of tokens in the list.</p>
		 * @return int <p>The number of tokens in the list.</p>
		 * @link https://php.net/manual/en/dom-tokenlist.count.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function count(): int {}

		/**
		 * Returns an iterator over the token list
		 * <p>Returns an iterator over the token list.</p>
		 * @return Iterator <p>An iterator over the token list.</p>
		 * @link https://php.net/manual/en/dom-tokenlist.getiterator.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function getIterator(): \Iterator {}

		/**
		 * Returns a token from the list
		 * <p>Returns a token from the list at <code>index</code>.</p>
		 * @param int $index The token index.
		 * @return ?string <p>Returns the token at <code>index</code> or <b><code>null</code></b> when the index is out of bounds.</p>
		 * @link https://php.net/manual/en/dom-tokenlist.item.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function item(int $index): ?string {}

		/**
		 * Removes the given tokens from the list
		 * <p>Removes the given <code>tokens</code> from the list, but ignores any that were not present.</p>
		 * @param string $tokens The tokens to remove.
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/dom-tokenlist.remove.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function remove(string ...$tokens): void {}

		/**
		 * Replaces a token in the list with another one
		 * <p>Replaces a token in the list with another one.</p>
		 * @param string $token The token to replace.
		 * @param string $newToken The new token.
		 * @return bool <p>Returns <b><code>true</code></b> if <code>token</code> was in the list, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/dom-tokenlist.replace.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function replace(string $token, string $newToken): bool {}

		/**
		 * Returns whether the given token is supported
		 * <p>Returns whether <code>token</code> is in the associated attribute's supported tokens.</p>
		 * @param string $token The token.
		 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
		 * @link https://php.net/manual/en/dom-tokenlist.supports.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function supports(string $token): bool {}

		/**
		 * Toggles the presence of a token in the list
		 * <p>Toggles the presence of <code>token</code> in the list.</p>
		 * @param string $token The token to toggle.
		 * @param ?bool $force If <code>force</code> is provided, setting it to <b><code>true</code></b> will add the token, and setting it to <b><code>false</code></b> will remove the token.
		 * @return bool <p>Returns <b><code>true</code></b> if the token is in the list after the call, <b><code>false</code></b> otherwise.</p>
		 * @link https://php.net/manual/en/dom-tokenlist.toggle.php
		 * @since PHP 8 >= 8.4.0
		 */
		public function toggle(string $token, ?bool $force = null): bool {}
	}

	/**
	 * <p>Represents an XML document.</p>
	 * @link https://php.net/manual/en/class.dom-xmldocument.php
	 * @since PHP 8 >= 8.4.0
	 */
	final class XMLDocument extends \Dom\Document {

		/**
		 * @var string <p>An attribute specifying, as part of the XML declaration, the encoding of this document. This is <b><code>null</code></b> when unspecified or when it is not known, such as when the Document was created in memory.</p>
		 * @link https://php.net/manual/en/class.dom-xmldocument.php#dom-xmldocument.props.xmlencoding
		 */
		public string $xmlEncoding;

		/**
		 * @var bool <p>An attribute specifying, as part of the XML declaration, whether this document is standalone. This is <b><code>false</code></b> when unspecified. A standalone document is one where there are no external markup declarations. An example of such a markup declaration is when the DTD declares an attribute with a default value.</p>
		 * @link https://php.net/manual/en/class.dom-xmldocument.php#dom-xmldocument.props.xmlstandalone
		 */
		public bool $xmlStandalone;

		/**
		 * @var string <p>An attribute specifying, as part of the XML declaration, the version number of this document. If there is no declaration and if this document supports the "XML" feature, the value is "1.0".</p>
		 * @link https://php.net/manual/en/class.dom-xmldocument.php#dom-xmldocument.props.xmlversion
		 */
		public string $xmlVersion;

		/**
		 * @var bool Nicely formats output with indentation and extra space.
		 * @link https://php.net/manual/en/class.dom-xmldocument.php#dom-xmldocument.props.formatoutput
		 */
		public bool $formatOutput;

		/**
		 * @var Dom\Implementation <p>The <code>DOMImplementation</code> object that handles this document.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.implementation
		 */
		public \Dom\Implementation $implementation;

		/**
		 * @var string Equivalent to documentURI.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.url
		 */
		public string $URL;

		/**
		 * @var string <p>The location of the document or <b><code>null</code></b> if undefined.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.documenturi
		 */
		public string $documentURI;

		/**
		 * @var string The encoding of the document used for serialization. Upon parsing a document, this is set to the input encoding of that document.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.characterset
		 */
		public string $characterSet;

		/**
		 * @var string Legacy alias for characterSet.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.charset
		 */
		public string $charset;

		/**
		 * @var string Legacy alias for characterSet.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.inputencoding
		 */
		public string $inputEncoding;

		/**
		 * @var ?Dom\DocumentType <p>The Document Type Declaration associated with this document.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.doctype
		 */
		public ?\Dom\DocumentType $doctype;

		/**
		 * @var ?Dom\Element The <code>Dom\Element</code> that is the document element. This evaluates to <b><code>null</code></b> for document without elements.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.documentelement
		 */
		public ?\Dom\Element $documentElement;

		/**
		 * @var ?Dom\Element <p>First child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.firstelementchild
		 */
		public ?\Dom\Element $firstElementChild;

		/**
		 * @var ?Dom\Element <p>Last child element or <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.lastelementchild
		 */
		public ?\Dom\Element $lastElementChild;

		/**
		 * @var int <p>The number of child elements.</p>
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.childelementcount
		 */
		public int $childElementCount;

		/**
		 * @var ?Dom\HTMLElement The first child of the <code>html</code> element that is either a <code>body</code> tag or a <code>frameset</code> tag. These need to be in the HTML namespace. If no element matches, this evaluates to <b><code>null</code></b>.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.body
		 */
		public ?\Dom\HTMLElement $body;

		/**
		 * @var ?Dom\HTMLElement The first <code>head</code> element that is a child of the <code>html</code> element. These need to be in the HTML namespace. If no element matches, this evaluates to <b><code>null</code></b>.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.head
		 */
		public ?\Dom\HTMLElement $head;

		/**
		 * @var string The title of the document as set by the <code>title</code> element for HTML or the SVG <code>title</code> element for SVG. If there is no title, this evaluates to the empty string.
		 * @link https://php.net/manual/en/class.dom-document.php#dom-document.props.title
		 */
		public string $title;

		/**
		 * @var int <p>Gets the type of the node. One of the predefined <b><code>XML_&#42;_NODE</code></b> constants</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodetype
		 */
		public int $nodeType;

		/**
		 * @var string Returns the most accurate name for the current node type. <ul> <li>For elements, this is the HTML-uppercased qualified name.</li> <li>For attributes, this is the qualified name.</li> <li>For processing instructions, this is the target.</li> <li>For document type nodes, this is the name.</li> </ul>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodename
		 */
		public string $nodeName;

		/**
		 * @var string <p>The absolute base URI of this node or <b><code>null</code></b> if the implementation wasn't able to obtain an absolute URI.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.baseuri
		 */
		public string $baseURI;

		/**
		 * @var bool <p>Whether the node is connected to a document</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.isconnected
		 */
		public bool $isConnected;

		/**
		 * @var ?Dom\Document The <code>Dom\Document</code> object associated with this node, or <b><code>null</code></b> if this node is a document.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.ownerdocument
		 */
		public ?\Dom\Document $ownerDocument;

		/**
		 * @var ?Dom\Node <p>The parent of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentnode
		 */
		public ?\Dom\Node $parentNode;

		/**
		 * @var ?Dom\Element <p>The parent element of this element. If there is no such element, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.parentelement
		 */
		public ?\Dom\Element $parentElement;

		/**
		 * @var Dom\NodeList A <code>Dom\NodeList</code> that contains all children of this node. If there are no children, this is an empty <code>Dom\NodeList</code>.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.childnodes
		 */
		public \Dom\NodeList $childNodes;

		/**
		 * @var ?Dom\Node <p>The first child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.firstchild
		 */
		public ?\Dom\Node $firstChild;

		/**
		 * @var ?Dom\Node <p>The last child of this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.lastchild
		 */
		public ?\Dom\Node $lastChild;

		/**
		 * @var ?Dom\Node <p>The node immediately preceding this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.previoussibling
		 */
		public ?\Dom\Node $previousSibling;

		/**
		 * @var ?Dom\Node <p>The node immediately following this node. If there is no such node, this returns <b><code>null</code></b>.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nextsibling
		 */
		public ?\Dom\Node $nextSibling;

		/**
		 * @var ?string The value of this node, depending on its type.
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.nodevalue
		 */
		public ?string $nodeValue;

		/**
		 * @var ?string <p>The text content of this node and its descendants.</p>
		 * @link https://php.net/manual/en/class.dom-node.php#dom-node.props.textcontent
		 */
		public ?string $textContent;
	}

	/**
	 * <p>Allows to use XPath 1.0 queries on HTML or XML documents.</p>
	 * <p>This is the modern, spec-compliant equivalent of <code>DOMXPath</code>.</p>
	 * @link https://php.net/manual/en/class.dom-xpath.php
	 * @since PHP 8 >= 8.4.0
	 */
	final class XPath {

		/**
		 * @var Dom\Document The document that is linked to this object.
		 * @link https://php.net/manual/en/class.dom-xpath.php#dom-xpath.props.document
		 */
		public \Dom\Document $document;

		/**
		 * @var bool When set to <b><code>true</code></b>, namespaces in the node are registered.
		 * @link https://php.net/manual/en/class.dom-xpath.php#dom-xpath.props.registernodenamespaces
		 */
		public bool $registerNodeNamespaces;
	}

	/**
	 * Gets a Dom\Attr or Dom\Element object from a SimpleXMLElement object
	 * <p>This function takes the given attribute or element <code>node</code> (a <code>SimpleXMLElement</code> instance) and creates a <code>Dom\Attr</code> or <code>Dom\Element</code> node, respectively. The new <code>Dom\Node</code> refers to the same underlying XML node as the <code>SimpleXMLElement</code>.</p>
	 * @param object $node <p>The attribute or element node to import (a <code>SimpleXMLElement</code> instance).</p>
	 * @return Dom\Attr|Dom\Element <p>The <code>Dom\Attr</code> or <code>Dom\Element</code>.</p>
	 * @link https://php.net/manual/en/function.dom-ns-import-simplexml.php
	 * @see simplexml_import_dom()
	 * @since PHP 8 >= 8.4.0
	 */
	function import_simplexml(object $node): \Dom\Attr|\Dom\Element {}

	/**
	 * If any node is inserted somewhere it doesn't belong
	 */
	const HIERARCHY_REQUEST_ERR = 3;

	/**
	 * If index or size is negative, or greater than the allowed value.
	 */
	const INDEX_SIZE_ERR = 1;

	/**
	 * If an attempt is made to add an attribute that is already in use elsewhere.
	 */
	const INUSE_ATTRIBUTE_ERR = 10;

	/**
	 * If a parameter or an operation is not supported by the underlying object.
	 */
	const INVALID_ACCESS_ERR = null;

	/**
	 * If an invalid or illegal character is specified, such as in a name.
	 */
	const INVALID_CHARACTER_ERR = 5;

	/**
	 * If an attempt is made to modify the type of the underlying object.
	 */
	const INVALID_MODIFICATION_ERR = 13;

	/**
	 * If an attempt is made to use an object that is not, or is no longer, usable.
	 */
	const INVALID_STATE_ERR = 11;

	/**
	 * If an attempt is made to create or change an object in a way which is incorrect with regard to namespaces.
	 */
	const NAMESPACE_ERR = 14;

	/**
	 * If data is specified for a node which does not support data.
	 */
	const NO_DATA_ALLOWED_ERR = 6;

	/**
	 * This disables setting the namespace of elements during parsing when using <code>Dom\HTMLDocument</code>. This exists for backwards compatibility with <code>DOMDocument</code>.  <b>Caution</b> <p>Some DOM methods depend on the HTML namespace being set. By using this parser option, the behaviour of those methods can be influenced.</p>
	 */
	const NO_DEFAULT_NS = null;

	/**
	 * If an attempt is made to modify an object where modifications are not allowed.
	 */
	const NO_MODIFICATION_ALLOWED_ERR = 7;

	/**
	 * If an attempt is made to reference a node in a context where it does not exist.
	 */
	const NOT_FOUND_ERR = 8;

	/**
	 * If the implementation does not support the requested type of object or operation.
	 */
	const NOT_SUPPORTED_ERR = 9;

	/**
	 * If the specified range of text does not fit into a <code>string</code>.
	 */
	const STRING_SIZE_ERR = 2;

	/**
	 * If an invalid or illegal string is specified.
	 */
	const SYNTAX_ERR = 12;

	/**
	 * If a call to a method such as insertBefore or removeChild would make the Node invalid with respect to "partial validity", this exception would be raised and the operation would not be done.
	 */
	const VALIDATION_ERR = 16;

	/**
	 * If a node is used in a different document than the one that created it.
	 */
	const WRONG_DOCUMENT_ERR = 4;

}
