/*
** Copyright (c) 2024 LunarG, Inc.
**
** Permission is hereby granted, free of charge, to any person obtaining a
** copy of this software and associated documentation files (the "Software"),
** to deal in the Software without restriction, including without limitation
** the rights to use, copy, modify, merge, publish, distribute, sublicense,
** and/or sell copies of the Software, and to permit persons to whom the
** Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in
** all copies or substantial portions of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
** AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
** LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
** FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
*/

#ifndef GFXRECON_DECODE_VULKAN_ADDRESS_REPLACER_SHADERS_H
#define GFXRECON_DECODE_VULKAN_ADDRESS_REPLACER_SHADERS_H

#include <array>

GFXRECON_BEGIN_NAMESPACE(gfxrecon)
GFXRECON_BEGIN_NAMESPACE(decode)

// g_replacer_sbt_comp: original GLSL source
#if 0
#version 460
#extension GL_EXT_buffer_reference2 : require

/// -> hash.glsl
uint xxhash32(uint lhs, uint rhs)
{
    const uint PRIME32_2 = 2246822519U, PRIME32_3 = 3266489917U;
    const uint PRIME32_4 = 668265263U, PRIME32_5 = 374761393U;
    uint h32 = lhs + PRIME32_5 + rhs * PRIME32_3;
    h32 = PRIME32_4 * ((h32 << 17) | (h32 >> (32 - 17)));
    h32 = PRIME32_2 * (h32 ^ (h32 >> 15));
    h32 = PRIME32_3 * (h32 ^ (h32 >> 13));
    return h32 ^ (h32 >> 16);
}

//! xxhash32 for arrays of uint32_t
#define DEFINE_XXHASH32(LEN)                   \
    uint xxhash32(const uint key[LEN], uint h) \
    {                                          \
        for (uint i = 0; i < LEN; ++i)         \
        {                                      \
            h = xxhash32(key[i], h);           \
        }                                      \
        return h;                              \
    }                                          \
/// hash.glsl

DEFINE_XXHASH32(8);

// 32-byte wide opaque handle
struct shader_group_handle_t
{
    uint data[8];
};
const shader_group_handle_t k_null_handle = shader_group_handle_t(uint[](0, 0, 0, 0, 0, 0, 0, 0));

bool equal(const shader_group_handle_t lhs, const shader_group_handle_t rhs)
{
    for(uint i = 0; i < 8; ++i)
    {
        if(lhs.data[i] != rhs.data[i])
        {
            return false;
        }
    }
    return true;
}

bool is_null(const shader_group_handle_t h){ return equal(h, k_null_handle); }

struct hashmap_item_t
{
    shader_group_handle_t key;
    shader_group_handle_t value;
};
layout(buffer_reference, std430) buffer readonly HashMapStorage{ hashmap_item_t v[]; };

//! 16 bytes
struct hashmap_t
{
    HashMapStorage storage;
    uint size;
    uint capacity;
};

uint hash(const shader_group_handle_t group_handle)
{
    return xxhash32(group_handle.data, 0);
}

shader_group_handle_t get(const hashmap_t hashmap, const shader_group_handle_t key)
{
    if(is_null(key) || hashmap.capacity == 0) { return k_null_handle; }

    for(uint idx = hash(key);; idx++)
    {
        idx &= hashmap.capacity - 1;
        const hashmap_item_t item = hashmap.storage.v[idx];
        if(is_null(item.key)) { return k_null_handle; }
        else if(equal(key, item.key) && !is_null(item.value)) { return item.value; }
    }
}

layout(buffer_reference, std430) buffer GroupHandlePtr{ shader_group_handle_t group_handle; };

layout(buffer_reference, std430) buffer readonly AddressArray{ GroupHandlePtr v[]; };

struct replacer_params_t
{
    hashmap_t shader_group_handle_map;

    // input-/output-arrays can be identical when sbt-alignments/strides match
    AddressArray input_handles, output_handles;
    uint num_handles;
};

layout(push_constant, std430) uniform pc
{
    replacer_params_t params;
};

layout(local_size_x = 32, local_size_y = 1, local_size_z = 1) in;

void main()
{
    uint gid = gl_GlobalInvocationID.x;
    if(gid >= params.num_handles){ return; }

    // hashmap lookup
    shader_group_handle_t input_handle = params.input_handles.v[gid].group_handle;
    shader_group_handle_t result = get(params.shader_group_handle_map, input_handle);

    // write remapped or original value to output-array
    params.output_handles.v[gid].group_handle = is_null(result) ? input_handle : result;
}
#endif // g_replacer_sbt_comp: original GLSL source

// g_replacer_bda_binary_comp: original GLSL source
#if 0
#version 460
#extension GL_EXT_buffer_reference2 : require
#extension GL_EXT_scalar_block_layout : require
#extension GL_EXT_shader_explicit_arithmetic_types_int64 : require
#extension GL_EXT_shader_atomic_int64 : require

/// -> hash.glsl
uint xxhash32(uint lhs, uint rhs)
{
    const uint PRIME32_2 = 2246822519U, PRIME32_3 = 3266489917U;
    const uint PRIME32_4 = 668265263U, PRIME32_5 = 374761393U;
    uint h32 = lhs + PRIME32_5 + rhs * PRIME32_3;
    h32 = PRIME32_4 * ((h32 << 17) | (h32 >> (32 - 17)));
    h32 = PRIME32_2 * (h32 ^ (h32 >> 15));
    h32 = PRIME32_3 * (h32 ^ (h32 >> 13));
    return h32 ^ (h32 >> 16);
}

bool is_null(const uint64_t h){ return h == 0ul; }
const uint64_t k_null_handle = 0ul;

struct hashmap_item_t
{
    uint64_t key;
    uint64_t value;
};
layout(buffer_reference, std430) buffer readonly HashMapStorage{ hashmap_item_t v[]; };

//! 16 bytes
// struct hashmap_t
layout(buffer_reference, std430) buffer HashMapPtr
{
    HashMapStorage storage;
    uint size;
    uint capacity;
};

uint hash(uint64_t device_address)
{
    const uint64_t mask32 = 4294967295ul;
    return xxhash32(uint(device_address >> 32ul), xxhash32(uint(device_address & mask32), 0));
}

uint64_t hashmap_get(const HashMapPtr hashmap, const uint64_t key)
{
    if(is_null(key) || hashmap.capacity == 0) { return k_null_handle; }

    for(uint idx = hash(key);; idx++)
    {
        idx &= hashmap.capacity - 1;
        const hashmap_item_t item = hashmap.storage.v[idx];
        if(is_null(item.key)) { return k_null_handle; }
        else if(key == item.key && !is_null(item.value)) { return item.value; }
    }
}

void hashmap_set(const HashMapPtr hashmap, const uint64_t key, const uint64_t value)
{
    if(is_null(key) || hashmap.capacity == 0) { return; }

    for(uint idx = hash(key);; idx++)
    {
        idx &= hashmap.capacity - 1;
        uint64_t probed_key = hashmap.storage.v[idx].key;
        if(probed_key != key)
        {
            // hit another valid entry, keep probing
            if(!is_null(probed_key) && !is_null(hashmap.storage.v[idx].value)) { continue; }

            // try to insert atomically
            probed_key = atomicCompSwap(hashmap.storage.v[idx].key, probed_key, key);
            if(!is_null(probed_key) && probed_key != key)
            {
                // another thread just stole it
                continue;
            }
            // atomically increase size
            atomicAdd(hashmap.size, 1);
        }
        hashmap.storage.v[idx].value = value;
        return;
    }
}

struct storage_item_t
{
    uint64_t key;
    uint64_t value;
    uint64_t size;

};
layout(buffer_reference, scalar) buffer readonly Storage{ storage_item_t v[]; };

struct sorted_array_t
{
    Storage storage;
    uint num_items;
    uint padding;
};

//! perform a binary search, take into account offsets
uint64_t map_address(const sorted_array_t array, uint64_t capture_address)
{
    if(array.num_items == 0){ return k_null_handle; }
    uint low = 0, high = array.num_items - 1;
    uint mid;
    while(low <= high)
    {
        mid = low + (high - low) / 2;

        if(capture_address >= array.storage.v[mid].key &&
           capture_address < (array.storage.v[mid].key + max(array.storage.v[mid].size, 1ul)))
        {
            // offset calculation
            uint64_t offset = capture_address - array.storage.v[mid].key;
            return array.storage.v[mid].value + offset;
        }
        else if(capture_address < array.storage.v[mid].key){ high = mid - 1; }
        else { low = mid + 1;}
    }
    return k_null_handle;
}

layout(buffer_reference, std430) buffer readonly AddressArray{ uint64_t v[]; };

struct replacer_params_bda_t
{
    sorted_array_t device_address_array;
    HashMapPtr address_blacklist;

    // input-/output-arrays 'can' be identical
    AddressArray input_handles;
    AddressArray output_handles;
    uint num_handles;
};

layout(push_constant, std430) uniform pc
{
    replacer_params_bda_t params;
};

layout(buffer_reference, std430) buffer BufferDeviceAddressPtr{ uint64_t buffer_device_address; };
layout(buffer_reference, std430) buffer PaddedBufferDeviceAddressPtr{ uint pad; uint64_t buffer_device_address; };

//! helper to read a VkDeviceAddress from address 'ptr', read-access is 8-byte aligned
uint64_t read_address(uint64_t ptr)
{
    return (ptr & 7UL) != 0 ? PaddedBufferDeviceAddressPtr(ptr & ~7).buffer_device_address :
                              BufferDeviceAddressPtr(ptr).buffer_device_address;
}

//! helper to write a VkDeviceAddress to address 'ptr', write-access is 8-byte aligned
void write_address(uint64_t ptr, uint64_t value)
{
    if((ptr & 7UL) != 0)
    {
        PaddedBufferDeviceAddressPtr pad_helper = PaddedBufferDeviceAddressPtr(ptr & ~7);
        pad_helper.buffer_device_address = value;
    }
    else
    {
        BufferDeviceAddressPtr(ptr).buffer_device_address = value;
    }
}

layout(local_size_x = 32, local_size_y = 1, local_size_z = 1) in;

void main()
{
    uint gid = gl_GlobalInvocationID.x;
    if(gid >= params.num_handles){ return; }

    uint64_t input_address = read_address(params.input_handles.v[gid]);

    // check if we have replaced + cached this address already, avoid double-replacements
    if(hashmap_get(params.address_blacklist, params.input_handles.v[gid]) != input_address)
    {
        // binary-search lookup
        uint64_t result = map_address(params.device_address_array, input_address);

        // keep track of mapped address
        hashmap_set(params.address_blacklist, params.input_handles.v[gid], result);

        // write remapped or original value to output-array
        write_address(params.output_handles.v[gid], is_null(result) ? input_address : result);
    }
}
#endif // g_replacer_bda_binary_comp

// g_replacer_rehash_comp: original GLSL source
#if 0
#version 460
#extension GL_EXT_buffer_reference2 : require
#extension GL_EXT_scalar_block_layout : require
#extension GL_EXT_shader_explicit_arithmetic_types_int64 : require
#extension GL_EXT_shader_atomic_int64 : require

/// -> hash.glsl
uint xxhash32(uint lhs, uint rhs)
{
    const uint PRIME32_2 = 2246822519U, PRIME32_3 = 3266489917U;
    const uint PRIME32_4 = 668265263U, PRIME32_5 = 374761393U;
    uint h32 = lhs + PRIME32_5 + rhs * PRIME32_3;
    h32 = PRIME32_4 * ((h32 << 17) | (h32 >> (32 - 17)));
    h32 = PRIME32_2 * (h32 ^ (h32 >> 15));
    h32 = PRIME32_3 * (h32 ^ (h32 >> 13));
    return h32 ^ (h32 >> 16);
}

bool is_null(const uint64_t h){ return h == 0ul; }
const uint64_t k_null_handle = 0ul;

struct hashmap_item_t
{
    uint64_t key;
    uint64_t value;
};
layout(buffer_reference, std430) buffer readonly HashMapStorage{ hashmap_item_t v[]; };

//! 16 bytes
// struct hashmap_t
layout(buffer_reference, std430) buffer HashMapPtr
{
    HashMapStorage storage;
    uint size;
    uint capacity;
};

uint hash(uint64_t device_address)
{
    const uint64_t mask32 = 4294967295ul;
    return xxhash32(uint(device_address >> 32ul), xxhash32(uint(device_address & mask32), 0));
}

uint64_t hashmap_get(const HashMapPtr hashmap, const uint64_t key)
{
    if(is_null(key) || hashmap.capacity == 0) { return k_null_handle; }

    for(uint idx = hash(key);; idx++)
    {
        idx &= hashmap.capacity - 1;
        const hashmap_item_t item = hashmap.storage.v[idx];
        if(is_null(item.key)) { return k_null_handle; }
        else if(key == item.key && !is_null(item.value)) { return item.value; }
    }
}

void hashmap_set(const HashMapPtr hashmap, const uint64_t key, const uint64_t value)
{
    if(is_null(key) || hashmap.capacity == 0) { return; }

    for(uint idx = hash(key);; idx++)
    {
        idx &= hashmap.capacity - 1;
        uint64_t probed_key = hashmap.storage.v[idx].key;
        if(probed_key != key)
        {
            // hit another valid entry, keep probing
            if(!is_null(probed_key) && !is_null(hashmap.storage.v[idx].value)) { continue; }

            // try to insert atomically
            probed_key = atomicCompSwap(hashmap.storage.v[idx].key, probed_key, key);
            if(!is_null(probed_key) && probed_key != key)
            {
                // another thread just stole it
                continue;
            }
            // atomically increase size
            atomicAdd(hashmap.size, 1);
        }
        hashmap.storage.v[idx].value = value;
        return;
    }
}

struct rehash_params
{
    HashMapPtr hashmap_old;
    HashMapPtr hashmap_new;
};

layout(push_constant, std430) uniform pc
{
    rehash_params params;
};

layout(local_size_x = 32, local_size_y = 1, local_size_z = 1) in;

void main()
{
    uint gid = gl_GlobalInvocationID.x;
    if(gid >= params.hashmap_old.capacity){ return; }

    // insert entries from hashmap_old into hashmap_new
    uint64_t key = params.hashmap_old.storage.v[gid].key;
    uint64_t value = params.hashmap_old.storage.v[gid].value;
    hashmap_set(params.hashmap_new, key, value);
}
#endif

static const std::array<uint32_t, 2176> g_replacer_sbt_comp = {
    0x07230203, 0x00010300, 0x0008000b, 0x00000451, 0x00000000, 0x00020011, 0x00000001, 0x00020011, 0x000014b6,
    0x00020011, 0x000014e3, 0x0008000a, 0x5f565053, 0x5f545845, 0x63736564, 0x74706972, 0x695f726f, 0x7865646e,
    0x00676e69, 0x0009000a, 0x5f565053, 0x5f52484b, 0x73796870, 0x6c616369, 0x6f74735f, 0x65676172, 0x6675625f,
    0x00726566, 0x0006000b, 0x00000001, 0x4c534c47, 0x6474732e, 0x3035342e, 0x00000000, 0x0003000e, 0x000014e4,
    0x00000001, 0x0006000f, 0x00000005, 0x00000004, 0x6e69616d, 0x00000000, 0x000000f9, 0x00060010, 0x00000004,
    0x00000011, 0x00000020, 0x00000001, 0x00000001, 0x00030003, 0x00000002, 0x000001cc, 0x00070004, 0x455f4c47,
    0x625f5458, 0x65666675, 0x65725f72, 0x65726566, 0x0065636e, 0x00080004, 0x455f4c47, 0x625f5458, 0x65666675,
    0x65725f72, 0x65726566, 0x3265636e, 0x00000000, 0x00040005, 0x00000004, 0x6e69616d, 0x00000000, 0x00080005,
    0x00000026, 0x64616873, 0x675f7265, 0x70756f72, 0x6e61685f, 0x5f656c64, 0x00000074, 0x00050006, 0x00000026,
    0x00000000, 0x61746164, 0x00000000, 0x00060005, 0x00000027, 0x68736168, 0x5f70616d, 0x6d657469, 0x0000745f,
    0x00040006, 0x00000027, 0x00000000, 0x0079656b, 0x00050006, 0x00000027, 0x00000001, 0x756c6176, 0x00000065,
    0x00060005, 0x00000029, 0x68736148, 0x5370614d, 0x61726f74, 0x00006567, 0x00040006, 0x00000029, 0x00000000,
    0x00000076, 0x00080005, 0x000000f9, 0x475f6c67, 0x61626f6c, 0x766e496c, 0x7461636f, 0x496e6f69, 0x00000044,
    0x00050005, 0x000000fe, 0x68736168, 0x5f70616d, 0x00000074, 0x00050006, 0x000000fe, 0x00000000, 0x726f7473,
    0x00656761, 0x00050006, 0x000000fe, 0x00000001, 0x657a6973, 0x00000000, 0x00060006, 0x000000fe, 0x00000002,
    0x61706163, 0x79746963, 0x00000000, 0x00070005, 0x00000100, 0x6c706572, 0x72656361, 0x7261705f, 0x5f736d61,
    0x00000074, 0x00090006, 0x00000100, 0x00000000, 0x64616873, 0x675f7265, 0x70756f72, 0x6e61685f, 0x5f656c64,
    0x0070616d, 0x00070006, 0x00000100, 0x00000001, 0x75706e69, 0x61685f74, 0x656c646e, 0x00000073, 0x00070006,
    0x00000100, 0x00000002, 0x7074756f, 0x685f7475, 0x6c646e61, 0x00007365, 0x00060006, 0x00000100, 0x00000003,
    0x5f6d756e, 0x646e6168, 0x0073656c, 0x00060005, 0x00000103, 0x72646441, 0x41737365, 0x79617272, 0x00000000,
    0x00040006, 0x00000103, 0x00000000, 0x00000076, 0x00080005, 0x00000105, 0x64616873, 0x675f7265, 0x70756f72,
    0x6e61685f, 0x5f656c64, 0x00000074, 0x00050006, 0x00000105, 0x00000000, 0x61746164, 0x00000000, 0x00060005,
    0x00000106, 0x756f7247, 0x6e614870, 0x50656c64, 0x00007274, 0x00070006, 0x00000106, 0x00000000, 0x756f7267,
    0x61685f70, 0x656c646e, 0x00000000, 0x00030005, 0x00000108, 0x00006370, 0x00050006, 0x00000108, 0x00000000,
    0x61726170, 0x0000736d, 0x00030005, 0x0000010a, 0x00000000, 0x00040047, 0x00000025, 0x00000006, 0x00000004,
    0x00050048, 0x00000026, 0x00000000, 0x00000023, 0x00000000, 0x00050048, 0x00000027, 0x00000000, 0x00000023,
    0x00000000, 0x00050048, 0x00000027, 0x00000001, 0x00000023, 0x00000020, 0x00040047, 0x00000028, 0x00000006,
    0x00000040, 0x00030047, 0x00000029, 0x00000002, 0x00040048, 0x00000029, 0x00000000, 0x00000018, 0x00050048,
    0x00000029, 0x00000000, 0x00000023, 0x00000000, 0x00040047, 0x000000f9, 0x0000000b, 0x0000001c, 0x00050048,
    0x000000fe, 0x00000000, 0x00000023, 0x00000000, 0x00050048, 0x000000fe, 0x00000001, 0x00000023, 0x00000008,
    0x00050048, 0x000000fe, 0x00000002, 0x00000023, 0x0000000c, 0x00050048, 0x00000100, 0x00000000, 0x00000023,
    0x00000000, 0x00050048, 0x00000100, 0x00000001, 0x00000023, 0x00000010, 0x00050048, 0x00000100, 0x00000002,
    0x00000023, 0x00000018, 0x00050048, 0x00000100, 0x00000003, 0x00000023, 0x00000020, 0x00040047, 0x00000102,
    0x00000006, 0x00000008, 0x00030047, 0x00000103, 0x00000002, 0x00040048, 0x00000103, 0x00000000, 0x00000018,
    0x00050048, 0x00000103, 0x00000000, 0x00000023, 0x00000000, 0x00040047, 0x00000104, 0x00000006, 0x00000004,
    0x00050048, 0x00000105, 0x00000000, 0x00000023, 0x00000000, 0x00030047, 0x00000106, 0x00000002, 0x00050048,
    0x00000106, 0x00000000, 0x00000023, 0x00000000, 0x00030047, 0x00000108, 0x00000002, 0x00050048, 0x00000108,
    0x00000000, 0x00000023, 0x00000000, 0x00040047, 0x00000163, 0x0000000b, 0x00000019, 0x00020013, 0x00000002,
    0x00030021, 0x00000003, 0x00000002, 0x00040015, 0x00000006, 0x00000020, 0x00000000, 0x00040020, 0x00000007,
    0x00000007, 0x00000006, 0x0004002b, 0x00000006, 0x0000000d, 0x00000008, 0x0004001c, 0x0000000e, 0x00000006,
    0x0000000d, 0x00020014, 0x00000015, 0x00030027, 0x00000023, 0x000014e5, 0x0004001c, 0x00000025, 0x00000006,
    0x0000000d, 0x0003001e, 0x00000026, 0x00000025, 0x0004001e, 0x00000027, 0x00000026, 0x00000026, 0x0003001d,
    0x00000028, 0x00000027, 0x0003001e, 0x00000029, 0x00000028, 0x00040020, 0x00000023, 0x000014e5, 0x00000029,
    0x0004002b, 0x00000006, 0x00000031, 0x165667b1, 0x0004002b, 0x00000006, 0x00000034, 0xc2b2ae3d, 0x0004002b,
    0x00000006, 0x00000037, 0x27d4eb2f, 0x00040015, 0x00000039, 0x00000020, 0x00000001, 0x0004002b, 0x00000039,
    0x0000003a, 0x00000011, 0x0004002b, 0x00000039, 0x0000003d, 0x0000000f, 0x0004002b, 0x00000006, 0x00000041,
    0x85ebca77, 0x0004002b, 0x00000039, 0x00000049, 0x0000000d, 0x0004002b, 0x00000039, 0x0000004f, 0x00000010,
    0x0004002b, 0x00000006, 0x00000055, 0x00000000, 0x00040020, 0x0000005f, 0x00000007, 0x0000000e, 0x0004002b,
    0x00000039, 0x00000067, 0x00000001, 0x0004002b, 0x00000039, 0x00000074, 0x00000000, 0x0003002a, 0x00000015,
    0x00000081, 0x00030029, 0x00000015, 0x00000085, 0x000b002c, 0x0000000e, 0x00000088, 0x00000055, 0x00000055,
    0x00000055, 0x00000055, 0x00000055, 0x00000055, 0x00000055, 0x00000055, 0x0004002b, 0x00000039, 0x00000096,
    0x00000002, 0x0004002b, 0x00000006, 0x000000a4, 0x00000001, 0x00040020, 0x000000ad, 0x000014e5, 0x00000027,
    0x0004002b, 0x00000039, 0x000000bb, 0x00000003, 0x0004002b, 0x00000039, 0x000000be, 0x00000004, 0x0004002b,
    0x00000039, 0x000000c1, 0x00000005, 0x0004002b, 0x00000039, 0x000000c4, 0x00000006, 0x0004002b, 0x00000039,
    0x000000c7, 0x00000007, 0x00040017, 0x000000f7, 0x00000006, 0x00000003, 0x00040020, 0x000000f8, 0x00000001,
    0x000000f7, 0x0004003b, 0x000000f8, 0x000000f9, 0x00000001, 0x00040020, 0x000000fa, 0x00000001, 0x00000006,
    0x0005001e, 0x000000fe, 0x00000023, 0x00000006, 0x00000006, 0x00030027, 0x000000ff, 0x000014e5, 0x0006001e,
    0x00000100, 0x000000fe, 0x000000ff, 0x000000ff, 0x00000006, 0x00030027, 0x00000101, 0x000014e5, 0x0003001d,
    0x00000102, 0x00000101, 0x0003001e, 0x00000103, 0x00000102, 0x0004001c, 0x00000104, 0x00000006, 0x0000000d,
    0x0003001e, 0x00000105, 0x00000104, 0x0003001e, 0x00000106, 0x00000105, 0x00040020, 0x00000101, 0x000014e5,
    0x00000106, 0x00040020, 0x000000ff, 0x000014e5, 0x00000103, 0x0003001e, 0x00000108, 0x00000100, 0x00040020,
    0x00000109, 0x00000009, 0x00000108, 0x0004003b, 0x00000109, 0x0000010a, 0x00000009, 0x00040020, 0x0000010b,
    0x00000009, 0x00000006, 0x00040020, 0x00000113, 0x00000009, 0x000000ff, 0x00040020, 0x00000117, 0x000014e5,
    0x00000101, 0x00040020, 0x0000011a, 0x000014e5, 0x00000105, 0x00040020, 0x00000130, 0x00000009, 0x000000fe,
    0x00040020, 0x0000014f, 0x000014e5, 0x00000104, 0x00040020, 0x00000152, 0x000014e5, 0x00000006, 0x0004002b,
    0x00000006, 0x00000162, 0x00000020, 0x0006002c, 0x000000f7, 0x00000163, 0x00000162, 0x000000a4, 0x000000a4,
    0x00030001, 0x00000015, 0x000003d1, 0x00050036, 0x00000002, 0x00000004, 0x00000000, 0x00000003, 0x000200f8,
    0x00000005, 0x0004003b, 0x0000005f, 0x000002c8, 0x00000007, 0x0004003b, 0x0000005f, 0x000002c5, 0x00000007,
    0x0004003b, 0x0000005f, 0x000002c2, 0x00000007, 0x0004003b, 0x0000005f, 0x000002bf, 0x00000007, 0x0004003b,
    0x0000005f, 0x000002bc, 0x00000007, 0x0004003b, 0x0000005f, 0x000002b9, 0x00000007, 0x0004003b, 0x0000005f,
    0x000002b6, 0x00000007, 0x0004003b, 0x0000005f, 0x000002b3, 0x00000007, 0x0004003b, 0x0000005f, 0x000002b0,
    0x00000007, 0x0004003b, 0x0000005f, 0x000002ad, 0x00000007, 0x0004003b, 0x0000005f, 0x000001fd, 0x00000007,
    0x000300f7, 0x00000164, 0x00000000, 0x000300fb, 0x00000055, 0x00000165, 0x000200f8, 0x00000165, 0x00050041,
    0x000000fa, 0x000000fb, 0x000000f9, 0x00000055, 0x0004003d, 0x00000006, 0x000000fc, 0x000000fb, 0x00060041,
    0x0000010b, 0x0000010c, 0x0000010a, 0x00000074, 0x000000bb, 0x0004003d, 0x00000006, 0x0000010d, 0x0000010c,
    0x000500ae, 0x00000015, 0x0000010e, 0x000000fc, 0x0000010d, 0x000300f7, 0x00000110, 0x00000000, 0x000400fa,
    0x0000010e, 0x0000010f, 0x00000110, 0x000200f8, 0x0000010f, 0x000200f9, 0x00000164, 0x000200f8, 0x00000110,
    0x00060041, 0x00000113, 0x00000114, 0x0000010a, 0x00000074, 0x00000067, 0x0004003d, 0x000000ff, 0x00000115,
    0x00000114, 0x00060041, 0x00000117, 0x00000118, 0x00000115, 0x00000074, 0x000000fc, 0x0006003d, 0x00000101,
    0x00000119, 0x00000118, 0x00000002, 0x00000008, 0x00050041, 0x0000011a, 0x0000011b, 0x00000119, 0x00000074,
    0x0006003d, 0x00000105, 0x0000011c, 0x0000011b, 0x00000002, 0x00000010, 0x00050051, 0x00000104, 0x0000011d,
    0x0000011c, 0x00000000, 0x00050051, 0x00000006, 0x0000011f, 0x0000011d, 0x00000000, 0x00050051, 0x00000006,
    0x00000121, 0x0000011d, 0x00000001, 0x00050051, 0x00000006, 0x00000123, 0x0000011d, 0x00000002, 0x00050051,
    0x00000006, 0x00000125, 0x0000011d, 0x00000003, 0x00050051, 0x00000006, 0x00000127, 0x0000011d, 0x00000004,
    0x00050051, 0x00000006, 0x00000129, 0x0000011d, 0x00000005, 0x00050051, 0x00000006, 0x0000012b, 0x0000011d,
    0x00000006, 0x00050051, 0x00000006, 0x0000012d, 0x0000011d, 0x00000007, 0x00060041, 0x00000130, 0x00000131,
    0x0000010a, 0x00000074, 0x00000074, 0x0004003d, 0x000000fe, 0x00000132, 0x00000131, 0x000b0050, 0x0000000e,
    0x0000034a, 0x0000011f, 0x00000121, 0x00000123, 0x00000125, 0x00000127, 0x00000129, 0x0000012b, 0x0000012d,
    0x00050051, 0x00000023, 0x00000136, 0x00000132, 0x00000000, 0x00050051, 0x00000006, 0x0000013b, 0x00000132,
    0x00000002, 0x000300f7, 0x000001d3, 0x00000000, 0x000300fb, 0x00000055, 0x0000017c, 0x000200f8, 0x0000017c,
    0x000300f7, 0x000001f4, 0x00000000, 0x000300fb, 0x00000055, 0x000001df, 0x000200f8, 0x000001df, 0x000200f9,
    0x000001e0, 0x000200f8, 0x000001e0, 0x000700f5, 0x00000006, 0x000003cb, 0x00000055, 0x000001df, 0x000001f0,
    0x000001ee, 0x000500b0, 0x00000015, 0x000001e3, 0x000003cb, 0x0000000d, 0x000400f6, 0x000001f1, 0x000001ee,
    0x00000000, 0x000400fa, 0x000001e3, 0x000001e4, 0x000001f1, 0x000200f8, 0x000001e4, 0x0003003e, 0x000002c5,
    0x0000034a, 0x00050041, 0x00000007, 0x000002c7, 0x000002c5, 0x000003cb, 0x0004003d, 0x00000006, 0x000001e7,
    0x000002c7, 0x0003003e, 0x000002c8, 0x00000088, 0x00050041, 0x00000007, 0x000002ca, 0x000002c8, 0x000003cb,
    0x0004003d, 0x00000006, 0x000001ea, 0x000002ca, 0x000500ab, 0x00000015, 0x000001eb, 0x000001e7, 0x000001ea,
    0x000300f7, 0x000001ed, 0x00000000, 0x000400fa, 0x000001eb, 0x000001ec, 0x000001ed, 0x000200f8, 0x000001ec,
    0x000200f9, 0x000001f1, 0x000200f8, 0x000001ed, 0x000200f9, 0x000001ee, 0x000200f8, 0x000001ee, 0x00050080,
    0x00000006, 0x000001f0, 0x000003cb, 0x00000067, 0x000200f9, 0x000001e0, 0x000200f8, 0x000001f1, 0x000700f5,
    0x00000015, 0x000003cf, 0x000003d1, 0x000001e0, 0x00000081, 0x000001ec, 0x000700f5, 0x00000015, 0x000003cc,
    0x00000081, 0x000001e0, 0x00000085, 0x000001ec, 0x000300f7, 0x000001f3, 0x00000000, 0x000400fa, 0x000003cc,
    0x000001f4, 0x000001f3, 0x000200f8, 0x000001f3, 0x000200f9, 0x000001f4, 0x000200f8, 0x000001f4, 0x000700f5,
    0x00000015, 0x000003ce, 0x000003cf, 0x000001f1, 0x00000085, 0x000001f3, 0x000400a8, 0x00000015, 0x0000017e,
    0x000003ce, 0x000300f7, 0x00000182, 0x00000000, 0x000400fa, 0x0000017e, 0x0000017f, 0x00000182, 0x000200f8,
    0x0000017f, 0x000500aa, 0x00000015, 0x00000181, 0x0000013b, 0x00000055, 0x000200f9, 0x00000182, 0x000200f8,
    0x00000182, 0x000700f5, 0x00000015, 0x00000183, 0x000003ce, 0x000001f4, 0x00000181, 0x0000017f, 0x000300f7,
    0x00000185, 0x00000000, 0x000400fa, 0x00000183, 0x00000184, 0x00000185, 0x000200f8, 0x00000184, 0x000200f9,
    0x000001d3, 0x000200f8, 0x00000185, 0x000200f9, 0x00000201, 0x000200f8, 0x00000201, 0x000700f5, 0x00000006,
    0x000003d3, 0x00000055, 0x00000185, 0x0000022b, 0x00000205, 0x000700f5, 0x00000006, 0x000003d2, 0x00000055,
    0x00000185, 0x0000020d, 0x00000205, 0x000500b0, 0x00000015, 0x00000204, 0x000003d2, 0x0000000d, 0x000400f6,
    0x0000020e, 0x00000205, 0x00000000, 0x000400fa, 0x00000204, 0x00000205, 0x0000020e, 0x000200f8, 0x00000205,
    0x0003003e, 0x000001fd, 0x0000034a, 0x00050041, 0x00000007, 0x00000207, 0x000001fd, 0x000003d2, 0x0004003d,
    0x00000006, 0x00000208, 0x00000207, 0x00050080, 0x00000006, 0x00000214, 0x00000208, 0x00000031, 0x00050084,
    0x00000006, 0x00000216, 0x000003d3, 0x00000034, 0x00050080, 0x00000006, 0x00000217, 0x00000214, 0x00000216,
    0x000500c4, 0x00000006, 0x00000219, 0x00000217, 0x0000003a, 0x000500c2, 0x00000006, 0x0000021b, 0x00000217,
    0x0000003d, 0x000500c5, 0x00000006, 0x0000021c, 0x00000219, 0x0000021b, 0x00050084, 0x00000006, 0x0000021d,
    0x00000037, 0x0000021c, 0x000500c2, 0x00000006, 0x00000220, 0x0000021d, 0x0000003d, 0x000500c6, 0x00000006,
    0x00000221, 0x0000021d, 0x00000220, 0x00050084, 0x00000006, 0x00000222, 0x00000041, 0x00000221, 0x000500c2,
    0x00000006, 0x00000225, 0x00000222, 0x00000049, 0x000500c6, 0x00000006, 0x00000226, 0x00000222, 0x00000225,
    0x00050084, 0x00000006, 0x00000227, 0x00000034, 0x00000226, 0x000500c2, 0x00000006, 0x0000022a, 0x00000227,
    0x0000004f, 0x000500c6, 0x00000006, 0x0000022b, 0x00000227, 0x0000022a, 0x00050080, 0x00000006, 0x0000020d,
    0x000003d2, 0x00000067, 0x000200f9, 0x00000201, 0x000200f8, 0x0000020e, 0x000200f9, 0x00000187, 0x000200f8,
    0x00000187, 0x000700f5, 0x00000015, 0x000003fc, 0x000003d1, 0x0000020e, 0x00000448, 0x000001cd, 0x000700f5,
    0x00000015, 0x000003ea, 0x000003d1, 0x0000020e, 0x000003e4, 0x000001cd, 0x000700f5, 0x00000015, 0x000003db,
    0x000003d1, 0x0000020e, 0x000003d8, 0x000001cd, 0x000700f5, 0x00000006, 0x000003d4, 0x000003d3, 0x0000020e,
    0x000001cf, 0x000001cd, 0x000400f6, 0x000001d0, 0x000001cd, 0x00000000, 0x000200f9, 0x00000188, 0x000200f8,
    0x00000188, 0x00050082, 0x00000006, 0x0000018a, 0x0000013b, 0x000000a4, 0x000500c7, 0x00000006, 0x0000018c,
    0x000003d4, 0x0000018a, 0x00060041, 0x000000ad, 0x0000018f, 0x00000136, 0x00000074, 0x0000018c, 0x0006003d,
    0x00000027, 0x00000190, 0x0000018f, 0x00000002, 0x00000010, 0x00050051, 0x00000026, 0x00000191, 0x00000190,
    0x00000000, 0x00050051, 0x00000025, 0x00000193, 0x00000191, 0x00000000, 0x00050051, 0x00000006, 0x00000195,
    0x00000193, 0x00000000, 0x00050051, 0x00000006, 0x00000197, 0x00000193, 0x00000001, 0x00050051, 0x00000006,
    0x00000199, 0x00000193, 0x00000002, 0x00050051, 0x00000006, 0x0000019b, 0x00000193, 0x00000003, 0x00050051,
    0x00000006, 0x0000019d, 0x00000193, 0x00000004, 0x00050051, 0x00000006, 0x0000019f, 0x00000193, 0x00000005,
    0x00050051, 0x00000006, 0x000001a1, 0x00000193, 0x00000006, 0x00050051, 0x00000006, 0x000001a3, 0x00000193,
    0x00000007, 0x00050051, 0x00000026, 0x000001a5, 0x00000190, 0x00000001, 0x00050051, 0x00000025, 0x000001a7,
    0x000001a5, 0x00000000, 0x00050051, 0x00000006, 0x000001a9, 0x000001a7, 0x00000000, 0x00050051, 0x00000006,
    0x000001ab, 0x000001a7, 0x00000001, 0x00050051, 0x00000006, 0x000001ad, 0x000001a7, 0x00000002, 0x00050051,
    0x00000006, 0x000001af, 0x000001a7, 0x00000003, 0x00050051, 0x00000006, 0x000001b1, 0x000001a7, 0x00000004,
    0x00050051, 0x00000006, 0x000001b3, 0x000001a7, 0x00000005, 0x00050051, 0x00000006, 0x000001b5, 0x000001a7,
    0x00000006, 0x00050051, 0x00000006, 0x000001b7, 0x000001a7, 0x00000007, 0x000b0050, 0x0000000e, 0x000003a7,
    0x00000195, 0x00000197, 0x00000199, 0x0000019b, 0x0000019d, 0x0000019f, 0x000001a1, 0x000001a3, 0x000300f7,
    0x0000024b, 0x00000000, 0x000300fb, 0x00000055, 0x00000236, 0x000200f8, 0x00000236, 0x000200f9, 0x00000237,
    0x000200f8, 0x00000237, 0x000700f5, 0x00000006, 0x000003d5, 0x00000055, 0x00000236, 0x00000247, 0x00000245,
    0x000500b0, 0x00000015, 0x0000023a, 0x000003d5, 0x0000000d, 0x000400f6, 0x00000248, 0x00000245, 0x00000000,
    0x000400fa, 0x0000023a, 0x0000023b, 0x00000248, 0x000200f8, 0x0000023b, 0x0003003e, 0x000002bf, 0x000003a7,
    0x00050041, 0x00000007, 0x000002c1, 0x000002bf, 0x000003d5, 0x0004003d, 0x00000006, 0x0000023e, 0x000002c1,
    0x0003003e, 0x000002c2, 0x00000088, 0x00050041, 0x00000007, 0x000002c4, 0x000002c2, 0x000003d5, 0x0004003d,
    0x00000006, 0x00000241, 0x000002c4, 0x000500ab, 0x00000015, 0x00000242, 0x0000023e, 0x00000241, 0x000300f7,
    0x00000244, 0x00000000, 0x000400fa, 0x00000242, 0x00000243, 0x00000244, 0x000200f8, 0x00000243, 0x000200f9,
    0x00000248, 0x000200f8, 0x00000244, 0x000200f9, 0x00000245, 0x000200f8, 0x00000245, 0x00050080, 0x00000006,
    0x00000247, 0x000003d5, 0x00000067, 0x000200f9, 0x00000237, 0x000200f8, 0x00000248, 0x000700f5, 0x00000015,
    0x000003d9, 0x000003db, 0x00000237, 0x00000081, 0x00000243, 0x000700f5, 0x00000015, 0x000003d6, 0x00000081,
    0x00000237, 0x00000085, 0x00000243, 0x000300f7, 0x0000024a, 0x00000000, 0x000400fa, 0x000003d6, 0x0000024b,
    0x0000024a, 0x000200f8, 0x0000024a, 0x000200f9, 0x0000024b, 0x000200f8, 0x0000024b, 0x000700f5, 0x00000015,
    0x000003d8, 0x000003d9, 0x00000248, 0x00000085, 0x0000024a, 0x000300f7, 0x000001cc, 0x00000000, 0x000400fa,
    0x000003d8, 0x000001bc, 0x000001bd, 0x000200f8, 0x000001bc, 0x000200f9, 0x000001d0, 0x000200f8, 0x000001bd,
    0x000300f7, 0x00000269, 0x00000000, 0x000300fb, 0x00000055, 0x00000254, 0x000200f8, 0x00000254, 0x000200f9,
    0x00000255, 0x000200f8, 0x00000255, 0x000700f5, 0x00000006, 0x000003e1, 0x00000055, 0x00000254, 0x00000265,
    0x00000263, 0x000500b0, 0x00000015, 0x00000258, 0x000003e1, 0x0000000d, 0x000400f6, 0x00000266, 0x00000263,
    0x00000000, 0x000400fa, 0x00000258, 0x00000259, 0x00000266, 0x000200f8, 0x00000259, 0x0003003e, 0x000002b9,
    0x0000034a, 0x00050041, 0x00000007, 0x000002bb, 0x000002b9, 0x000003e1, 0x0004003d, 0x00000006, 0x0000025c,
    0x000002bb, 0x0003003e, 0x000002bc, 0x000003a7, 0x00050041, 0x00000007, 0x000002be, 0x000002bc, 0x000003e1,
    0x0004003d, 0x00000006, 0x0000025f, 0x000002be, 0x000500ab, 0x00000015, 0x00000260, 0x0000025c, 0x0000025f,
    0x000300f7, 0x00000262, 0x00000000, 0x000400fa, 0x00000260, 0x00000261, 0x00000262, 0x000200f8, 0x00000261,
    0x000200f9, 0x00000266, 0x000200f8, 0x00000262, 0x000200f9, 0x00000263, 0x000200f8, 0x00000263, 0x00050080,
    0x00000006, 0x00000265, 0x000003e1, 0x00000067, 0x000200f9, 0x00000255, 0x000200f8, 0x00000266, 0x000700f5,
    0x00000015, 0x000003e5, 0x000003ea, 0x00000255, 0x00000081, 0x00000261, 0x000700f5, 0x00000015, 0x000003e2,
    0x00000081, 0x00000255, 0x00000085, 0x00000261, 0x000300f7, 0x00000268, 0x00000000, 0x000400fa, 0x000003e2,
    0x00000269, 0x00000268, 0x000200f8, 0x00000268, 0x000200f9, 0x00000269, 0x000200f8, 0x00000269, 0x000700f5,
    0x00000015, 0x000003e4, 0x000003e5, 0x00000266, 0x00000085, 0x00000268, 0x000300f7, 0x000001c6, 0x00000000,
    0x000400fa, 0x000003e4, 0x000001c1, 0x000001c6, 0x000200f8, 0x000001c1, 0x000b0050, 0x0000000e, 0x000003c1,
    0x000001a9, 0x000001ab, 0x000001ad, 0x000001af, 0x000001b1, 0x000001b3, 0x000001b5, 0x000001b7, 0x000300f7,
    0x0000028a, 0x00000000, 0x000300fb, 0x00000055, 0x00000275, 0x000200f8, 0x00000275, 0x000200f9, 0x00000276,
    0x000200f8, 0x00000276, 0x000700f5, 0x00000006, 0x000003f0, 0x00000055, 0x00000275, 0x00000286, 0x00000284,
    0x000500b0, 0x00000015, 0x00000279, 0x000003f0, 0x0000000d, 0x000400f6, 0x00000287, 0x00000284, 0x00000000,
    0x000400fa, 0x00000279, 0x0000027a, 0x00000287, 0x000200f8, 0x0000027a, 0x0003003e, 0x000002b3, 0x000003c1,
    0x00050041, 0x00000007, 0x000002b5, 0x000002b3, 0x000003f0, 0x0004003d, 0x00000006, 0x0000027d, 0x000002b5,
    0x0003003e, 0x000002b6, 0x00000088, 0x00050041, 0x00000007, 0x000002b8, 0x000002b6, 0x000003f0, 0x0004003d,
    0x00000006, 0x00000280, 0x000002b8, 0x000500ab, 0x00000015, 0x00000281, 0x0000027d, 0x00000280, 0x000300f7,
    0x00000283, 0x00000000, 0x000400fa, 0x00000281, 0x00000282, 0x00000283, 0x000200f8, 0x00000282, 0x000200f9,
    0x00000287, 0x000200f8, 0x00000283, 0x000200f9, 0x00000284, 0x000200f8, 0x00000284, 0x00050080, 0x00000006,
    0x00000286, 0x000003f0, 0x00000067, 0x000200f9, 0x00000276, 0x000200f8, 0x00000287, 0x000700f5, 0x00000015,
    0x000003f4, 0x000003fc, 0x00000276, 0x00000081, 0x00000282, 0x000700f5, 0x00000015, 0x000003f1, 0x00000081,
    0x00000276, 0x00000085, 0x00000282, 0x000300f7, 0x00000289, 0x00000000, 0x000400fa, 0x000003f1, 0x0000028a,
    0x00000289, 0x000200f8, 0x00000289, 0x000200f9, 0x0000028a, 0x000200f8, 0x0000028a, 0x000700f5, 0x00000015,
    0x000003f3, 0x000003f4, 0x00000287, 0x00000085, 0x00000289, 0x000400a8, 0x00000015, 0x000001c5, 0x000003f3,
    0x000200f9, 0x000001c6, 0x000200f8, 0x000001c6, 0x000700f5, 0x00000015, 0x00000448, 0x000003fc, 0x00000269,
    0x000003f3, 0x0000028a, 0x000700f5, 0x00000015, 0x000001c7, 0x000003e4, 0x00000269, 0x000001c5, 0x0000028a,
    0x000300f7, 0x000001cb, 0x00000000, 0x000400fa, 0x000001c7, 0x000001c8, 0x000001cb, 0x000200f8, 0x000001c8,
    0x000200f9, 0x000001d0, 0x000200f8, 0x000001cb, 0x000200f9, 0x000001cc, 0x000200f8, 0x000001cc, 0x000200f9,
    0x000001cd, 0x000200f8, 0x000001cd, 0x00050080, 0x00000006, 0x000001cf, 0x0000018c, 0x00000067, 0x000200f9,
    0x00000187, 0x000200f8, 0x000001d0, 0x000700f5, 0x00000006, 0x0000041c, 0x00000055, 0x000001bc, 0x000001a9,
    0x000001c8, 0x000700f5, 0x00000006, 0x0000041a, 0x00000055, 0x000001bc, 0x000001ab, 0x000001c8, 0x000700f5,
    0x00000006, 0x00000418, 0x00000055, 0x000001bc, 0x000001ad, 0x000001c8, 0x000700f5, 0x00000006, 0x00000416,
    0x00000055, 0x000001bc, 0x000001af, 0x000001c8, 0x000700f5, 0x00000006, 0x00000414, 0x00000055, 0x000001bc,
    0x000001b1, 0x000001c8, 0x000700f5, 0x00000006, 0x00000412, 0x00000055, 0x000001bc, 0x000001b3, 0x000001c8,
    0x000700f5, 0x00000006, 0x00000410, 0x00000055, 0x000001bc, 0x000001b5, 0x000001c8, 0x000700f5, 0x00000006,
    0x0000040e, 0x00000055, 0x000001bc, 0x000001b7, 0x000001c8, 0x000300f7, 0x000001d2, 0x00000000, 0x000400fa,
    0x00000085, 0x000001d3, 0x000001d2, 0x000200f8, 0x000001d2, 0x000200f9, 0x000001d3, 0x000200f8, 0x000001d3,
    0x000900f5, 0x00000006, 0x0000041b, 0x00000055, 0x00000184, 0x0000041c, 0x000001d0, 0x0000041c, 0x000001d2,
    0x000900f5, 0x00000006, 0x00000419, 0x00000055, 0x00000184, 0x0000041a, 0x000001d0, 0x0000041a, 0x000001d2,
    0x000900f5, 0x00000006, 0x00000417, 0x00000055, 0x00000184, 0x00000418, 0x000001d0, 0x00000418, 0x000001d2,
    0x000900f5, 0x00000006, 0x00000415, 0x00000055, 0x00000184, 0x00000416, 0x000001d0, 0x00000416, 0x000001d2,
    0x000900f5, 0x00000006, 0x00000413, 0x00000055, 0x00000184, 0x00000414, 0x000001d0, 0x00000414, 0x000001d2,
    0x000900f5, 0x00000006, 0x00000411, 0x00000055, 0x00000184, 0x00000412, 0x000001d0, 0x00000412, 0x000001d2,
    0x000900f5, 0x00000006, 0x0000040f, 0x00000055, 0x00000184, 0x00000410, 0x000001d0, 0x00000410, 0x000001d2,
    0x000900f5, 0x00000006, 0x0000040d, 0x00000055, 0x00000184, 0x0000040e, 0x000001d0, 0x0000040e, 0x000001d2,
    0x00060041, 0x00000113, 0x0000013f, 0x0000010a, 0x00000074, 0x00000096, 0x0004003d, 0x000000ff, 0x00000140,
    0x0000013f, 0x00060041, 0x00000117, 0x00000142, 0x00000140, 0x00000074, 0x000000fc, 0x0006003d, 0x00000101,
    0x00000143, 0x00000142, 0x00000002, 0x00000008, 0x000b0050, 0x0000000e, 0x0000036c, 0x0000041b, 0x00000419,
    0x00000417, 0x00000415, 0x00000413, 0x00000411, 0x0000040f, 0x0000040d, 0x000300f7, 0x000002ab, 0x00000000,
    0x000300fb, 0x00000055, 0x00000296, 0x000200f8, 0x00000296, 0x000200f9, 0x00000297, 0x000200f8, 0x00000297,
    0x000700f5, 0x00000006, 0x0000041d, 0x00000055, 0x00000296, 0x000002a7, 0x000002a5, 0x000500b0, 0x00000015,
    0x0000029a, 0x0000041d, 0x0000000d, 0x000400f6, 0x000002a8, 0x000002a5, 0x00000000, 0x000400fa, 0x0000029a,
    0x0000029b, 0x000002a8, 0x000200f8, 0x0000029b, 0x0003003e, 0x000002ad, 0x0000036c, 0x00050041, 0x00000007,
    0x000002af, 0x000002ad, 0x0000041d, 0x0004003d, 0x00000006, 0x0000029e, 0x000002af, 0x0003003e, 0x000002b0,
    0x00000088, 0x00050041, 0x00000007, 0x000002b2, 0x000002b0, 0x0000041d, 0x0004003d, 0x00000006, 0x000002a1,
    0x000002b2, 0x000500ab, 0x00000015, 0x000002a2, 0x0000029e, 0x000002a1, 0x000300f7, 0x000002a4, 0x00000000,
    0x000400fa, 0x000002a2, 0x000002a3, 0x000002a4, 0x000200f8, 0x000002a3, 0x000200f9, 0x000002a8, 0x000200f8,
    0x000002a4, 0x000200f9, 0x000002a5, 0x000200f8, 0x000002a5, 0x00050080, 0x00000006, 0x000002a7, 0x0000041d,
    0x00000067, 0x000200f9, 0x00000297, 0x000200f8, 0x000002a8, 0x000700f5, 0x00000015, 0x00000421, 0x000003d1,
    0x00000297, 0x00000081, 0x000002a3, 0x000700f5, 0x00000015, 0x0000041e, 0x00000081, 0x00000297, 0x00000085,
    0x000002a3, 0x000300f7, 0x000002aa, 0x00000000, 0x000400fa, 0x0000041e, 0x000002ab, 0x000002aa, 0x000200f8,
    0x000002aa, 0x000200f9, 0x000002ab, 0x000200f8, 0x000002ab, 0x000700f5, 0x00000015, 0x00000420, 0x00000421,
    0x000002a8, 0x00000085, 0x000002aa, 0x000200f9, 0x00000148, 0x000200f8, 0x00000148, 0x000600a9, 0x00000006,
    0x00000449, 0x00000420, 0x0000011f, 0x0000041b, 0x000600a9, 0x00000006, 0x0000044a, 0x00000420, 0x00000121,
    0x00000419, 0x000600a9, 0x00000006, 0x0000044b, 0x00000420, 0x00000123, 0x00000417, 0x000600a9, 0x00000006,
    0x0000044c, 0x00000420, 0x00000125, 0x00000415, 0x000600a9, 0x00000006, 0x0000044d, 0x00000420, 0x00000127,
    0x00000413, 0x000600a9, 0x00000006, 0x0000044e, 0x00000420, 0x00000129, 0x00000411, 0x000600a9, 0x00000006,
    0x0000044f, 0x00000420, 0x0000012b, 0x0000040f, 0x000600a9, 0x00000006, 0x00000450, 0x00000420, 0x0000012d,
    0x0000040d, 0x00050041, 0x0000011a, 0x0000014d, 0x00000143, 0x00000074, 0x00050041, 0x0000014f, 0x00000150,
    0x0000014d, 0x00000074, 0x00050041, 0x00000152, 0x00000153, 0x00000150, 0x00000074, 0x0005003e, 0x00000153,
    0x00000449, 0x00000002, 0x00000004, 0x00050041, 0x00000152, 0x00000155, 0x00000150, 0x00000067, 0x0005003e,
    0x00000155, 0x0000044a, 0x00000002, 0x00000004, 0x00050041, 0x00000152, 0x00000157, 0x00000150, 0x00000096,
    0x0005003e, 0x00000157, 0x0000044b, 0x00000002, 0x00000004, 0x00050041, 0x00000152, 0x00000159, 0x00000150,
    0x000000bb, 0x0005003e, 0x00000159, 0x0000044c, 0x00000002, 0x00000004, 0x00050041, 0x00000152, 0x0000015b,
    0x00000150, 0x000000be, 0x0005003e, 0x0000015b, 0x0000044d, 0x00000002, 0x00000004, 0x00050041, 0x00000152,
    0x0000015d, 0x00000150, 0x000000c1, 0x0005003e, 0x0000015d, 0x0000044e, 0x00000002, 0x00000004, 0x00050041,
    0x00000152, 0x0000015f, 0x00000150, 0x000000c4, 0x0005003e, 0x0000015f, 0x0000044f, 0x00000002, 0x00000004,
    0x00050041, 0x00000152, 0x00000161, 0x00000150, 0x000000c7, 0x0005003e, 0x00000161, 0x00000450, 0x00000002,
    0x00000004, 0x000200f9, 0x00000164, 0x000200f8, 0x00000164, 0x000100fd, 0x00010038
};

const std::vector<uint32_t> g_replacer_bda_binary_comp = {
    0x07230203, 0x00010300, 0x0008000b, 0x000003bc, 0x00000000, 0x00020011, 0x00000001, 0x00020011, 0x0000000b,
    0x00020011, 0x0000000c, 0x00020011, 0x000014e3, 0x0009000a, 0x5f565053, 0x5f52484b, 0x73796870, 0x6c616369,
    0x6f74735f, 0x65676172, 0x6675625f, 0x00726566, 0x0006000b, 0x00000001, 0x4c534c47, 0x6474732e, 0x3035342e,
    0x00000000, 0x0003000e, 0x000014e4, 0x00000001, 0x0006000f, 0x00000005, 0x00000004, 0x6e69616d, 0x00000000,
    0x00000180, 0x00060010, 0x00000004, 0x00000011, 0x00000020, 0x00000001, 0x00000001, 0x00030003, 0x00000002,
    0x000001cc, 0x00070004, 0x455f4c47, 0x625f5458, 0x65666675, 0x65725f72, 0x65726566, 0x0065636e, 0x00080004,
    0x455f4c47, 0x625f5458, 0x65666675, 0x65725f72, 0x65726566, 0x3265636e, 0x00000000, 0x00080004, 0x455f4c47,
    0x735f5458, 0x616c6163, 0x6c625f72, 0x5f6b636f, 0x6f79616c, 0x00007475, 0x00080004, 0x455f4c47, 0x735f5458,
    0x65646168, 0x74615f72, 0x63696d6f, 0x746e695f, 0x00003436, 0x000d0004, 0x455f4c47, 0x735f5458, 0x65646168,
    0x78655f72, 0x63696c70, 0x615f7469, 0x68746972, 0x6974656d, 0x79745f63, 0x5f736570, 0x36746e69, 0x00000034,
    0x00040005, 0x00000004, 0x6e69616d, 0x00000000, 0x00050005, 0x0000001a, 0x68736148, 0x5070614d, 0x00007274,
    0x00050006, 0x0000001a, 0x00000000, 0x726f7473, 0x00656761, 0x00050006, 0x0000001a, 0x00000001, 0x657a6973,
    0x00000000, 0x00060006, 0x0000001a, 0x00000002, 0x61706163, 0x79746963, 0x00000000, 0x00060005, 0x0000001b,
    0x68736168, 0x5f70616d, 0x6d657469, 0x0000745f, 0x00040006, 0x0000001b, 0x00000000, 0x0079656b, 0x00050006,
    0x0000001b, 0x00000001, 0x756c6176, 0x00000065, 0x00060005, 0x0000001d, 0x68736148, 0x5370614d, 0x61726f74,
    0x00006567, 0x00040006, 0x0000001d, 0x00000000, 0x00000076, 0x00060005, 0x0000002b, 0x726f7473, 0x5f656761,
    0x6d657469, 0x0000745f, 0x00040006, 0x0000002b, 0x00000000, 0x0079656b, 0x00050006, 0x0000002b, 0x00000001,
    0x756c6176, 0x00000065, 0x00050006, 0x0000002b, 0x00000002, 0x657a6973, 0x00000000, 0x00040005, 0x0000002d,
    0x726f7453, 0x00656761, 0x00040006, 0x0000002d, 0x00000000, 0x00000076, 0x000a0005, 0x0000015d, 0x64646150,
    0x75426465, 0x72656666, 0x69766544, 0x64416563, 0x73657264, 0x72745073, 0x00000000, 0x00040006, 0x0000015d,
    0x00000000, 0x00646170, 0x00090006, 0x0000015d, 0x00000001, 0x66667562, 0x645f7265, 0x63697665, 0x64615f65,
    0x73657264, 0x00000073, 0x00080005, 0x00000164, 0x66667542, 0x65447265, 0x65636976, 0x72646441, 0x50737365,
    0x00007274, 0x00090006, 0x00000164, 0x00000000, 0x66667562, 0x645f7265, 0x63697665, 0x64615f65, 0x73657264,
    0x00000073, 0x00080005, 0x00000180, 0x475f6c67, 0x61626f6c, 0x766e496c, 0x7461636f, 0x496e6f69, 0x00000044,
    0x00060005, 0x00000185, 0x74726f73, 0x615f6465, 0x79617272, 0x0000745f, 0x00050006, 0x00000185, 0x00000000,
    0x726f7473, 0x00656761, 0x00060006, 0x00000185, 0x00000001, 0x5f6d756e, 0x6d657469, 0x00000073, 0x00050006,
    0x00000185, 0x00000002, 0x64646170, 0x00676e69, 0x00080005, 0x00000187, 0x6c706572, 0x72656361, 0x7261705f,
    0x5f736d61, 0x5f616462, 0x00000074, 0x00090006, 0x00000187, 0x00000000, 0x69766564, 0x615f6563, 0x65726464,
    0x615f7373, 0x79617272, 0x00000000, 0x00080006, 0x00000187, 0x00000001, 0x72646461, 0x5f737365, 0x63616c62,
    0x73696c6b, 0x00000074, 0x00070006, 0x00000187, 0x00000002, 0x75706e69, 0x61685f74, 0x656c646e, 0x00000073,
    0x00070006, 0x00000187, 0x00000003, 0x7074756f, 0x685f7475, 0x6c646e61, 0x00007365, 0x00060006, 0x00000187,
    0x00000004, 0x5f6d756e, 0x646e6168, 0x0073656c, 0x00060005, 0x00000189, 0x72646441, 0x41737365, 0x79617272,
    0x00000000, 0x00040006, 0x00000189, 0x00000000, 0x00000076, 0x00030005, 0x0000018a, 0x00006370, 0x00050006,
    0x0000018a, 0x00000000, 0x61726170, 0x0000736d, 0x00030005, 0x0000018c, 0x00000000, 0x00030047, 0x0000001a,
    0x00000002, 0x00050048, 0x0000001a, 0x00000000, 0x00000023, 0x00000000, 0x00050048, 0x0000001a, 0x00000001,
    0x00000023, 0x00000008, 0x00050048, 0x0000001a, 0x00000002, 0x00000023, 0x0000000c, 0x00050048, 0x0000001b,
    0x00000000, 0x00000023, 0x00000000, 0x00050048, 0x0000001b, 0x00000001, 0x00000023, 0x00000008, 0x00040047,
    0x0000001c, 0x00000006, 0x00000010, 0x00030047, 0x0000001d, 0x00000002, 0x00040048, 0x0000001d, 0x00000000,
    0x00000018, 0x00050048, 0x0000001d, 0x00000000, 0x00000023, 0x00000000, 0x00050048, 0x0000002b, 0x00000000,
    0x00000023, 0x00000000, 0x00050048, 0x0000002b, 0x00000001, 0x00000023, 0x00000008, 0x00050048, 0x0000002b,
    0x00000002, 0x00000023, 0x00000010, 0x00040047, 0x0000002c, 0x00000006, 0x00000018, 0x00030047, 0x0000002d,
    0x00000002, 0x00040048, 0x0000002d, 0x00000000, 0x00000018, 0x00050048, 0x0000002d, 0x00000000, 0x00000023,
    0x00000000, 0x00030047, 0x0000015d, 0x00000002, 0x00050048, 0x0000015d, 0x00000000, 0x00000023, 0x00000000,
    0x00050048, 0x0000015d, 0x00000001, 0x00000023, 0x00000008, 0x00030047, 0x00000164, 0x00000002, 0x00050048,
    0x00000164, 0x00000000, 0x00000023, 0x00000000, 0x00040047, 0x00000180, 0x0000000b, 0x0000001c, 0x00050048,
    0x00000185, 0x00000000, 0x00000023, 0x00000000, 0x00050048, 0x00000185, 0x00000001, 0x00000023, 0x00000008,
    0x00050048, 0x00000185, 0x00000002, 0x00000023, 0x0000000c, 0x00050048, 0x00000187, 0x00000000, 0x00000023,
    0x00000000, 0x00050048, 0x00000187, 0x00000001, 0x00000023, 0x00000010, 0x00050048, 0x00000187, 0x00000002,
    0x00000023, 0x00000018, 0x00050048, 0x00000187, 0x00000003, 0x00000023, 0x00000020, 0x00050048, 0x00000187,
    0x00000004, 0x00000023, 0x00000028, 0x00040047, 0x00000188, 0x00000006, 0x00000008, 0x00030047, 0x00000189,
    0x00000002, 0x00040048, 0x00000189, 0x00000000, 0x00000018, 0x00050048, 0x00000189, 0x00000000, 0x00000023,
    0x00000000, 0x00030047, 0x0000018a, 0x00000002, 0x00050048, 0x0000018a, 0x00000000, 0x00000023, 0x00000000,
    0x00040047, 0x000001d4, 0x0000000b, 0x00000019, 0x00020013, 0x00000002, 0x00030021, 0x00000003, 0x00000002,
    0x00040015, 0x00000006, 0x00000020, 0x00000000, 0x00040015, 0x0000000d, 0x00000040, 0x00000000, 0x00020014,
    0x0000000e, 0x00030027, 0x00000018, 0x000014e5, 0x00030027, 0x00000019, 0x000014e5, 0x0005001e, 0x0000001a,
    0x00000019, 0x00000006, 0x00000006, 0x0004001e, 0x0000001b, 0x0000000d, 0x0000000d, 0x0003001d, 0x0000001c,
    0x0000001b, 0x0003001e, 0x0000001d, 0x0000001c, 0x00040020, 0x00000019, 0x000014e5, 0x0000001d, 0x00040020,
    0x00000018, 0x000014e5, 0x0000001a, 0x00030027, 0x00000029, 0x000014e5, 0x0005001e, 0x0000002b, 0x0000000d,
    0x0000000d, 0x0000000d, 0x0003001d, 0x0000002c, 0x0000002b, 0x0003001e, 0x0000002d, 0x0000002c, 0x00040020,
    0x00000029, 0x000014e5, 0x0000002d, 0x0004002b, 0x00000006, 0x0000003e, 0x165667b1, 0x0004002b, 0x00000006,
    0x00000041, 0xc2b2ae3d, 0x0004002b, 0x00000006, 0x00000044, 0x27d4eb2f, 0x00040015, 0x00000046, 0x00000020,
    0x00000001, 0x0004002b, 0x00000046, 0x00000047, 0x00000011, 0x0004002b, 0x00000046, 0x0000004a, 0x0000000f,
    0x0004002b, 0x00000006, 0x0000004e, 0x85ebca77, 0x0004002b, 0x00000046, 0x00000056, 0x0000000d, 0x0004002b,
    0x00000046, 0x0000005c, 0x00000010, 0x0005002b, 0x0000000d, 0x00000061, 0x00000000, 0x00000000, 0x0005002b,
    0x0000000d, 0x00000066, 0x00000020, 0x00000000, 0x0005002b, 0x0000000d, 0x0000006a, 0xffffffff, 0x00000000,
    0x0004002b, 0x00000006, 0x0000006d, 0x00000000, 0x0004002b, 0x00000046, 0x0000007a, 0x00000002, 0x00040020,
    0x0000007b, 0x000014e5, 0x00000006, 0x0004002b, 0x00000006, 0x0000008c, 0x00000001, 0x0004002b, 0x00000046,
    0x00000093, 0x00000000, 0x00040020, 0x00000094, 0x000014e5, 0x00000019, 0x00040020, 0x00000098, 0x000014e5,
    0x0000001b, 0x0004002b, 0x00000046, 0x0000009e, 0x00000001, 0x00040020, 0x000000d4, 0x000014e5, 0x0000000d,
    0x0004002b, 0x00000006, 0x00000119, 0x00000002, 0x0005002b, 0x0000000d, 0x0000012d, 0x00000001, 0x00000000,
    0x0005002b, 0x0000000d, 0x00000153, 0x00000007, 0x00000000, 0x0005002b, 0x0000000d, 0x0000015a, 0xfffffff8,
    0xffffffff, 0x00030027, 0x0000015c, 0x000014e5, 0x0004001e, 0x0000015d, 0x00000006, 0x0000000d, 0x00040020,
    0x0000015c, 0x000014e5, 0x0000015d, 0x00030027, 0x00000163, 0x000014e5, 0x0003001e, 0x00000164, 0x0000000d,
    0x00040020, 0x00000163, 0x000014e5, 0x00000164, 0x00040017, 0x0000017e, 0x00000006, 0x00000003, 0x00040020,
    0x0000017f, 0x00000001, 0x0000017e, 0x0004003b, 0x0000017f, 0x00000180, 0x00000001, 0x00040020, 0x00000181,
    0x00000001, 0x00000006, 0x0005001e, 0x00000185, 0x00000029, 0x00000006, 0x00000006, 0x00030027, 0x00000186,
    0x000014e5, 0x0007001e, 0x00000187, 0x00000185, 0x00000018, 0x00000186, 0x00000186, 0x00000006, 0x0003001d,
    0x00000188, 0x0000000d, 0x0003001e, 0x00000189, 0x00000188, 0x00040020, 0x00000186, 0x000014e5, 0x00000189,
    0x0003001e, 0x0000018a, 0x00000187, 0x00040020, 0x0000018b, 0x00000009, 0x0000018a, 0x0004003b, 0x0000018b,
    0x0000018c, 0x00000009, 0x0004002b, 0x00000046, 0x0000018d, 0x00000004, 0x00040020, 0x0000018e, 0x00000009,
    0x00000006, 0x00040020, 0x00000196, 0x00000009, 0x00000186, 0x00040020, 0x0000019e, 0x00000009, 0x00000018,
    0x00040020, 0x000001ac, 0x00000009, 0x00000185, 0x0004002b, 0x00000046, 0x000001c5, 0x00000003, 0x0004002b,
    0x00000006, 0x000001d3, 0x00000020, 0x0006002c, 0x0000017e, 0x000001d4, 0x000001d3, 0x0000008c, 0x0000008c,
    0x0003002a, 0x0000000e, 0x000001d7, 0x00030029, 0x0000000e, 0x000001da, 0x00030001, 0x0000000d, 0x000003ae,
    0x00050036, 0x00000002, 0x00000004, 0x00000000, 0x00000003, 0x000200f8, 0x00000005, 0x000300f7, 0x000001d5,
    0x00000000, 0x000300fb, 0x0000006d, 0x000001d6, 0x000200f8, 0x000001d6, 0x00050041, 0x00000181, 0x00000182,
    0x00000180, 0x0000006d, 0x0004003d, 0x00000006, 0x00000183, 0x00000182, 0x00060041, 0x0000018e, 0x0000018f,
    0x0000018c, 0x00000093, 0x0000018d, 0x0004003d, 0x00000006, 0x00000190, 0x0000018f, 0x000500ae, 0x0000000e,
    0x00000191, 0x00000183, 0x00000190, 0x000300f7, 0x00000193, 0x00000000, 0x000400fa, 0x00000191, 0x00000192,
    0x00000193, 0x000200f8, 0x00000192, 0x000200f9, 0x000001d5, 0x000200f8, 0x00000193, 0x00060041, 0x00000196,
    0x00000197, 0x0000018c, 0x00000093, 0x0000007a, 0x0004003d, 0x00000186, 0x00000198, 0x00000197, 0x00060041,
    0x000000d4, 0x0000019b, 0x00000198, 0x00000093, 0x00000183, 0x0006003d, 0x0000000d, 0x0000019c, 0x0000019b,
    0x00000002, 0x00000008, 0x000500c7, 0x0000000d, 0x000001f2, 0x0000019c, 0x00000153, 0x000500ab, 0x0000000e,
    0x000001f3, 0x000001f2, 0x00000061, 0x000300f7, 0x000001ff, 0x00000000, 0x000400fa, 0x000001f3, 0x000001f4,
    0x000001fa, 0x000200f8, 0x000001f4, 0x000500c7, 0x0000000d, 0x000001f6, 0x0000019c, 0x0000015a, 0x00040078,
    0x0000015c, 0x000001f7, 0x000001f6, 0x00050041, 0x000000d4, 0x000001f8, 0x000001f7, 0x0000009e, 0x0006003d,
    0x0000000d, 0x000001f9, 0x000001f8, 0x00000002, 0x00000008, 0x000200f9, 0x000001ff, 0x000200f8, 0x000001fa,
    0x00040078, 0x00000163, 0x000001fc, 0x0000019c, 0x00050041, 0x000000d4, 0x000001fd, 0x000001fc, 0x00000093,
    0x0006003d, 0x0000000d, 0x000001fe, 0x000001fd, 0x00000002, 0x00000010, 0x000200f9, 0x000001ff, 0x000200f8,
    0x000001ff, 0x000700f5, 0x0000000d, 0x0000039c, 0x000001f9, 0x000001f4, 0x000001fe, 0x000001fa, 0x00060041,
    0x0000019e, 0x0000019f, 0x0000018c, 0x00000093, 0x0000009e, 0x0004003d, 0x00000018, 0x000001a0, 0x0000019f,
    0x0006003d, 0x0000000d, 0x000001a5, 0x0000019b, 0x00000002, 0x00000008, 0x000300f7, 0x0000023e, 0x00000000,
    0x000300fb, 0x0000006d, 0x00000208, 0x000200f8, 0x00000208, 0x000500aa, 0x0000000e, 0x00000242, 0x000001a5,
    0x00000061, 0x000400a8, 0x0000000e, 0x0000020a, 0x00000242, 0x000300f7, 0x0000020f, 0x00000000, 0x000400fa,
    0x0000020a, 0x0000020b, 0x0000020f, 0x000200f8, 0x0000020b, 0x00050041, 0x0000007b, 0x0000020c, 0x000001a0,
    0x0000007a, 0x0006003d, 0x00000006, 0x0000020d, 0x0000020c, 0x00000002, 0x00000004, 0x000500aa, 0x0000000e,
    0x0000020e, 0x0000020d, 0x0000006d, 0x000200f9, 0x0000020f, 0x000200f8, 0x0000020f, 0x000700f5, 0x0000000e,
    0x00000210, 0x00000242, 0x00000208, 0x0000020e, 0x0000020b, 0x000300f7, 0x00000212, 0x00000000, 0x000400fa,
    0x00000210, 0x00000211, 0x00000212, 0x000200f8, 0x00000211, 0x000200f9, 0x0000023e, 0x000200f8, 0x00000212,
    0x000500c2, 0x0000000d, 0x0000024a, 0x000001a5, 0x00000066, 0x00040071, 0x00000006, 0x0000024b, 0x0000024a,
    0x000500c7, 0x0000000d, 0x0000024d, 0x000001a5, 0x0000006a, 0x00040071, 0x00000006, 0x0000024e, 0x0000024d,
    0x00050080, 0x00000006, 0x00000255, 0x0000024e, 0x0000003e, 0x000500c4, 0x00000006, 0x0000025a, 0x00000255,
    0x00000047, 0x000500c2, 0x00000006, 0x0000025c, 0x00000255, 0x0000004a, 0x000500c5, 0x00000006, 0x0000025d,
    0x0000025a, 0x0000025c, 0x00050084, 0x00000006, 0x0000025e, 0x00000044, 0x0000025d, 0x000500c2, 0x00000006,
    0x00000261, 0x0000025e, 0x0000004a, 0x000500c6, 0x00000006, 0x00000262, 0x0000025e, 0x00000261, 0x00050084,
    0x00000006, 0x00000263, 0x0000004e, 0x00000262, 0x000500c2, 0x00000006, 0x00000266, 0x00000263, 0x00000056,
    0x000500c6, 0x00000006, 0x00000267, 0x00000263, 0x00000266, 0x00050084, 0x00000006, 0x00000268, 0x00000041,
    0x00000267, 0x000500c2, 0x00000006, 0x0000026b, 0x00000268, 0x0000005c, 0x000500c6, 0x00000006, 0x0000026c,
    0x00000268, 0x0000026b, 0x00050080, 0x00000006, 0x00000271, 0x0000024b, 0x0000003e, 0x00050084, 0x00000006,
    0x00000273, 0x0000026c, 0x00000041, 0x00050080, 0x00000006, 0x00000274, 0x00000271, 0x00000273, 0x000500c4,
    0x00000006, 0x00000276, 0x00000274, 0x00000047, 0x000500c2, 0x00000006, 0x00000278, 0x00000274, 0x0000004a,
    0x000500c5, 0x00000006, 0x00000279, 0x00000276, 0x00000278, 0x00050084, 0x00000006, 0x0000027a, 0x00000044,
    0x00000279, 0x000500c2, 0x00000006, 0x0000027d, 0x0000027a, 0x0000004a, 0x000500c6, 0x00000006, 0x0000027e,
    0x0000027a, 0x0000027d, 0x00050084, 0x00000006, 0x0000027f, 0x0000004e, 0x0000027e, 0x000500c2, 0x00000006,
    0x00000282, 0x0000027f, 0x00000056, 0x000500c6, 0x00000006, 0x00000283, 0x0000027f, 0x00000282, 0x00050084,
    0x00000006, 0x00000284, 0x00000041, 0x00000283, 0x000500c2, 0x00000006, 0x00000287, 0x00000284, 0x0000005c,
    0x000500c6, 0x00000006, 0x00000288, 0x00000284, 0x00000287, 0x000200f9, 0x00000214, 0x000200f8, 0x00000214,
    0x000700f5, 0x00000006, 0x0000039d, 0x00000288, 0x00000212, 0x0000023a, 0x00000238, 0x000400f6, 0x0000023b,
    0x00000238, 0x00000000, 0x000200f9, 0x00000215, 0x000200f8, 0x00000215, 0x00050041, 0x0000007b, 0x00000216,
    0x000001a0, 0x0000007a, 0x0006003d, 0x00000006, 0x00000217, 0x00000216, 0x00000002, 0x00000004, 0x00050082,
    0x00000006, 0x00000218, 0x00000217, 0x0000008c, 0x000500c7, 0x00000006, 0x0000021a, 0x0000039d, 0x00000218,
    0x00050041, 0x00000094, 0x0000021b, 0x000001a0, 0x00000093, 0x0006003d, 0x00000019, 0x0000021c, 0x0000021b,
    0x00000002, 0x00000010, 0x00060041, 0x00000098, 0x0000021e, 0x0000021c, 0x00000093, 0x0000021a, 0x0006003d,
    0x0000001b, 0x0000021f, 0x0000021e, 0x00000002, 0x00000010, 0x00050051, 0x0000000d, 0x00000220, 0x0000021f,
    0x00000000, 0x00050051, 0x0000000d, 0x00000222, 0x0000021f, 0x00000001, 0x000500aa, 0x0000000e, 0x0000028b,
    0x00000220, 0x00000061, 0x000300f7, 0x00000237, 0x00000000, 0x000400fa, 0x0000028b, 0x00000227, 0x00000228,
    0x000200f8, 0x00000227, 0x000200f9, 0x0000023b, 0x000200f8, 0x00000228, 0x000500aa, 0x0000000e, 0x0000022b,
    0x000001a5, 0x00000220, 0x000300f7, 0x00000231, 0x00000000, 0x000400fa, 0x0000022b, 0x0000022c, 0x00000231,
    0x000200f8, 0x0000022c, 0x000500aa, 0x0000000e, 0x0000028e, 0x00000222, 0x00000061, 0x000400a8, 0x0000000e,
    0x00000230, 0x0000028e, 0x000200f9, 0x00000231, 0x000200f8, 0x00000231, 0x000700f5, 0x0000000e, 0x00000232,
    0x0000022b, 0x00000228, 0x00000230, 0x0000022c, 0x000300f7, 0x00000236, 0x00000000, 0x000400fa, 0x00000232,
    0x00000233, 0x00000236, 0x000200f8, 0x00000233, 0x000200f9, 0x0000023b, 0x000200f8, 0x00000236, 0x000200f9,
    0x00000237, 0x000200f8, 0x00000237, 0x000200f9, 0x00000238, 0x000200f8, 0x00000238, 0x00050080, 0x00000006,
    0x0000023a, 0x0000021a, 0x0000009e, 0x000200f9, 0x00000214, 0x000200f8, 0x0000023b, 0x000700f5, 0x0000000d,
    0x000003a1, 0x00000061, 0x00000227, 0x00000222, 0x00000233, 0x000300f7, 0x0000023d, 0x00000000, 0x000400fa,
    0x000001da, 0x0000023e, 0x0000023d, 0x000200f8, 0x0000023d, 0x000200f9, 0x0000023e, 0x000200f8, 0x0000023e,
    0x000900f5, 0x0000000d, 0x000003a0, 0x00000061, 0x00000211, 0x000003a1, 0x0000023b, 0x000003a1, 0x0000023d,
    0x000500ab, 0x0000000e, 0x000001a8, 0x000003a0, 0x0000039c, 0x000300f7, 0x000001aa, 0x00000000, 0x000400fa,
    0x000001a8, 0x000001a9, 0x000001aa, 0x000200f8, 0x000001a9, 0x00060041, 0x000001ac, 0x000001ad, 0x0000018c,
    0x00000093, 0x00000093, 0x0004003d, 0x00000185, 0x000001ae, 0x000001ad, 0x00050051, 0x00000029, 0x000001b1,
    0x000001ae, 0x00000000, 0x00050051, 0x00000006, 0x000001b4, 0x000001ae, 0x00000001, 0x000300f7, 0x000002df,
    0x00000000, 0x000300fb, 0x0000006d, 0x00000297, 0x000200f8, 0x00000297, 0x000500aa, 0x0000000e, 0x00000299,
    0x000001b4, 0x0000006d, 0x000300f7, 0x0000029b, 0x00000000, 0x000400fa, 0x00000299, 0x0000029a, 0x0000029b,
    0x000200f8, 0x0000029a, 0x000200f9, 0x000002df, 0x000200f8, 0x0000029b, 0x00050082, 0x00000006, 0x0000029d,
    0x000001b4, 0x0000008c, 0x000200f9, 0x0000029e, 0x000200f8, 0x0000029e, 0x000700f5, 0x00000006, 0x000003a3,
    0x0000029d, 0x0000029b, 0x000003bb, 0x000002db, 0x000700f5, 0x00000006, 0x000003a2, 0x0000006d, 0x0000029b,
    0x000003b9, 0x000002db, 0x000500b2, 0x0000000e, 0x000002a2, 0x000003a2, 0x000003a3, 0x000400f6, 0x000002dc,
    0x000002db, 0x00000000, 0x000400fa, 0x000002a2, 0x000002a3, 0x000002dc, 0x000200f8, 0x000002a3, 0x00050082,
    0x00000006, 0x000002a7, 0x000003a3, 0x000003a2, 0x00050086, 0x00000006, 0x000002a8, 0x000002a7, 0x00000119,
    0x00050080, 0x00000006, 0x000002a9, 0x000003a2, 0x000002a8, 0x00070041, 0x000000d4, 0x000002ad, 0x000001b1,
    0x00000093, 0x000002a9, 0x00000093, 0x0006003d, 0x0000000d, 0x000002ae, 0x000002ad, 0x00000002, 0x00000008,
    0x000500ae, 0x0000000e, 0x000002af, 0x0000039c, 0x000002ae, 0x000300f7, 0x000002bd, 0x00000000, 0x000400fa,
    0x000002af, 0x000002b0, 0x000002bd, 0x000200f8, 0x000002b0, 0x0006003d, 0x0000000d, 0x000002b5, 0x000002ad,
    0x00000002, 0x00000008, 0x00070041, 0x000000d4, 0x000002b8, 0x000001b1, 0x00000093, 0x000002a9, 0x0000007a,
    0x0006003d, 0x0000000d, 0x000002b9, 0x000002b8, 0x00000002, 0x00000008, 0x0007000c, 0x0000000d, 0x000002ba,
    0x00000001, 0x00000029, 0x000002b9, 0x0000012d, 0x00050080, 0x0000000d, 0x000002bb, 0x000002b5, 0x000002ba,
    0x000500b0, 0x0000000e, 0x000002bc, 0x0000039c, 0x000002bb, 0x000200f9, 0x000002bd, 0x000200f8, 0x000002bd,
    0x000700f5, 0x0000000e, 0x000002be, 0x000002af, 0x000002a3, 0x000002bc, 0x000002b0, 0x000300f7, 0x000002da,
    0x00000000, 0x000400fa, 0x000002be, 0x000002bf, 0x000002cc, 0x000200f8, 0x000002bf, 0x0006003d, 0x0000000d,
    0x000002c4, 0x000002ad, 0x00000002, 0x00000008, 0x00050082, 0x0000000d, 0x000002c5, 0x0000039c, 0x000002c4,
    0x00070041, 0x000000d4, 0x000002c8, 0x000001b1, 0x00000093, 0x000002a9, 0x0000009e, 0x0006003d, 0x0000000d,
    0x000002c9, 0x000002c8, 0x00000002, 0x00000008, 0x00050080, 0x0000000d, 0x000002cb, 0x000002c9, 0x000002c5,
    0x000200f9, 0x000002dc, 0x000200f8, 0x000002cc, 0x0006003d, 0x0000000d, 0x000002d1, 0x000002ad, 0x00000002,
    0x00000008, 0x000500b0, 0x0000000e, 0x000002d2, 0x0000039c, 0x000002d1, 0x000300f7, 0x000002d9, 0x00000000,
    0x000400fa, 0x000002d2, 0x000002d3, 0x000002d6, 0x000200f8, 0x000002d3, 0x00050082, 0x00000006, 0x000002d5,
    0x000002a9, 0x0000008c, 0x000200f9, 0x000002d9, 0x000200f8, 0x000002d6, 0x00050080, 0x00000006, 0x000002d8,
    0x000002a9, 0x0000008c, 0x000200f9, 0x000002d9, 0x000200f8, 0x000002d9, 0x000700f5, 0x00000006, 0x000003bb,
    0x000002d5, 0x000002d3, 0x000003a3, 0x000002d6, 0x000700f5, 0x00000006, 0x000003b9, 0x000003a2, 0x000002d3,
    0x000002d8, 0x000002d6, 0x000200f9, 0x000002da, 0x000200f8, 0x000002da, 0x000200f9, 0x000002db, 0x000200f8,
    0x000002db, 0x000200f9, 0x0000029e, 0x000200f8, 0x000002dc, 0x000700f5, 0x0000000d, 0x000003a9, 0x000003ae,
    0x0000029e, 0x000002cb, 0x000002bf, 0x000700f5, 0x0000000e, 0x000003a4, 0x000001d7, 0x0000029e, 0x000001da,
    0x000002bf, 0x000300f7, 0x000002de, 0x00000000, 0x000400fa, 0x000003a4, 0x000002df, 0x000002de, 0x000200f8,
    0x000002de, 0x000200f9, 0x000002df, 0x000200f8, 0x000002df, 0x000900f5, 0x0000000d, 0x000003a8, 0x00000061,
    0x0000029a, 0x000003a9, 0x000002dc, 0x00000061, 0x000002de, 0x0006003d, 0x0000000d, 0x000001c2, 0x0000019b,
    0x00000002, 0x00000008, 0x000300f7, 0x0000032b, 0x00000000, 0x000300fb, 0x0000006d, 0x000002e6, 0x000200f8,
    0x000002e6, 0x000500aa, 0x0000000e, 0x0000032e, 0x000001c2, 0x00000061, 0x000400a8, 0x0000000e, 0x000002e8,
    0x0000032e, 0x000300f7, 0x000002ed, 0x00000000, 0x000400fa, 0x000002e8, 0x000002e9, 0x000002ed, 0x000200f8,
    0x000002e9, 0x00050041, 0x0000007b, 0x000002ea, 0x000001a0, 0x0000007a, 0x0006003d, 0x00000006, 0x000002eb,
    0x000002ea, 0x00000002, 0x00000004, 0x000500aa, 0x0000000e, 0x000002ec, 0x000002eb, 0x0000006d, 0x000200f9,
    0x000002ed, 0x000200f8, 0x000002ed, 0x000700f5, 0x0000000e, 0x000002ee, 0x0000032e, 0x000002e6, 0x000002ec,
    0x000002e9, 0x000300f7, 0x000002f0, 0x00000000, 0x000400fa, 0x000002ee, 0x000002ef, 0x000002f0, 0x000200f8,
    0x000002ef, 0x000200f9, 0x0000032b, 0x000200f8, 0x000002f0, 0x000500c2, 0x0000000d, 0x00000336, 0x000001c2,
    0x00000066, 0x00040071, 0x00000006, 0x00000337, 0x00000336, 0x000500c7, 0x0000000d, 0x00000339, 0x000001c2,
    0x0000006a, 0x00040071, 0x00000006, 0x0000033a, 0x00000339, 0x00050080, 0x00000006, 0x00000341, 0x0000033a,
    0x0000003e, 0x000500c4, 0x00000006, 0x00000346, 0x00000341, 0x00000047, 0x000500c2, 0x00000006, 0x00000348,
    0x00000341, 0x0000004a, 0x000500c5, 0x00000006, 0x00000349, 0x00000346, 0x00000348, 0x00050084, 0x00000006,
    0x0000034a, 0x00000044, 0x00000349, 0x000500c2, 0x00000006, 0x0000034d, 0x0000034a, 0x0000004a, 0x000500c6,
    0x00000006, 0x0000034e, 0x0000034a, 0x0000034d, 0x00050084, 0x00000006, 0x0000034f, 0x0000004e, 0x0000034e,
    0x000500c2, 0x00000006, 0x00000352, 0x0000034f, 0x00000056, 0x000500c6, 0x00000006, 0x00000353, 0x0000034f,
    0x00000352, 0x00050084, 0x00000006, 0x00000354, 0x00000041, 0x00000353, 0x000500c2, 0x00000006, 0x00000357,
    0x00000354, 0x0000005c, 0x000500c6, 0x00000006, 0x00000358, 0x00000354, 0x00000357, 0x00050080, 0x00000006,
    0x0000035d, 0x00000337, 0x0000003e, 0x00050084, 0x00000006, 0x0000035f, 0x00000358, 0x00000041, 0x00050080,
    0x00000006, 0x00000360, 0x0000035d, 0x0000035f, 0x000500c4, 0x00000006, 0x00000362, 0x00000360, 0x00000047,
    0x000500c2, 0x00000006, 0x00000364, 0x00000360, 0x0000004a, 0x000500c5, 0x00000006, 0x00000365, 0x00000362,
    0x00000364, 0x00050084, 0x00000006, 0x00000366, 0x00000044, 0x00000365, 0x000500c2, 0x00000006, 0x00000369,
    0x00000366, 0x0000004a, 0x000500c6, 0x00000006, 0x0000036a, 0x00000366, 0x00000369, 0x00050084, 0x00000006,
    0x0000036b, 0x0000004e, 0x0000036a, 0x000500c2, 0x00000006, 0x0000036e, 0x0000036b, 0x00000056, 0x000500c6,
    0x00000006, 0x0000036f, 0x0000036b, 0x0000036e, 0x00050084, 0x00000006, 0x00000370, 0x00000041, 0x0000036f,
    0x000500c2, 0x00000006, 0x00000373, 0x00000370, 0x0000005c, 0x000500c6, 0x00000006, 0x00000374, 0x00000370,
    0x00000373, 0x000200f9, 0x000002f2, 0x000200f8, 0x000002f2, 0x000700f5, 0x00000006, 0x000003b4, 0x00000374,
    0x000002f0, 0x00000327, 0x00000325, 0x000400f6, 0x00000328, 0x00000325, 0x00000000, 0x000200f9, 0x000002f3,
    0x000200f8, 0x000002f3, 0x00050041, 0x0000007b, 0x000002f4, 0x000001a0, 0x0000007a, 0x0006003d, 0x00000006,
    0x000002f5, 0x000002f4, 0x00000002, 0x00000004, 0x00050082, 0x00000006, 0x000002f6, 0x000002f5, 0x0000008c,
    0x000500c7, 0x00000006, 0x000002f8, 0x000003b4, 0x000002f6, 0x00050041, 0x00000094, 0x000002f9, 0x000001a0,
    0x00000093, 0x0006003d, 0x00000019, 0x000002fa, 0x000002f9, 0x00000002, 0x00000010, 0x00070041, 0x000000d4,
    0x000002fc, 0x000002fa, 0x00000093, 0x000002f8, 0x00000093, 0x0006003d, 0x0000000d, 0x000002fd, 0x000002fc,
    0x00000002, 0x00000010, 0x000500ab, 0x0000000e, 0x000002ff, 0x000002fd, 0x000001c2, 0x000300f7, 0x00000320,
    0x00000000, 0x000400fa, 0x000002ff, 0x00000300, 0x00000320, 0x000200f8, 0x00000300, 0x000500aa, 0x0000000e,
    0x00000377, 0x000002fd, 0x00000061, 0x000400a8, 0x0000000e, 0x00000303, 0x00000377, 0x000300f7, 0x0000030c,
    0x00000000, 0x000400fa, 0x00000303, 0x00000304, 0x0000030c, 0x000200f8, 0x00000304, 0x0006003d, 0x00000019,
    0x00000306, 0x000002f9, 0x00000002, 0x00000010, 0x00070041, 0x000000d4, 0x00000308, 0x00000306, 0x00000093,
    0x000002f8, 0x0000009e, 0x0006003d, 0x0000000d, 0x00000309, 0x00000308, 0x00000002, 0x00000008, 0x000500aa,
    0x0000000e, 0x0000037a, 0x00000309, 0x00000061, 0x000400a8, 0x0000000e, 0x0000030b, 0x0000037a, 0x000200f9,
    0x0000030c, 0x000200f8, 0x0000030c, 0x000700f5, 0x0000000e, 0x0000030d, 0x00000303, 0x00000300, 0x0000030b,
    0x00000304, 0x000300f7, 0x0000030f, 0x00000000, 0x000400fa, 0x0000030d, 0x0000030e, 0x0000030f, 0x000200f8,
    0x0000030e, 0x000200f9, 0x00000325, 0x000200f8, 0x0000030f, 0x0006003d, 0x00000019, 0x00000311, 0x000002f9,
    0x00000002, 0x00000010, 0x00070041, 0x000000d4, 0x00000313, 0x00000311, 0x00000093, 0x000002f8, 0x00000093,
    0x000900e6, 0x0000000d, 0x00000315, 0x00000313, 0x0000008c, 0x0000006d, 0x0000006d, 0x000001c2, 0x000002fd,
    0x000500aa, 0x0000000e, 0x0000037d, 0x00000315, 0x00000061, 0x000400a8, 0x0000000e, 0x00000318, 0x0000037d,
    0x000500ab, 0x0000000e, 0x0000031a, 0x00000315, 0x000001c2, 0x000500a7, 0x0000000e, 0x0000031b, 0x00000318,
    0x0000031a, 0x000300f7, 0x0000031d, 0x00000000, 0x000400fa, 0x0000031b, 0x0000031c, 0x0000031d, 0x000200f8,
    0x0000031c, 0x000200f9, 0x00000325, 0x000200f8, 0x0000031d, 0x00050041, 0x0000007b, 0x0000031e, 0x000001a0,
    0x0000009e, 0x000700ea, 0x00000006, 0x0000031f, 0x0000031e, 0x0000008c, 0x0000006d, 0x0000008c, 0x000200f9,
    0x00000320, 0x000200f8, 0x00000320, 0x0006003d, 0x00000019, 0x00000322, 0x000002f9, 0x00000002, 0x00000010,
    0x00070041, 0x000000d4, 0x00000324, 0x00000322, 0x00000093, 0x000002f8, 0x0000009e, 0x0005003e, 0x00000324,
    0x000003a8, 0x00000002, 0x00000008, 0x000200f9, 0x00000328, 0x000200f8, 0x00000325, 0x00050080, 0x00000006,
    0x00000327, 0x000002f8, 0x0000009e, 0x000200f9, 0x000002f2, 0x000200f8, 0x00000328, 0x000300f7, 0x0000032a,
    0x00000000, 0x000400fa, 0x000001da, 0x0000032b, 0x0000032a, 0x000200f8, 0x0000032a, 0x000200f9, 0x0000032b,
    0x000200f8, 0x0000032b, 0x00060041, 0x00000196, 0x000001c6, 0x0000018c, 0x00000093, 0x000001c5, 0x0004003d,
    0x00000186, 0x000001c7, 0x000001c6, 0x000500aa, 0x0000000e, 0x00000380, 0x000003a8, 0x00000061, 0x000600a9,
    0x0000000d, 0x000001cd, 0x00000380, 0x0000039c, 0x000003a8, 0x00060041, 0x000000d4, 0x000001cf, 0x000001c7,
    0x00000093, 0x00000183, 0x0006003d, 0x0000000d, 0x000001d0, 0x000001cf, 0x00000002, 0x00000008, 0x000500c7,
    0x0000000d, 0x00000384, 0x000001d0, 0x00000153, 0x000500ab, 0x0000000e, 0x00000385, 0x00000384, 0x00000061,
    0x000300f7, 0x00000392, 0x00000000, 0x000400fa, 0x00000385, 0x00000386, 0x0000038d, 0x000200f8, 0x00000386,
    0x000500c7, 0x0000000d, 0x00000388, 0x000001d0, 0x0000015a, 0x00040078, 0x0000015c, 0x00000389, 0x00000388,
    0x00050041, 0x000000d4, 0x0000038c, 0x00000389, 0x0000009e, 0x0005003e, 0x0000038c, 0x000001cd, 0x00000002,
    0x00000008, 0x000200f9, 0x00000392, 0x000200f8, 0x0000038d, 0x00040078, 0x00000163, 0x0000038f, 0x000001d0,
    0x00050041, 0x000000d4, 0x00000391, 0x0000038f, 0x00000093, 0x0005003e, 0x00000391, 0x000001cd, 0x00000002,
    0x00000010, 0x000200f9, 0x00000392, 0x000200f8, 0x00000392, 0x000200f9, 0x000001aa, 0x000200f8, 0x000001aa,
    0x000200f9, 0x000001d5, 0x000200f8, 0x000001d5, 0x000100fd, 0x00010038
};

const std::vector<uint32_t> g_replacer_rehash_comp = {
    0x07230203, 0x00010300, 0x0008000b, 0x00000183, 0x00000000, 0x00020011, 0x00000001, 0x00020011, 0x0000000b,
    0x00020011, 0x0000000c, 0x00020011, 0x000014e3, 0x0009000a, 0x5f565053, 0x5f52484b, 0x73796870, 0x6c616369,
    0x6f74735f, 0x65676172, 0x6675625f, 0x00726566, 0x0006000b, 0x00000001, 0x4c534c47, 0x6474732e, 0x3035342e,
    0x00000000, 0x0003000e, 0x000014e4, 0x00000001, 0x0006000f, 0x00000005, 0x00000004, 0x6e69616d, 0x00000000,
    0x000000b1, 0x00060010, 0x00000004, 0x00000011, 0x00000020, 0x00000001, 0x00000001, 0x00030003, 0x00000002,
    0x000001cc, 0x00070004, 0x455f4c47, 0x625f5458, 0x65666675, 0x65725f72, 0x65726566, 0x0065636e, 0x00080004,
    0x455f4c47, 0x625f5458, 0x65666675, 0x65725f72, 0x65726566, 0x3265636e, 0x00000000, 0x00080004, 0x455f4c47,
    0x735f5458, 0x616c6163, 0x6c625f72, 0x5f6b636f, 0x6f79616c, 0x00007475, 0x00080004, 0x455f4c47, 0x735f5458,
    0x65646168, 0x74615f72, 0x63696d6f, 0x746e695f, 0x00003436, 0x000d0004, 0x455f4c47, 0x735f5458, 0x65646168,
    0x78655f72, 0x63696c70, 0x615f7469, 0x68746972, 0x6974656d, 0x79745f63, 0x5f736570, 0x36746e69, 0x00000034,
    0x00040005, 0x00000004, 0x6e69616d, 0x00000000, 0x00050005, 0x0000001a, 0x68736148, 0x5070614d, 0x00007274,
    0x00050006, 0x0000001a, 0x00000000, 0x726f7473, 0x00656761, 0x00050006, 0x0000001a, 0x00000001, 0x657a6973,
    0x00000000, 0x00060006, 0x0000001a, 0x00000002, 0x61706163, 0x79746963, 0x00000000, 0x00060005, 0x0000001b,
    0x68736168, 0x5f70616d, 0x6d657469, 0x0000745f, 0x00040006, 0x0000001b, 0x00000000, 0x0079656b, 0x00050006,
    0x0000001b, 0x00000001, 0x756c6176, 0x00000065, 0x00060005, 0x0000001d, 0x68736148, 0x5370614d, 0x61726f74,
    0x00006567, 0x00040006, 0x0000001d, 0x00000000, 0x00000076, 0x00080005, 0x000000b1, 0x475f6c67, 0x61626f6c,
    0x766e496c, 0x7461636f, 0x496e6f69, 0x00000044, 0x00060005, 0x000000b6, 0x61686572, 0x705f6873, 0x6d617261,
    0x00000073, 0x00060006, 0x000000b6, 0x00000000, 0x68736168, 0x5f70616d, 0x00646c6f, 0x00060006, 0x000000b6,
    0x00000001, 0x68736168, 0x5f70616d, 0x0077656e, 0x00030005, 0x000000b7, 0x00006370, 0x00050006, 0x000000b7,
    0x00000000, 0x61726170, 0x0000736d, 0x00030005, 0x000000b9, 0x00000000, 0x00030047, 0x0000001a, 0x00000002,
    0x00050048, 0x0000001a, 0x00000000, 0x00000023, 0x00000000, 0x00050048, 0x0000001a, 0x00000001, 0x00000023,
    0x00000008, 0x00050048, 0x0000001a, 0x00000002, 0x00000023, 0x0000000c, 0x00050048, 0x0000001b, 0x00000000,
    0x00000023, 0x00000000, 0x00050048, 0x0000001b, 0x00000001, 0x00000023, 0x00000008, 0x00040047, 0x0000001c,
    0x00000006, 0x00000010, 0x00030047, 0x0000001d, 0x00000002, 0x00040048, 0x0000001d, 0x00000000, 0x00000018,
    0x00050048, 0x0000001d, 0x00000000, 0x00000023, 0x00000000, 0x00040047, 0x000000b1, 0x0000000b, 0x0000001c,
    0x00050048, 0x000000b6, 0x00000000, 0x00000023, 0x00000000, 0x00050048, 0x000000b6, 0x00000001, 0x00000023,
    0x00000008, 0x00030047, 0x000000b7, 0x00000002, 0x00050048, 0x000000b7, 0x00000000, 0x00000023, 0x00000000,
    0x00040047, 0x000000d9, 0x0000000b, 0x00000019, 0x00020013, 0x00000002, 0x00030021, 0x00000003, 0x00000002,
    0x00040015, 0x00000006, 0x00000020, 0x00000000, 0x00040015, 0x0000000d, 0x00000040, 0x00000000, 0x00020014,
    0x0000000e, 0x00030027, 0x00000018, 0x000014e5, 0x00030027, 0x00000019, 0x000014e5, 0x0005001e, 0x0000001a,
    0x00000019, 0x00000006, 0x00000006, 0x0004001e, 0x0000001b, 0x0000000d, 0x0000000d, 0x0003001d, 0x0000001c,
    0x0000001b, 0x0003001e, 0x0000001d, 0x0000001c, 0x00040020, 0x00000019, 0x000014e5, 0x0000001d, 0x00040020,
    0x00000018, 0x000014e5, 0x0000001a, 0x0004002b, 0x00000006, 0x00000026, 0x165667b1, 0x0004002b, 0x00000006,
    0x00000029, 0xc2b2ae3d, 0x0004002b, 0x00000006, 0x0000002c, 0x27d4eb2f, 0x00040015, 0x0000002e, 0x00000020,
    0x00000001, 0x0004002b, 0x0000002e, 0x0000002f, 0x00000011, 0x0004002b, 0x0000002e, 0x00000032, 0x0000000f,
    0x0004002b, 0x00000006, 0x00000036, 0x85ebca77, 0x0004002b, 0x0000002e, 0x0000003e, 0x0000000d, 0x0004002b,
    0x0000002e, 0x00000044, 0x00000010, 0x0005002b, 0x0000000d, 0x00000049, 0x00000000, 0x00000000, 0x0005002b,
    0x0000000d, 0x0000004e, 0x00000020, 0x00000000, 0x0005002b, 0x0000000d, 0x00000052, 0xffffffff, 0x00000000,
    0x0004002b, 0x00000006, 0x00000055, 0x00000000, 0x0004002b, 0x0000002e, 0x00000062, 0x00000002, 0x00040020,
    0x00000063, 0x000014e5, 0x00000006, 0x0004002b, 0x00000006, 0x00000074, 0x00000001, 0x0004002b, 0x0000002e,
    0x00000079, 0x00000000, 0x00040020, 0x0000007a, 0x000014e5, 0x00000019, 0x00040020, 0x0000007e, 0x000014e5,
    0x0000000d, 0x0004002b, 0x0000002e, 0x0000008d, 0x00000001, 0x00040017, 0x000000af, 0x00000006, 0x00000003,
    0x00040020, 0x000000b0, 0x00000001, 0x000000af, 0x0004003b, 0x000000b0, 0x000000b1, 0x00000001, 0x00040020,
    0x000000b2, 0x00000001, 0x00000006, 0x0004001e, 0x000000b6, 0x00000018, 0x00000018, 0x0003001e, 0x000000b7,
    0x000000b6, 0x00040020, 0x000000b8, 0x00000009, 0x000000b7, 0x0004003b, 0x000000b8, 0x000000b9, 0x00000009,
    0x00040020, 0x000000ba, 0x00000009, 0x00000018, 0x0004002b, 0x00000006, 0x000000d8, 0x00000020, 0x0006002c,
    0x000000af, 0x000000d9, 0x000000d8, 0x00000074, 0x00000074, 0x00030029, 0x0000000e, 0x000000df, 0x00050036,
    0x00000002, 0x00000004, 0x00000000, 0x00000003, 0x000200f8, 0x00000005, 0x000300f7, 0x000000da, 0x00000000,
    0x000300fb, 0x00000055, 0x000000db, 0x000200f8, 0x000000db, 0x00050041, 0x000000b2, 0x000000b3, 0x000000b1,
    0x00000055, 0x0004003d, 0x00000006, 0x000000b4, 0x000000b3, 0x00060041, 0x000000ba, 0x000000bb, 0x000000b9,
    0x00000079, 0x00000079, 0x0004003d, 0x00000018, 0x000000bc, 0x000000bb, 0x00050041, 0x00000063, 0x000000bd,
    0x000000bc, 0x00000062, 0x0006003d, 0x00000006, 0x000000be, 0x000000bd, 0x00000002, 0x00000004, 0x000500ae,
    0x0000000e, 0x000000bf, 0x000000b4, 0x000000be, 0x000300f7, 0x000000c1, 0x00000000, 0x000400fa, 0x000000bf,
    0x000000c0, 0x000000c1, 0x000200f8, 0x000000c0, 0x000200f9, 0x000000da, 0x000200f8, 0x000000c1, 0x00050041,
    0x0000007a, 0x000000c6, 0x000000bc, 0x00000079, 0x0006003d, 0x00000019, 0x000000c7, 0x000000c6, 0x00000002,
    0x00000010, 0x00070041, 0x0000007e, 0x000000c9, 0x000000c7, 0x00000079, 0x000000b4, 0x00000079, 0x0006003d,
    0x0000000d, 0x000000ca, 0x000000c9, 0x00000002, 0x00000010, 0x0006003d, 0x00000019, 0x000000cf, 0x000000c6,
    0x00000002, 0x00000010, 0x00070041, 0x0000007e, 0x000000d1, 0x000000cf, 0x00000079, 0x000000b4, 0x0000008d,
    0x0006003d, 0x0000000d, 0x000000d2, 0x000000d1, 0x00000002, 0x00000008, 0x00060041, 0x000000ba, 0x000000d3,
    0x000000b9, 0x00000079, 0x0000008d, 0x0004003d, 0x00000018, 0x000000d4, 0x000000d3, 0x000300f7, 0x0000012f,
    0x00000000, 0x000300fb, 0x00000055, 0x000000ea, 0x000200f8, 0x000000ea, 0x000500aa, 0x0000000e, 0x00000132,
    0x000000ca, 0x00000049, 0x000400a8, 0x0000000e, 0x000000ec, 0x00000132, 0x000300f7, 0x000000f1, 0x00000000,
    0x000400fa, 0x000000ec, 0x000000ed, 0x000000f1, 0x000200f8, 0x000000ed, 0x00050041, 0x00000063, 0x000000ee,
    0x000000d4, 0x00000062, 0x0006003d, 0x00000006, 0x000000ef, 0x000000ee, 0x00000002, 0x00000004, 0x000500aa,
    0x0000000e, 0x000000f0, 0x000000ef, 0x00000055, 0x000200f9, 0x000000f1, 0x000200f8, 0x000000f1, 0x000700f5,
    0x0000000e, 0x000000f2, 0x00000132, 0x000000ea, 0x000000f0, 0x000000ed, 0x000300f7, 0x000000f4, 0x00000000,
    0x000400fa, 0x000000f2, 0x000000f3, 0x000000f4, 0x000200f8, 0x000000f3, 0x000200f9, 0x0000012f, 0x000200f8,
    0x000000f4, 0x000500c2, 0x0000000d, 0x0000013a, 0x000000ca, 0x0000004e, 0x00040071, 0x00000006, 0x0000013b,
    0x0000013a, 0x000500c7, 0x0000000d, 0x0000013d, 0x000000ca, 0x00000052, 0x00040071, 0x00000006, 0x0000013e,
    0x0000013d, 0x00050080, 0x00000006, 0x00000145, 0x0000013e, 0x00000026, 0x000500c4, 0x00000006, 0x0000014a,
    0x00000145, 0x0000002f, 0x000500c2, 0x00000006, 0x0000014c, 0x00000145, 0x00000032, 0x000500c5, 0x00000006,
    0x0000014d, 0x0000014a, 0x0000014c, 0x00050084, 0x00000006, 0x0000014e, 0x0000002c, 0x0000014d, 0x000500c2,
    0x00000006, 0x00000151, 0x0000014e, 0x00000032, 0x000500c6, 0x00000006, 0x00000152, 0x0000014e, 0x00000151,
    0x00050084, 0x00000006, 0x00000153, 0x00000036, 0x00000152, 0x000500c2, 0x00000006, 0x00000156, 0x00000153,
    0x0000003e, 0x000500c6, 0x00000006, 0x00000157, 0x00000153, 0x00000156, 0x00050084, 0x00000006, 0x00000158,
    0x00000029, 0x00000157, 0x000500c2, 0x00000006, 0x0000015b, 0x00000158, 0x00000044, 0x000500c6, 0x00000006,
    0x0000015c, 0x00000158, 0x0000015b, 0x00050080, 0x00000006, 0x00000161, 0x0000013b, 0x00000026, 0x00050084,
    0x00000006, 0x00000163, 0x0000015c, 0x00000029, 0x00050080, 0x00000006, 0x00000164, 0x00000161, 0x00000163,
    0x000500c4, 0x00000006, 0x00000166, 0x00000164, 0x0000002f, 0x000500c2, 0x00000006, 0x00000168, 0x00000164,
    0x00000032, 0x000500c5, 0x00000006, 0x00000169, 0x00000166, 0x00000168, 0x00050084, 0x00000006, 0x0000016a,
    0x0000002c, 0x00000169, 0x000500c2, 0x00000006, 0x0000016d, 0x0000016a, 0x00000032, 0x000500c6, 0x00000006,
    0x0000016e, 0x0000016a, 0x0000016d, 0x00050084, 0x00000006, 0x0000016f, 0x00000036, 0x0000016e, 0x000500c2,
    0x00000006, 0x00000172, 0x0000016f, 0x0000003e, 0x000500c6, 0x00000006, 0x00000173, 0x0000016f, 0x00000172,
    0x00050084, 0x00000006, 0x00000174, 0x00000029, 0x00000173, 0x000500c2, 0x00000006, 0x00000177, 0x00000174,
    0x00000044, 0x000500c6, 0x00000006, 0x00000178, 0x00000174, 0x00000177, 0x000200f9, 0x000000f6, 0x000200f8,
    0x000000f6, 0x000700f5, 0x00000006, 0x00000182, 0x00000178, 0x000000f4, 0x0000012b, 0x00000129, 0x000400f6,
    0x0000012c, 0x00000129, 0x00000000, 0x000200f9, 0x000000f7, 0x000200f8, 0x000000f7, 0x00050041, 0x00000063,
    0x000000f8, 0x000000d4, 0x00000062, 0x0006003d, 0x00000006, 0x000000f9, 0x000000f8, 0x00000002, 0x00000004,
    0x00050082, 0x00000006, 0x000000fa, 0x000000f9, 0x00000074, 0x000500c7, 0x00000006, 0x000000fc, 0x00000182,
    0x000000fa, 0x00050041, 0x0000007a, 0x000000fd, 0x000000d4, 0x00000079, 0x0006003d, 0x00000019, 0x000000fe,
    0x000000fd, 0x00000002, 0x00000010, 0x00070041, 0x0000007e, 0x00000100, 0x000000fe, 0x00000079, 0x000000fc,
    0x00000079, 0x0006003d, 0x0000000d, 0x00000101, 0x00000100, 0x00000002, 0x00000010, 0x000500ab, 0x0000000e,
    0x00000103, 0x00000101, 0x000000ca, 0x000300f7, 0x00000124, 0x00000000, 0x000400fa, 0x00000103, 0x00000104,
    0x00000124, 0x000200f8, 0x00000104, 0x000500aa, 0x0000000e, 0x0000017b, 0x00000101, 0x00000049, 0x000400a8,
    0x0000000e, 0x00000107, 0x0000017b, 0x000300f7, 0x00000110, 0x00000000, 0x000400fa, 0x00000107, 0x00000108,
    0x00000110, 0x000200f8, 0x00000108, 0x0006003d, 0x00000019, 0x0000010a, 0x000000fd, 0x00000002, 0x00000010,
    0x00070041, 0x0000007e, 0x0000010c, 0x0000010a, 0x00000079, 0x000000fc, 0x0000008d, 0x0006003d, 0x0000000d,
    0x0000010d, 0x0000010c, 0x00000002, 0x00000008, 0x000500aa, 0x0000000e, 0x0000017e, 0x0000010d, 0x00000049,
    0x000400a8, 0x0000000e, 0x0000010f, 0x0000017e, 0x000200f9, 0x00000110, 0x000200f8, 0x00000110, 0x000700f5,
    0x0000000e, 0x00000111, 0x00000107, 0x00000104, 0x0000010f, 0x00000108, 0x000300f7, 0x00000113, 0x00000000,
    0x000400fa, 0x00000111, 0x00000112, 0x00000113, 0x000200f8, 0x00000112, 0x000200f9, 0x00000129, 0x000200f8,
    0x00000113, 0x0006003d, 0x00000019, 0x00000115, 0x000000fd, 0x00000002, 0x00000010, 0x00070041, 0x0000007e,
    0x00000117, 0x00000115, 0x00000079, 0x000000fc, 0x00000079, 0x000900e6, 0x0000000d, 0x00000119, 0x00000117,
    0x00000074, 0x00000055, 0x00000055, 0x000000ca, 0x00000101, 0x000500aa, 0x0000000e, 0x00000181, 0x00000119,
    0x00000049, 0x000400a8, 0x0000000e, 0x0000011c, 0x00000181, 0x000500ab, 0x0000000e, 0x0000011e, 0x00000119,
    0x000000ca, 0x000500a7, 0x0000000e, 0x0000011f, 0x0000011c, 0x0000011e, 0x000300f7, 0x00000121, 0x00000000,
    0x000400fa, 0x0000011f, 0x00000120, 0x00000121, 0x000200f8, 0x00000120, 0x000200f9, 0x00000129, 0x000200f8,
    0x00000121, 0x00050041, 0x00000063, 0x00000122, 0x000000d4, 0x0000008d, 0x000700ea, 0x00000006, 0x00000123,
    0x00000122, 0x00000074, 0x00000055, 0x00000074, 0x000200f9, 0x00000124, 0x000200f8, 0x00000124, 0x0006003d,
    0x00000019, 0x00000126, 0x000000fd, 0x00000002, 0x00000010, 0x00070041, 0x0000007e, 0x00000128, 0x00000126,
    0x00000079, 0x000000fc, 0x0000008d, 0x0005003e, 0x00000128, 0x000000d2, 0x00000002, 0x00000008, 0x000200f9,
    0x0000012c, 0x000200f8, 0x00000129, 0x00050080, 0x00000006, 0x0000012b, 0x000000fc, 0x0000008d, 0x000200f9,
    0x000000f6, 0x000200f8, 0x0000012c, 0x000300f7, 0x0000012e, 0x00000000, 0x000400fa, 0x000000df, 0x0000012f,
    0x0000012e, 0x000200f8, 0x0000012e, 0x000200f9, 0x0000012f, 0x000200f8, 0x0000012f, 0x000200f9, 0x000000da,
    0x000200f8, 0x000000da, 0x000100fd, 0x00010038
};

GFXRECON_END_NAMESPACE(decode)
GFXRECON_END_NAMESPACE(gfxrecon)

#endif // GFXRECON_DECODE_VULKAN_ADDRESS_REPLACER_SHADERS_H
