package parse

import (
	"testing"

	"github.com/go-graphite/go-carbon/points"
)

// benchmarks
// same for all protocols

/* bench1
data = [
	[('param1', (1423931224, 60.2))],
	[('param1', (1423931224, 60.2), (1423931225, 50.2), (1423931226, 40.2))],
	[('param1', (1423931224, 60.2)), ('param2', (1423931224, -15))],
	[('param1', (1423931224, 60.2), (1423931284, 42)), ('param2', (1423931224, -15))],
]
*/

var pickle1 = [][]byte{
	[]byte("(lp0\n(S'param1'\np1\n(I1423931224\nF60.2\ntp2\ntp3\na."),
	[]byte("(lp0\n(S'param1'\np1\n(I1423931224\nF60.2\ntp2\n(I1423931225\nF50.2\ntp3\n(I1423931226\nF40.2\ntp4\ntp5\na."),
	[]byte("(lp0\n(S'param1'\np1\n(I1423931224\nF60.2\ntp2\ntp3\na(S'param2'\np4\n(I1423931224\nI-15\ntp5\ntp6\na."),
	[]byte("(lp0\n(S'param1'\np1\n(I1423931224\nF60.2\ntp2\n(I1423931284\nI42\ntp3\ntp4\na(S'param2'\np5\n(I1423931224\nI-15\ntp6\ntp7\na."),
}

var protobuf1 = [][]byte{
	[]byte("\n\x19\n\x06param1\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x19N@"),
	[]byte("\n;\n\x06param1\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x19N@\x12\x0f\x08\xd9\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x19I@\x12\x0f\x08\xda\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x19D@"),
	[]byte("\n\x19\n\x06param1\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x19N@\n\x19\n\x06param2\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x00\x00\x00\x00\x00\x00.\xc0"),
	[]byte("\n*\n\x06param1\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x19N@\x12\x0f\x08\x94\xef\xfd\xa6\x05\x11\x00\x00\x00\x00\x00\x00E@\n\x19\n\x06param2\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x00\x00\x00\x00\x00\x00.\xc0"),
}

var plain1 = [][]byte{
	[]byte("param1 60.2 1423931224\n"),
	[]byte("param1 60.2 1423931224\nparam1 50.2 1423931225\nparam1 40.2 1423931226\n"),
	[]byte("param1 60.2 1423931224\nparam2 -15 1423931224\n"),
	[]byte("param1 60.2 1423931224\nparam1 42 1423931284\nparam2 -15 1423931224\n"),
}

func runBenchmark(b *testing.B, parser func(body []byte) ([]*points.Points, error), data [][]byte) {
	// run function b.N times
	for n := 0; n < b.N; n++ {
		_, err := parser(data[n%len(data)])
		if err != nil {
			b.Fatalf("Error raised while benchmarking")
		}
	}
}

func BenchmarkPickle1(b *testing.B) {
	runBenchmark(b, Pickle, pickle1)
}

func BenchmarkProtobuf1(b *testing.B) {
	runBenchmark(b, Protobuf, protobuf1)
}

func BenchmarkPlain1(b *testing.B) {
	runBenchmark(b, Plain, plain1)
}

func BenchmarkOldPlain1(b *testing.B) {
	runBenchmark(b, oldPlain, plain1)
}

var pickle100 = "(lp0\n(S'carbon.agents.graph1.tcp.metricReceived0'\np1\n(I1423931224\nF42.2\ntp2\ntp3\na(S'carbon.agents.graph1.tcp.metricReceived1'\np4\n(I1423931224\nF43.2\ntp5\ntp6\na(S'carbon.agents.graph1.tcp.metricReceived2'\np7\n(I1423931224\nF44.2\ntp8\ntp9\na(S'carbon.agents.graph1.tcp.metricReceived3'\np10\n(I1423931224\nF45.2\ntp11\ntp12\na(S'carbon.agents.graph1.tcp.metricReceived4'\np13\n(I1423931224\nF46.2\ntp14\ntp15\na(S'carbon.agents.graph1.tcp.metricReceived5'\np16\n(I1423931224\nF47.2\ntp17\ntp18\na(S'carbon.agents.graph1.tcp.metricReceived6'\np19\n(I1423931224\nF48.2\ntp20\ntp21\na(S'carbon.agents.graph1.tcp.metricReceived7'\np22\n(I1423931224\nF49.2\ntp23\ntp24\na(S'carbon.agents.graph1.tcp.metricReceived8'\np25\n(I1423931224\nF50.2\ntp26\ntp27\na(S'carbon.agents.graph1.tcp.metricReceived9'\np28\n(I1423931224\nF51.2\ntp29\ntp30\na(S'carbon.agents.graph1.tcp.metricReceived10'\np31\n(I1423931224\nF52.2\ntp32\ntp33\na(S'carbon.agents.graph1.tcp.metricReceived11'\np34\n(I1423931224\nF53.2\ntp35\ntp36\na(S'carbon.agents.graph1.tcp.metricReceived12'\np37\n(I1423931224\nF54.2\ntp38\ntp39\na(S'carbon.agents.graph1.tcp.metricReceived13'\np40\n(I1423931224\nF55.2\ntp41\ntp42\na(S'carbon.agents.graph1.tcp.metricReceived14'\np43\n(I1423931224\nF56.2\ntp44\ntp45\na(S'carbon.agents.graph1.tcp.metricReceived15'\np46\n(I1423931224\nF57.2\ntp47\ntp48\na(S'carbon.agents.graph1.tcp.metricReceived16'\np49\n(I1423931224\nF58.2\ntp50\ntp51\na(S'carbon.agents.graph1.tcp.metricReceived17'\np52\n(I1423931224\nF59.2\ntp53\ntp54\na(S'carbon.agents.graph1.tcp.metricReceived18'\np55\n(I1423931224\nF60.2\ntp56\ntp57\na(S'carbon.agents.graph1.tcp.metricReceived19'\np58\n(I1423931224\nF61.2\ntp59\ntp60\na(S'carbon.agents.graph1.tcp.metricReceived20'\np61\n(I1423931224\nF62.2\ntp62\ntp63\na(S'carbon.agents.graph1.tcp.metricReceived21'\np64\n(I1423931224\nF63.2\ntp65\ntp66\na(S'carbon.agents.graph1.tcp.metricReceived22'\np67\n(I1423931224\nF64.2\ntp68\ntp69\na(S'carbon.agents.graph1.tcp.metricReceived23'\np70\n(I1423931224\nF65.2\ntp71\ntp72\na(S'carbon.agents.graph1.tcp.metricReceived24'\np73\n(I1423931224\nF66.2\ntp74\ntp75\na(S'carbon.agents.graph1.tcp.metricReceived25'\np76\n(I1423931224\nF67.2\ntp77\ntp78\na(S'carbon.agents.graph1.tcp.metricReceived26'\np79\n(I1423931224\nF68.2\ntp80\ntp81\na(S'carbon.agents.graph1.tcp.metricReceived27'\np82\n(I1423931224\nF69.2\ntp83\ntp84\na(S'carbon.agents.graph1.tcp.metricReceived28'\np85\n(I1423931224\nF70.2\ntp86\ntp87\na(S'carbon.agents.graph1.tcp.metricReceived29'\np88\n(I1423931224\nF71.2\ntp89\ntp90\na(S'carbon.agents.graph1.tcp.metricReceived30'\np91\n(I1423931224\nF72.2\ntp92\ntp93\na(S'carbon.agents.graph1.tcp.metricReceived31'\np94\n(I1423931224\nF73.2\ntp95\ntp96\na(S'carbon.agents.graph1.tcp.metricReceived32'\np97\n(I1423931224\nF74.2\ntp98\ntp99\na(S'carbon.agents.graph1.tcp.metricReceived33'\np100\n(I1423931224\nF75.2\ntp101\ntp102\na(S'carbon.agents.graph1.tcp.metricReceived34'\np103\n(I1423931224\nF76.2\ntp104\ntp105\na(S'carbon.agents.graph1.tcp.metricReceived35'\np106\n(I1423931224\nF77.2\ntp107\ntp108\na(S'carbon.agents.graph1.tcp.metricReceived36'\np109\n(I1423931224\nF78.2\ntp110\ntp111\na(S'carbon.agents.graph1.tcp.metricReceived37'\np112\n(I1423931224\nF79.2\ntp113\ntp114\na(S'carbon.agents.graph1.tcp.metricReceived38'\np115\n(I1423931224\nF80.2\ntp116\ntp117\na(S'carbon.agents.graph1.tcp.metricReceived39'\np118\n(I1423931224\nF81.2\ntp119\ntp120\na(S'carbon.agents.graph1.tcp.metricReceived40'\np121\n(I1423931224\nF82.2\ntp122\ntp123\na(S'carbon.agents.graph1.tcp.metricReceived41'\np124\n(I1423931224\nF83.2\ntp125\ntp126\na(S'carbon.agents.graph1.tcp.metricReceived42'\np127\n(I1423931224\nF84.2\ntp128\ntp129\na(S'carbon.agents.graph1.tcp.metricReceived43'\np130\n(I1423931224\nF85.2\ntp131\ntp132\na(S'carbon.agents.graph1.tcp.metricReceived44'\np133\n(I1423931224\nF86.2\ntp134\ntp135\na(S'carbon.agents.graph1.tcp.metricReceived45'\np136\n(I1423931224\nF87.2\ntp137\ntp138\na(S'carbon.agents.graph1.tcp.metricReceived46'\np139\n(I1423931224\nF88.2\ntp140\ntp141\na(S'carbon.agents.graph1.tcp.metricReceived47'\np142\n(I1423931224\nF89.2\ntp143\ntp144\na(S'carbon.agents.graph1.tcp.metricReceived48'\np145\n(I1423931224\nF90.2\ntp146\ntp147\na(S'carbon.agents.graph1.tcp.metricReceived49'\np148\n(I1423931224\nF91.2\ntp149\ntp150\na(S'carbon.agents.graph1.tcp.metricReceived50'\np151\n(I1423931224\nF92.2\ntp152\ntp153\na(S'carbon.agents.graph1.tcp.metricReceived51'\np154\n(I1423931224\nF93.2\ntp155\ntp156\na(S'carbon.agents.graph1.tcp.metricReceived52'\np157\n(I1423931224\nF94.2\ntp158\ntp159\na(S'carbon.agents.graph1.tcp.metricReceived53'\np160\n(I1423931224\nF95.2\ntp161\ntp162\na(S'carbon.agents.graph1.tcp.metricReceived54'\np163\n(I1423931224\nF96.2\ntp164\ntp165\na(S'carbon.agents.graph1.tcp.metricReceived55'\np166\n(I1423931224\nF97.2\ntp167\ntp168\na(S'carbon.agents.graph1.tcp.metricReceived56'\np169\n(I1423931224\nF98.2\ntp170\ntp171\na(S'carbon.agents.graph1.tcp.metricReceived57'\np172\n(I1423931224\nF99.2\ntp173\ntp174\na(S'carbon.agents.graph1.tcp.metricReceived58'\np175\n(I1423931224\nF100.2\ntp176\ntp177\na(S'carbon.agents.graph1.tcp.metricReceived59'\np178\n(I1423931224\nF101.2\ntp179\ntp180\na(S'carbon.agents.graph1.tcp.metricReceived60'\np181\n(I1423931224\nF102.2\ntp182\ntp183\na(S'carbon.agents.graph1.tcp.metricReceived61'\np184\n(I1423931224\nF103.2\ntp185\ntp186\na(S'carbon.agents.graph1.tcp.metricReceived62'\np187\n(I1423931224\nF104.2\ntp188\ntp189\na(S'carbon.agents.graph1.tcp.metricReceived63'\np190\n(I1423931224\nF105.2\ntp191\ntp192\na(S'carbon.agents.graph1.tcp.metricReceived64'\np193\n(I1423931224\nF106.2\ntp194\ntp195\na(S'carbon.agents.graph1.tcp.metricReceived65'\np196\n(I1423931224\nF107.2\ntp197\ntp198\na(S'carbon.agents.graph1.tcp.metricReceived66'\np199\n(I1423931224\nF108.2\ntp200\ntp201\na(S'carbon.agents.graph1.tcp.metricReceived67'\np202\n(I1423931224\nF109.2\ntp203\ntp204\na(S'carbon.agents.graph1.tcp.metricReceived68'\np205\n(I1423931224\nF110.2\ntp206\ntp207\na(S'carbon.agents.graph1.tcp.metricReceived69'\np208\n(I1423931224\nF111.2\ntp209\ntp210\na(S'carbon.agents.graph1.tcp.metricReceived70'\np211\n(I1423931224\nF112.2\ntp212\ntp213\na(S'carbon.agents.graph1.tcp.metricReceived71'\np214\n(I1423931224\nF113.2\ntp215\ntp216\na(S'carbon.agents.graph1.tcp.metricReceived72'\np217\n(I1423931224\nF114.2\ntp218\ntp219\na(S'carbon.agents.graph1.tcp.metricReceived73'\np220\n(I1423931224\nF115.2\ntp221\ntp222\na(S'carbon.agents.graph1.tcp.metricReceived74'\np223\n(I1423931224\nF116.2\ntp224\ntp225\na(S'carbon.agents.graph1.tcp.metricReceived75'\np226\n(I1423931224\nF117.2\ntp227\ntp228\na(S'carbon.agents.graph1.tcp.metricReceived76'\np229\n(I1423931224\nF118.2\ntp230\ntp231\na(S'carbon.agents.graph1.tcp.metricReceived77'\np232\n(I1423931224\nF119.2\ntp233\ntp234\na(S'carbon.agents.graph1.tcp.metricReceived78'\np235\n(I1423931224\nF120.2\ntp236\ntp237\na(S'carbon.agents.graph1.tcp.metricReceived79'\np238\n(I1423931224\nF121.2\ntp239\ntp240\na(S'carbon.agents.graph1.tcp.metricReceived80'\np241\n(I1423931224\nF122.2\ntp242\ntp243\na(S'carbon.agents.graph1.tcp.metricReceived81'\np244\n(I1423931224\nF123.2\ntp245\ntp246\na(S'carbon.agents.graph1.tcp.metricReceived82'\np247\n(I1423931224\nF124.2\ntp248\ntp249\na(S'carbon.agents.graph1.tcp.metricReceived83'\np250\n(I1423931224\nF125.2\ntp251\ntp252\na(S'carbon.agents.graph1.tcp.metricReceived84'\np253\n(I1423931224\nF126.2\ntp254\ntp255\na(S'carbon.agents.graph1.tcp.metricReceived85'\np256\n(I1423931224\nF127.2\ntp257\ntp258\na(S'carbon.agents.graph1.tcp.metricReceived86'\np259\n(I1423931224\nF128.2\ntp260\ntp261\na(S'carbon.agents.graph1.tcp.metricReceived87'\np262\n(I1423931224\nF129.2\ntp263\ntp264\na(S'carbon.agents.graph1.tcp.metricReceived88'\np265\n(I1423931224\nF130.2\ntp266\ntp267\na(S'carbon.agents.graph1.tcp.metricReceived89'\np268\n(I1423931224\nF131.2\ntp269\ntp270\na(S'carbon.agents.graph1.tcp.metricReceived90'\np271\n(I1423931224\nF132.2\ntp272\ntp273\na(S'carbon.agents.graph1.tcp.metricReceived91'\np274\n(I1423931224\nF133.2\ntp275\ntp276\na(S'carbon.agents.graph1.tcp.metricReceived92'\np277\n(I1423931224\nF134.2\ntp278\ntp279\na(S'carbon.agents.graph1.tcp.metricReceived93'\np280\n(I1423931224\nF135.2\ntp281\ntp282\na(S'carbon.agents.graph1.tcp.metricReceived94'\np283\n(I1423931224\nF136.2\ntp284\ntp285\na(S'carbon.agents.graph1.tcp.metricReceived95'\np286\n(I1423931224\nF137.2\ntp287\ntp288\na(S'carbon.agents.graph1.tcp.metricReceived96'\np289\n(I1423931224\nF138.2\ntp290\ntp291\na(S'carbon.agents.graph1.tcp.metricReceived97'\np292\n(I1423931224\nF139.2\ntp293\ntp294\na(S'carbon.agents.graph1.tcp.metricReceived98'\np295\n(I1423931224\nF140.2\ntp296\ntp297\na(S'carbon.agents.graph1.tcp.metricReceived99'\np298\n(I1423931224\nF141.2\ntp299\ntp300\na."
var protobuf100 = "\n;\n(carbon.agents.graph1.tcp.metricReceived0\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x19E@\n;\n(carbon.agents.graph1.tcp.metricReceived1\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x99E@\n;\n(carbon.agents.graph1.tcp.metricReceived2\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x19F@\n;\n(carbon.agents.graph1.tcp.metricReceived3\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x99F@\n;\n(carbon.agents.graph1.tcp.metricReceived4\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x19G@\n;\n(carbon.agents.graph1.tcp.metricReceived5\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x99G@\n;\n(carbon.agents.graph1.tcp.metricReceived6\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x19H@\n;\n(carbon.agents.graph1.tcp.metricReceived7\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x99H@\n;\n(carbon.agents.graph1.tcp.metricReceived8\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x19I@\n;\n(carbon.agents.graph1.tcp.metricReceived9\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x99I@\n<\n)carbon.agents.graph1.tcp.metricReceived10\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x19J@\n<\n)carbon.agents.graph1.tcp.metricReceived11\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x99J@\n<\n)carbon.agents.graph1.tcp.metricReceived12\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x19K@\n<\n)carbon.agents.graph1.tcp.metricReceived13\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x99K@\n<\n)carbon.agents.graph1.tcp.metricReceived14\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x19L@\n<\n)carbon.agents.graph1.tcp.metricReceived15\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x99L@\n<\n)carbon.agents.graph1.tcp.metricReceived16\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x19M@\n<\n)carbon.agents.graph1.tcp.metricReceived17\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x99M@\n<\n)carbon.agents.graph1.tcp.metricReceived18\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x19N@\n<\n)carbon.agents.graph1.tcp.metricReceived19\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x99N@\n<\n)carbon.agents.graph1.tcp.metricReceived20\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x19O@\n<\n)carbon.agents.graph1.tcp.metricReceived21\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\x9a\x99\x99\x99\x99\x99O@\n<\n)carbon.agents.graph1.tcp.metricReceived22\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x0cP@\n<\n)carbon.agents.graph1.tcp.metricReceived23\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xccLP@\n<\n)carbon.agents.graph1.tcp.metricReceived24\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x8cP@\n<\n)carbon.agents.graph1.tcp.metricReceived25\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\xccP@\n<\n)carbon.agents.graph1.tcp.metricReceived26\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x0cQ@\n<\n)carbon.agents.graph1.tcp.metricReceived27\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xccLQ@\n<\n)carbon.agents.graph1.tcp.metricReceived28\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x8cQ@\n<\n)carbon.agents.graph1.tcp.metricReceived29\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\xccQ@\n<\n)carbon.agents.graph1.tcp.metricReceived30\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x0cR@\n<\n)carbon.agents.graph1.tcp.metricReceived31\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xccLR@\n<\n)carbon.agents.graph1.tcp.metricReceived32\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x8cR@\n<\n)carbon.agents.graph1.tcp.metricReceived33\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\xccR@\n<\n)carbon.agents.graph1.tcp.metricReceived34\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x0cS@\n<\n)carbon.agents.graph1.tcp.metricReceived35\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xccLS@\n<\n)carbon.agents.graph1.tcp.metricReceived36\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x8cS@\n<\n)carbon.agents.graph1.tcp.metricReceived37\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\xccS@\n<\n)carbon.agents.graph1.tcp.metricReceived38\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x0cT@\n<\n)carbon.agents.graph1.tcp.metricReceived39\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xccLT@\n<\n)carbon.agents.graph1.tcp.metricReceived40\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x8cT@\n<\n)carbon.agents.graph1.tcp.metricReceived41\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\xccT@\n<\n)carbon.agents.graph1.tcp.metricReceived42\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x0cU@\n<\n)carbon.agents.graph1.tcp.metricReceived43\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xccLU@\n<\n)carbon.agents.graph1.tcp.metricReceived44\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x8cU@\n<\n)carbon.agents.graph1.tcp.metricReceived45\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\xccU@\n<\n)carbon.agents.graph1.tcp.metricReceived46\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x0cV@\n<\n)carbon.agents.graph1.tcp.metricReceived47\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xccLV@\n<\n)carbon.agents.graph1.tcp.metricReceived48\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x8cV@\n<\n)carbon.agents.graph1.tcp.metricReceived49\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\xccV@\n<\n)carbon.agents.graph1.tcp.metricReceived50\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x0cW@\n<\n)carbon.agents.graph1.tcp.metricReceived51\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xccLW@\n<\n)carbon.agents.graph1.tcp.metricReceived52\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x8cW@\n<\n)carbon.agents.graph1.tcp.metricReceived53\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\xccW@\n<\n)carbon.agents.graph1.tcp.metricReceived54\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x0cX@\n<\n)carbon.agents.graph1.tcp.metricReceived55\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xccLX@\n<\n)carbon.agents.graph1.tcp.metricReceived56\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x8cX@\n<\n)carbon.agents.graph1.tcp.metricReceived57\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\xccX@\n<\n)carbon.agents.graph1.tcp.metricReceived58\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x0cY@\n<\n)carbon.agents.graph1.tcp.metricReceived59\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xccLY@\n<\n)carbon.agents.graph1.tcp.metricReceived60\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x8cY@\n<\n)carbon.agents.graph1.tcp.metricReceived61\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\xccY@\n<\n)carbon.agents.graph1.tcp.metricReceived62\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x0cZ@\n<\n)carbon.agents.graph1.tcp.metricReceived63\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xccLZ@\n<\n)carbon.agents.graph1.tcp.metricReceived64\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x8cZ@\n<\n)carbon.agents.graph1.tcp.metricReceived65\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\xccZ@\n<\n)carbon.agents.graph1.tcp.metricReceived66\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x0c[@\n<\n)carbon.agents.graph1.tcp.metricReceived67\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xccL[@\n<\n)carbon.agents.graph1.tcp.metricReceived68\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x8c[@\n<\n)carbon.agents.graph1.tcp.metricReceived69\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\xcc[@\n<\n)carbon.agents.graph1.tcp.metricReceived70\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x0c\\@\n<\n)carbon.agents.graph1.tcp.metricReceived71\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xccL\\@\n<\n)carbon.agents.graph1.tcp.metricReceived72\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x8c\\@\n<\n)carbon.agents.graph1.tcp.metricReceived73\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\xcc\\@\n<\n)carbon.agents.graph1.tcp.metricReceived74\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x0c]@\n<\n)carbon.agents.graph1.tcp.metricReceived75\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xccL]@\n<\n)carbon.agents.graph1.tcp.metricReceived76\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x8c]@\n<\n)carbon.agents.graph1.tcp.metricReceived77\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\xcc]@\n<\n)carbon.agents.graph1.tcp.metricReceived78\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x0c^@\n<\n)carbon.agents.graph1.tcp.metricReceived79\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xccL^@\n<\n)carbon.agents.graph1.tcp.metricReceived80\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x8c^@\n<\n)carbon.agents.graph1.tcp.metricReceived81\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\xcc^@\n<\n)carbon.agents.graph1.tcp.metricReceived82\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x0c_@\n<\n)carbon.agents.graph1.tcp.metricReceived83\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xccL_@\n<\n)carbon.agents.graph1.tcp.metricReceived84\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\x8c_@\n<\n)carbon.agents.graph1.tcp.metricReceived85\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11\xcd\xcc\xcc\xcc\xcc\xcc_@\n<\n)carbon.agents.graph1.tcp.metricReceived86\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11fffff\x06`@\n<\n)carbon.agents.graph1.tcp.metricReceived87\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11fffff&`@\n<\n)carbon.agents.graph1.tcp.metricReceived88\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11fffffF`@\n<\n)carbon.agents.graph1.tcp.metricReceived89\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11ffffff`@\n<\n)carbon.agents.graph1.tcp.metricReceived90\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11fffff\x86`@\n<\n)carbon.agents.graph1.tcp.metricReceived91\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11fffff\xa6`@\n<\n)carbon.agents.graph1.tcp.metricReceived92\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11fffff\xc6`@\n<\n)carbon.agents.graph1.tcp.metricReceived93\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11fffff\xe6`@\n<\n)carbon.agents.graph1.tcp.metricReceived94\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11fffff\x06a@\n<\n)carbon.agents.graph1.tcp.metricReceived95\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11fffff&a@\n<\n)carbon.agents.graph1.tcp.metricReceived96\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11fffffFa@\n<\n)carbon.agents.graph1.tcp.metricReceived97\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11ffffffa@\n<\n)carbon.agents.graph1.tcp.metricReceived98\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11fffff\x86a@\n<\n)carbon.agents.graph1.tcp.metricReceived99\x12\x0f\x08\xd8\xee\xfd\xa6\x05\x11fffff\xa6a@"
var plain100 = "carbon.agents.graph1.tcp.metricReceived0 42.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived1 43.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived2 44.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived3 45.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived4 46.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived5 47.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived6 48.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived7 49.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived8 50.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived9 51.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived10 52.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived11 53.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived12 54.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived13 55.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived14 56.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived15 57.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived16 58.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived17 59.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived18 60.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived19 61.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived20 62.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived21 63.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived22 64.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived23 65.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived24 66.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived25 67.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived26 68.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived27 69.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived28 70.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived29 71.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived30 72.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived31 73.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived32 74.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived33 75.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived34 76.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived35 77.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived36 78.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived37 79.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived38 80.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived39 81.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived40 82.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived41 83.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived42 84.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived43 85.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived44 86.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived45 87.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived46 88.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived47 89.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived48 90.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived49 91.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived50 92.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived51 93.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived52 94.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived53 95.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived54 96.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived55 97.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived56 98.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived57 99.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived58 100.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived59 101.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived60 102.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived61 103.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived62 104.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived63 105.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived64 106.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived65 107.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived66 108.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived67 109.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived68 110.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived69 111.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived70 112.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived71 113.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived72 114.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived73 115.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived74 116.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived75 117.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived76 118.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived77 119.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived78 120.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived79 121.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived80 122.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived81 123.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived82 124.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived83 125.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived84 126.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived85 127.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived86 128.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived87 129.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived88 130.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived89 131.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived90 132.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived91 133.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived92 134.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived93 135.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived94 136.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived95 137.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived96 138.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived97 139.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived98 140.2 1423931224\ncarbon.agents.graph1.tcp.metricReceived99 141.2 1423931224\n"

func BenchmarkPickle100(b *testing.B) {
	runBenchmark(b, Pickle, [][]byte{[]byte(pickle100)})
}

func BenchmarkProtobuf100(b *testing.B) {
	runBenchmark(b, Protobuf, [][]byte{[]byte(protobuf100)})
}

func BenchmarkPlain100(b *testing.B) {
	runBenchmark(b, Plain, [][]byte{[]byte(plain100)})
}

func BenchmarkOldPlain100(b *testing.B) {
	runBenchmark(b, oldPlain, [][]byte{[]byte(plain100)})
}
