/* Copyright 2017 The TensorFlow Authors. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/
#include "tensorflow/lite/kernels/transpose_test_utils.h"

#include <vector>

#include <gmock/gmock.h>
#include <gtest/gtest.h>
#include "absl/algorithm/container.h"

using testing::ElementsAreArray;

namespace tflite {
namespace {

TEST(TransposeTest, TestRefOps1D) {
  // Basic 1D identity.
  EXPECT_THAT(RunTestPermutation<float>({3}, {0}), ElementsAreArray({0, 1, 2}));
}

TEST(TransposeTest, TestRefOps2D) {
  // Basic 2D.
  EXPECT_THAT(RunTestPermutation<float>({3, 2}, {1, 0}),
              ElementsAreArray({0, 2, 4, 1, 3, 5}));
  // Identity.
  EXPECT_THAT(RunTestPermutation<float>({3, 2}, {0, 1}),
              ElementsAreArray({0, 1, 2, 3, 4, 5}));
}

TEST(TransposeTest, TestRefOps3D) {
  {
    EXPECT_THAT(
        RunTestPermutation<float>(/*shape=*/{2, 3, 4}, /*perms=*/{2, 0, 1}),
        ElementsAreArray({0, 4, 8,  12, 16, 20, 1, 5, 9,  13, 17, 21,
                          2, 6, 10, 14, 18, 22, 3, 7, 11, 15, 19, 23}));
  }

  // Test 3 dimensional identity transform
  {
    std::vector<float> out =
        RunTestPermutation<float>(/*shape=*/{2, 3, 4}, /*perms=*/{0, 1, 2});
    std::vector<float> ref(out.size());
    absl::c_iota(ref, 0);
    EXPECT_THAT(out, ElementsAreArray(ref));
  }

  /**
   * Additional tests that mimic first case, but with different perm.
   */
  {
    EXPECT_THAT(
        RunTestPermutation<float>(/*shape=*/{2, 3, 4}, /*perms=*/{1, 2, 0}),
        ElementsAreArray({0, 12, 1, 13, 2, 14, 3, 15, 4,  16, 5,  17,
                          6, 18, 7, 19, 8, 20, 9, 21, 10, 22, 11, 23}));
  }

  {
    EXPECT_THAT(
        RunTestPermutation<float>(/*shape=*/{2, 3, 4}, /*perms=*/{0, 2, 1}),
        ElementsAreArray({0,  4,  8,  1,  5,  9,  2,  6,  10, 3,  7,  11,
                          12, 16, 20, 13, 17, 21, 14, 18, 22, 15, 19, 23}));
  }

  {
    EXPECT_THAT(
        RunTestPermutation<float>(/*shape=*/{2, 3, 4}, /*perms=*/{1, 0, 2}),
        ElementsAreArray({0,  1,  2,  3,  12, 13, 14, 15, 4,  5,  6,  7,
                          16, 17, 18, 19, 8,  9,  10, 11, 20, 21, 22, 23}));
  }

  {
    EXPECT_THAT(
        RunTestPermutation<float>(/*shape=*/{2, 3, 4}, /*perms=*/{2, 1, 0}),
        ElementsAreArray({0, 12, 4, 16, 8,  20, 1, 13, 5, 17, 9,  21,
                          2, 14, 6, 18, 10, 22, 3, 15, 7, 19, 11, 23}));
  }
}

TEST(TransposeTest, TestRefOps3D_OneInDimension) {
  // Shape with 1 as first dim -> transposed.
  {
    EXPECT_THAT(
        RunTestPermutation<float>(/*shape=*/{1, 2, 3}, /*perms=*/{2, 0, 1}),
        ElementsAreArray({0, 3, 1, 4, 2, 5}));
  }
  // Shape with 1 as first dim -> identity.
  {
    EXPECT_THAT(
        RunTestPermutation<float>(/*shape=*/{1, 2, 3}, /*perms=*/{1, 2, 0}),
        ElementsAreArray({0, 1, 2, 3, 4, 5}));
  }
  // Shape with 1 as third dim -> transposed.
  {
    EXPECT_THAT(
        RunTestPermutation<float>(/*shape=*/{2, 3, 1}, /*perms=*/{1, 2, 0}),
        ElementsAreArray({0, 3, 1, 4, 2, 5}));
  }
  // Shape with 1 as third dim -> identity.
  {
    EXPECT_THAT(
        RunTestPermutation<float>(/*shape=*/{2, 3, 1}, /*perms=*/{2, 0, 1}),
        ElementsAreArray({0, 1, 2, 3, 4, 5}));
  }
}

template <typename T>
void TransposeTestTestRefOps4D() {
  // Basic 4d.
  EXPECT_THAT(
      RunTestPermutation<T>({2, 3, 4, 5}, {2, 0, 1, 3}),
      ElementsAreArray(
          {0,  1,  2,  3,  4,  20, 21, 22, 23, 24, 40,  41,  42,  43,  44,
           60, 61, 62, 63, 64, 80, 81, 82, 83, 84, 100, 101, 102, 103, 104,
           5,  6,  7,  8,  9,  25, 26, 27, 28, 29, 45,  46,  47,  48,  49,
           65, 66, 67, 68, 69, 85, 86, 87, 88, 89, 105, 106, 107, 108, 109,
           10, 11, 12, 13, 14, 30, 31, 32, 33, 34, 50,  51,  52,  53,  54,
           70, 71, 72, 73, 74, 90, 91, 92, 93, 94, 110, 111, 112, 113, 114,
           15, 16, 17, 18, 19, 35, 36, 37, 38, 39, 55,  56,  57,  58,  59,
           75, 76, 77, 78, 79, 95, 96, 97, 98, 99, 115, 116, 117, 118, 119}));
  {
    // Basic identity.
    std::vector<T> out = RunTestPermutation<T>({2, 3, 4, 5}, {0, 1, 2, 3});
    std::vector<T> ref(out.size());
    absl::c_iota(ref, 0);
    EXPECT_THAT(out, ElementsAreArray(ref));
  }
}

TEST(TransposeTest, TestRefOps4D) { TransposeTestTestRefOps4D<float>(); }

TEST(TransposeTest, TestRefOps4DInt8) { TransposeTestTestRefOps4D<int8_t>(); }

TEST(TransposeTest, TestRefOps4DInt16) { TransposeTestTestRefOps4D<int16_t>(); }

TEST(TransposeTest, TestRefOps1D0) {
  EXPECT_THAT(RunTestPermutation<float>(/*shape=*/{2}, /*perms=*/{0}),
              ElementsAreArray({0, 1}));
}

TEST(TransposeTest, TestRefOps2D0) {
  EXPECT_THAT(RunTestPermutation<float>(/*shape=*/{2, 3}, /*perms=*/{0, 1}),
              ElementsAreArray({0, 1, 2, 3, 4, 5}));
}

TEST(TransposeTest, TestRefOps2D1) {
  EXPECT_THAT(RunTestPermutation<float>(/*shape=*/{2, 3}, /*perms=*/{1, 0}),
              ElementsAreArray({0, 3, 1, 4, 2, 5}));
}

TEST(TransposeTest, TestRefOps3D0) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4}, /*perms=*/{0, 1, 2}),
      ElementsAreArray({0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11,
                        12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23}));
}

TEST(TransposeTest, TestRefOps3D1) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4}, /*perms=*/{0, 2, 1}),
      ElementsAreArray({0,  4,  8,  1,  5,  9,  2,  6,  10, 3,  7,  11,
                        12, 16, 20, 13, 17, 21, 14, 18, 22, 15, 19, 23}));
}

TEST(TransposeTest, TestRefOps3D2) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4}, /*perms=*/{1, 0, 2}),
      ElementsAreArray({0,  1,  2,  3,  12, 13, 14, 15, 4,  5,  6,  7,
                        16, 17, 18, 19, 8,  9,  10, 11, 20, 21, 22, 23}));
}

TEST(TransposeTest, TestRefOps3D3) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4}, /*perms=*/{1, 2, 0}),
      ElementsAreArray({0, 12, 1, 13, 2, 14, 3, 15, 4,  16, 5,  17,
                        6, 18, 7, 19, 8, 20, 9, 21, 10, 22, 11, 23}));
}

TEST(TransposeTest, TestRefOps3D4) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4}, /*perms=*/{2, 0, 1}),
      ElementsAreArray({0, 4, 8,  12, 16, 20, 1, 5, 9,  13, 17, 21,
                        2, 6, 10, 14, 18, 22, 3, 7, 11, 15, 19, 23}));
}

TEST(TransposeTest, TestRefOps3D5) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4}, /*perms=*/{2, 1, 0}),
      ElementsAreArray({0, 12, 4, 16, 8,  20, 1, 13, 5, 17, 9,  21,
                        2, 14, 6, 18, 10, 22, 3, 15, 7, 19, 11, 23}));
}

TEST(TransposeTest, TestRefOps4D0) {
  const std::vector<float> ref = [] {
    std::vector<float> ref(120);
    absl::c_iota(ref, 0);
    return ref;
  }();
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{0, 1, 2, 3}),
      ElementsAreArray(ref));
}

TEST(TransposeTest, TestRefOps4D1) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{0, 1, 3, 2}),
      ElementsAreArray(
          {0,   5,   10,  15,  1,   6,   11,  16,  2,   7,   12,  17,  3,   8,
           13,  18,  4,   9,   14,  19,  20,  25,  30,  35,  21,  26,  31,  36,
           22,  27,  32,  37,  23,  28,  33,  38,  24,  29,  34,  39,  40,  45,
           50,  55,  41,  46,  51,  56,  42,  47,  52,  57,  43,  48,  53,  58,
           44,  49,  54,  59,  60,  65,  70,  75,  61,  66,  71,  76,  62,  67,
           72,  77,  63,  68,  73,  78,  64,  69,  74,  79,  80,  85,  90,  95,
           81,  86,  91,  96,  82,  87,  92,  97,  83,  88,  93,  98,  84,  89,
           94,  99,  100, 105, 110, 115, 101, 106, 111, 116, 102, 107, 112, 117,
           103, 108, 113, 118, 104, 109, 114, 119}));
}

TEST(TransposeTest, TestRefOps4D2) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{0, 2, 1, 3}),
      ElementsAreArray(
          {0,  1,  2,  3,  4,  20, 21, 22, 23, 24, 40,  41,  42,  43,  44,
           5,  6,  7,  8,  9,  25, 26, 27, 28, 29, 45,  46,  47,  48,  49,
           10, 11, 12, 13, 14, 30, 31, 32, 33, 34, 50,  51,  52,  53,  54,
           15, 16, 17, 18, 19, 35, 36, 37, 38, 39, 55,  56,  57,  58,  59,
           60, 61, 62, 63, 64, 80, 81, 82, 83, 84, 100, 101, 102, 103, 104,
           65, 66, 67, 68, 69, 85, 86, 87, 88, 89, 105, 106, 107, 108, 109,
           70, 71, 72, 73, 74, 90, 91, 92, 93, 94, 110, 111, 112, 113, 114,
           75, 76, 77, 78, 79, 95, 96, 97, 98, 99, 115, 116, 117, 118, 119}));
}

TEST(TransposeTest, TestRefOps4D3) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{0, 2, 3, 1}),
      ElementsAreArray(
          {0,  20, 40,  1,  21, 41,  2,  22, 42,  3,  23, 43,  4,  24, 44,
           5,  25, 45,  6,  26, 46,  7,  27, 47,  8,  28, 48,  9,  29, 49,
           10, 30, 50,  11, 31, 51,  12, 32, 52,  13, 33, 53,  14, 34, 54,
           15, 35, 55,  16, 36, 56,  17, 37, 57,  18, 38, 58,  19, 39, 59,
           60, 80, 100, 61, 81, 101, 62, 82, 102, 63, 83, 103, 64, 84, 104,
           65, 85, 105, 66, 86, 106, 67, 87, 107, 68, 88, 108, 69, 89, 109,
           70, 90, 110, 71, 91, 111, 72, 92, 112, 73, 93, 113, 74, 94, 114,
           75, 95, 115, 76, 96, 116, 77, 97, 117, 78, 98, 118, 79, 99, 119}));
}

TEST(TransposeTest, TestRefOps4D4) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{0, 3, 1, 2}),
      ElementsAreArray({0,  5,  10, 15, 20, 25, 30, 35, 40,  45,  50,  55,
                        1,  6,  11, 16, 21, 26, 31, 36, 41,  46,  51,  56,
                        2,  7,  12, 17, 22, 27, 32, 37, 42,  47,  52,  57,
                        3,  8,  13, 18, 23, 28, 33, 38, 43,  48,  53,  58,
                        4,  9,  14, 19, 24, 29, 34, 39, 44,  49,  54,  59,
                        60, 65, 70, 75, 80, 85, 90, 95, 100, 105, 110, 115,
                        61, 66, 71, 76, 81, 86, 91, 96, 101, 106, 111, 116,
                        62, 67, 72, 77, 82, 87, 92, 97, 102, 107, 112, 117,
                        63, 68, 73, 78, 83, 88, 93, 98, 103, 108, 113, 118,
                        64, 69, 74, 79, 84, 89, 94, 99, 104, 109, 114, 119}));
}

TEST(TransposeTest, TestRefOps4D5) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{0, 3, 2, 1}),
      ElementsAreArray(
          {0,  20, 40,  5,  25, 45,  10, 30, 50,  15, 35, 55,  1,  21, 41,
           6,  26, 46,  11, 31, 51,  16, 36, 56,  2,  22, 42,  7,  27, 47,
           12, 32, 52,  17, 37, 57,  3,  23, 43,  8,  28, 48,  13, 33, 53,
           18, 38, 58,  4,  24, 44,  9,  29, 49,  14, 34, 54,  19, 39, 59,
           60, 80, 100, 65, 85, 105, 70, 90, 110, 75, 95, 115, 61, 81, 101,
           66, 86, 106, 71, 91, 111, 76, 96, 116, 62, 82, 102, 67, 87, 107,
           72, 92, 112, 77, 97, 117, 63, 83, 103, 68, 88, 108, 73, 93, 113,
           78, 98, 118, 64, 84, 104, 69, 89, 109, 74, 94, 114, 79, 99, 119}));
}

TEST(TransposeTest, TestRefOps4D6) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{1, 0, 2, 3}),
      ElementsAreArray(
          {0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   10,  11,  12,  13,
           14,  15,  16,  17,  18,  19,  60,  61,  62,  63,  64,  65,  66,  67,
           68,  69,  70,  71,  72,  73,  74,  75,  76,  77,  78,  79,  20,  21,
           22,  23,  24,  25,  26,  27,  28,  29,  30,  31,  32,  33,  34,  35,
           36,  37,  38,  39,  80,  81,  82,  83,  84,  85,  86,  87,  88,  89,
           90,  91,  92,  93,  94,  95,  96,  97,  98,  99,  40,  41,  42,  43,
           44,  45,  46,  47,  48,  49,  50,  51,  52,  53,  54,  55,  56,  57,
           58,  59,  100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111,
           112, 113, 114, 115, 116, 117, 118, 119}));
}

TEST(TransposeTest, TestRefOps4D7) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{1, 0, 3, 2}),
      ElementsAreArray(
          {0,   5,   10,  15,  1,   6,   11,  16,  2,   7,   12,  17,  3,   8,
           13,  18,  4,   9,   14,  19,  60,  65,  70,  75,  61,  66,  71,  76,
           62,  67,  72,  77,  63,  68,  73,  78,  64,  69,  74,  79,  20,  25,
           30,  35,  21,  26,  31,  36,  22,  27,  32,  37,  23,  28,  33,  38,
           24,  29,  34,  39,  80,  85,  90,  95,  81,  86,  91,  96,  82,  87,
           92,  97,  83,  88,  93,  98,  84,  89,  94,  99,  40,  45,  50,  55,
           41,  46,  51,  56,  42,  47,  52,  57,  43,  48,  53,  58,  44,  49,
           54,  59,  100, 105, 110, 115, 101, 106, 111, 116, 102, 107, 112, 117,
           103, 108, 113, 118, 104, 109, 114, 119}));
}

TEST(TransposeTest, TestRefOps4D8) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{1, 2, 0, 3}),
      ElementsAreArray(
          {0,   1,   2,   3,   4,   60,  61,  62,  63,  64,  5,   6,   7,   8,
           9,   65,  66,  67,  68,  69,  10,  11,  12,  13,  14,  70,  71,  72,
           73,  74,  15,  16,  17,  18,  19,  75,  76,  77,  78,  79,  20,  21,
           22,  23,  24,  80,  81,  82,  83,  84,  25,  26,  27,  28,  29,  85,
           86,  87,  88,  89,  30,  31,  32,  33,  34,  90,  91,  92,  93,  94,
           35,  36,  37,  38,  39,  95,  96,  97,  98,  99,  40,  41,  42,  43,
           44,  100, 101, 102, 103, 104, 45,  46,  47,  48,  49,  105, 106, 107,
           108, 109, 50,  51,  52,  53,  54,  110, 111, 112, 113, 114, 55,  56,
           57,  58,  59,  115, 116, 117, 118, 119}));
}

TEST(TransposeTest, TestRefOps4D9) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{1, 2, 3, 0}),
      ElementsAreArray({0,  60,  1,  61,  2,  62,  3,  63,  4,  64,  5,  65,
                        6,  66,  7,  67,  8,  68,  9,  69,  10, 70,  11, 71,
                        12, 72,  13, 73,  14, 74,  15, 75,  16, 76,  17, 77,
                        18, 78,  19, 79,  20, 80,  21, 81,  22, 82,  23, 83,
                        24, 84,  25, 85,  26, 86,  27, 87,  28, 88,  29, 89,
                        30, 90,  31, 91,  32, 92,  33, 93,  34, 94,  35, 95,
                        36, 96,  37, 97,  38, 98,  39, 99,  40, 100, 41, 101,
                        42, 102, 43, 103, 44, 104, 45, 105, 46, 106, 47, 107,
                        48, 108, 49, 109, 50, 110, 51, 111, 52, 112, 53, 113,
                        54, 114, 55, 115, 56, 116, 57, 117, 58, 118, 59, 119}));
}

TEST(TransposeTest, TestRefOps4D10) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{1, 3, 0, 2}),
      ElementsAreArray(
          {0,   5,   10,  15,  60,  65,  70,  75, 1,   6,   11,  16,  61,  66,
           71,  76,  2,   7,   12,  17,  62,  67, 72,  77,  3,   8,   13,  18,
           63,  68,  73,  78,  4,   9,   14,  19, 64,  69,  74,  79,  20,  25,
           30,  35,  80,  85,  90,  95,  21,  26, 31,  36,  81,  86,  91,  96,
           22,  27,  32,  37,  82,  87,  92,  97, 23,  28,  33,  38,  83,  88,
           93,  98,  24,  29,  34,  39,  84,  89, 94,  99,  40,  45,  50,  55,
           100, 105, 110, 115, 41,  46,  51,  56, 101, 106, 111, 116, 42,  47,
           52,  57,  102, 107, 112, 117, 43,  48, 53,  58,  103, 108, 113, 118,
           44,  49,  54,  59,  104, 109, 114, 119}));
}

TEST(TransposeTest, TestRefOps4D11) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{1, 3, 2, 0}),
      ElementsAreArray({0,  60,  5,  65,  10, 70,  15, 75,  1,  61,  6,  66,
                        11, 71,  16, 76,  2,  62,  7,  67,  12, 72,  17, 77,
                        3,  63,  8,  68,  13, 73,  18, 78,  4,  64,  9,  69,
                        14, 74,  19, 79,  20, 80,  25, 85,  30, 90,  35, 95,
                        21, 81,  26, 86,  31, 91,  36, 96,  22, 82,  27, 87,
                        32, 92,  37, 97,  23, 83,  28, 88,  33, 93,  38, 98,
                        24, 84,  29, 89,  34, 94,  39, 99,  40, 100, 45, 105,
                        50, 110, 55, 115, 41, 101, 46, 106, 51, 111, 56, 116,
                        42, 102, 47, 107, 52, 112, 57, 117, 43, 103, 48, 108,
                        53, 113, 58, 118, 44, 104, 49, 109, 54, 114, 59, 119}));
}

TEST(TransposeTest, TestRefOps4D12) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{2, 0, 1, 3}),
      ElementsAreArray(
          {0,  1,  2,  3,  4,  20, 21, 22, 23, 24, 40,  41,  42,  43,  44,
           60, 61, 62, 63, 64, 80, 81, 82, 83, 84, 100, 101, 102, 103, 104,
           5,  6,  7,  8,  9,  25, 26, 27, 28, 29, 45,  46,  47,  48,  49,
           65, 66, 67, 68, 69, 85, 86, 87, 88, 89, 105, 106, 107, 108, 109,
           10, 11, 12, 13, 14, 30, 31, 32, 33, 34, 50,  51,  52,  53,  54,
           70, 71, 72, 73, 74, 90, 91, 92, 93, 94, 110, 111, 112, 113, 114,
           15, 16, 17, 18, 19, 35, 36, 37, 38, 39, 55,  56,  57,  58,  59,
           75, 76, 77, 78, 79, 95, 96, 97, 98, 99, 115, 116, 117, 118, 119}));
}

TEST(TransposeTest, TestRefOps4D13) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{2, 0, 3, 1}),
      ElementsAreArray(
          {0,  20, 40,  1,  21, 41,  2,  22, 42,  3,  23, 43,  4,  24, 44,
           60, 80, 100, 61, 81, 101, 62, 82, 102, 63, 83, 103, 64, 84, 104,
           5,  25, 45,  6,  26, 46,  7,  27, 47,  8,  28, 48,  9,  29, 49,
           65, 85, 105, 66, 86, 106, 67, 87, 107, 68, 88, 108, 69, 89, 109,
           10, 30, 50,  11, 31, 51,  12, 32, 52,  13, 33, 53,  14, 34, 54,
           70, 90, 110, 71, 91, 111, 72, 92, 112, 73, 93, 113, 74, 94, 114,
           15, 35, 55,  16, 36, 56,  17, 37, 57,  18, 38, 58,  19, 39, 59,
           75, 95, 115, 76, 96, 116, 77, 97, 117, 78, 98, 118, 79, 99, 119}));
}

TEST(TransposeTest, TestRefOps4D14) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{2, 1, 0, 3}),
      ElementsAreArray(
          {0,  1,  2,  3,  4,  60, 61, 62, 63, 64, 20,  21,  22,  23,  24,
           80, 81, 82, 83, 84, 40, 41, 42, 43, 44, 100, 101, 102, 103, 104,
           5,  6,  7,  8,  9,  65, 66, 67, 68, 69, 25,  26,  27,  28,  29,
           85, 86, 87, 88, 89, 45, 46, 47, 48, 49, 105, 106, 107, 108, 109,
           10, 11, 12, 13, 14, 70, 71, 72, 73, 74, 30,  31,  32,  33,  34,
           90, 91, 92, 93, 94, 50, 51, 52, 53, 54, 110, 111, 112, 113, 114,
           15, 16, 17, 18, 19, 75, 76, 77, 78, 79, 35,  36,  37,  38,  39,
           95, 96, 97, 98, 99, 55, 56, 57, 58, 59, 115, 116, 117, 118, 119}));
}

TEST(TransposeTest, TestRefOps4D15) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{2, 1, 3, 0}),
      ElementsAreArray(
          {0,  60, 1,  61, 2,  62, 3,   63, 4,   64, 20,  80, 21,  81, 22,
           82, 23, 83, 24, 84, 40, 100, 41, 101, 42, 102, 43, 103, 44, 104,
           5,  65, 6,  66, 7,  67, 8,   68, 9,   69, 25,  85, 26,  86, 27,
           87, 28, 88, 29, 89, 45, 105, 46, 106, 47, 107, 48, 108, 49, 109,
           10, 70, 11, 71, 12, 72, 13,  73, 14,  74, 30,  90, 31,  91, 32,
           92, 33, 93, 34, 94, 50, 110, 51, 111, 52, 112, 53, 113, 54, 114,
           15, 75, 16, 76, 17, 77, 18,  78, 19,  79, 35,  95, 36,  96, 37,
           97, 38, 98, 39, 99, 55, 115, 56, 116, 57, 117, 58, 118, 59, 119}));
}

TEST(TransposeTest, TestRefOps4D16) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{2, 3, 0, 1}),
      ElementsAreArray(
          {0,  20, 40,  60, 80, 100, 1,  21, 41,  61, 81, 101, 2,  22, 42,
           62, 82, 102, 3,  23, 43,  63, 83, 103, 4,  24, 44,  64, 84, 104,
           5,  25, 45,  65, 85, 105, 6,  26, 46,  66, 86, 106, 7,  27, 47,
           67, 87, 107, 8,  28, 48,  68, 88, 108, 9,  29, 49,  69, 89, 109,
           10, 30, 50,  70, 90, 110, 11, 31, 51,  71, 91, 111, 12, 32, 52,
           72, 92, 112, 13, 33, 53,  73, 93, 113, 14, 34, 54,  74, 94, 114,
           15, 35, 55,  75, 95, 115, 16, 36, 56,  76, 96, 116, 17, 37, 57,
           77, 97, 117, 18, 38, 58,  78, 98, 118, 19, 39, 59,  79, 99, 119}));
}

TEST(TransposeTest, TestRefOps4D17) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{2, 3, 1, 0}),
      ElementsAreArray(
          {0,  60, 20,  80, 40, 100, 1,  61, 21,  81, 41, 101, 2,  62, 22,
           82, 42, 102, 3,  63, 23,  83, 43, 103, 4,  64, 24,  84, 44, 104,
           5,  65, 25,  85, 45, 105, 6,  66, 26,  86, 46, 106, 7,  67, 27,
           87, 47, 107, 8,  68, 28,  88, 48, 108, 9,  69, 29,  89, 49, 109,
           10, 70, 30,  90, 50, 110, 11, 71, 31,  91, 51, 111, 12, 72, 32,
           92, 52, 112, 13, 73, 33,  93, 53, 113, 14, 74, 34,  94, 54, 114,
           15, 75, 35,  95, 55, 115, 16, 76, 36,  96, 56, 116, 17, 77, 37,
           97, 57, 117, 18, 78, 38,  98, 58, 118, 19, 79, 39,  99, 59, 119}));
}

TEST(TransposeTest, TestRefOps4D18) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{3, 0, 1, 2}),
      ElementsAreArray({0,  5,  10, 15, 20, 25, 30, 35, 40,  45,  50,  55,
                        60, 65, 70, 75, 80, 85, 90, 95, 100, 105, 110, 115,
                        1,  6,  11, 16, 21, 26, 31, 36, 41,  46,  51,  56,
                        61, 66, 71, 76, 81, 86, 91, 96, 101, 106, 111, 116,
                        2,  7,  12, 17, 22, 27, 32, 37, 42,  47,  52,  57,
                        62, 67, 72, 77, 82, 87, 92, 97, 102, 107, 112, 117,
                        3,  8,  13, 18, 23, 28, 33, 38, 43,  48,  53,  58,
                        63, 68, 73, 78, 83, 88, 93, 98, 103, 108, 113, 118,
                        4,  9,  14, 19, 24, 29, 34, 39, 44,  49,  54,  59,
                        64, 69, 74, 79, 84, 89, 94, 99, 104, 109, 114, 119}));
}

TEST(TransposeTest, TestRefOps4D19) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{3, 0, 2, 1}),
      ElementsAreArray(
          {0,  20, 40,  5,  25, 45,  10, 30, 50,  15, 35, 55,  60, 80, 100,
           65, 85, 105, 70, 90, 110, 75, 95, 115, 1,  21, 41,  6,  26, 46,
           11, 31, 51,  16, 36, 56,  61, 81, 101, 66, 86, 106, 71, 91, 111,
           76, 96, 116, 2,  22, 42,  7,  27, 47,  12, 32, 52,  17, 37, 57,
           62, 82, 102, 67, 87, 107, 72, 92, 112, 77, 97, 117, 3,  23, 43,
           8,  28, 48,  13, 33, 53,  18, 38, 58,  63, 83, 103, 68, 88, 108,
           73, 93, 113, 78, 98, 118, 4,  24, 44,  9,  29, 49,  14, 34, 54,
           19, 39, 59,  64, 84, 104, 69, 89, 109, 74, 94, 114, 79, 99, 119}));
}

TEST(TransposeTest, TestRefOps4D20) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{3, 1, 0, 2}),
      ElementsAreArray({0,  5,  10, 15, 60, 65, 70, 75, 20,  25,  30,  35,
                        80, 85, 90, 95, 40, 45, 50, 55, 100, 105, 110, 115,
                        1,  6,  11, 16, 61, 66, 71, 76, 21,  26,  31,  36,
                        81, 86, 91, 96, 41, 46, 51, 56, 101, 106, 111, 116,
                        2,  7,  12, 17, 62, 67, 72, 77, 22,  27,  32,  37,
                        82, 87, 92, 97, 42, 47, 52, 57, 102, 107, 112, 117,
                        3,  8,  13, 18, 63, 68, 73, 78, 23,  28,  33,  38,
                        83, 88, 93, 98, 43, 48, 53, 58, 103, 108, 113, 118,
                        4,  9,  14, 19, 64, 69, 74, 79, 24,  29,  34,  39,
                        84, 89, 94, 99, 44, 49, 54, 59, 104, 109, 114, 119}));
}

TEST(TransposeTest, TestRefOps4D21) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{3, 1, 2, 0}),
      ElementsAreArray({0,  60, 5,  65, 10, 70,  15, 75,  20, 80,  25, 85,
                        30, 90, 35, 95, 40, 100, 45, 105, 50, 110, 55, 115,
                        1,  61, 6,  66, 11, 71,  16, 76,  21, 81,  26, 86,
                        31, 91, 36, 96, 41, 101, 46, 106, 51, 111, 56, 116,
                        2,  62, 7,  67, 12, 72,  17, 77,  22, 82,  27, 87,
                        32, 92, 37, 97, 42, 102, 47, 107, 52, 112, 57, 117,
                        3,  63, 8,  68, 13, 73,  18, 78,  23, 83,  28, 88,
                        33, 93, 38, 98, 43, 103, 48, 108, 53, 113, 58, 118,
                        4,  64, 9,  69, 14, 74,  19, 79,  24, 84,  29, 89,
                        34, 94, 39, 99, 44, 104, 49, 109, 54, 114, 59, 119}));
}

TEST(TransposeTest, TestRefOps4D22) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{3, 2, 0, 1}),
      ElementsAreArray(
          {0,  20, 40,  60, 80, 100, 5,  25, 45,  65, 85, 105, 10, 30, 50,
           70, 90, 110, 15, 35, 55,  75, 95, 115, 1,  21, 41,  61, 81, 101,
           6,  26, 46,  66, 86, 106, 11, 31, 51,  71, 91, 111, 16, 36, 56,
           76, 96, 116, 2,  22, 42,  62, 82, 102, 7,  27, 47,  67, 87, 107,
           12, 32, 52,  72, 92, 112, 17, 37, 57,  77, 97, 117, 3,  23, 43,
           63, 83, 103, 8,  28, 48,  68, 88, 108, 13, 33, 53,  73, 93, 113,
           18, 38, 58,  78, 98, 118, 4,  24, 44,  64, 84, 104, 9,  29, 49,
           69, 89, 109, 14, 34, 54,  74, 94, 114, 19, 39, 59,  79, 99, 119}));
}

TEST(TransposeTest, TestRefOps4D23) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5}, /*perms=*/{3, 2, 1, 0}),
      ElementsAreArray(
          {0,  60, 20,  80, 40, 100, 5,  65, 25,  85, 45, 105, 10, 70, 30,
           90, 50, 110, 15, 75, 35,  95, 55, 115, 1,  61, 21,  81, 41, 101,
           6,  66, 26,  86, 46, 106, 11, 71, 31,  91, 51, 111, 16, 76, 36,
           96, 56, 116, 2,  62, 22,  82, 42, 102, 7,  67, 27,  87, 47, 107,
           12, 72, 32,  92, 52, 112, 17, 77, 37,  97, 57, 117, 3,  63, 23,
           83, 43, 103, 8,  68, 28,  88, 48, 108, 13, 73, 33,  93, 53, 113,
           18, 78, 38,  98, 58, 118, 4,  64, 24,  84, 44, 104, 9,  69, 29,
           89, 49, 109, 14, 74, 34,  94, 54, 114, 19, 79, 39,  99, 59, 119}));
}

TEST(TransposeTest, TestRefOps5D0) {
  const std::vector<float> ref = [] {
    std::vector<float> ref(720);
    absl::c_iota(ref, 0);
    return ref;
  }();
  EXPECT_THAT(RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5, 6},
                                        /*perms=*/{0, 1, 2, 3, 4}),
              ElementsAreArray(ref));
}

TEST(TransposeTest, TestRefOps5D1) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5, 6},
                                /*perms=*/{4, 3, 2, 1, 0}),
      ElementsAreArray(
          {0,   360, 120, 480, 240, 600, 30,  390, 150, 510, 270, 630, 60,  420,
           180, 540, 300, 660, 90,  450, 210, 570, 330, 690, 6,   366, 126, 486,
           246, 606, 36,  396, 156, 516, 276, 636, 66,  426, 186, 546, 306, 666,
           96,  456, 216, 576, 336, 696, 12,  372, 132, 492, 252, 612, 42,  402,
           162, 522, 282, 642, 72,  432, 192, 552, 312, 672, 102, 462, 222, 582,
           342, 702, 18,  378, 138, 498, 258, 618, 48,  408, 168, 528, 288, 648,
           78,  438, 198, 558, 318, 678, 108, 468, 228, 588, 348, 708, 24,  384,
           144, 504, 264, 624, 54,  414, 174, 534, 294, 654, 84,  444, 204, 564,
           324, 684, 114, 474, 234, 594, 354, 714, 1,   361, 121, 481, 241, 601,
           31,  391, 151, 511, 271, 631, 61,  421, 181, 541, 301, 661, 91,  451,
           211, 571, 331, 691, 7,   367, 127, 487, 247, 607, 37,  397, 157, 517,
           277, 637, 67,  427, 187, 547, 307, 667, 97,  457, 217, 577, 337, 697,
           13,  373, 133, 493, 253, 613, 43,  403, 163, 523, 283, 643, 73,  433,
           193, 553, 313, 673, 103, 463, 223, 583, 343, 703, 19,  379, 139, 499,
           259, 619, 49,  409, 169, 529, 289, 649, 79,  439, 199, 559, 319, 679,
           109, 469, 229, 589, 349, 709, 25,  385, 145, 505, 265, 625, 55,  415,
           175, 535, 295, 655, 85,  445, 205, 565, 325, 685, 115, 475, 235, 595,
           355, 715, 2,   362, 122, 482, 242, 602, 32,  392, 152, 512, 272, 632,
           62,  422, 182, 542, 302, 662, 92,  452, 212, 572, 332, 692, 8,   368,
           128, 488, 248, 608, 38,  398, 158, 518, 278, 638, 68,  428, 188, 548,
           308, 668, 98,  458, 218, 578, 338, 698, 14,  374, 134, 494, 254, 614,
           44,  404, 164, 524, 284, 644, 74,  434, 194, 554, 314, 674, 104, 464,
           224, 584, 344, 704, 20,  380, 140, 500, 260, 620, 50,  410, 170, 530,
           290, 650, 80,  440, 200, 560, 320, 680, 110, 470, 230, 590, 350, 710,
           26,  386, 146, 506, 266, 626, 56,  416, 176, 536, 296, 656, 86,  446,
           206, 566, 326, 686, 116, 476, 236, 596, 356, 716, 3,   363, 123, 483,
           243, 603, 33,  393, 153, 513, 273, 633, 63,  423, 183, 543, 303, 663,
           93,  453, 213, 573, 333, 693, 9,   369, 129, 489, 249, 609, 39,  399,
           159, 519, 279, 639, 69,  429, 189, 549, 309, 669, 99,  459, 219, 579,
           339, 699, 15,  375, 135, 495, 255, 615, 45,  405, 165, 525, 285, 645,
           75,  435, 195, 555, 315, 675, 105, 465, 225, 585, 345, 705, 21,  381,
           141, 501, 261, 621, 51,  411, 171, 531, 291, 651, 81,  441, 201, 561,
           321, 681, 111, 471, 231, 591, 351, 711, 27,  387, 147, 507, 267, 627,
           57,  417, 177, 537, 297, 657, 87,  447, 207, 567, 327, 687, 117, 477,
           237, 597, 357, 717, 4,   364, 124, 484, 244, 604, 34,  394, 154, 514,
           274, 634, 64,  424, 184, 544, 304, 664, 94,  454, 214, 574, 334, 694,
           10,  370, 130, 490, 250, 610, 40,  400, 160, 520, 280, 640, 70,  430,
           190, 550, 310, 670, 100, 460, 220, 580, 340, 700, 16,  376, 136, 496,
           256, 616, 46,  406, 166, 526, 286, 646, 76,  436, 196, 556, 316, 676,
           106, 466, 226, 586, 346, 706, 22,  382, 142, 502, 262, 622, 52,  412,
           172, 532, 292, 652, 82,  442, 202, 562, 322, 682, 112, 472, 232, 592,
           352, 712, 28,  388, 148, 508, 268, 628, 58,  418, 178, 538, 298, 658,
           88,  448, 208, 568, 328, 688, 118, 478, 238, 598, 358, 718, 5,   365,
           125, 485, 245, 605, 35,  395, 155, 515, 275, 635, 65,  425, 185, 545,
           305, 665, 95,  455, 215, 575, 335, 695, 11,  371, 131, 491, 251, 611,
           41,  401, 161, 521, 281, 641, 71,  431, 191, 551, 311, 671, 101, 461,
           221, 581, 341, 701, 17,  377, 137, 497, 257, 617, 47,  407, 167, 527,
           287, 647, 77,  437, 197, 557, 317, 677, 107, 467, 227, 587, 347, 707,
           23,  383, 143, 503, 263, 623, 53,  413, 173, 533, 293, 653, 83,  443,
           203, 563, 323, 683, 113, 473, 233, 593, 353, 713, 29,  389, 149, 509,
           269, 629, 59,  419, 179, 539, 299, 659, 89,  449, 209, 569, 329, 689,
           119, 479, 239, 599, 359, 719}));
}

TEST(TransposeTest, TestRefOps5D2) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5, 6},
                                /*perms=*/{1, 0, 2, 3, 4}),
      ElementsAreArray(
          {0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   10,  11,  12,  13,
           14,  15,  16,  17,  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,
           28,  29,  30,  31,  32,  33,  34,  35,  36,  37,  38,  39,  40,  41,
           42,  43,  44,  45,  46,  47,  48,  49,  50,  51,  52,  53,  54,  55,
           56,  57,  58,  59,  60,  61,  62,  63,  64,  65,  66,  67,  68,  69,
           70,  71,  72,  73,  74,  75,  76,  77,  78,  79,  80,  81,  82,  83,
           84,  85,  86,  87,  88,  89,  90,  91,  92,  93,  94,  95,  96,  97,
           98,  99,  100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111,
           112, 113, 114, 115, 116, 117, 118, 119, 360, 361, 362, 363, 364, 365,
           366, 367, 368, 369, 370, 371, 372, 373, 374, 375, 376, 377, 378, 379,
           380, 381, 382, 383, 384, 385, 386, 387, 388, 389, 390, 391, 392, 393,
           394, 395, 396, 397, 398, 399, 400, 401, 402, 403, 404, 405, 406, 407,
           408, 409, 410, 411, 412, 413, 414, 415, 416, 417, 418, 419, 420, 421,
           422, 423, 424, 425, 426, 427, 428, 429, 430, 431, 432, 433, 434, 435,
           436, 437, 438, 439, 440, 441, 442, 443, 444, 445, 446, 447, 448, 449,
           450, 451, 452, 453, 454, 455, 456, 457, 458, 459, 460, 461, 462, 463,
           464, 465, 466, 467, 468, 469, 470, 471, 472, 473, 474, 475, 476, 477,
           478, 479, 120, 121, 122, 123, 124, 125, 126, 127, 128, 129, 130, 131,
           132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145,
           146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159,
           160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173,
           174, 175, 176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187,
           188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201,
           202, 203, 204, 205, 206, 207, 208, 209, 210, 211, 212, 213, 214, 215,
           216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229,
           230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 480, 481, 482, 483,
           484, 485, 486, 487, 488, 489, 490, 491, 492, 493, 494, 495, 496, 497,
           498, 499, 500, 501, 502, 503, 504, 505, 506, 507, 508, 509, 510, 511,
           512, 513, 514, 515, 516, 517, 518, 519, 520, 521, 522, 523, 524, 525,
           526, 527, 528, 529, 530, 531, 532, 533, 534, 535, 536, 537, 538, 539,
           540, 541, 542, 543, 544, 545, 546, 547, 548, 549, 550, 551, 552, 553,
           554, 555, 556, 557, 558, 559, 560, 561, 562, 563, 564, 565, 566, 567,
           568, 569, 570, 571, 572, 573, 574, 575, 576, 577, 578, 579, 580, 581,
           582, 583, 584, 585, 586, 587, 588, 589, 590, 591, 592, 593, 594, 595,
           596, 597, 598, 599, 240, 241, 242, 243, 244, 245, 246, 247, 248, 249,
           250, 251, 252, 253, 254, 255, 256, 257, 258, 259, 260, 261, 262, 263,
           264, 265, 266, 267, 268, 269, 270, 271, 272, 273, 274, 275, 276, 277,
           278, 279, 280, 281, 282, 283, 284, 285, 286, 287, 288, 289, 290, 291,
           292, 293, 294, 295, 296, 297, 298, 299, 300, 301, 302, 303, 304, 305,
           306, 307, 308, 309, 310, 311, 312, 313, 314, 315, 316, 317, 318, 319,
           320, 321, 322, 323, 324, 325, 326, 327, 328, 329, 330, 331, 332, 333,
           334, 335, 336, 337, 338, 339, 340, 341, 342, 343, 344, 345, 346, 347,
           348, 349, 350, 351, 352, 353, 354, 355, 356, 357, 358, 359, 600, 601,
           602, 603, 604, 605, 606, 607, 608, 609, 610, 611, 612, 613, 614, 615,
           616, 617, 618, 619, 620, 621, 622, 623, 624, 625, 626, 627, 628, 629,
           630, 631, 632, 633, 634, 635, 636, 637, 638, 639, 640, 641, 642, 643,
           644, 645, 646, 647, 648, 649, 650, 651, 652, 653, 654, 655, 656, 657,
           658, 659, 660, 661, 662, 663, 664, 665, 666, 667, 668, 669, 670, 671,
           672, 673, 674, 675, 676, 677, 678, 679, 680, 681, 682, 683, 684, 685,
           686, 687, 688, 689, 690, 691, 692, 693, 694, 695, 696, 697, 698, 699,
           700, 701, 702, 703, 704, 705, 706, 707, 708, 709, 710, 711, 712, 713,
           714, 715, 716, 717, 718, 719}));
}

TEST(TransposeTest, TestRefOps5D3) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5, 6},
                                /*perms=*/{1, 2, 0, 3, 4}),
      ElementsAreArray(
          {0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   10,  11,  12,  13,
           14,  15,  16,  17,  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,
           28,  29,  360, 361, 362, 363, 364, 365, 366, 367, 368, 369, 370, 371,
           372, 373, 374, 375, 376, 377, 378, 379, 380, 381, 382, 383, 384, 385,
           386, 387, 388, 389, 30,  31,  32,  33,  34,  35,  36,  37,  38,  39,
           40,  41,  42,  43,  44,  45,  46,  47,  48,  49,  50,  51,  52,  53,
           54,  55,  56,  57,  58,  59,  390, 391, 392, 393, 394, 395, 396, 397,
           398, 399, 400, 401, 402, 403, 404, 405, 406, 407, 408, 409, 410, 411,
           412, 413, 414, 415, 416, 417, 418, 419, 60,  61,  62,  63,  64,  65,
           66,  67,  68,  69,  70,  71,  72,  73,  74,  75,  76,  77,  78,  79,
           80,  81,  82,  83,  84,  85,  86,  87,  88,  89,  420, 421, 422, 423,
           424, 425, 426, 427, 428, 429, 430, 431, 432, 433, 434, 435, 436, 437,
           438, 439, 440, 441, 442, 443, 444, 445, 446, 447, 448, 449, 90,  91,
           92,  93,  94,  95,  96,  97,  98,  99,  100, 101, 102, 103, 104, 105,
           106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119,
           450, 451, 452, 453, 454, 455, 456, 457, 458, 459, 460, 461, 462, 463,
           464, 465, 466, 467, 468, 469, 470, 471, 472, 473, 474, 475, 476, 477,
           478, 479, 120, 121, 122, 123, 124, 125, 126, 127, 128, 129, 130, 131,
           132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145,
           146, 147, 148, 149, 480, 481, 482, 483, 484, 485, 486, 487, 488, 489,
           490, 491, 492, 493, 494, 495, 496, 497, 498, 499, 500, 501, 502, 503,
           504, 505, 506, 507, 508, 509, 150, 151, 152, 153, 154, 155, 156, 157,
           158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171,
           172, 173, 174, 175, 176, 177, 178, 179, 510, 511, 512, 513, 514, 515,
           516, 517, 518, 519, 520, 521, 522, 523, 524, 525, 526, 527, 528, 529,
           530, 531, 532, 533, 534, 535, 536, 537, 538, 539, 180, 181, 182, 183,
           184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194, 195, 196, 197,
           198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 209, 540, 541,
           542, 543, 544, 545, 546, 547, 548, 549, 550, 551, 552, 553, 554, 555,
           556, 557, 558, 559, 560, 561, 562, 563, 564, 565, 566, 567, 568, 569,
           210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223,
           224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237,
           238, 239, 570, 571, 572, 573, 574, 575, 576, 577, 578, 579, 580, 581,
           582, 583, 584, 585, 586, 587, 588, 589, 590, 591, 592, 593, 594, 595,
           596, 597, 598, 599, 240, 241, 242, 243, 244, 245, 246, 247, 248, 249,
           250, 251, 252, 253, 254, 255, 256, 257, 258, 259, 260, 261, 262, 263,
           264, 265, 266, 267, 268, 269, 600, 601, 602, 603, 604, 605, 606, 607,
           608, 609, 610, 611, 612, 613, 614, 615, 616, 617, 618, 619, 620, 621,
           622, 623, 624, 625, 626, 627, 628, 629, 270, 271, 272, 273, 274, 275,
           276, 277, 278, 279, 280, 281, 282, 283, 284, 285, 286, 287, 288, 289,
           290, 291, 292, 293, 294, 295, 296, 297, 298, 299, 630, 631, 632, 633,
           634, 635, 636, 637, 638, 639, 640, 641, 642, 643, 644, 645, 646, 647,
           648, 649, 650, 651, 652, 653, 654, 655, 656, 657, 658, 659, 300, 301,
           302, 303, 304, 305, 306, 307, 308, 309, 310, 311, 312, 313, 314, 315,
           316, 317, 318, 319, 320, 321, 322, 323, 324, 325, 326, 327, 328, 329,
           660, 661, 662, 663, 664, 665, 666, 667, 668, 669, 670, 671, 672, 673,
           674, 675, 676, 677, 678, 679, 680, 681, 682, 683, 684, 685, 686, 687,
           688, 689, 330, 331, 332, 333, 334, 335, 336, 337, 338, 339, 340, 341,
           342, 343, 344, 345, 346, 347, 348, 349, 350, 351, 352, 353, 354, 355,
           356, 357, 358, 359, 690, 691, 692, 693, 694, 695, 696, 697, 698, 699,
           700, 701, 702, 703, 704, 705, 706, 707, 708, 709, 710, 711, 712, 713,
           714, 715, 716, 717, 718, 719}));
}

TEST(TransposeTest, TestRefOps5D4) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5, 6},
                                /*perms=*/{1, 2, 3, 0, 4}),
      ElementsAreArray(
          {0,   1,   2,   3,   4,   5,   360, 361, 362, 363, 364, 365, 6,   7,
           8,   9,   10,  11,  366, 367, 368, 369, 370, 371, 12,  13,  14,  15,
           16,  17,  372, 373, 374, 375, 376, 377, 18,  19,  20,  21,  22,  23,
           378, 379, 380, 381, 382, 383, 24,  25,  26,  27,  28,  29,  384, 385,
           386, 387, 388, 389, 30,  31,  32,  33,  34,  35,  390, 391, 392, 393,
           394, 395, 36,  37,  38,  39,  40,  41,  396, 397, 398, 399, 400, 401,
           42,  43,  44,  45,  46,  47,  402, 403, 404, 405, 406, 407, 48,  49,
           50,  51,  52,  53,  408, 409, 410, 411, 412, 413, 54,  55,  56,  57,
           58,  59,  414, 415, 416, 417, 418, 419, 60,  61,  62,  63,  64,  65,
           420, 421, 422, 423, 424, 425, 66,  67,  68,  69,  70,  71,  426, 427,
           428, 429, 430, 431, 72,  73,  74,  75,  76,  77,  432, 433, 434, 435,
           436, 437, 78,  79,  80,  81,  82,  83,  438, 439, 440, 441, 442, 443,
           84,  85,  86,  87,  88,  89,  444, 445, 446, 447, 448, 449, 90,  91,
           92,  93,  94,  95,  450, 451, 452, 453, 454, 455, 96,  97,  98,  99,
           100, 101, 456, 457, 458, 459, 460, 461, 102, 103, 104, 105, 106, 107,
           462, 463, 464, 465, 466, 467, 108, 109, 110, 111, 112, 113, 468, 469,
           470, 471, 472, 473, 114, 115, 116, 117, 118, 119, 474, 475, 476, 477,
           478, 479, 120, 121, 122, 123, 124, 125, 480, 481, 482, 483, 484, 485,
           126, 127, 128, 129, 130, 131, 486, 487, 488, 489, 490, 491, 132, 133,
           134, 135, 136, 137, 492, 493, 494, 495, 496, 497, 138, 139, 140, 141,
           142, 143, 498, 499, 500, 501, 502, 503, 144, 145, 146, 147, 148, 149,
           504, 505, 506, 507, 508, 509, 150, 151, 152, 153, 154, 155, 510, 511,
           512, 513, 514, 515, 156, 157, 158, 159, 160, 161, 516, 517, 518, 519,
           520, 521, 162, 163, 164, 165, 166, 167, 522, 523, 524, 525, 526, 527,
           168, 169, 170, 171, 172, 173, 528, 529, 530, 531, 532, 533, 174, 175,
           176, 177, 178, 179, 534, 535, 536, 537, 538, 539, 180, 181, 182, 183,
           184, 185, 540, 541, 542, 543, 544, 545, 186, 187, 188, 189, 190, 191,
           546, 547, 548, 549, 550, 551, 192, 193, 194, 195, 196, 197, 552, 553,
           554, 555, 556, 557, 198, 199, 200, 201, 202, 203, 558, 559, 560, 561,
           562, 563, 204, 205, 206, 207, 208, 209, 564, 565, 566, 567, 568, 569,
           210, 211, 212, 213, 214, 215, 570, 571, 572, 573, 574, 575, 216, 217,
           218, 219, 220, 221, 576, 577, 578, 579, 580, 581, 222, 223, 224, 225,
           226, 227, 582, 583, 584, 585, 586, 587, 228, 229, 230, 231, 232, 233,
           588, 589, 590, 591, 592, 593, 234, 235, 236, 237, 238, 239, 594, 595,
           596, 597, 598, 599, 240, 241, 242, 243, 244, 245, 600, 601, 602, 603,
           604, 605, 246, 247, 248, 249, 250, 251, 606, 607, 608, 609, 610, 611,
           252, 253, 254, 255, 256, 257, 612, 613, 614, 615, 616, 617, 258, 259,
           260, 261, 262, 263, 618, 619, 620, 621, 622, 623, 264, 265, 266, 267,
           268, 269, 624, 625, 626, 627, 628, 629, 270, 271, 272, 273, 274, 275,
           630, 631, 632, 633, 634, 635, 276, 277, 278, 279, 280, 281, 636, 637,
           638, 639, 640, 641, 282, 283, 284, 285, 286, 287, 642, 643, 644, 645,
           646, 647, 288, 289, 290, 291, 292, 293, 648, 649, 650, 651, 652, 653,
           294, 295, 296, 297, 298, 299, 654, 655, 656, 657, 658, 659, 300, 301,
           302, 303, 304, 305, 660, 661, 662, 663, 664, 665, 306, 307, 308, 309,
           310, 311, 666, 667, 668, 669, 670, 671, 312, 313, 314, 315, 316, 317,
           672, 673, 674, 675, 676, 677, 318, 319, 320, 321, 322, 323, 678, 679,
           680, 681, 682, 683, 324, 325, 326, 327, 328, 329, 684, 685, 686, 687,
           688, 689, 330, 331, 332, 333, 334, 335, 690, 691, 692, 693, 694, 695,
           336, 337, 338, 339, 340, 341, 696, 697, 698, 699, 700, 701, 342, 343,
           344, 345, 346, 347, 702, 703, 704, 705, 706, 707, 348, 349, 350, 351,
           352, 353, 708, 709, 710, 711, 712, 713, 354, 355, 356, 357, 358, 359,
           714, 715, 716, 717, 718, 719}));
}

TEST(TransposeTest, TestRefOps5D5) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5, 6},
                                /*perms=*/{1, 2, 3, 4, 0}),
      ElementsAreArray(
          {0,   360, 1,   361, 2,   362, 3,   363, 4,   364, 5,   365, 6,   366,
           7,   367, 8,   368, 9,   369, 10,  370, 11,  371, 12,  372, 13,  373,
           14,  374, 15,  375, 16,  376, 17,  377, 18,  378, 19,  379, 20,  380,
           21,  381, 22,  382, 23,  383, 24,  384, 25,  385, 26,  386, 27,  387,
           28,  388, 29,  389, 30,  390, 31,  391, 32,  392, 33,  393, 34,  394,
           35,  395, 36,  396, 37,  397, 38,  398, 39,  399, 40,  400, 41,  401,
           42,  402, 43,  403, 44,  404, 45,  405, 46,  406, 47,  407, 48,  408,
           49,  409, 50,  410, 51,  411, 52,  412, 53,  413, 54,  414, 55,  415,
           56,  416, 57,  417, 58,  418, 59,  419, 60,  420, 61,  421, 62,  422,
           63,  423, 64,  424, 65,  425, 66,  426, 67,  427, 68,  428, 69,  429,
           70,  430, 71,  431, 72,  432, 73,  433, 74,  434, 75,  435, 76,  436,
           77,  437, 78,  438, 79,  439, 80,  440, 81,  441, 82,  442, 83,  443,
           84,  444, 85,  445, 86,  446, 87,  447, 88,  448, 89,  449, 90,  450,
           91,  451, 92,  452, 93,  453, 94,  454, 95,  455, 96,  456, 97,  457,
           98,  458, 99,  459, 100, 460, 101, 461, 102, 462, 103, 463, 104, 464,
           105, 465, 106, 466, 107, 467, 108, 468, 109, 469, 110, 470, 111, 471,
           112, 472, 113, 473, 114, 474, 115, 475, 116, 476, 117, 477, 118, 478,
           119, 479, 120, 480, 121, 481, 122, 482, 123, 483, 124, 484, 125, 485,
           126, 486, 127, 487, 128, 488, 129, 489, 130, 490, 131, 491, 132, 492,
           133, 493, 134, 494, 135, 495, 136, 496, 137, 497, 138, 498, 139, 499,
           140, 500, 141, 501, 142, 502, 143, 503, 144, 504, 145, 505, 146, 506,
           147, 507, 148, 508, 149, 509, 150, 510, 151, 511, 152, 512, 153, 513,
           154, 514, 155, 515, 156, 516, 157, 517, 158, 518, 159, 519, 160, 520,
           161, 521, 162, 522, 163, 523, 164, 524, 165, 525, 166, 526, 167, 527,
           168, 528, 169, 529, 170, 530, 171, 531, 172, 532, 173, 533, 174, 534,
           175, 535, 176, 536, 177, 537, 178, 538, 179, 539, 180, 540, 181, 541,
           182, 542, 183, 543, 184, 544, 185, 545, 186, 546, 187, 547, 188, 548,
           189, 549, 190, 550, 191, 551, 192, 552, 193, 553, 194, 554, 195, 555,
           196, 556, 197, 557, 198, 558, 199, 559, 200, 560, 201, 561, 202, 562,
           203, 563, 204, 564, 205, 565, 206, 566, 207, 567, 208, 568, 209, 569,
           210, 570, 211, 571, 212, 572, 213, 573, 214, 574, 215, 575, 216, 576,
           217, 577, 218, 578, 219, 579, 220, 580, 221, 581, 222, 582, 223, 583,
           224, 584, 225, 585, 226, 586, 227, 587, 228, 588, 229, 589, 230, 590,
           231, 591, 232, 592, 233, 593, 234, 594, 235, 595, 236, 596, 237, 597,
           238, 598, 239, 599, 240, 600, 241, 601, 242, 602, 243, 603, 244, 604,
           245, 605, 246, 606, 247, 607, 248, 608, 249, 609, 250, 610, 251, 611,
           252, 612, 253, 613, 254, 614, 255, 615, 256, 616, 257, 617, 258, 618,
           259, 619, 260, 620, 261, 621, 262, 622, 263, 623, 264, 624, 265, 625,
           266, 626, 267, 627, 268, 628, 269, 629, 270, 630, 271, 631, 272, 632,
           273, 633, 274, 634, 275, 635, 276, 636, 277, 637, 278, 638, 279, 639,
           280, 640, 281, 641, 282, 642, 283, 643, 284, 644, 285, 645, 286, 646,
           287, 647, 288, 648, 289, 649, 290, 650, 291, 651, 292, 652, 293, 653,
           294, 654, 295, 655, 296, 656, 297, 657, 298, 658, 299, 659, 300, 660,
           301, 661, 302, 662, 303, 663, 304, 664, 305, 665, 306, 666, 307, 667,
           308, 668, 309, 669, 310, 670, 311, 671, 312, 672, 313, 673, 314, 674,
           315, 675, 316, 676, 317, 677, 318, 678, 319, 679, 320, 680, 321, 681,
           322, 682, 323, 683, 324, 684, 325, 685, 326, 686, 327, 687, 328, 688,
           329, 689, 330, 690, 331, 691, 332, 692, 333, 693, 334, 694, 335, 695,
           336, 696, 337, 697, 338, 698, 339, 699, 340, 700, 341, 701, 342, 702,
           343, 703, 344, 704, 345, 705, 346, 706, 347, 707, 348, 708, 349, 709,
           350, 710, 351, 711, 352, 712, 353, 713, 354, 714, 355, 715, 356, 716,
           357, 717, 358, 718, 359, 719}));
}

TEST(TransposeTest, TestRefOps6D0) {
  const std::vector<float> ref = [] {
    std::vector<float> ref(5040);
    absl::c_iota(ref, 0);
    return ref;
  }();
  EXPECT_THAT(RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5, 6, 7},
                                        /*perms=*/{0, 1, 2, 3, 4, 5}),
              ElementsAreArray(ref));
}

TEST(TransposeTest, TestRefOps6D1) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5, 6, 7},
                                /*perms=*/{5, 4, 3, 2, 1, 0}),
      ElementsAreArray(
          {0,    2520, 840,  3360, 1680, 4200, 210,  2730, 1050, 3570, 1890,
           4410, 420,  2940, 1260, 3780, 2100, 4620, 630,  3150, 1470, 3990,
           2310, 4830, 42,   2562, 882,  3402, 1722, 4242, 252,  2772, 1092,
           3612, 1932, 4452, 462,  2982, 1302, 3822, 2142, 4662, 672,  3192,
           1512, 4032, 2352, 4872, 84,   2604, 924,  3444, 1764, 4284, 294,
           2814, 1134, 3654, 1974, 4494, 504,  3024, 1344, 3864, 2184, 4704,
           714,  3234, 1554, 4074, 2394, 4914, 126,  2646, 966,  3486, 1806,
           4326, 336,  2856, 1176, 3696, 2016, 4536, 546,  3066, 1386, 3906,
           2226, 4746, 756,  3276, 1596, 4116, 2436, 4956, 168,  2688, 1008,
           3528, 1848, 4368, 378,  2898, 1218, 3738, 2058, 4578, 588,  3108,
           1428, 3948, 2268, 4788, 798,  3318, 1638, 4158, 2478, 4998, 7,
           2527, 847,  3367, 1687, 4207, 217,  2737, 1057, 3577, 1897, 4417,
           427,  2947, 1267, 3787, 2107, 4627, 637,  3157, 1477, 3997, 2317,
           4837, 49,   2569, 889,  3409, 1729, 4249, 259,  2779, 1099, 3619,
           1939, 4459, 469,  2989, 1309, 3829, 2149, 4669, 679,  3199, 1519,
           4039, 2359, 4879, 91,   2611, 931,  3451, 1771, 4291, 301,  2821,
           1141, 3661, 1981, 4501, 511,  3031, 1351, 3871, 2191, 4711, 721,
           3241, 1561, 4081, 2401, 4921, 133,  2653, 973,  3493, 1813, 4333,
           343,  2863, 1183, 3703, 2023, 4543, 553,  3073, 1393, 3913, 2233,
           4753, 763,  3283, 1603, 4123, 2443, 4963, 175,  2695, 1015, 3535,
           1855, 4375, 385,  2905, 1225, 3745, 2065, 4585, 595,  3115, 1435,
           3955, 2275, 4795, 805,  3325, 1645, 4165, 2485, 5005, 14,   2534,
           854,  3374, 1694, 4214, 224,  2744, 1064, 3584, 1904, 4424, 434,
           2954, 1274, 3794, 2114, 4634, 644,  3164, 1484, 4004, 2324, 4844,
           56,   2576, 896,  3416, 1736, 4256, 266,  2786, 1106, 3626, 1946,
           4466, 476,  2996, 1316, 3836, 2156, 4676, 686,  3206, 1526, 4046,
           2366, 4886, 98,   2618, 938,  3458, 1778, 4298, 308,  2828, 1148,
           3668, 1988, 4508, 518,  3038, 1358, 3878, 2198, 4718, 728,  3248,
           1568, 4088, 2408, 4928, 140,  2660, 980,  3500, 1820, 4340, 350,
           2870, 1190, 3710, 2030, 4550, 560,  3080, 1400, 3920, 2240, 4760,
           770,  3290, 1610, 4130, 2450, 4970, 182,  2702, 1022, 3542, 1862,
           4382, 392,  2912, 1232, 3752, 2072, 4592, 602,  3122, 1442, 3962,
           2282, 4802, 812,  3332, 1652, 4172, 2492, 5012, 21,   2541, 861,
           3381, 1701, 4221, 231,  2751, 1071, 3591, 1911, 4431, 441,  2961,
           1281, 3801, 2121, 4641, 651,  3171, 1491, 4011, 2331, 4851, 63,
           2583, 903,  3423, 1743, 4263, 273,  2793, 1113, 3633, 1953, 4473,
           483,  3003, 1323, 3843, 2163, 4683, 693,  3213, 1533, 4053, 2373,
           4893, 105,  2625, 945,  3465, 1785, 4305, 315,  2835, 1155, 3675,
           1995, 4515, 525,  3045, 1365, 3885, 2205, 4725, 735,  3255, 1575,
           4095, 2415, 4935, 147,  2667, 987,  3507, 1827, 4347, 357,  2877,
           1197, 3717, 2037, 4557, 567,  3087, 1407, 3927, 2247, 4767, 777,
           3297, 1617, 4137, 2457, 4977, 189,  2709, 1029, 3549, 1869, 4389,
           399,  2919, 1239, 3759, 2079, 4599, 609,  3129, 1449, 3969, 2289,
           4809, 819,  3339, 1659, 4179, 2499, 5019, 28,   2548, 868,  3388,
           1708, 4228, 238,  2758, 1078, 3598, 1918, 4438, 448,  2968, 1288,
           3808, 2128, 4648, 658,  3178, 1498, 4018, 2338, 4858, 70,   2590,
           910,  3430, 1750, 4270, 280,  2800, 1120, 3640, 1960, 4480, 490,
           3010, 1330, 3850, 2170, 4690, 700,  3220, 1540, 4060, 2380, 4900,
           112,  2632, 952,  3472, 1792, 4312, 322,  2842, 1162, 3682, 2002,
           4522, 532,  3052, 1372, 3892, 2212, 4732, 742,  3262, 1582, 4102,
           2422, 4942, 154,  2674, 994,  3514, 1834, 4354, 364,  2884, 1204,
           3724, 2044, 4564, 574,  3094, 1414, 3934, 2254, 4774, 784,  3304,
           1624, 4144, 2464, 4984, 196,  2716, 1036, 3556, 1876, 4396, 406,
           2926, 1246, 3766, 2086, 4606, 616,  3136, 1456, 3976, 2296, 4816,
           826,  3346, 1666, 4186, 2506, 5026, 35,   2555, 875,  3395, 1715,
           4235, 245,  2765, 1085, 3605, 1925, 4445, 455,  2975, 1295, 3815,
           2135, 4655, 665,  3185, 1505, 4025, 2345, 4865, 77,   2597, 917,
           3437, 1757, 4277, 287,  2807, 1127, 3647, 1967, 4487, 497,  3017,
           1337, 3857, 2177, 4697, 707,  3227, 1547, 4067, 2387, 4907, 119,
           2639, 959,  3479, 1799, 4319, 329,  2849, 1169, 3689, 2009, 4529,
           539,  3059, 1379, 3899, 2219, 4739, 749,  3269, 1589, 4109, 2429,
           4949, 161,  2681, 1001, 3521, 1841, 4361, 371,  2891, 1211, 3731,
           2051, 4571, 581,  3101, 1421, 3941, 2261, 4781, 791,  3311, 1631,
           4151, 2471, 4991, 203,  2723, 1043, 3563, 1883, 4403, 413,  2933,
           1253, 3773, 2093, 4613, 623,  3143, 1463, 3983, 2303, 4823, 833,
           3353, 1673, 4193, 2513, 5033, 1,    2521, 841,  3361, 1681, 4201,
           211,  2731, 1051, 3571, 1891, 4411, 421,  2941, 1261, 3781, 2101,
           4621, 631,  3151, 1471, 3991, 2311, 4831, 43,   2563, 883,  3403,
           1723, 4243, 253,  2773, 1093, 3613, 1933, 4453, 463,  2983, 1303,
           3823, 2143, 4663, 673,  3193, 1513, 4033, 2353, 4873, 85,   2605,
           925,  3445, 1765, 4285, 295,  2815, 1135, 3655, 1975, 4495, 505,
           3025, 1345, 3865, 2185, 4705, 715,  3235, 1555, 4075, 2395, 4915,
           127,  2647, 967,  3487, 1807, 4327, 337,  2857, 1177, 3697, 2017,
           4537, 547,  3067, 1387, 3907, 2227, 4747, 757,  3277, 1597, 4117,
           2437, 4957, 169,  2689, 1009, 3529, 1849, 4369, 379,  2899, 1219,
           3739, 2059, 4579, 589,  3109, 1429, 3949, 2269, 4789, 799,  3319,
           1639, 4159, 2479, 4999, 8,    2528, 848,  3368, 1688, 4208, 218,
           2738, 1058, 3578, 1898, 4418, 428,  2948, 1268, 3788, 2108, 4628,
           638,  3158, 1478, 3998, 2318, 4838, 50,   2570, 890,  3410, 1730,
           4250, 260,  2780, 1100, 3620, 1940, 4460, 470,  2990, 1310, 3830,
           2150, 4670, 680,  3200, 1520, 4040, 2360, 4880, 92,   2612, 932,
           3452, 1772, 4292, 302,  2822, 1142, 3662, 1982, 4502, 512,  3032,
           1352, 3872, 2192, 4712, 722,  3242, 1562, 4082, 2402, 4922, 134,
           2654, 974,  3494, 1814, 4334, 344,  2864, 1184, 3704, 2024, 4544,
           554,  3074, 1394, 3914, 2234, 4754, 764,  3284, 1604, 4124, 2444,
           4964, 176,  2696, 1016, 3536, 1856, 4376, 386,  2906, 1226, 3746,
           2066, 4586, 596,  3116, 1436, 3956, 2276, 4796, 806,  3326, 1646,
           4166, 2486, 5006, 15,   2535, 855,  3375, 1695, 4215, 225,  2745,
           1065, 3585, 1905, 4425, 435,  2955, 1275, 3795, 2115, 4635, 645,
           3165, 1485, 4005, 2325, 4845, 57,   2577, 897,  3417, 1737, 4257,
           267,  2787, 1107, 3627, 1947, 4467, 477,  2997, 1317, 3837, 2157,
           4677, 687,  3207, 1527, 4047, 2367, 4887, 99,   2619, 939,  3459,
           1779, 4299, 309,  2829, 1149, 3669, 1989, 4509, 519,  3039, 1359,
           3879, 2199, 4719, 729,  3249, 1569, 4089, 2409, 4929, 141,  2661,
           981,  3501, 1821, 4341, 351,  2871, 1191, 3711, 2031, 4551, 561,
           3081, 1401, 3921, 2241, 4761, 771,  3291, 1611, 4131, 2451, 4971,
           183,  2703, 1023, 3543, 1863, 4383, 393,  2913, 1233, 3753, 2073,
           4593, 603,  3123, 1443, 3963, 2283, 4803, 813,  3333, 1653, 4173,
           2493, 5013, 22,   2542, 862,  3382, 1702, 4222, 232,  2752, 1072,
           3592, 1912, 4432, 442,  2962, 1282, 3802, 2122, 4642, 652,  3172,
           1492, 4012, 2332, 4852, 64,   2584, 904,  3424, 1744, 4264, 274,
           2794, 1114, 3634, 1954, 4474, 484,  3004, 1324, 3844, 2164, 4684,
           694,  3214, 1534, 4054, 2374, 4894, 106,  2626, 946,  3466, 1786,
           4306, 316,  2836, 1156, 3676, 1996, 4516, 526,  3046, 1366, 3886,
           2206, 4726, 736,  3256, 1576, 4096, 2416, 4936, 148,  2668, 988,
           3508, 1828, 4348, 358,  2878, 1198, 3718, 2038, 4558, 568,  3088,
           1408, 3928, 2248, 4768, 778,  3298, 1618, 4138, 2458, 4978, 190,
           2710, 1030, 3550, 1870, 4390, 400,  2920, 1240, 3760, 2080, 4600,
           610,  3130, 1450, 3970, 2290, 4810, 820,  3340, 1660, 4180, 2500,
           5020, 29,   2549, 869,  3389, 1709, 4229, 239,  2759, 1079, 3599,
           1919, 4439, 449,  2969, 1289, 3809, 2129, 4649, 659,  3179, 1499,
           4019, 2339, 4859, 71,   2591, 911,  3431, 1751, 4271, 281,  2801,
           1121, 3641, 1961, 4481, 491,  3011, 1331, 3851, 2171, 4691, 701,
           3221, 1541, 4061, 2381, 4901, 113,  2633, 953,  3473, 1793, 4313,
           323,  2843, 1163, 3683, 2003, 4523, 533,  3053, 1373, 3893, 2213,
           4733, 743,  3263, 1583, 4103, 2423, 4943, 155,  2675, 995,  3515,
           1835, 4355, 365,  2885, 1205, 3725, 2045, 4565, 575,  3095, 1415,
           3935, 2255, 4775, 785,  3305, 1625, 4145, 2465, 4985, 197,  2717,
           1037, 3557, 1877, 4397, 407,  2927, 1247, 3767, 2087, 4607, 617,
           3137, 1457, 3977, 2297, 4817, 827,  3347, 1667, 4187, 2507, 5027,
           36,   2556, 876,  3396, 1716, 4236, 246,  2766, 1086, 3606, 1926,
           4446, 456,  2976, 1296, 3816, 2136, 4656, 666,  3186, 1506, 4026,
           2346, 4866, 78,   2598, 918,  3438, 1758, 4278, 288,  2808, 1128,
           3648, 1968, 4488, 498,  3018, 1338, 3858, 2178, 4698, 708,  3228,
           1548, 4068, 2388, 4908, 120,  2640, 960,  3480, 1800, 4320, 330,
           2850, 1170, 3690, 2010, 4530, 540,  3060, 1380, 3900, 2220, 4740,
           750,  3270, 1590, 4110, 2430, 4950, 162,  2682, 1002, 3522, 1842,
           4362, 372,  2892, 1212, 3732, 2052, 4572, 582,  3102, 1422, 3942,
           2262, 4782, 792,  3312, 1632, 4152, 2472, 4992, 204,  2724, 1044,
           3564, 1884, 4404, 414,  2934, 1254, 3774, 2094, 4614, 624,  3144,
           1464, 3984, 2304, 4824, 834,  3354, 1674, 4194, 2514, 5034, 2,
           2522, 842,  3362, 1682, 4202, 212,  2732, 1052, 3572, 1892, 4412,
           422,  2942, 1262, 3782, 2102, 4622, 632,  3152, 1472, 3992, 2312,
           4832, 44,   2564, 884,  3404, 1724, 4244, 254,  2774, 1094, 3614,
           1934, 4454, 464,  2984, 1304, 3824, 2144, 4664, 674,  3194, 1514,
           4034, 2354, 4874, 86,   2606, 926,  3446, 1766, 4286, 296,  2816,
           1136, 3656, 1976, 4496, 506,  3026, 1346, 3866, 2186, 4706, 716,
           3236, 1556, 4076, 2396, 4916, 128,  2648, 968,  3488, 1808, 4328,
           338,  2858, 1178, 3698, 2018, 4538, 548,  3068, 1388, 3908, 2228,
           4748, 758,  3278, 1598, 4118, 2438, 4958, 170,  2690, 1010, 3530,
           1850, 4370, 380,  2900, 1220, 3740, 2060, 4580, 590,  3110, 1430,
           3950, 2270, 4790, 800,  3320, 1640, 4160, 2480, 5000, 9,    2529,
           849,  3369, 1689, 4209, 219,  2739, 1059, 3579, 1899, 4419, 429,
           2949, 1269, 3789, 2109, 4629, 639,  3159, 1479, 3999, 2319, 4839,
           51,   2571, 891,  3411, 1731, 4251, 261,  2781, 1101, 3621, 1941,
           4461, 471,  2991, 1311, 3831, 2151, 4671, 681,  3201, 1521, 4041,
           2361, 4881, 93,   2613, 933,  3453, 1773, 4293, 303,  2823, 1143,
           3663, 1983, 4503, 513,  3033, 1353, 3873, 2193, 4713, 723,  3243,
           1563, 4083, 2403, 4923, 135,  2655, 975,  3495, 1815, 4335, 345,
           2865, 1185, 3705, 2025, 4545, 555,  3075, 1395, 3915, 2235, 4755,
           765,  3285, 1605, 4125, 2445, 4965, 177,  2697, 1017, 3537, 1857,
           4377, 387,  2907, 1227, 3747, 2067, 4587, 597,  3117, 1437, 3957,
           2277, 4797, 807,  3327, 1647, 4167, 2487, 5007, 16,   2536, 856,
           3376, 1696, 4216, 226,  2746, 1066, 3586, 1906, 4426, 436,  2956,
           1276, 3796, 2116, 4636, 646,  3166, 1486, 4006, 2326, 4846, 58,
           2578, 898,  3418, 1738, 4258, 268,  2788, 1108, 3628, 1948, 4468,
           478,  2998, 1318, 3838, 2158, 4678, 688,  3208, 1528, 4048, 2368,
           4888, 100,  2620, 940,  3460, 1780, 4300, 310,  2830, 1150, 3670,
           1990, 4510, 520,  3040, 1360, 3880, 2200, 4720, 730,  3250, 1570,
           4090, 2410, 4930, 142,  2662, 982,  3502, 1822, 4342, 352,  2872,
           1192, 3712, 2032, 4552, 562,  3082, 1402, 3922, 2242, 4762, 772,
           3292, 1612, 4132, 2452, 4972, 184,  2704, 1024, 3544, 1864, 4384,
           394,  2914, 1234, 3754, 2074, 4594, 604,  3124, 1444, 3964, 2284,
           4804, 814,  3334, 1654, 4174, 2494, 5014, 23,   2543, 863,  3383,
           1703, 4223, 233,  2753, 1073, 3593, 1913, 4433, 443,  2963, 1283,
           3803, 2123, 4643, 653,  3173, 1493, 4013, 2333, 4853, 65,   2585,
           905,  3425, 1745, 4265, 275,  2795, 1115, 3635, 1955, 4475, 485,
           3005, 1325, 3845, 2165, 4685, 695,  3215, 1535, 4055, 2375, 4895,
           107,  2627, 947,  3467, 1787, 4307, 317,  2837, 1157, 3677, 1997,
           4517, 527,  3047, 1367, 3887, 2207, 4727, 737,  3257, 1577, 4097,
           2417, 4937, 149,  2669, 989,  3509, 1829, 4349, 359,  2879, 1199,
           3719, 2039, 4559, 569,  3089, 1409, 3929, 2249, 4769, 779,  3299,
           1619, 4139, 2459, 4979, 191,  2711, 1031, 3551, 1871, 4391, 401,
           2921, 1241, 3761, 2081, 4601, 611,  3131, 1451, 3971, 2291, 4811,
           821,  3341, 1661, 4181, 2501, 5021, 30,   2550, 870,  3390, 1710,
           4230, 240,  2760, 1080, 3600, 1920, 4440, 450,  2970, 1290, 3810,
           2130, 4650, 660,  3180, 1500, 4020, 2340, 4860, 72,   2592, 912,
           3432, 1752, 4272, 282,  2802, 1122, 3642, 1962, 4482, 492,  3012,
           1332, 3852, 2172, 4692, 702,  3222, 1542, 4062, 2382, 4902, 114,
           2634, 954,  3474, 1794, 4314, 324,  2844, 1164, 3684, 2004, 4524,
           534,  3054, 1374, 3894, 2214, 4734, 744,  3264, 1584, 4104, 2424,
           4944, 156,  2676, 996,  3516, 1836, 4356, 366,  2886, 1206, 3726,
           2046, 4566, 576,  3096, 1416, 3936, 2256, 4776, 786,  3306, 1626,
           4146, 2466, 4986, 198,  2718, 1038, 3558, 1878, 4398, 408,  2928,
           1248, 3768, 2088, 4608, 618,  3138, 1458, 3978, 2298, 4818, 828,
           3348, 1668, 4188, 2508, 5028, 37,   2557, 877,  3397, 1717, 4237,
           247,  2767, 1087, 3607, 1927, 4447, 457,  2977, 1297, 3817, 2137,
           4657, 667,  3187, 1507, 4027, 2347, 4867, 79,   2599, 919,  3439,
           1759, 4279, 289,  2809, 1129, 3649, 1969, 4489, 499,  3019, 1339,
           3859, 2179, 4699, 709,  3229, 1549, 4069, 2389, 4909, 121,  2641,
           961,  3481, 1801, 4321, 331,  2851, 1171, 3691, 2011, 4531, 541,
           3061, 1381, 3901, 2221, 4741, 751,  3271, 1591, 4111, 2431, 4951,
           163,  2683, 1003, 3523, 1843, 4363, 373,  2893, 1213, 3733, 2053,
           4573, 583,  3103, 1423, 3943, 2263, 4783, 793,  3313, 1633, 4153,
           2473, 4993, 205,  2725, 1045, 3565, 1885, 4405, 415,  2935, 1255,
           3775, 2095, 4615, 625,  3145, 1465, 3985, 2305, 4825, 835,  3355,
           1675, 4195, 2515, 5035, 3,    2523, 843,  3363, 1683, 4203, 213,
           2733, 1053, 3573, 1893, 4413, 423,  2943, 1263, 3783, 2103, 4623,
           633,  3153, 1473, 3993, 2313, 4833, 45,   2565, 885,  3405, 1725,
           4245, 255,  2775, 1095, 3615, 1935, 4455, 465,  2985, 1305, 3825,
           2145, 4665, 675,  3195, 1515, 4035, 2355, 4875, 87,   2607, 927,
           3447, 1767, 4287, 297,  2817, 1137, 3657, 1977, 4497, 507,  3027,
           1347, 3867, 2187, 4707, 717,  3237, 1557, 4077, 2397, 4917, 129,
           2649, 969,  3489, 1809, 4329, 339,  2859, 1179, 3699, 2019, 4539,
           549,  3069, 1389, 3909, 2229, 4749, 759,  3279, 1599, 4119, 2439,
           4959, 171,  2691, 1011, 3531, 1851, 4371, 381,  2901, 1221, 3741,
           2061, 4581, 591,  3111, 1431, 3951, 2271, 4791, 801,  3321, 1641,
           4161, 2481, 5001, 10,   2530, 850,  3370, 1690, 4210, 220,  2740,
           1060, 3580, 1900, 4420, 430,  2950, 1270, 3790, 2110, 4630, 640,
           3160, 1480, 4000, 2320, 4840, 52,   2572, 892,  3412, 1732, 4252,
           262,  2782, 1102, 3622, 1942, 4462, 472,  2992, 1312, 3832, 2152,
           4672, 682,  3202, 1522, 4042, 2362, 4882, 94,   2614, 934,  3454,
           1774, 4294, 304,  2824, 1144, 3664, 1984, 4504, 514,  3034, 1354,
           3874, 2194, 4714, 724,  3244, 1564, 4084, 2404, 4924, 136,  2656,
           976,  3496, 1816, 4336, 346,  2866, 1186, 3706, 2026, 4546, 556,
           3076, 1396, 3916, 2236, 4756, 766,  3286, 1606, 4126, 2446, 4966,
           178,  2698, 1018, 3538, 1858, 4378, 388,  2908, 1228, 3748, 2068,
           4588, 598,  3118, 1438, 3958, 2278, 4798, 808,  3328, 1648, 4168,
           2488, 5008, 17,   2537, 857,  3377, 1697, 4217, 227,  2747, 1067,
           3587, 1907, 4427, 437,  2957, 1277, 3797, 2117, 4637, 647,  3167,
           1487, 4007, 2327, 4847, 59,   2579, 899,  3419, 1739, 4259, 269,
           2789, 1109, 3629, 1949, 4469, 479,  2999, 1319, 3839, 2159, 4679,
           689,  3209, 1529, 4049, 2369, 4889, 101,  2621, 941,  3461, 1781,
           4301, 311,  2831, 1151, 3671, 1991, 4511, 521,  3041, 1361, 3881,
           2201, 4721, 731,  3251, 1571, 4091, 2411, 4931, 143,  2663, 983,
           3503, 1823, 4343, 353,  2873, 1193, 3713, 2033, 4553, 563,  3083,
           1403, 3923, 2243, 4763, 773,  3293, 1613, 4133, 2453, 4973, 185,
           2705, 1025, 3545, 1865, 4385, 395,  2915, 1235, 3755, 2075, 4595,
           605,  3125, 1445, 3965, 2285, 4805, 815,  3335, 1655, 4175, 2495,
           5015, 24,   2544, 864,  3384, 1704, 4224, 234,  2754, 1074, 3594,
           1914, 4434, 444,  2964, 1284, 3804, 2124, 4644, 654,  3174, 1494,
           4014, 2334, 4854, 66,   2586, 906,  3426, 1746, 4266, 276,  2796,
           1116, 3636, 1956, 4476, 486,  3006, 1326, 3846, 2166, 4686, 696,
           3216, 1536, 4056, 2376, 4896, 108,  2628, 948,  3468, 1788, 4308,
           318,  2838, 1158, 3678, 1998, 4518, 528,  3048, 1368, 3888, 2208,
           4728, 738,  3258, 1578, 4098, 2418, 4938, 150,  2670, 990,  3510,
           1830, 4350, 360,  2880, 1200, 3720, 2040, 4560, 570,  3090, 1410,
           3930, 2250, 4770, 780,  3300, 1620, 4140, 2460, 4980, 192,  2712,
           1032, 3552, 1872, 4392, 402,  2922, 1242, 3762, 2082, 4602, 612,
           3132, 1452, 3972, 2292, 4812, 822,  3342, 1662, 4182, 2502, 5022,
           31,   2551, 871,  3391, 1711, 4231, 241,  2761, 1081, 3601, 1921,
           4441, 451,  2971, 1291, 3811, 2131, 4651, 661,  3181, 1501, 4021,
           2341, 4861, 73,   2593, 913,  3433, 1753, 4273, 283,  2803, 1123,
           3643, 1963, 4483, 493,  3013, 1333, 3853, 2173, 4693, 703,  3223,
           1543, 4063, 2383, 4903, 115,  2635, 955,  3475, 1795, 4315, 325,
           2845, 1165, 3685, 2005, 4525, 535,  3055, 1375, 3895, 2215, 4735,
           745,  3265, 1585, 4105, 2425, 4945, 157,  2677, 997,  3517, 1837,
           4357, 367,  2887, 1207, 3727, 2047, 4567, 577,  3097, 1417, 3937,
           2257, 4777, 787,  3307, 1627, 4147, 2467, 4987, 199,  2719, 1039,
           3559, 1879, 4399, 409,  2929, 1249, 3769, 2089, 4609, 619,  3139,
           1459, 3979, 2299, 4819, 829,  3349, 1669, 4189, 2509, 5029, 38,
           2558, 878,  3398, 1718, 4238, 248,  2768, 1088, 3608, 1928, 4448,
           458,  2978, 1298, 3818, 2138, 4658, 668,  3188, 1508, 4028, 2348,
           4868, 80,   2600, 920,  3440, 1760, 4280, 290,  2810, 1130, 3650,
           1970, 4490, 500,  3020, 1340, 3860, 2180, 4700, 710,  3230, 1550,
           4070, 2390, 4910, 122,  2642, 962,  3482, 1802, 4322, 332,  2852,
           1172, 3692, 2012, 4532, 542,  3062, 1382, 3902, 2222, 4742, 752,
           3272, 1592, 4112, 2432, 4952, 164,  2684, 1004, 3524, 1844, 4364,
           374,  2894, 1214, 3734, 2054, 4574, 584,  3104, 1424, 3944, 2264,
           4784, 794,  3314, 1634, 4154, 2474, 4994, 206,  2726, 1046, 3566,
           1886, 4406, 416,  2936, 1256, 3776, 2096, 4616, 626,  3146, 1466,
           3986, 2306, 4826, 836,  3356, 1676, 4196, 2516, 5036, 4,    2524,
           844,  3364, 1684, 4204, 214,  2734, 1054, 3574, 1894, 4414, 424,
           2944, 1264, 3784, 2104, 4624, 634,  3154, 1474, 3994, 2314, 4834,
           46,   2566, 886,  3406, 1726, 4246, 256,  2776, 1096, 3616, 1936,
           4456, 466,  2986, 1306, 3826, 2146, 4666, 676,  3196, 1516, 4036,
           2356, 4876, 88,   2608, 928,  3448, 1768, 4288, 298,  2818, 1138,
           3658, 1978, 4498, 508,  3028, 1348, 3868, 2188, 4708, 718,  3238,
           1558, 4078, 2398, 4918, 130,  2650, 970,  3490, 1810, 4330, 340,
           2860, 1180, 3700, 2020, 4540, 550,  3070, 1390, 3910, 2230, 4750,
           760,  3280, 1600, 4120, 2440, 4960, 172,  2692, 1012, 3532, 1852,
           4372, 382,  2902, 1222, 3742, 2062, 4582, 592,  3112, 1432, 3952,
           2272, 4792, 802,  3322, 1642, 4162, 2482, 5002, 11,   2531, 851,
           3371, 1691, 4211, 221,  2741, 1061, 3581, 1901, 4421, 431,  2951,
           1271, 3791, 2111, 4631, 641,  3161, 1481, 4001, 2321, 4841, 53,
           2573, 893,  3413, 1733, 4253, 263,  2783, 1103, 3623, 1943, 4463,
           473,  2993, 1313, 3833, 2153, 4673, 683,  3203, 1523, 4043, 2363,
           4883, 95,   2615, 935,  3455, 1775, 4295, 305,  2825, 1145, 3665,
           1985, 4505, 515,  3035, 1355, 3875, 2195, 4715, 725,  3245, 1565,
           4085, 2405, 4925, 137,  2657, 977,  3497, 1817, 4337, 347,  2867,
           1187, 3707, 2027, 4547, 557,  3077, 1397, 3917, 2237, 4757, 767,
           3287, 1607, 4127, 2447, 4967, 179,  2699, 1019, 3539, 1859, 4379,
           389,  2909, 1229, 3749, 2069, 4589, 599,  3119, 1439, 3959, 2279,
           4799, 809,  3329, 1649, 4169, 2489, 5009, 18,   2538, 858,  3378,
           1698, 4218, 228,  2748, 1068, 3588, 1908, 4428, 438,  2958, 1278,
           3798, 2118, 4638, 648,  3168, 1488, 4008, 2328, 4848, 60,   2580,
           900,  3420, 1740, 4260, 270,  2790, 1110, 3630, 1950, 4470, 480,
           3000, 1320, 3840, 2160, 4680, 690,  3210, 1530, 4050, 2370, 4890,
           102,  2622, 942,  3462, 1782, 4302, 312,  2832, 1152, 3672, 1992,
           4512, 522,  3042, 1362, 3882, 2202, 4722, 732,  3252, 1572, 4092,
           2412, 4932, 144,  2664, 984,  3504, 1824, 4344, 354,  2874, 1194,
           3714, 2034, 4554, 564,  3084, 1404, 3924, 2244, 4764, 774,  3294,
           1614, 4134, 2454, 4974, 186,  2706, 1026, 3546, 1866, 4386, 396,
           2916, 1236, 3756, 2076, 4596, 606,  3126, 1446, 3966, 2286, 4806,
           816,  3336, 1656, 4176, 2496, 5016, 25,   2545, 865,  3385, 1705,
           4225, 235,  2755, 1075, 3595, 1915, 4435, 445,  2965, 1285, 3805,
           2125, 4645, 655,  3175, 1495, 4015, 2335, 4855, 67,   2587, 907,
           3427, 1747, 4267, 277,  2797, 1117, 3637, 1957, 4477, 487,  3007,
           1327, 3847, 2167, 4687, 697,  3217, 1537, 4057, 2377, 4897, 109,
           2629, 949,  3469, 1789, 4309, 319,  2839, 1159, 3679, 1999, 4519,
           529,  3049, 1369, 3889, 2209, 4729, 739,  3259, 1579, 4099, 2419,
           4939, 151,  2671, 991,  3511, 1831, 4351, 361,  2881, 1201, 3721,
           2041, 4561, 571,  3091, 1411, 3931, 2251, 4771, 781,  3301, 1621,
           4141, 2461, 4981, 193,  2713, 1033, 3553, 1873, 4393, 403,  2923,
           1243, 3763, 2083, 4603, 613,  3133, 1453, 3973, 2293, 4813, 823,
           3343, 1663, 4183, 2503, 5023, 32,   2552, 872,  3392, 1712, 4232,
           242,  2762, 1082, 3602, 1922, 4442, 452,  2972, 1292, 3812, 2132,
           4652, 662,  3182, 1502, 4022, 2342, 4862, 74,   2594, 914,  3434,
           1754, 4274, 284,  2804, 1124, 3644, 1964, 4484, 494,  3014, 1334,
           3854, 2174, 4694, 704,  3224, 1544, 4064, 2384, 4904, 116,  2636,
           956,  3476, 1796, 4316, 326,  2846, 1166, 3686, 2006, 4526, 536,
           3056, 1376, 3896, 2216, 4736, 746,  3266, 1586, 4106, 2426, 4946,
           158,  2678, 998,  3518, 1838, 4358, 368,  2888, 1208, 3728, 2048,
           4568, 578,  3098, 1418, 3938, 2258, 4778, 788,  3308, 1628, 4148,
           2468, 4988, 200,  2720, 1040, 3560, 1880, 4400, 410,  2930, 1250,
           3770, 2090, 4610, 620,  3140, 1460, 3980, 2300, 4820, 830,  3350,
           1670, 4190, 2510, 5030, 39,   2559, 879,  3399, 1719, 4239, 249,
           2769, 1089, 3609, 1929, 4449, 459,  2979, 1299, 3819, 2139, 4659,
           669,  3189, 1509, 4029, 2349, 4869, 81,   2601, 921,  3441, 1761,
           4281, 291,  2811, 1131, 3651, 1971, 4491, 501,  3021, 1341, 3861,
           2181, 4701, 711,  3231, 1551, 4071, 2391, 4911, 123,  2643, 963,
           3483, 1803, 4323, 333,  2853, 1173, 3693, 2013, 4533, 543,  3063,
           1383, 3903, 2223, 4743, 753,  3273, 1593, 4113, 2433, 4953, 165,
           2685, 1005, 3525, 1845, 4365, 375,  2895, 1215, 3735, 2055, 4575,
           585,  3105, 1425, 3945, 2265, 4785, 795,  3315, 1635, 4155, 2475,
           4995, 207,  2727, 1047, 3567, 1887, 4407, 417,  2937, 1257, 3777,
           2097, 4617, 627,  3147, 1467, 3987, 2307, 4827, 837,  3357, 1677,
           4197, 2517, 5037, 5,    2525, 845,  3365, 1685, 4205, 215,  2735,
           1055, 3575, 1895, 4415, 425,  2945, 1265, 3785, 2105, 4625, 635,
           3155, 1475, 3995, 2315, 4835, 47,   2567, 887,  3407, 1727, 4247,
           257,  2777, 1097, 3617, 1937, 4457, 467,  2987, 1307, 3827, 2147,
           4667, 677,  3197, 1517, 4037, 2357, 4877, 89,   2609, 929,  3449,
           1769, 4289, 299,  2819, 1139, 3659, 1979, 4499, 509,  3029, 1349,
           3869, 2189, 4709, 719,  3239, 1559, 4079, 2399, 4919, 131,  2651,
           971,  3491, 1811, 4331, 341,  2861, 1181, 3701, 2021, 4541, 551,
           3071, 1391, 3911, 2231, 4751, 761,  3281, 1601, 4121, 2441, 4961,
           173,  2693, 1013, 3533, 1853, 4373, 383,  2903, 1223, 3743, 2063,
           4583, 593,  3113, 1433, 3953, 2273, 4793, 803,  3323, 1643, 4163,
           2483, 5003, 12,   2532, 852,  3372, 1692, 4212, 222,  2742, 1062,
           3582, 1902, 4422, 432,  2952, 1272, 3792, 2112, 4632, 642,  3162,
           1482, 4002, 2322, 4842, 54,   2574, 894,  3414, 1734, 4254, 264,
           2784, 1104, 3624, 1944, 4464, 474,  2994, 1314, 3834, 2154, 4674,
           684,  3204, 1524, 4044, 2364, 4884, 96,   2616, 936,  3456, 1776,
           4296, 306,  2826, 1146, 3666, 1986, 4506, 516,  3036, 1356, 3876,
           2196, 4716, 726,  3246, 1566, 4086, 2406, 4926, 138,  2658, 978,
           3498, 1818, 4338, 348,  2868, 1188, 3708, 2028, 4548, 558,  3078,
           1398, 3918, 2238, 4758, 768,  3288, 1608, 4128, 2448, 4968, 180,
           2700, 1020, 3540, 1860, 4380, 390,  2910, 1230, 3750, 2070, 4590,
           600,  3120, 1440, 3960, 2280, 4800, 810,  3330, 1650, 4170, 2490,
           5010, 19,   2539, 859,  3379, 1699, 4219, 229,  2749, 1069, 3589,
           1909, 4429, 439,  2959, 1279, 3799, 2119, 4639, 649,  3169, 1489,
           4009, 2329, 4849, 61,   2581, 901,  3421, 1741, 4261, 271,  2791,
           1111, 3631, 1951, 4471, 481,  3001, 1321, 3841, 2161, 4681, 691,
           3211, 1531, 4051, 2371, 4891, 103,  2623, 943,  3463, 1783, 4303,
           313,  2833, 1153, 3673, 1993, 4513, 523,  3043, 1363, 3883, 2203,
           4723, 733,  3253, 1573, 4093, 2413, 4933, 145,  2665, 985,  3505,
           1825, 4345, 355,  2875, 1195, 3715, 2035, 4555, 565,  3085, 1405,
           3925, 2245, 4765, 775,  3295, 1615, 4135, 2455, 4975, 187,  2707,
           1027, 3547, 1867, 4387, 397,  2917, 1237, 3757, 2077, 4597, 607,
           3127, 1447, 3967, 2287, 4807, 817,  3337, 1657, 4177, 2497, 5017,
           26,   2546, 866,  3386, 1706, 4226, 236,  2756, 1076, 3596, 1916,
           4436, 446,  2966, 1286, 3806, 2126, 4646, 656,  3176, 1496, 4016,
           2336, 4856, 68,   2588, 908,  3428, 1748, 4268, 278,  2798, 1118,
           3638, 1958, 4478, 488,  3008, 1328, 3848, 2168, 4688, 698,  3218,
           1538, 4058, 2378, 4898, 110,  2630, 950,  3470, 1790, 4310, 320,
           2840, 1160, 3680, 2000, 4520, 530,  3050, 1370, 3890, 2210, 4730,
           740,  3260, 1580, 4100, 2420, 4940, 152,  2672, 992,  3512, 1832,
           4352, 362,  2882, 1202, 3722, 2042, 4562, 572,  3092, 1412, 3932,
           2252, 4772, 782,  3302, 1622, 4142, 2462, 4982, 194,  2714, 1034,
           3554, 1874, 4394, 404,  2924, 1244, 3764, 2084, 4604, 614,  3134,
           1454, 3974, 2294, 4814, 824,  3344, 1664, 4184, 2504, 5024, 33,
           2553, 873,  3393, 1713, 4233, 243,  2763, 1083, 3603, 1923, 4443,
           453,  2973, 1293, 3813, 2133, 4653, 663,  3183, 1503, 4023, 2343,
           4863, 75,   2595, 915,  3435, 1755, 4275, 285,  2805, 1125, 3645,
           1965, 4485, 495,  3015, 1335, 3855, 2175, 4695, 705,  3225, 1545,
           4065, 2385, 4905, 117,  2637, 957,  3477, 1797, 4317, 327,  2847,
           1167, 3687, 2007, 4527, 537,  3057, 1377, 3897, 2217, 4737, 747,
           3267, 1587, 4107, 2427, 4947, 159,  2679, 999,  3519, 1839, 4359,
           369,  2889, 1209, 3729, 2049, 4569, 579,  3099, 1419, 3939, 2259,
           4779, 789,  3309, 1629, 4149, 2469, 4989, 201,  2721, 1041, 3561,
           1881, 4401, 411,  2931, 1251, 3771, 2091, 4611, 621,  3141, 1461,
           3981, 2301, 4821, 831,  3351, 1671, 4191, 2511, 5031, 40,   2560,
           880,  3400, 1720, 4240, 250,  2770, 1090, 3610, 1930, 4450, 460,
           2980, 1300, 3820, 2140, 4660, 670,  3190, 1510, 4030, 2350, 4870,
           82,   2602, 922,  3442, 1762, 4282, 292,  2812, 1132, 3652, 1972,
           4492, 502,  3022, 1342, 3862, 2182, 4702, 712,  3232, 1552, 4072,
           2392, 4912, 124,  2644, 964,  3484, 1804, 4324, 334,  2854, 1174,
           3694, 2014, 4534, 544,  3064, 1384, 3904, 2224, 4744, 754,  3274,
           1594, 4114, 2434, 4954, 166,  2686, 1006, 3526, 1846, 4366, 376,
           2896, 1216, 3736, 2056, 4576, 586,  3106, 1426, 3946, 2266, 4786,
           796,  3316, 1636, 4156, 2476, 4996, 208,  2728, 1048, 3568, 1888,
           4408, 418,  2938, 1258, 3778, 2098, 4618, 628,  3148, 1468, 3988,
           2308, 4828, 838,  3358, 1678, 4198, 2518, 5038, 6,    2526, 846,
           3366, 1686, 4206, 216,  2736, 1056, 3576, 1896, 4416, 426,  2946,
           1266, 3786, 2106, 4626, 636,  3156, 1476, 3996, 2316, 4836, 48,
           2568, 888,  3408, 1728, 4248, 258,  2778, 1098, 3618, 1938, 4458,
           468,  2988, 1308, 3828, 2148, 4668, 678,  3198, 1518, 4038, 2358,
           4878, 90,   2610, 930,  3450, 1770, 4290, 300,  2820, 1140, 3660,
           1980, 4500, 510,  3030, 1350, 3870, 2190, 4710, 720,  3240, 1560,
           4080, 2400, 4920, 132,  2652, 972,  3492, 1812, 4332, 342,  2862,
           1182, 3702, 2022, 4542, 552,  3072, 1392, 3912, 2232, 4752, 762,
           3282, 1602, 4122, 2442, 4962, 174,  2694, 1014, 3534, 1854, 4374,
           384,  2904, 1224, 3744, 2064, 4584, 594,  3114, 1434, 3954, 2274,
           4794, 804,  3324, 1644, 4164, 2484, 5004, 13,   2533, 853,  3373,
           1693, 4213, 223,  2743, 1063, 3583, 1903, 4423, 433,  2953, 1273,
           3793, 2113, 4633, 643,  3163, 1483, 4003, 2323, 4843, 55,   2575,
           895,  3415, 1735, 4255, 265,  2785, 1105, 3625, 1945, 4465, 475,
           2995, 1315, 3835, 2155, 4675, 685,  3205, 1525, 4045, 2365, 4885,
           97,   2617, 937,  3457, 1777, 4297, 307,  2827, 1147, 3667, 1987,
           4507, 517,  3037, 1357, 3877, 2197, 4717, 727,  3247, 1567, 4087,
           2407, 4927, 139,  2659, 979,  3499, 1819, 4339, 349,  2869, 1189,
           3709, 2029, 4549, 559,  3079, 1399, 3919, 2239, 4759, 769,  3289,
           1609, 4129, 2449, 4969, 181,  2701, 1021, 3541, 1861, 4381, 391,
           2911, 1231, 3751, 2071, 4591, 601,  3121, 1441, 3961, 2281, 4801,
           811,  3331, 1651, 4171, 2491, 5011, 20,   2540, 860,  3380, 1700,
           4220, 230,  2750, 1070, 3590, 1910, 4430, 440,  2960, 1280, 3800,
           2120, 4640, 650,  3170, 1490, 4010, 2330, 4850, 62,   2582, 902,
           3422, 1742, 4262, 272,  2792, 1112, 3632, 1952, 4472, 482,  3002,
           1322, 3842, 2162, 4682, 692,  3212, 1532, 4052, 2372, 4892, 104,
           2624, 944,  3464, 1784, 4304, 314,  2834, 1154, 3674, 1994, 4514,
           524,  3044, 1364, 3884, 2204, 4724, 734,  3254, 1574, 4094, 2414,
           4934, 146,  2666, 986,  3506, 1826, 4346, 356,  2876, 1196, 3716,
           2036, 4556, 566,  3086, 1406, 3926, 2246, 4766, 776,  3296, 1616,
           4136, 2456, 4976, 188,  2708, 1028, 3548, 1868, 4388, 398,  2918,
           1238, 3758, 2078, 4598, 608,  3128, 1448, 3968, 2288, 4808, 818,
           3338, 1658, 4178, 2498, 5018, 27,   2547, 867,  3387, 1707, 4227,
           237,  2757, 1077, 3597, 1917, 4437, 447,  2967, 1287, 3807, 2127,
           4647, 657,  3177, 1497, 4017, 2337, 4857, 69,   2589, 909,  3429,
           1749, 4269, 279,  2799, 1119, 3639, 1959, 4479, 489,  3009, 1329,
           3849, 2169, 4689, 699,  3219, 1539, 4059, 2379, 4899, 111,  2631,
           951,  3471, 1791, 4311, 321,  2841, 1161, 3681, 2001, 4521, 531,
           3051, 1371, 3891, 2211, 4731, 741,  3261, 1581, 4101, 2421, 4941,
           153,  2673, 993,  3513, 1833, 4353, 363,  2883, 1203, 3723, 2043,
           4563, 573,  3093, 1413, 3933, 2253, 4773, 783,  3303, 1623, 4143,
           2463, 4983, 195,  2715, 1035, 3555, 1875, 4395, 405,  2925, 1245,
           3765, 2085, 4605, 615,  3135, 1455, 3975, 2295, 4815, 825,  3345,
           1665, 4185, 2505, 5025, 34,   2554, 874,  3394, 1714, 4234, 244,
           2764, 1084, 3604, 1924, 4444, 454,  2974, 1294, 3814, 2134, 4654,
           664,  3184, 1504, 4024, 2344, 4864, 76,   2596, 916,  3436, 1756,
           4276, 286,  2806, 1126, 3646, 1966, 4486, 496,  3016, 1336, 3856,
           2176, 4696, 706,  3226, 1546, 4066, 2386, 4906, 118,  2638, 958,
           3478, 1798, 4318, 328,  2848, 1168, 3688, 2008, 4528, 538,  3058,
           1378, 3898, 2218, 4738, 748,  3268, 1588, 4108, 2428, 4948, 160,
           2680, 1000, 3520, 1840, 4360, 370,  2890, 1210, 3730, 2050, 4570,
           580,  3100, 1420, 3940, 2260, 4780, 790,  3310, 1630, 4150, 2470,
           4990, 202,  2722, 1042, 3562, 1882, 4402, 412,  2932, 1252, 3772,
           2092, 4612, 622,  3142, 1462, 3982, 2302, 4822, 832,  3352, 1672,
           4192, 2512, 5032, 41,   2561, 881,  3401, 1721, 4241, 251,  2771,
           1091, 3611, 1931, 4451, 461,  2981, 1301, 3821, 2141, 4661, 671,
           3191, 1511, 4031, 2351, 4871, 83,   2603, 923,  3443, 1763, 4283,
           293,  2813, 1133, 3653, 1973, 4493, 503,  3023, 1343, 3863, 2183,
           4703, 713,  3233, 1553, 4073, 2393, 4913, 125,  2645, 965,  3485,
           1805, 4325, 335,  2855, 1175, 3695, 2015, 4535, 545,  3065, 1385,
           3905, 2225, 4745, 755,  3275, 1595, 4115, 2435, 4955, 167,  2687,
           1007, 3527, 1847, 4367, 377,  2897, 1217, 3737, 2057, 4577, 587,
           3107, 1427, 3947, 2267, 4787, 797,  3317, 1637, 4157, 2477, 4997,
           209,  2729, 1049, 3569, 1889, 4409, 419,  2939, 1259, 3779, 2099,
           4619, 629,  3149, 1469, 3989, 2309, 4829, 839,  3359, 1679, 4199,
           2519, 5039}));
}

TEST(TransposeTest, TestRefOps6D2) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5, 6, 7},
                                /*perms=*/{1, 0, 2, 3, 4, 5}),
      ElementsAreArray(
          {0,    1,    2,    3,    4,    5,    6,    7,    8,    9,    10,
           11,   12,   13,   14,   15,   16,   17,   18,   19,   20,   21,
           22,   23,   24,   25,   26,   27,   28,   29,   30,   31,   32,
           33,   34,   35,   36,   37,   38,   39,   40,   41,   42,   43,
           44,   45,   46,   47,   48,   49,   50,   51,   52,   53,   54,
           55,   56,   57,   58,   59,   60,   61,   62,   63,   64,   65,
           66,   67,   68,   69,   70,   71,   72,   73,   74,   75,   76,
           77,   78,   79,   80,   81,   82,   83,   84,   85,   86,   87,
           88,   89,   90,   91,   92,   93,   94,   95,   96,   97,   98,
           99,   100,  101,  102,  103,  104,  105,  106,  107,  108,  109,
           110,  111,  112,  113,  114,  115,  116,  117,  118,  119,  120,
           121,  122,  123,  124,  125,  126,  127,  128,  129,  130,  131,
           132,  133,  134,  135,  136,  137,  138,  139,  140,  141,  142,
           143,  144,  145,  146,  147,  148,  149,  150,  151,  152,  153,
           154,  155,  156,  157,  158,  159,  160,  161,  162,  163,  164,
           165,  166,  167,  168,  169,  170,  171,  172,  173,  174,  175,
           176,  177,  178,  179,  180,  181,  182,  183,  184,  185,  186,
           187,  188,  189,  190,  191,  192,  193,  194,  195,  196,  197,
           198,  199,  200,  201,  202,  203,  204,  205,  206,  207,  208,
           209,  210,  211,  212,  213,  214,  215,  216,  217,  218,  219,
           220,  221,  222,  223,  224,  225,  226,  227,  228,  229,  230,
           231,  232,  233,  234,  235,  236,  237,  238,  239,  240,  241,
           242,  243,  244,  245,  246,  247,  248,  249,  250,  251,  252,
           253,  254,  255,  256,  257,  258,  259,  260,  261,  262,  263,
           264,  265,  266,  267,  268,  269,  270,  271,  272,  273,  274,
           275,  276,  277,  278,  279,  280,  281,  282,  283,  284,  285,
           286,  287,  288,  289,  290,  291,  292,  293,  294,  295,  296,
           297,  298,  299,  300,  301,  302,  303,  304,  305,  306,  307,
           308,  309,  310,  311,  312,  313,  314,  315,  316,  317,  318,
           319,  320,  321,  322,  323,  324,  325,  326,  327,  328,  329,
           330,  331,  332,  333,  334,  335,  336,  337,  338,  339,  340,
           341,  342,  343,  344,  345,  346,  347,  348,  349,  350,  351,
           352,  353,  354,  355,  356,  357,  358,  359,  360,  361,  362,
           363,  364,  365,  366,  367,  368,  369,  370,  371,  372,  373,
           374,  375,  376,  377,  378,  379,  380,  381,  382,  383,  384,
           385,  386,  387,  388,  389,  390,  391,  392,  393,  394,  395,
           396,  397,  398,  399,  400,  401,  402,  403,  404,  405,  406,
           407,  408,  409,  410,  411,  412,  413,  414,  415,  416,  417,
           418,  419,  420,  421,  422,  423,  424,  425,  426,  427,  428,
           429,  430,  431,  432,  433,  434,  435,  436,  437,  438,  439,
           440,  441,  442,  443,  444,  445,  446,  447,  448,  449,  450,
           451,  452,  453,  454,  455,  456,  457,  458,  459,  460,  461,
           462,  463,  464,  465,  466,  467,  468,  469,  470,  471,  472,
           473,  474,  475,  476,  477,  478,  479,  480,  481,  482,  483,
           484,  485,  486,  487,  488,  489,  490,  491,  492,  493,  494,
           495,  496,  497,  498,  499,  500,  501,  502,  503,  504,  505,
           506,  507,  508,  509,  510,  511,  512,  513,  514,  515,  516,
           517,  518,  519,  520,  521,  522,  523,  524,  525,  526,  527,
           528,  529,  530,  531,  532,  533,  534,  535,  536,  537,  538,
           539,  540,  541,  542,  543,  544,  545,  546,  547,  548,  549,
           550,  551,  552,  553,  554,  555,  556,  557,  558,  559,  560,
           561,  562,  563,  564,  565,  566,  567,  568,  569,  570,  571,
           572,  573,  574,  575,  576,  577,  578,  579,  580,  581,  582,
           583,  584,  585,  586,  587,  588,  589,  590,  591,  592,  593,
           594,  595,  596,  597,  598,  599,  600,  601,  602,  603,  604,
           605,  606,  607,  608,  609,  610,  611,  612,  613,  614,  615,
           616,  617,  618,  619,  620,  621,  622,  623,  624,  625,  626,
           627,  628,  629,  630,  631,  632,  633,  634,  635,  636,  637,
           638,  639,  640,  641,  642,  643,  644,  645,  646,  647,  648,
           649,  650,  651,  652,  653,  654,  655,  656,  657,  658,  659,
           660,  661,  662,  663,  664,  665,  666,  667,  668,  669,  670,
           671,  672,  673,  674,  675,  676,  677,  678,  679,  680,  681,
           682,  683,  684,  685,  686,  687,  688,  689,  690,  691,  692,
           693,  694,  695,  696,  697,  698,  699,  700,  701,  702,  703,
           704,  705,  706,  707,  708,  709,  710,  711,  712,  713,  714,
           715,  716,  717,  718,  719,  720,  721,  722,  723,  724,  725,
           726,  727,  728,  729,  730,  731,  732,  733,  734,  735,  736,
           737,  738,  739,  740,  741,  742,  743,  744,  745,  746,  747,
           748,  749,  750,  751,  752,  753,  754,  755,  756,  757,  758,
           759,  760,  761,  762,  763,  764,  765,  766,  767,  768,  769,
           770,  771,  772,  773,  774,  775,  776,  777,  778,  779,  780,
           781,  782,  783,  784,  785,  786,  787,  788,  789,  790,  791,
           792,  793,  794,  795,  796,  797,  798,  799,  800,  801,  802,
           803,  804,  805,  806,  807,  808,  809,  810,  811,  812,  813,
           814,  815,  816,  817,  818,  819,  820,  821,  822,  823,  824,
           825,  826,  827,  828,  829,  830,  831,  832,  833,  834,  835,
           836,  837,  838,  839,  2520, 2521, 2522, 2523, 2524, 2525, 2526,
           2527, 2528, 2529, 2530, 2531, 2532, 2533, 2534, 2535, 2536, 2537,
           2538, 2539, 2540, 2541, 2542, 2543, 2544, 2545, 2546, 2547, 2548,
           2549, 2550, 2551, 2552, 2553, 2554, 2555, 2556, 2557, 2558, 2559,
           2560, 2561, 2562, 2563, 2564, 2565, 2566, 2567, 2568, 2569, 2570,
           2571, 2572, 2573, 2574, 2575, 2576, 2577, 2578, 2579, 2580, 2581,
           2582, 2583, 2584, 2585, 2586, 2587, 2588, 2589, 2590, 2591, 2592,
           2593, 2594, 2595, 2596, 2597, 2598, 2599, 2600, 2601, 2602, 2603,
           2604, 2605, 2606, 2607, 2608, 2609, 2610, 2611, 2612, 2613, 2614,
           2615, 2616, 2617, 2618, 2619, 2620, 2621, 2622, 2623, 2624, 2625,
           2626, 2627, 2628, 2629, 2630, 2631, 2632, 2633, 2634, 2635, 2636,
           2637, 2638, 2639, 2640, 2641, 2642, 2643, 2644, 2645, 2646, 2647,
           2648, 2649, 2650, 2651, 2652, 2653, 2654, 2655, 2656, 2657, 2658,
           2659, 2660, 2661, 2662, 2663, 2664, 2665, 2666, 2667, 2668, 2669,
           2670, 2671, 2672, 2673, 2674, 2675, 2676, 2677, 2678, 2679, 2680,
           2681, 2682, 2683, 2684, 2685, 2686, 2687, 2688, 2689, 2690, 2691,
           2692, 2693, 2694, 2695, 2696, 2697, 2698, 2699, 2700, 2701, 2702,
           2703, 2704, 2705, 2706, 2707, 2708, 2709, 2710, 2711, 2712, 2713,
           2714, 2715, 2716, 2717, 2718, 2719, 2720, 2721, 2722, 2723, 2724,
           2725, 2726, 2727, 2728, 2729, 2730, 2731, 2732, 2733, 2734, 2735,
           2736, 2737, 2738, 2739, 2740, 2741, 2742, 2743, 2744, 2745, 2746,
           2747, 2748, 2749, 2750, 2751, 2752, 2753, 2754, 2755, 2756, 2757,
           2758, 2759, 2760, 2761, 2762, 2763, 2764, 2765, 2766, 2767, 2768,
           2769, 2770, 2771, 2772, 2773, 2774, 2775, 2776, 2777, 2778, 2779,
           2780, 2781, 2782, 2783, 2784, 2785, 2786, 2787, 2788, 2789, 2790,
           2791, 2792, 2793, 2794, 2795, 2796, 2797, 2798, 2799, 2800, 2801,
           2802, 2803, 2804, 2805, 2806, 2807, 2808, 2809, 2810, 2811, 2812,
           2813, 2814, 2815, 2816, 2817, 2818, 2819, 2820, 2821, 2822, 2823,
           2824, 2825, 2826, 2827, 2828, 2829, 2830, 2831, 2832, 2833, 2834,
           2835, 2836, 2837, 2838, 2839, 2840, 2841, 2842, 2843, 2844, 2845,
           2846, 2847, 2848, 2849, 2850, 2851, 2852, 2853, 2854, 2855, 2856,
           2857, 2858, 2859, 2860, 2861, 2862, 2863, 2864, 2865, 2866, 2867,
           2868, 2869, 2870, 2871, 2872, 2873, 2874, 2875, 2876, 2877, 2878,
           2879, 2880, 2881, 2882, 2883, 2884, 2885, 2886, 2887, 2888, 2889,
           2890, 2891, 2892, 2893, 2894, 2895, 2896, 2897, 2898, 2899, 2900,
           2901, 2902, 2903, 2904, 2905, 2906, 2907, 2908, 2909, 2910, 2911,
           2912, 2913, 2914, 2915, 2916, 2917, 2918, 2919, 2920, 2921, 2922,
           2923, 2924, 2925, 2926, 2927, 2928, 2929, 2930, 2931, 2932, 2933,
           2934, 2935, 2936, 2937, 2938, 2939, 2940, 2941, 2942, 2943, 2944,
           2945, 2946, 2947, 2948, 2949, 2950, 2951, 2952, 2953, 2954, 2955,
           2956, 2957, 2958, 2959, 2960, 2961, 2962, 2963, 2964, 2965, 2966,
           2967, 2968, 2969, 2970, 2971, 2972, 2973, 2974, 2975, 2976, 2977,
           2978, 2979, 2980, 2981, 2982, 2983, 2984, 2985, 2986, 2987, 2988,
           2989, 2990, 2991, 2992, 2993, 2994, 2995, 2996, 2997, 2998, 2999,
           3000, 3001, 3002, 3003, 3004, 3005, 3006, 3007, 3008, 3009, 3010,
           3011, 3012, 3013, 3014, 3015, 3016, 3017, 3018, 3019, 3020, 3021,
           3022, 3023, 3024, 3025, 3026, 3027, 3028, 3029, 3030, 3031, 3032,
           3033, 3034, 3035, 3036, 3037, 3038, 3039, 3040, 3041, 3042, 3043,
           3044, 3045, 3046, 3047, 3048, 3049, 3050, 3051, 3052, 3053, 3054,
           3055, 3056, 3057, 3058, 3059, 3060, 3061, 3062, 3063, 3064, 3065,
           3066, 3067, 3068, 3069, 3070, 3071, 3072, 3073, 3074, 3075, 3076,
           3077, 3078, 3079, 3080, 3081, 3082, 3083, 3084, 3085, 3086, 3087,
           3088, 3089, 3090, 3091, 3092, 3093, 3094, 3095, 3096, 3097, 3098,
           3099, 3100, 3101, 3102, 3103, 3104, 3105, 3106, 3107, 3108, 3109,
           3110, 3111, 3112, 3113, 3114, 3115, 3116, 3117, 3118, 3119, 3120,
           3121, 3122, 3123, 3124, 3125, 3126, 3127, 3128, 3129, 3130, 3131,
           3132, 3133, 3134, 3135, 3136, 3137, 3138, 3139, 3140, 3141, 3142,
           3143, 3144, 3145, 3146, 3147, 3148, 3149, 3150, 3151, 3152, 3153,
           3154, 3155, 3156, 3157, 3158, 3159, 3160, 3161, 3162, 3163, 3164,
           3165, 3166, 3167, 3168, 3169, 3170, 3171, 3172, 3173, 3174, 3175,
           3176, 3177, 3178, 3179, 3180, 3181, 3182, 3183, 3184, 3185, 3186,
           3187, 3188, 3189, 3190, 3191, 3192, 3193, 3194, 3195, 3196, 3197,
           3198, 3199, 3200, 3201, 3202, 3203, 3204, 3205, 3206, 3207, 3208,
           3209, 3210, 3211, 3212, 3213, 3214, 3215, 3216, 3217, 3218, 3219,
           3220, 3221, 3222, 3223, 3224, 3225, 3226, 3227, 3228, 3229, 3230,
           3231, 3232, 3233, 3234, 3235, 3236, 3237, 3238, 3239, 3240, 3241,
           3242, 3243, 3244, 3245, 3246, 3247, 3248, 3249, 3250, 3251, 3252,
           3253, 3254, 3255, 3256, 3257, 3258, 3259, 3260, 3261, 3262, 3263,
           3264, 3265, 3266, 3267, 3268, 3269, 3270, 3271, 3272, 3273, 3274,
           3275, 3276, 3277, 3278, 3279, 3280, 3281, 3282, 3283, 3284, 3285,
           3286, 3287, 3288, 3289, 3290, 3291, 3292, 3293, 3294, 3295, 3296,
           3297, 3298, 3299, 3300, 3301, 3302, 3303, 3304, 3305, 3306, 3307,
           3308, 3309, 3310, 3311, 3312, 3313, 3314, 3315, 3316, 3317, 3318,
           3319, 3320, 3321, 3322, 3323, 3324, 3325, 3326, 3327, 3328, 3329,
           3330, 3331, 3332, 3333, 3334, 3335, 3336, 3337, 3338, 3339, 3340,
           3341, 3342, 3343, 3344, 3345, 3346, 3347, 3348, 3349, 3350, 3351,
           3352, 3353, 3354, 3355, 3356, 3357, 3358, 3359, 840,  841,  842,
           843,  844,  845,  846,  847,  848,  849,  850,  851,  852,  853,
           854,  855,  856,  857,  858,  859,  860,  861,  862,  863,  864,
           865,  866,  867,  868,  869,  870,  871,  872,  873,  874,  875,
           876,  877,  878,  879,  880,  881,  882,  883,  884,  885,  886,
           887,  888,  889,  890,  891,  892,  893,  894,  895,  896,  897,
           898,  899,  900,  901,  902,  903,  904,  905,  906,  907,  908,
           909,  910,  911,  912,  913,  914,  915,  916,  917,  918,  919,
           920,  921,  922,  923,  924,  925,  926,  927,  928,  929,  930,
           931,  932,  933,  934,  935,  936,  937,  938,  939,  940,  941,
           942,  943,  944,  945,  946,  947,  948,  949,  950,  951,  952,
           953,  954,  955,  956,  957,  958,  959,  960,  961,  962,  963,
           964,  965,  966,  967,  968,  969,  970,  971,  972,  973,  974,
           975,  976,  977,  978,  979,  980,  981,  982,  983,  984,  985,
           986,  987,  988,  989,  990,  991,  992,  993,  994,  995,  996,
           997,  998,  999,  1000, 1001, 1002, 1003, 1004, 1005, 1006, 1007,
           1008, 1009, 1010, 1011, 1012, 1013, 1014, 1015, 1016, 1017, 1018,
           1019, 1020, 1021, 1022, 1023, 1024, 1025, 1026, 1027, 1028, 1029,
           1030, 1031, 1032, 1033, 1034, 1035, 1036, 1037, 1038, 1039, 1040,
           1041, 1042, 1043, 1044, 1045, 1046, 1047, 1048, 1049, 1050, 1051,
           1052, 1053, 1054, 1055, 1056, 1057, 1058, 1059, 1060, 1061, 1062,
           1063, 1064, 1065, 1066, 1067, 1068, 1069, 1070, 1071, 1072, 1073,
           1074, 1075, 1076, 1077, 1078, 1079, 1080, 1081, 1082, 1083, 1084,
           1085, 1086, 1087, 1088, 1089, 1090, 1091, 1092, 1093, 1094, 1095,
           1096, 1097, 1098, 1099, 1100, 1101, 1102, 1103, 1104, 1105, 1106,
           1107, 1108, 1109, 1110, 1111, 1112, 1113, 1114, 1115, 1116, 1117,
           1118, 1119, 1120, 1121, 1122, 1123, 1124, 1125, 1126, 1127, 1128,
           1129, 1130, 1131, 1132, 1133, 1134, 1135, 1136, 1137, 1138, 1139,
           1140, 1141, 1142, 1143, 1144, 1145, 1146, 1147, 1148, 1149, 1150,
           1151, 1152, 1153, 1154, 1155, 1156, 1157, 1158, 1159, 1160, 1161,
           1162, 1163, 1164, 1165, 1166, 1167, 1168, 1169, 1170, 1171, 1172,
           1173, 1174, 1175, 1176, 1177, 1178, 1179, 1180, 1181, 1182, 1183,
           1184, 1185, 1186, 1187, 1188, 1189, 1190, 1191, 1192, 1193, 1194,
           1195, 1196, 1197, 1198, 1199, 1200, 1201, 1202, 1203, 1204, 1205,
           1206, 1207, 1208, 1209, 1210, 1211, 1212, 1213, 1214, 1215, 1216,
           1217, 1218, 1219, 1220, 1221, 1222, 1223, 1224, 1225, 1226, 1227,
           1228, 1229, 1230, 1231, 1232, 1233, 1234, 1235, 1236, 1237, 1238,
           1239, 1240, 1241, 1242, 1243, 1244, 1245, 1246, 1247, 1248, 1249,
           1250, 1251, 1252, 1253, 1254, 1255, 1256, 1257, 1258, 1259, 1260,
           1261, 1262, 1263, 1264, 1265, 1266, 1267, 1268, 1269, 1270, 1271,
           1272, 1273, 1274, 1275, 1276, 1277, 1278, 1279, 1280, 1281, 1282,
           1283, 1284, 1285, 1286, 1287, 1288, 1289, 1290, 1291, 1292, 1293,
           1294, 1295, 1296, 1297, 1298, 1299, 1300, 1301, 1302, 1303, 1304,
           1305, 1306, 1307, 1308, 1309, 1310, 1311, 1312, 1313, 1314, 1315,
           1316, 1317, 1318, 1319, 1320, 1321, 1322, 1323, 1324, 1325, 1326,
           1327, 1328, 1329, 1330, 1331, 1332, 1333, 1334, 1335, 1336, 1337,
           1338, 1339, 1340, 1341, 1342, 1343, 1344, 1345, 1346, 1347, 1348,
           1349, 1350, 1351, 1352, 1353, 1354, 1355, 1356, 1357, 1358, 1359,
           1360, 1361, 1362, 1363, 1364, 1365, 1366, 1367, 1368, 1369, 1370,
           1371, 1372, 1373, 1374, 1375, 1376, 1377, 1378, 1379, 1380, 1381,
           1382, 1383, 1384, 1385, 1386, 1387, 1388, 1389, 1390, 1391, 1392,
           1393, 1394, 1395, 1396, 1397, 1398, 1399, 1400, 1401, 1402, 1403,
           1404, 1405, 1406, 1407, 1408, 1409, 1410, 1411, 1412, 1413, 1414,
           1415, 1416, 1417, 1418, 1419, 1420, 1421, 1422, 1423, 1424, 1425,
           1426, 1427, 1428, 1429, 1430, 1431, 1432, 1433, 1434, 1435, 1436,
           1437, 1438, 1439, 1440, 1441, 1442, 1443, 1444, 1445, 1446, 1447,
           1448, 1449, 1450, 1451, 1452, 1453, 1454, 1455, 1456, 1457, 1458,
           1459, 1460, 1461, 1462, 1463, 1464, 1465, 1466, 1467, 1468, 1469,
           1470, 1471, 1472, 1473, 1474, 1475, 1476, 1477, 1478, 1479, 1480,
           1481, 1482, 1483, 1484, 1485, 1486, 1487, 1488, 1489, 1490, 1491,
           1492, 1493, 1494, 1495, 1496, 1497, 1498, 1499, 1500, 1501, 1502,
           1503, 1504, 1505, 1506, 1507, 1508, 1509, 1510, 1511, 1512, 1513,
           1514, 1515, 1516, 1517, 1518, 1519, 1520, 1521, 1522, 1523, 1524,
           1525, 1526, 1527, 1528, 1529, 1530, 1531, 1532, 1533, 1534, 1535,
           1536, 1537, 1538, 1539, 1540, 1541, 1542, 1543, 1544, 1545, 1546,
           1547, 1548, 1549, 1550, 1551, 1552, 1553, 1554, 1555, 1556, 1557,
           1558, 1559, 1560, 1561, 1562, 1563, 1564, 1565, 1566, 1567, 1568,
           1569, 1570, 1571, 1572, 1573, 1574, 1575, 1576, 1577, 1578, 1579,
           1580, 1581, 1582, 1583, 1584, 1585, 1586, 1587, 1588, 1589, 1590,
           1591, 1592, 1593, 1594, 1595, 1596, 1597, 1598, 1599, 1600, 1601,
           1602, 1603, 1604, 1605, 1606, 1607, 1608, 1609, 1610, 1611, 1612,
           1613, 1614, 1615, 1616, 1617, 1618, 1619, 1620, 1621, 1622, 1623,
           1624, 1625, 1626, 1627, 1628, 1629, 1630, 1631, 1632, 1633, 1634,
           1635, 1636, 1637, 1638, 1639, 1640, 1641, 1642, 1643, 1644, 1645,
           1646, 1647, 1648, 1649, 1650, 1651, 1652, 1653, 1654, 1655, 1656,
           1657, 1658, 1659, 1660, 1661, 1662, 1663, 1664, 1665, 1666, 1667,
           1668, 1669, 1670, 1671, 1672, 1673, 1674, 1675, 1676, 1677, 1678,
           1679, 3360, 3361, 3362, 3363, 3364, 3365, 3366, 3367, 3368, 3369,
           3370, 3371, 3372, 3373, 3374, 3375, 3376, 3377, 3378, 3379, 3380,
           3381, 3382, 3383, 3384, 3385, 3386, 3387, 3388, 3389, 3390, 3391,
           3392, 3393, 3394, 3395, 3396, 3397, 3398, 3399, 3400, 3401, 3402,
           3403, 3404, 3405, 3406, 3407, 3408, 3409, 3410, 3411, 3412, 3413,
           3414, 3415, 3416, 3417, 3418, 3419, 3420, 3421, 3422, 3423, 3424,
           3425, 3426, 3427, 3428, 3429, 3430, 3431, 3432, 3433, 3434, 3435,
           3436, 3437, 3438, 3439, 3440, 3441, 3442, 3443, 3444, 3445, 3446,
           3447, 3448, 3449, 3450, 3451, 3452, 3453, 3454, 3455, 3456, 3457,
           3458, 3459, 3460, 3461, 3462, 3463, 3464, 3465, 3466, 3467, 3468,
           3469, 3470, 3471, 3472, 3473, 3474, 3475, 3476, 3477, 3478, 3479,
           3480, 3481, 3482, 3483, 3484, 3485, 3486, 3487, 3488, 3489, 3490,
           3491, 3492, 3493, 3494, 3495, 3496, 3497, 3498, 3499, 3500, 3501,
           3502, 3503, 3504, 3505, 3506, 3507, 3508, 3509, 3510, 3511, 3512,
           3513, 3514, 3515, 3516, 3517, 3518, 3519, 3520, 3521, 3522, 3523,
           3524, 3525, 3526, 3527, 3528, 3529, 3530, 3531, 3532, 3533, 3534,
           3535, 3536, 3537, 3538, 3539, 3540, 3541, 3542, 3543, 3544, 3545,
           3546, 3547, 3548, 3549, 3550, 3551, 3552, 3553, 3554, 3555, 3556,
           3557, 3558, 3559, 3560, 3561, 3562, 3563, 3564, 3565, 3566, 3567,
           3568, 3569, 3570, 3571, 3572, 3573, 3574, 3575, 3576, 3577, 3578,
           3579, 3580, 3581, 3582, 3583, 3584, 3585, 3586, 3587, 3588, 3589,
           3590, 3591, 3592, 3593, 3594, 3595, 3596, 3597, 3598, 3599, 3600,
           3601, 3602, 3603, 3604, 3605, 3606, 3607, 3608, 3609, 3610, 3611,
           3612, 3613, 3614, 3615, 3616, 3617, 3618, 3619, 3620, 3621, 3622,
           3623, 3624, 3625, 3626, 3627, 3628, 3629, 3630, 3631, 3632, 3633,
           3634, 3635, 3636, 3637, 3638, 3639, 3640, 3641, 3642, 3643, 3644,
           3645, 3646, 3647, 3648, 3649, 3650, 3651, 3652, 3653, 3654, 3655,
           3656, 3657, 3658, 3659, 3660, 3661, 3662, 3663, 3664, 3665, 3666,
           3667, 3668, 3669, 3670, 3671, 3672, 3673, 3674, 3675, 3676, 3677,
           3678, 3679, 3680, 3681, 3682, 3683, 3684, 3685, 3686, 3687, 3688,
           3689, 3690, 3691, 3692, 3693, 3694, 3695, 3696, 3697, 3698, 3699,
           3700, 3701, 3702, 3703, 3704, 3705, 3706, 3707, 3708, 3709, 3710,
           3711, 3712, 3713, 3714, 3715, 3716, 3717, 3718, 3719, 3720, 3721,
           3722, 3723, 3724, 3725, 3726, 3727, 3728, 3729, 3730, 3731, 3732,
           3733, 3734, 3735, 3736, 3737, 3738, 3739, 3740, 3741, 3742, 3743,
           3744, 3745, 3746, 3747, 3748, 3749, 3750, 3751, 3752, 3753, 3754,
           3755, 3756, 3757, 3758, 3759, 3760, 3761, 3762, 3763, 3764, 3765,
           3766, 3767, 3768, 3769, 3770, 3771, 3772, 3773, 3774, 3775, 3776,
           3777, 3778, 3779, 3780, 3781, 3782, 3783, 3784, 3785, 3786, 3787,
           3788, 3789, 3790, 3791, 3792, 3793, 3794, 3795, 3796, 3797, 3798,
           3799, 3800, 3801, 3802, 3803, 3804, 3805, 3806, 3807, 3808, 3809,
           3810, 3811, 3812, 3813, 3814, 3815, 3816, 3817, 3818, 3819, 3820,
           3821, 3822, 3823, 3824, 3825, 3826, 3827, 3828, 3829, 3830, 3831,
           3832, 3833, 3834, 3835, 3836, 3837, 3838, 3839, 3840, 3841, 3842,
           3843, 3844, 3845, 3846, 3847, 3848, 3849, 3850, 3851, 3852, 3853,
           3854, 3855, 3856, 3857, 3858, 3859, 3860, 3861, 3862, 3863, 3864,
           3865, 3866, 3867, 3868, 3869, 3870, 3871, 3872, 3873, 3874, 3875,
           3876, 3877, 3878, 3879, 3880, 3881, 3882, 3883, 3884, 3885, 3886,
           3887, 3888, 3889, 3890, 3891, 3892, 3893, 3894, 3895, 3896, 3897,
           3898, 3899, 3900, 3901, 3902, 3903, 3904, 3905, 3906, 3907, 3908,
           3909, 3910, 3911, 3912, 3913, 3914, 3915, 3916, 3917, 3918, 3919,
           3920, 3921, 3922, 3923, 3924, 3925, 3926, 3927, 3928, 3929, 3930,
           3931, 3932, 3933, 3934, 3935, 3936, 3937, 3938, 3939, 3940, 3941,
           3942, 3943, 3944, 3945, 3946, 3947, 3948, 3949, 3950, 3951, 3952,
           3953, 3954, 3955, 3956, 3957, 3958, 3959, 3960, 3961, 3962, 3963,
           3964, 3965, 3966, 3967, 3968, 3969, 3970, 3971, 3972, 3973, 3974,
           3975, 3976, 3977, 3978, 3979, 3980, 3981, 3982, 3983, 3984, 3985,
           3986, 3987, 3988, 3989, 3990, 3991, 3992, 3993, 3994, 3995, 3996,
           3997, 3998, 3999, 4000, 4001, 4002, 4003, 4004, 4005, 4006, 4007,
           4008, 4009, 4010, 4011, 4012, 4013, 4014, 4015, 4016, 4017, 4018,
           4019, 4020, 4021, 4022, 4023, 4024, 4025, 4026, 4027, 4028, 4029,
           4030, 4031, 4032, 4033, 4034, 4035, 4036, 4037, 4038, 4039, 4040,
           4041, 4042, 4043, 4044, 4045, 4046, 4047, 4048, 4049, 4050, 4051,
           4052, 4053, 4054, 4055, 4056, 4057, 4058, 4059, 4060, 4061, 4062,
           4063, 4064, 4065, 4066, 4067, 4068, 4069, 4070, 4071, 4072, 4073,
           4074, 4075, 4076, 4077, 4078, 4079, 4080, 4081, 4082, 4083, 4084,
           4085, 4086, 4087, 4088, 4089, 4090, 4091, 4092, 4093, 4094, 4095,
           4096, 4097, 4098, 4099, 4100, 4101, 4102, 4103, 4104, 4105, 4106,
           4107, 4108, 4109, 4110, 4111, 4112, 4113, 4114, 4115, 4116, 4117,
           4118, 4119, 4120, 4121, 4122, 4123, 4124, 4125, 4126, 4127, 4128,
           4129, 4130, 4131, 4132, 4133, 4134, 4135, 4136, 4137, 4138, 4139,
           4140, 4141, 4142, 4143, 4144, 4145, 4146, 4147, 4148, 4149, 4150,
           4151, 4152, 4153, 4154, 4155, 4156, 4157, 4158, 4159, 4160, 4161,
           4162, 4163, 4164, 4165, 4166, 4167, 4168, 4169, 4170, 4171, 4172,
           4173, 4174, 4175, 4176, 4177, 4178, 4179, 4180, 4181, 4182, 4183,
           4184, 4185, 4186, 4187, 4188, 4189, 4190, 4191, 4192, 4193, 4194,
           4195, 4196, 4197, 4198, 4199, 1680, 1681, 1682, 1683, 1684, 1685,
           1686, 1687, 1688, 1689, 1690, 1691, 1692, 1693, 1694, 1695, 1696,
           1697, 1698, 1699, 1700, 1701, 1702, 1703, 1704, 1705, 1706, 1707,
           1708, 1709, 1710, 1711, 1712, 1713, 1714, 1715, 1716, 1717, 1718,
           1719, 1720, 1721, 1722, 1723, 1724, 1725, 1726, 1727, 1728, 1729,
           1730, 1731, 1732, 1733, 1734, 1735, 1736, 1737, 1738, 1739, 1740,
           1741, 1742, 1743, 1744, 1745, 1746, 1747, 1748, 1749, 1750, 1751,
           1752, 1753, 1754, 1755, 1756, 1757, 1758, 1759, 1760, 1761, 1762,
           1763, 1764, 1765, 1766, 1767, 1768, 1769, 1770, 1771, 1772, 1773,
           1774, 1775, 1776, 1777, 1778, 1779, 1780, 1781, 1782, 1783, 1784,
           1785, 1786, 1787, 1788, 1789, 1790, 1791, 1792, 1793, 1794, 1795,
           1796, 1797, 1798, 1799, 1800, 1801, 1802, 1803, 1804, 1805, 1806,
           1807, 1808, 1809, 1810, 1811, 1812, 1813, 1814, 1815, 1816, 1817,
           1818, 1819, 1820, 1821, 1822, 1823, 1824, 1825, 1826, 1827, 1828,
           1829, 1830, 1831, 1832, 1833, 1834, 1835, 1836, 1837, 1838, 1839,
           1840, 1841, 1842, 1843, 1844, 1845, 1846, 1847, 1848, 1849, 1850,
           1851, 1852, 1853, 1854, 1855, 1856, 1857, 1858, 1859, 1860, 1861,
           1862, 1863, 1864, 1865, 1866, 1867, 1868, 1869, 1870, 1871, 1872,
           1873, 1874, 1875, 1876, 1877, 1878, 1879, 1880, 1881, 1882, 1883,
           1884, 1885, 1886, 1887, 1888, 1889, 1890, 1891, 1892, 1893, 1894,
           1895, 1896, 1897, 1898, 1899, 1900, 1901, 1902, 1903, 1904, 1905,
           1906, 1907, 1908, 1909, 1910, 1911, 1912, 1913, 1914, 1915, 1916,
           1917, 1918, 1919, 1920, 1921, 1922, 1923, 1924, 1925, 1926, 1927,
           1928, 1929, 1930, 1931, 1932, 1933, 1934, 1935, 1936, 1937, 1938,
           1939, 1940, 1941, 1942, 1943, 1944, 1945, 1946, 1947, 1948, 1949,
           1950, 1951, 1952, 1953, 1954, 1955, 1956, 1957, 1958, 1959, 1960,
           1961, 1962, 1963, 1964, 1965, 1966, 1967, 1968, 1969, 1970, 1971,
           1972, 1973, 1974, 1975, 1976, 1977, 1978, 1979, 1980, 1981, 1982,
           1983, 1984, 1985, 1986, 1987, 1988, 1989, 1990, 1991, 1992, 1993,
           1994, 1995, 1996, 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004,
           2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015,
           2016, 2017, 2018, 2019, 2020, 2021, 2022, 2023, 2024, 2025, 2026,
           2027, 2028, 2029, 2030, 2031, 2032, 2033, 2034, 2035, 2036, 2037,
           2038, 2039, 2040, 2041, 2042, 2043, 2044, 2045, 2046, 2047, 2048,
           2049, 2050, 2051, 2052, 2053, 2054, 2055, 2056, 2057, 2058, 2059,
           2060, 2061, 2062, 2063, 2064, 2065, 2066, 2067, 2068, 2069, 2070,
           2071, 2072, 2073, 2074, 2075, 2076, 2077, 2078, 2079, 2080, 2081,
           2082, 2083, 2084, 2085, 2086, 2087, 2088, 2089, 2090, 2091, 2092,
           2093, 2094, 2095, 2096, 2097, 2098, 2099, 2100, 2101, 2102, 2103,
           2104, 2105, 2106, 2107, 2108, 2109, 2110, 2111, 2112, 2113, 2114,
           2115, 2116, 2117, 2118, 2119, 2120, 2121, 2122, 2123, 2124, 2125,
           2126, 2127, 2128, 2129, 2130, 2131, 2132, 2133, 2134, 2135, 2136,
           2137, 2138, 2139, 2140, 2141, 2142, 2143, 2144, 2145, 2146, 2147,
           2148, 2149, 2150, 2151, 2152, 2153, 2154, 2155, 2156, 2157, 2158,
           2159, 2160, 2161, 2162, 2163, 2164, 2165, 2166, 2167, 2168, 2169,
           2170, 2171, 2172, 2173, 2174, 2175, 2176, 2177, 2178, 2179, 2180,
           2181, 2182, 2183, 2184, 2185, 2186, 2187, 2188, 2189, 2190, 2191,
           2192, 2193, 2194, 2195, 2196, 2197, 2198, 2199, 2200, 2201, 2202,
           2203, 2204, 2205, 2206, 2207, 2208, 2209, 2210, 2211, 2212, 2213,
           2214, 2215, 2216, 2217, 2218, 2219, 2220, 2221, 2222, 2223, 2224,
           2225, 2226, 2227, 2228, 2229, 2230, 2231, 2232, 2233, 2234, 2235,
           2236, 2237, 2238, 2239, 2240, 2241, 2242, 2243, 2244, 2245, 2246,
           2247, 2248, 2249, 2250, 2251, 2252, 2253, 2254, 2255, 2256, 2257,
           2258, 2259, 2260, 2261, 2262, 2263, 2264, 2265, 2266, 2267, 2268,
           2269, 2270, 2271, 2272, 2273, 2274, 2275, 2276, 2277, 2278, 2279,
           2280, 2281, 2282, 2283, 2284, 2285, 2286, 2287, 2288, 2289, 2290,
           2291, 2292, 2293, 2294, 2295, 2296, 2297, 2298, 2299, 2300, 2301,
           2302, 2303, 2304, 2305, 2306, 2307, 2308, 2309, 2310, 2311, 2312,
           2313, 2314, 2315, 2316, 2317, 2318, 2319, 2320, 2321, 2322, 2323,
           2324, 2325, 2326, 2327, 2328, 2329, 2330, 2331, 2332, 2333, 2334,
           2335, 2336, 2337, 2338, 2339, 2340, 2341, 2342, 2343, 2344, 2345,
           2346, 2347, 2348, 2349, 2350, 2351, 2352, 2353, 2354, 2355, 2356,
           2357, 2358, 2359, 2360, 2361, 2362, 2363, 2364, 2365, 2366, 2367,
           2368, 2369, 2370, 2371, 2372, 2373, 2374, 2375, 2376, 2377, 2378,
           2379, 2380, 2381, 2382, 2383, 2384, 2385, 2386, 2387, 2388, 2389,
           2390, 2391, 2392, 2393, 2394, 2395, 2396, 2397, 2398, 2399, 2400,
           2401, 2402, 2403, 2404, 2405, 2406, 2407, 2408, 2409, 2410, 2411,
           2412, 2413, 2414, 2415, 2416, 2417, 2418, 2419, 2420, 2421, 2422,
           2423, 2424, 2425, 2426, 2427, 2428, 2429, 2430, 2431, 2432, 2433,
           2434, 2435, 2436, 2437, 2438, 2439, 2440, 2441, 2442, 2443, 2444,
           2445, 2446, 2447, 2448, 2449, 2450, 2451, 2452, 2453, 2454, 2455,
           2456, 2457, 2458, 2459, 2460, 2461, 2462, 2463, 2464, 2465, 2466,
           2467, 2468, 2469, 2470, 2471, 2472, 2473, 2474, 2475, 2476, 2477,
           2478, 2479, 2480, 2481, 2482, 2483, 2484, 2485, 2486, 2487, 2488,
           2489, 2490, 2491, 2492, 2493, 2494, 2495, 2496, 2497, 2498, 2499,
           2500, 2501, 2502, 2503, 2504, 2505, 2506, 2507, 2508, 2509, 2510,
           2511, 2512, 2513, 2514, 2515, 2516, 2517, 2518, 2519, 4200, 4201,
           4202, 4203, 4204, 4205, 4206, 4207, 4208, 4209, 4210, 4211, 4212,
           4213, 4214, 4215, 4216, 4217, 4218, 4219, 4220, 4221, 4222, 4223,
           4224, 4225, 4226, 4227, 4228, 4229, 4230, 4231, 4232, 4233, 4234,
           4235, 4236, 4237, 4238, 4239, 4240, 4241, 4242, 4243, 4244, 4245,
           4246, 4247, 4248, 4249, 4250, 4251, 4252, 4253, 4254, 4255, 4256,
           4257, 4258, 4259, 4260, 4261, 4262, 4263, 4264, 4265, 4266, 4267,
           4268, 4269, 4270, 4271, 4272, 4273, 4274, 4275, 4276, 4277, 4278,
           4279, 4280, 4281, 4282, 4283, 4284, 4285, 4286, 4287, 4288, 4289,
           4290, 4291, 4292, 4293, 4294, 4295, 4296, 4297, 4298, 4299, 4300,
           4301, 4302, 4303, 4304, 4305, 4306, 4307, 4308, 4309, 4310, 4311,
           4312, 4313, 4314, 4315, 4316, 4317, 4318, 4319, 4320, 4321, 4322,
           4323, 4324, 4325, 4326, 4327, 4328, 4329, 4330, 4331, 4332, 4333,
           4334, 4335, 4336, 4337, 4338, 4339, 4340, 4341, 4342, 4343, 4344,
           4345, 4346, 4347, 4348, 4349, 4350, 4351, 4352, 4353, 4354, 4355,
           4356, 4357, 4358, 4359, 4360, 4361, 4362, 4363, 4364, 4365, 4366,
           4367, 4368, 4369, 4370, 4371, 4372, 4373, 4374, 4375, 4376, 4377,
           4378, 4379, 4380, 4381, 4382, 4383, 4384, 4385, 4386, 4387, 4388,
           4389, 4390, 4391, 4392, 4393, 4394, 4395, 4396, 4397, 4398, 4399,
           4400, 4401, 4402, 4403, 4404, 4405, 4406, 4407, 4408, 4409, 4410,
           4411, 4412, 4413, 4414, 4415, 4416, 4417, 4418, 4419, 4420, 4421,
           4422, 4423, 4424, 4425, 4426, 4427, 4428, 4429, 4430, 4431, 4432,
           4433, 4434, 4435, 4436, 4437, 4438, 4439, 4440, 4441, 4442, 4443,
           4444, 4445, 4446, 4447, 4448, 4449, 4450, 4451, 4452, 4453, 4454,
           4455, 4456, 4457, 4458, 4459, 4460, 4461, 4462, 4463, 4464, 4465,
           4466, 4467, 4468, 4469, 4470, 4471, 4472, 4473, 4474, 4475, 4476,
           4477, 4478, 4479, 4480, 4481, 4482, 4483, 4484, 4485, 4486, 4487,
           4488, 4489, 4490, 4491, 4492, 4493, 4494, 4495, 4496, 4497, 4498,
           4499, 4500, 4501, 4502, 4503, 4504, 4505, 4506, 4507, 4508, 4509,
           4510, 4511, 4512, 4513, 4514, 4515, 4516, 4517, 4518, 4519, 4520,
           4521, 4522, 4523, 4524, 4525, 4526, 4527, 4528, 4529, 4530, 4531,
           4532, 4533, 4534, 4535, 4536, 4537, 4538, 4539, 4540, 4541, 4542,
           4543, 4544, 4545, 4546, 4547, 4548, 4549, 4550, 4551, 4552, 4553,
           4554, 4555, 4556, 4557, 4558, 4559, 4560, 4561, 4562, 4563, 4564,
           4565, 4566, 4567, 4568, 4569, 4570, 4571, 4572, 4573, 4574, 4575,
           4576, 4577, 4578, 4579, 4580, 4581, 4582, 4583, 4584, 4585, 4586,
           4587, 4588, 4589, 4590, 4591, 4592, 4593, 4594, 4595, 4596, 4597,
           4598, 4599, 4600, 4601, 4602, 4603, 4604, 4605, 4606, 4607, 4608,
           4609, 4610, 4611, 4612, 4613, 4614, 4615, 4616, 4617, 4618, 4619,
           4620, 4621, 4622, 4623, 4624, 4625, 4626, 4627, 4628, 4629, 4630,
           4631, 4632, 4633, 4634, 4635, 4636, 4637, 4638, 4639, 4640, 4641,
           4642, 4643, 4644, 4645, 4646, 4647, 4648, 4649, 4650, 4651, 4652,
           4653, 4654, 4655, 4656, 4657, 4658, 4659, 4660, 4661, 4662, 4663,
           4664, 4665, 4666, 4667, 4668, 4669, 4670, 4671, 4672, 4673, 4674,
           4675, 4676, 4677, 4678, 4679, 4680, 4681, 4682, 4683, 4684, 4685,
           4686, 4687, 4688, 4689, 4690, 4691, 4692, 4693, 4694, 4695, 4696,
           4697, 4698, 4699, 4700, 4701, 4702, 4703, 4704, 4705, 4706, 4707,
           4708, 4709, 4710, 4711, 4712, 4713, 4714, 4715, 4716, 4717, 4718,
           4719, 4720, 4721, 4722, 4723, 4724, 4725, 4726, 4727, 4728, 4729,
           4730, 4731, 4732, 4733, 4734, 4735, 4736, 4737, 4738, 4739, 4740,
           4741, 4742, 4743, 4744, 4745, 4746, 4747, 4748, 4749, 4750, 4751,
           4752, 4753, 4754, 4755, 4756, 4757, 4758, 4759, 4760, 4761, 4762,
           4763, 4764, 4765, 4766, 4767, 4768, 4769, 4770, 4771, 4772, 4773,
           4774, 4775, 4776, 4777, 4778, 4779, 4780, 4781, 4782, 4783, 4784,
           4785, 4786, 4787, 4788, 4789, 4790, 4791, 4792, 4793, 4794, 4795,
           4796, 4797, 4798, 4799, 4800, 4801, 4802, 4803, 4804, 4805, 4806,
           4807, 4808, 4809, 4810, 4811, 4812, 4813, 4814, 4815, 4816, 4817,
           4818, 4819, 4820, 4821, 4822, 4823, 4824, 4825, 4826, 4827, 4828,
           4829, 4830, 4831, 4832, 4833, 4834, 4835, 4836, 4837, 4838, 4839,
           4840, 4841, 4842, 4843, 4844, 4845, 4846, 4847, 4848, 4849, 4850,
           4851, 4852, 4853, 4854, 4855, 4856, 4857, 4858, 4859, 4860, 4861,
           4862, 4863, 4864, 4865, 4866, 4867, 4868, 4869, 4870, 4871, 4872,
           4873, 4874, 4875, 4876, 4877, 4878, 4879, 4880, 4881, 4882, 4883,
           4884, 4885, 4886, 4887, 4888, 4889, 4890, 4891, 4892, 4893, 4894,
           4895, 4896, 4897, 4898, 4899, 4900, 4901, 4902, 4903, 4904, 4905,
           4906, 4907, 4908, 4909, 4910, 4911, 4912, 4913, 4914, 4915, 4916,
           4917, 4918, 4919, 4920, 4921, 4922, 4923, 4924, 4925, 4926, 4927,
           4928, 4929, 4930, 4931, 4932, 4933, 4934, 4935, 4936, 4937, 4938,
           4939, 4940, 4941, 4942, 4943, 4944, 4945, 4946, 4947, 4948, 4949,
           4950, 4951, 4952, 4953, 4954, 4955, 4956, 4957, 4958, 4959, 4960,
           4961, 4962, 4963, 4964, 4965, 4966, 4967, 4968, 4969, 4970, 4971,
           4972, 4973, 4974, 4975, 4976, 4977, 4978, 4979, 4980, 4981, 4982,
           4983, 4984, 4985, 4986, 4987, 4988, 4989, 4990, 4991, 4992, 4993,
           4994, 4995, 4996, 4997, 4998, 4999, 5000, 5001, 5002, 5003, 5004,
           5005, 5006, 5007, 5008, 5009, 5010, 5011, 5012, 5013, 5014, 5015,
           5016, 5017, 5018, 5019, 5020, 5021, 5022, 5023, 5024, 5025, 5026,
           5027, 5028, 5029, 5030, 5031, 5032, 5033, 5034, 5035, 5036, 5037,
           5038, 5039}));
}

TEST(TransposeTest, TestRefOps6D3) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5, 6, 7},
                                /*perms=*/{1, 2, 0, 3, 4, 5}),
      ElementsAreArray(
          {0,    1,    2,    3,    4,    5,    6,    7,    8,    9,    10,
           11,   12,   13,   14,   15,   16,   17,   18,   19,   20,   21,
           22,   23,   24,   25,   26,   27,   28,   29,   30,   31,   32,
           33,   34,   35,   36,   37,   38,   39,   40,   41,   42,   43,
           44,   45,   46,   47,   48,   49,   50,   51,   52,   53,   54,
           55,   56,   57,   58,   59,   60,   61,   62,   63,   64,   65,
           66,   67,   68,   69,   70,   71,   72,   73,   74,   75,   76,
           77,   78,   79,   80,   81,   82,   83,   84,   85,   86,   87,
           88,   89,   90,   91,   92,   93,   94,   95,   96,   97,   98,
           99,   100,  101,  102,  103,  104,  105,  106,  107,  108,  109,
           110,  111,  112,  113,  114,  115,  116,  117,  118,  119,  120,
           121,  122,  123,  124,  125,  126,  127,  128,  129,  130,  131,
           132,  133,  134,  135,  136,  137,  138,  139,  140,  141,  142,
           143,  144,  145,  146,  147,  148,  149,  150,  151,  152,  153,
           154,  155,  156,  157,  158,  159,  160,  161,  162,  163,  164,
           165,  166,  167,  168,  169,  170,  171,  172,  173,  174,  175,
           176,  177,  178,  179,  180,  181,  182,  183,  184,  185,  186,
           187,  188,  189,  190,  191,  192,  193,  194,  195,  196,  197,
           198,  199,  200,  201,  202,  203,  204,  205,  206,  207,  208,
           209,  2520, 2521, 2522, 2523, 2524, 2525, 2526, 2527, 2528, 2529,
           2530, 2531, 2532, 2533, 2534, 2535, 2536, 2537, 2538, 2539, 2540,
           2541, 2542, 2543, 2544, 2545, 2546, 2547, 2548, 2549, 2550, 2551,
           2552, 2553, 2554, 2555, 2556, 2557, 2558, 2559, 2560, 2561, 2562,
           2563, 2564, 2565, 2566, 2567, 2568, 2569, 2570, 2571, 2572, 2573,
           2574, 2575, 2576, 2577, 2578, 2579, 2580, 2581, 2582, 2583, 2584,
           2585, 2586, 2587, 2588, 2589, 2590, 2591, 2592, 2593, 2594, 2595,
           2596, 2597, 2598, 2599, 2600, 2601, 2602, 2603, 2604, 2605, 2606,
           2607, 2608, 2609, 2610, 2611, 2612, 2613, 2614, 2615, 2616, 2617,
           2618, 2619, 2620, 2621, 2622, 2623, 2624, 2625, 2626, 2627, 2628,
           2629, 2630, 2631, 2632, 2633, 2634, 2635, 2636, 2637, 2638, 2639,
           2640, 2641, 2642, 2643, 2644, 2645, 2646, 2647, 2648, 2649, 2650,
           2651, 2652, 2653, 2654, 2655, 2656, 2657, 2658, 2659, 2660, 2661,
           2662, 2663, 2664, 2665, 2666, 2667, 2668, 2669, 2670, 2671, 2672,
           2673, 2674, 2675, 2676, 2677, 2678, 2679, 2680, 2681, 2682, 2683,
           2684, 2685, 2686, 2687, 2688, 2689, 2690, 2691, 2692, 2693, 2694,
           2695, 2696, 2697, 2698, 2699, 2700, 2701, 2702, 2703, 2704, 2705,
           2706, 2707, 2708, 2709, 2710, 2711, 2712, 2713, 2714, 2715, 2716,
           2717, 2718, 2719, 2720, 2721, 2722, 2723, 2724, 2725, 2726, 2727,
           2728, 2729, 210,  211,  212,  213,  214,  215,  216,  217,  218,
           219,  220,  221,  222,  223,  224,  225,  226,  227,  228,  229,
           230,  231,  232,  233,  234,  235,  236,  237,  238,  239,  240,
           241,  242,  243,  244,  245,  246,  247,  248,  249,  250,  251,
           252,  253,  254,  255,  256,  257,  258,  259,  260,  261,  262,
           263,  264,  265,  266,  267,  268,  269,  270,  271,  272,  273,
           274,  275,  276,  277,  278,  279,  280,  281,  282,  283,  284,
           285,  286,  287,  288,  289,  290,  291,  292,  293,  294,  295,
           296,  297,  298,  299,  300,  301,  302,  303,  304,  305,  306,
           307,  308,  309,  310,  311,  312,  313,  314,  315,  316,  317,
           318,  319,  320,  321,  322,  323,  324,  325,  326,  327,  328,
           329,  330,  331,  332,  333,  334,  335,  336,  337,  338,  339,
           340,  341,  342,  343,  344,  345,  346,  347,  348,  349,  350,
           351,  352,  353,  354,  355,  356,  357,  358,  359,  360,  361,
           362,  363,  364,  365,  366,  367,  368,  369,  370,  371,  372,
           373,  374,  375,  376,  377,  378,  379,  380,  381,  382,  383,
           384,  385,  386,  387,  388,  389,  390,  391,  392,  393,  394,
           395,  396,  397,  398,  399,  400,  401,  402,  403,  404,  405,
           406,  407,  408,  409,  410,  411,  412,  413,  414,  415,  416,
           417,  418,  419,  2730, 2731, 2732, 2733, 2734, 2735, 2736, 2737,
           2738, 2739, 2740, 2741, 2742, 2743, 2744, 2745, 2746, 2747, 2748,
           2749, 2750, 2751, 2752, 2753, 2754, 2755, 2756, 2757, 2758, 2759,
           2760, 2761, 2762, 2763, 2764, 2765, 2766, 2767, 2768, 2769, 2770,
           2771, 2772, 2773, 2774, 2775, 2776, 2777, 2778, 2779, 2780, 2781,
           2782, 2783, 2784, 2785, 2786, 2787, 2788, 2789, 2790, 2791, 2792,
           2793, 2794, 2795, 2796, 2797, 2798, 2799, 2800, 2801, 2802, 2803,
           2804, 2805, 2806, 2807, 2808, 2809, 2810, 2811, 2812, 2813, 2814,
           2815, 2816, 2817, 2818, 2819, 2820, 2821, 2822, 2823, 2824, 2825,
           2826, 2827, 2828, 2829, 2830, 2831, 2832, 2833, 2834, 2835, 2836,
           2837, 2838, 2839, 2840, 2841, 2842, 2843, 2844, 2845, 2846, 2847,
           2848, 2849, 2850, 2851, 2852, 2853, 2854, 2855, 2856, 2857, 2858,
           2859, 2860, 2861, 2862, 2863, 2864, 2865, 2866, 2867, 2868, 2869,
           2870, 2871, 2872, 2873, 2874, 2875, 2876, 2877, 2878, 2879, 2880,
           2881, 2882, 2883, 2884, 2885, 2886, 2887, 2888, 2889, 2890, 2891,
           2892, 2893, 2894, 2895, 2896, 2897, 2898, 2899, 2900, 2901, 2902,
           2903, 2904, 2905, 2906, 2907, 2908, 2909, 2910, 2911, 2912, 2913,
           2914, 2915, 2916, 2917, 2918, 2919, 2920, 2921, 2922, 2923, 2924,
           2925, 2926, 2927, 2928, 2929, 2930, 2931, 2932, 2933, 2934, 2935,
           2936, 2937, 2938, 2939, 420,  421,  422,  423,  424,  425,  426,
           427,  428,  429,  430,  431,  432,  433,  434,  435,  436,  437,
           438,  439,  440,  441,  442,  443,  444,  445,  446,  447,  448,
           449,  450,  451,  452,  453,  454,  455,  456,  457,  458,  459,
           460,  461,  462,  463,  464,  465,  466,  467,  468,  469,  470,
           471,  472,  473,  474,  475,  476,  477,  478,  479,  480,  481,
           482,  483,  484,  485,  486,  487,  488,  489,  490,  491,  492,
           493,  494,  495,  496,  497,  498,  499,  500,  501,  502,  503,
           504,  505,  506,  507,  508,  509,  510,  511,  512,  513,  514,
           515,  516,  517,  518,  519,  520,  521,  522,  523,  524,  525,
           526,  527,  528,  529,  530,  531,  532,  533,  534,  535,  536,
           537,  538,  539,  540,  541,  542,  543,  544,  545,  546,  547,
           548,  549,  550,  551,  552,  553,  554,  555,  556,  557,  558,
           559,  560,  561,  562,  563,  564,  565,  566,  567,  568,  569,
           570,  571,  572,  573,  574,  575,  576,  577,  578,  579,  580,
           581,  582,  583,  584,  585,  586,  587,  588,  589,  590,  591,
           592,  593,  594,  595,  596,  597,  598,  599,  600,  601,  602,
           603,  604,  605,  606,  607,  608,  609,  610,  611,  612,  613,
           614,  615,  616,  617,  618,  619,  620,  621,  622,  623,  624,
           625,  626,  627,  628,  629,  2940, 2941, 2942, 2943, 2944, 2945,
           2946, 2947, 2948, 2949, 2950, 2951, 2952, 2953, 2954, 2955, 2956,
           2957, 2958, 2959, 2960, 2961, 2962, 2963, 2964, 2965, 2966, 2967,
           2968, 2969, 2970, 2971, 2972, 2973, 2974, 2975, 2976, 2977, 2978,
           2979, 2980, 2981, 2982, 2983, 2984, 2985, 2986, 2987, 2988, 2989,
           2990, 2991, 2992, 2993, 2994, 2995, 2996, 2997, 2998, 2999, 3000,
           3001, 3002, 3003, 3004, 3005, 3006, 3007, 3008, 3009, 3010, 3011,
           3012, 3013, 3014, 3015, 3016, 3017, 3018, 3019, 3020, 3021, 3022,
           3023, 3024, 3025, 3026, 3027, 3028, 3029, 3030, 3031, 3032, 3033,
           3034, 3035, 3036, 3037, 3038, 3039, 3040, 3041, 3042, 3043, 3044,
           3045, 3046, 3047, 3048, 3049, 3050, 3051, 3052, 3053, 3054, 3055,
           3056, 3057, 3058, 3059, 3060, 3061, 3062, 3063, 3064, 3065, 3066,
           3067, 3068, 3069, 3070, 3071, 3072, 3073, 3074, 3075, 3076, 3077,
           3078, 3079, 3080, 3081, 3082, 3083, 3084, 3085, 3086, 3087, 3088,
           3089, 3090, 3091, 3092, 3093, 3094, 3095, 3096, 3097, 3098, 3099,
           3100, 3101, 3102, 3103, 3104, 3105, 3106, 3107, 3108, 3109, 3110,
           3111, 3112, 3113, 3114, 3115, 3116, 3117, 3118, 3119, 3120, 3121,
           3122, 3123, 3124, 3125, 3126, 3127, 3128, 3129, 3130, 3131, 3132,
           3133, 3134, 3135, 3136, 3137, 3138, 3139, 3140, 3141, 3142, 3143,
           3144, 3145, 3146, 3147, 3148, 3149, 630,  631,  632,  633,  634,
           635,  636,  637,  638,  639,  640,  641,  642,  643,  644,  645,
           646,  647,  648,  649,  650,  651,  652,  653,  654,  655,  656,
           657,  658,  659,  660,  661,  662,  663,  664,  665,  666,  667,
           668,  669,  670,  671,  672,  673,  674,  675,  676,  677,  678,
           679,  680,  681,  682,  683,  684,  685,  686,  687,  688,  689,
           690,  691,  692,  693,  694,  695,  696,  697,  698,  699,  700,
           701,  702,  703,  704,  705,  706,  707,  708,  709,  710,  711,
           712,  713,  714,  715,  716,  717,  718,  719,  720,  721,  722,
           723,  724,  725,  726,  727,  728,  729,  730,  731,  732,  733,
           734,  735,  736,  737,  738,  739,  740,  741,  742,  743,  744,
           745,  746,  747,  748,  749,  750,  751,  752,  753,  754,  755,
           756,  757,  758,  759,  760,  761,  762,  763,  764,  765,  766,
           767,  768,  769,  770,  771,  772,  773,  774,  775,  776,  777,
           778,  779,  780,  781,  782,  783,  784,  785,  786,  787,  788,
           789,  790,  791,  792,  793,  794,  795,  796,  797,  798,  799,
           800,  801,  802,  803,  804,  805,  806,  807,  808,  809,  810,
           811,  812,  813,  814,  815,  816,  817,  818,  819,  820,  821,
           822,  823,  824,  825,  826,  827,  828,  829,  830,  831,  832,
           833,  834,  835,  836,  837,  838,  839,  3150, 3151, 3152, 3153,
           3154, 3155, 3156, 3157, 3158, 3159, 3160, 3161, 3162, 3163, 3164,
           3165, 3166, 3167, 3168, 3169, 3170, 3171, 3172, 3173, 3174, 3175,
           3176, 3177, 3178, 3179, 3180, 3181, 3182, 3183, 3184, 3185, 3186,
           3187, 3188, 3189, 3190, 3191, 3192, 3193, 3194, 3195, 3196, 3197,
           3198, 3199, 3200, 3201, 3202, 3203, 3204, 3205, 3206, 3207, 3208,
           3209, 3210, 3211, 3212, 3213, 3214, 3215, 3216, 3217, 3218, 3219,
           3220, 3221, 3222, 3223, 3224, 3225, 3226, 3227, 3228, 3229, 3230,
           3231, 3232, 3233, 3234, 3235, 3236, 3237, 3238, 3239, 3240, 3241,
           3242, 3243, 3244, 3245, 3246, 3247, 3248, 3249, 3250, 3251, 3252,
           3253, 3254, 3255, 3256, 3257, 3258, 3259, 3260, 3261, 3262, 3263,
           3264, 3265, 3266, 3267, 3268, 3269, 3270, 3271, 3272, 3273, 3274,
           3275, 3276, 3277, 3278, 3279, 3280, 3281, 3282, 3283, 3284, 3285,
           3286, 3287, 3288, 3289, 3290, 3291, 3292, 3293, 3294, 3295, 3296,
           3297, 3298, 3299, 3300, 3301, 3302, 3303, 3304, 3305, 3306, 3307,
           3308, 3309, 3310, 3311, 3312, 3313, 3314, 3315, 3316, 3317, 3318,
           3319, 3320, 3321, 3322, 3323, 3324, 3325, 3326, 3327, 3328, 3329,
           3330, 3331, 3332, 3333, 3334, 3335, 3336, 3337, 3338, 3339, 3340,
           3341, 3342, 3343, 3344, 3345, 3346, 3347, 3348, 3349, 3350, 3351,
           3352, 3353, 3354, 3355, 3356, 3357, 3358, 3359, 840,  841,  842,
           843,  844,  845,  846,  847,  848,  849,  850,  851,  852,  853,
           854,  855,  856,  857,  858,  859,  860,  861,  862,  863,  864,
           865,  866,  867,  868,  869,  870,  871,  872,  873,  874,  875,
           876,  877,  878,  879,  880,  881,  882,  883,  884,  885,  886,
           887,  888,  889,  890,  891,  892,  893,  894,  895,  896,  897,
           898,  899,  900,  901,  902,  903,  904,  905,  906,  907,  908,
           909,  910,  911,  912,  913,  914,  915,  916,  917,  918,  919,
           920,  921,  922,  923,  924,  925,  926,  927,  928,  929,  930,
           931,  932,  933,  934,  935,  936,  937,  938,  939,  940,  941,
           942,  943,  944,  945,  946,  947,  948,  949,  950,  951,  952,
           953,  954,  955,  956,  957,  958,  959,  960,  961,  962,  963,
           964,  965,  966,  967,  968,  969,  970,  971,  972,  973,  974,
           975,  976,  977,  978,  979,  980,  981,  982,  983,  984,  985,
           986,  987,  988,  989,  990,  991,  992,  993,  994,  995,  996,
           997,  998,  999,  1000, 1001, 1002, 1003, 1004, 1005, 1006, 1007,
           1008, 1009, 1010, 1011, 1012, 1013, 1014, 1015, 1016, 1017, 1018,
           1019, 1020, 1021, 1022, 1023, 1024, 1025, 1026, 1027, 1028, 1029,
           1030, 1031, 1032, 1033, 1034, 1035, 1036, 1037, 1038, 1039, 1040,
           1041, 1042, 1043, 1044, 1045, 1046, 1047, 1048, 1049, 3360, 3361,
           3362, 3363, 3364, 3365, 3366, 3367, 3368, 3369, 3370, 3371, 3372,
           3373, 3374, 3375, 3376, 3377, 3378, 3379, 3380, 3381, 3382, 3383,
           3384, 3385, 3386, 3387, 3388, 3389, 3390, 3391, 3392, 3393, 3394,
           3395, 3396, 3397, 3398, 3399, 3400, 3401, 3402, 3403, 3404, 3405,
           3406, 3407, 3408, 3409, 3410, 3411, 3412, 3413, 3414, 3415, 3416,
           3417, 3418, 3419, 3420, 3421, 3422, 3423, 3424, 3425, 3426, 3427,
           3428, 3429, 3430, 3431, 3432, 3433, 3434, 3435, 3436, 3437, 3438,
           3439, 3440, 3441, 3442, 3443, 3444, 3445, 3446, 3447, 3448, 3449,
           3450, 3451, 3452, 3453, 3454, 3455, 3456, 3457, 3458, 3459, 3460,
           3461, 3462, 3463, 3464, 3465, 3466, 3467, 3468, 3469, 3470, 3471,
           3472, 3473, 3474, 3475, 3476, 3477, 3478, 3479, 3480, 3481, 3482,
           3483, 3484, 3485, 3486, 3487, 3488, 3489, 3490, 3491, 3492, 3493,
           3494, 3495, 3496, 3497, 3498, 3499, 3500, 3501, 3502, 3503, 3504,
           3505, 3506, 3507, 3508, 3509, 3510, 3511, 3512, 3513, 3514, 3515,
           3516, 3517, 3518, 3519, 3520, 3521, 3522, 3523, 3524, 3525, 3526,
           3527, 3528, 3529, 3530, 3531, 3532, 3533, 3534, 3535, 3536, 3537,
           3538, 3539, 3540, 3541, 3542, 3543, 3544, 3545, 3546, 3547, 3548,
           3549, 3550, 3551, 3552, 3553, 3554, 3555, 3556, 3557, 3558, 3559,
           3560, 3561, 3562, 3563, 3564, 3565, 3566, 3567, 3568, 3569, 1050,
           1051, 1052, 1053, 1054, 1055, 1056, 1057, 1058, 1059, 1060, 1061,
           1062, 1063, 1064, 1065, 1066, 1067, 1068, 1069, 1070, 1071, 1072,
           1073, 1074, 1075, 1076, 1077, 1078, 1079, 1080, 1081, 1082, 1083,
           1084, 1085, 1086, 1087, 1088, 1089, 1090, 1091, 1092, 1093, 1094,
           1095, 1096, 1097, 1098, 1099, 1100, 1101, 1102, 1103, 1104, 1105,
           1106, 1107, 1108, 1109, 1110, 1111, 1112, 1113, 1114, 1115, 1116,
           1117, 1118, 1119, 1120, 1121, 1122, 1123, 1124, 1125, 1126, 1127,
           1128, 1129, 1130, 1131, 1132, 1133, 1134, 1135, 1136, 1137, 1138,
           1139, 1140, 1141, 1142, 1143, 1144, 1145, 1146, 1147, 1148, 1149,
           1150, 1151, 1152, 1153, 1154, 1155, 1156, 1157, 1158, 1159, 1160,
           1161, 1162, 1163, 1164, 1165, 1166, 1167, 1168, 1169, 1170, 1171,
           1172, 1173, 1174, 1175, 1176, 1177, 1178, 1179, 1180, 1181, 1182,
           1183, 1184, 1185, 1186, 1187, 1188, 1189, 1190, 1191, 1192, 1193,
           1194, 1195, 1196, 1197, 1198, 1199, 1200, 1201, 1202, 1203, 1204,
           1205, 1206, 1207, 1208, 1209, 1210, 1211, 1212, 1213, 1214, 1215,
           1216, 1217, 1218, 1219, 1220, 1221, 1222, 1223, 1224, 1225, 1226,
           1227, 1228, 1229, 1230, 1231, 1232, 1233, 1234, 1235, 1236, 1237,
           1238, 1239, 1240, 1241, 1242, 1243, 1244, 1245, 1246, 1247, 1248,
           1249, 1250, 1251, 1252, 1253, 1254, 1255, 1256, 1257, 1258, 1259,
           3570, 3571, 3572, 3573, 3574, 3575, 3576, 3577, 3578, 3579, 3580,
           3581, 3582, 3583, 3584, 3585, 3586, 3587, 3588, 3589, 3590, 3591,
           3592, 3593, 3594, 3595, 3596, 3597, 3598, 3599, 3600, 3601, 3602,
           3603, 3604, 3605, 3606, 3607, 3608, 3609, 3610, 3611, 3612, 3613,
           3614, 3615, 3616, 3617, 3618, 3619, 3620, 3621, 3622, 3623, 3624,
           3625, 3626, 3627, 3628, 3629, 3630, 3631, 3632, 3633, 3634, 3635,
           3636, 3637, 3638, 3639, 3640, 3641, 3642, 3643, 3644, 3645, 3646,
           3647, 3648, 3649, 3650, 3651, 3652, 3653, 3654, 3655, 3656, 3657,
           3658, 3659, 3660, 3661, 3662, 3663, 3664, 3665, 3666, 3667, 3668,
           3669, 3670, 3671, 3672, 3673, 3674, 3675, 3676, 3677, 3678, 3679,
           3680, 3681, 3682, 3683, 3684, 3685, 3686, 3687, 3688, 3689, 3690,
           3691, 3692, 3693, 3694, 3695, 3696, 3697, 3698, 3699, 3700, 3701,
           3702, 3703, 3704, 3705, 3706, 3707, 3708, 3709, 3710, 3711, 3712,
           3713, 3714, 3715, 3716, 3717, 3718, 3719, 3720, 3721, 3722, 3723,
           3724, 3725, 3726, 3727, 3728, 3729, 3730, 3731, 3732, 3733, 3734,
           3735, 3736, 3737, 3738, 3739, 3740, 3741, 3742, 3743, 3744, 3745,
           3746, 3747, 3748, 3749, 3750, 3751, 3752, 3753, 3754, 3755, 3756,
           3757, 3758, 3759, 3760, 3761, 3762, 3763, 3764, 3765, 3766, 3767,
           3768, 3769, 3770, 3771, 3772, 3773, 3774, 3775, 3776, 3777, 3778,
           3779, 1260, 1261, 1262, 1263, 1264, 1265, 1266, 1267, 1268, 1269,
           1270, 1271, 1272, 1273, 1274, 1275, 1276, 1277, 1278, 1279, 1280,
           1281, 1282, 1283, 1284, 1285, 1286, 1287, 1288, 1289, 1290, 1291,
           1292, 1293, 1294, 1295, 1296, 1297, 1298, 1299, 1300, 1301, 1302,
           1303, 1304, 1305, 1306, 1307, 1308, 1309, 1310, 1311, 1312, 1313,
           1314, 1315, 1316, 1317, 1318, 1319, 1320, 1321, 1322, 1323, 1324,
           1325, 1326, 1327, 1328, 1329, 1330, 1331, 1332, 1333, 1334, 1335,
           1336, 1337, 1338, 1339, 1340, 1341, 1342, 1343, 1344, 1345, 1346,
           1347, 1348, 1349, 1350, 1351, 1352, 1353, 1354, 1355, 1356, 1357,
           1358, 1359, 1360, 1361, 1362, 1363, 1364, 1365, 1366, 1367, 1368,
           1369, 1370, 1371, 1372, 1373, 1374, 1375, 1376, 1377, 1378, 1379,
           1380, 1381, 1382, 1383, 1384, 1385, 1386, 1387, 1388, 1389, 1390,
           1391, 1392, 1393, 1394, 1395, 1396, 1397, 1398, 1399, 1400, 1401,
           1402, 1403, 1404, 1405, 1406, 1407, 1408, 1409, 1410, 1411, 1412,
           1413, 1414, 1415, 1416, 1417, 1418, 1419, 1420, 1421, 1422, 1423,
           1424, 1425, 1426, 1427, 1428, 1429, 1430, 1431, 1432, 1433, 1434,
           1435, 1436, 1437, 1438, 1439, 1440, 1441, 1442, 1443, 1444, 1445,
           1446, 1447, 1448, 1449, 1450, 1451, 1452, 1453, 1454, 1455, 1456,
           1457, 1458, 1459, 1460, 1461, 1462, 1463, 1464, 1465, 1466, 1467,
           1468, 1469, 3780, 3781, 3782, 3783, 3784, 3785, 3786, 3787, 3788,
           3789, 3790, 3791, 3792, 3793, 3794, 3795, 3796, 3797, 3798, 3799,
           3800, 3801, 3802, 3803, 3804, 3805, 3806, 3807, 3808, 3809, 3810,
           3811, 3812, 3813, 3814, 3815, 3816, 3817, 3818, 3819, 3820, 3821,
           3822, 3823, 3824, 3825, 3826, 3827, 3828, 3829, 3830, 3831, 3832,
           3833, 3834, 3835, 3836, 3837, 3838, 3839, 3840, 3841, 3842, 3843,
           3844, 3845, 3846, 3847, 3848, 3849, 3850, 3851, 3852, 3853, 3854,
           3855, 3856, 3857, 3858, 3859, 3860, 3861, 3862, 3863, 3864, 3865,
           3866, 3867, 3868, 3869, 3870, 3871, 3872, 3873, 3874, 3875, 3876,
           3877, 3878, 3879, 3880, 3881, 3882, 3883, 3884, 3885, 3886, 3887,
           3888, 3889, 3890, 3891, 3892, 3893, 3894, 3895, 3896, 3897, 3898,
           3899, 3900, 3901, 3902, 3903, 3904, 3905, 3906, 3907, 3908, 3909,
           3910, 3911, 3912, 3913, 3914, 3915, 3916, 3917, 3918, 3919, 3920,
           3921, 3922, 3923, 3924, 3925, 3926, 3927, 3928, 3929, 3930, 3931,
           3932, 3933, 3934, 3935, 3936, 3937, 3938, 3939, 3940, 3941, 3942,
           3943, 3944, 3945, 3946, 3947, 3948, 3949, 3950, 3951, 3952, 3953,
           3954, 3955, 3956, 3957, 3958, 3959, 3960, 3961, 3962, 3963, 3964,
           3965, 3966, 3967, 3968, 3969, 3970, 3971, 3972, 3973, 3974, 3975,
           3976, 3977, 3978, 3979, 3980, 3981, 3982, 3983, 3984, 3985, 3986,
           3987, 3988, 3989, 1470, 1471, 1472, 1473, 1474, 1475, 1476, 1477,
           1478, 1479, 1480, 1481, 1482, 1483, 1484, 1485, 1486, 1487, 1488,
           1489, 1490, 1491, 1492, 1493, 1494, 1495, 1496, 1497, 1498, 1499,
           1500, 1501, 1502, 1503, 1504, 1505, 1506, 1507, 1508, 1509, 1510,
           1511, 1512, 1513, 1514, 1515, 1516, 1517, 1518, 1519, 1520, 1521,
           1522, 1523, 1524, 1525, 1526, 1527, 1528, 1529, 1530, 1531, 1532,
           1533, 1534, 1535, 1536, 1537, 1538, 1539, 1540, 1541, 1542, 1543,
           1544, 1545, 1546, 1547, 1548, 1549, 1550, 1551, 1552, 1553, 1554,
           1555, 1556, 1557, 1558, 1559, 1560, 1561, 1562, 1563, 1564, 1565,
           1566, 1567, 1568, 1569, 1570, 1571, 1572, 1573, 1574, 1575, 1576,
           1577, 1578, 1579, 1580, 1581, 1582, 1583, 1584, 1585, 1586, 1587,
           1588, 1589, 1590, 1591, 1592, 1593, 1594, 1595, 1596, 1597, 1598,
           1599, 1600, 1601, 1602, 1603, 1604, 1605, 1606, 1607, 1608, 1609,
           1610, 1611, 1612, 1613, 1614, 1615, 1616, 1617, 1618, 1619, 1620,
           1621, 1622, 1623, 1624, 1625, 1626, 1627, 1628, 1629, 1630, 1631,
           1632, 1633, 1634, 1635, 1636, 1637, 1638, 1639, 1640, 1641, 1642,
           1643, 1644, 1645, 1646, 1647, 1648, 1649, 1650, 1651, 1652, 1653,
           1654, 1655, 1656, 1657, 1658, 1659, 1660, 1661, 1662, 1663, 1664,
           1665, 1666, 1667, 1668, 1669, 1670, 1671, 1672, 1673, 1674, 1675,
           1676, 1677, 1678, 1679, 3990, 3991, 3992, 3993, 3994, 3995, 3996,
           3997, 3998, 3999, 4000, 4001, 4002, 4003, 4004, 4005, 4006, 4007,
           4008, 4009, 4010, 4011, 4012, 4013, 4014, 4015, 4016, 4017, 4018,
           4019, 4020, 4021, 4022, 4023, 4024, 4025, 4026, 4027, 4028, 4029,
           4030, 4031, 4032, 4033, 4034, 4035, 4036, 4037, 4038, 4039, 4040,
           4041, 4042, 4043, 4044, 4045, 4046, 4047, 4048, 4049, 4050, 4051,
           4052, 4053, 4054, 4055, 4056, 4057, 4058, 4059, 4060, 4061, 4062,
           4063, 4064, 4065, 4066, 4067, 4068, 4069, 4070, 4071, 4072, 4073,
           4074, 4075, 4076, 4077, 4078, 4079, 4080, 4081, 4082, 4083, 4084,
           4085, 4086, 4087, 4088, 4089, 4090, 4091, 4092, 4093, 4094, 4095,
           4096, 4097, 4098, 4099, 4100, 4101, 4102, 4103, 4104, 4105, 4106,
           4107, 4108, 4109, 4110, 4111, 4112, 4113, 4114, 4115, 4116, 4117,
           4118, 4119, 4120, 4121, 4122, 4123, 4124, 4125, 4126, 4127, 4128,
           4129, 4130, 4131, 4132, 4133, 4134, 4135, 4136, 4137, 4138, 4139,
           4140, 4141, 4142, 4143, 4144, 4145, 4146, 4147, 4148, 4149, 4150,
           4151, 4152, 4153, 4154, 4155, 4156, 4157, 4158, 4159, 4160, 4161,
           4162, 4163, 4164, 4165, 4166, 4167, 4168, 4169, 4170, 4171, 4172,
           4173, 4174, 4175, 4176, 4177, 4178, 4179, 4180, 4181, 4182, 4183,
           4184, 4185, 4186, 4187, 4188, 4189, 4190, 4191, 4192, 4193, 4194,
           4195, 4196, 4197, 4198, 4199, 1680, 1681, 1682, 1683, 1684, 1685,
           1686, 1687, 1688, 1689, 1690, 1691, 1692, 1693, 1694, 1695, 1696,
           1697, 1698, 1699, 1700, 1701, 1702, 1703, 1704, 1705, 1706, 1707,
           1708, 1709, 1710, 1711, 1712, 1713, 1714, 1715, 1716, 1717, 1718,
           1719, 1720, 1721, 1722, 1723, 1724, 1725, 1726, 1727, 1728, 1729,
           1730, 1731, 1732, 1733, 1734, 1735, 1736, 1737, 1738, 1739, 1740,
           1741, 1742, 1743, 1744, 1745, 1746, 1747, 1748, 1749, 1750, 1751,
           1752, 1753, 1754, 1755, 1756, 1757, 1758, 1759, 1760, 1761, 1762,
           1763, 1764, 1765, 1766, 1767, 1768, 1769, 1770, 1771, 1772, 1773,
           1774, 1775, 1776, 1777, 1778, 1779, 1780, 1781, 1782, 1783, 1784,
           1785, 1786, 1787, 1788, 1789, 1790, 1791, 1792, 1793, 1794, 1795,
           1796, 1797, 1798, 1799, 1800, 1801, 1802, 1803, 1804, 1805, 1806,
           1807, 1808, 1809, 1810, 1811, 1812, 1813, 1814, 1815, 1816, 1817,
           1818, 1819, 1820, 1821, 1822, 1823, 1824, 1825, 1826, 1827, 1828,
           1829, 1830, 1831, 1832, 1833, 1834, 1835, 1836, 1837, 1838, 1839,
           1840, 1841, 1842, 1843, 1844, 1845, 1846, 1847, 1848, 1849, 1850,
           1851, 1852, 1853, 1854, 1855, 1856, 1857, 1858, 1859, 1860, 1861,
           1862, 1863, 1864, 1865, 1866, 1867, 1868, 1869, 1870, 1871, 1872,
           1873, 1874, 1875, 1876, 1877, 1878, 1879, 1880, 1881, 1882, 1883,
           1884, 1885, 1886, 1887, 1888, 1889, 4200, 4201, 4202, 4203, 4204,
           4205, 4206, 4207, 4208, 4209, 4210, 4211, 4212, 4213, 4214, 4215,
           4216, 4217, 4218, 4219, 4220, 4221, 4222, 4223, 4224, 4225, 4226,
           4227, 4228, 4229, 4230, 4231, 4232, 4233, 4234, 4235, 4236, 4237,
           4238, 4239, 4240, 4241, 4242, 4243, 4244, 4245, 4246, 4247, 4248,
           4249, 4250, 4251, 4252, 4253, 4254, 4255, 4256, 4257, 4258, 4259,
           4260, 4261, 4262, 4263, 4264, 4265, 4266, 4267, 4268, 4269, 4270,
           4271, 4272, 4273, 4274, 4275, 4276, 4277, 4278, 4279, 4280, 4281,
           4282, 4283, 4284, 4285, 4286, 4287, 4288, 4289, 4290, 4291, 4292,
           4293, 4294, 4295, 4296, 4297, 4298, 4299, 4300, 4301, 4302, 4303,
           4304, 4305, 4306, 4307, 4308, 4309, 4310, 4311, 4312, 4313, 4314,
           4315, 4316, 4317, 4318, 4319, 4320, 4321, 4322, 4323, 4324, 4325,
           4326, 4327, 4328, 4329, 4330, 4331, 4332, 4333, 4334, 4335, 4336,
           4337, 4338, 4339, 4340, 4341, 4342, 4343, 4344, 4345, 4346, 4347,
           4348, 4349, 4350, 4351, 4352, 4353, 4354, 4355, 4356, 4357, 4358,
           4359, 4360, 4361, 4362, 4363, 4364, 4365, 4366, 4367, 4368, 4369,
           4370, 4371, 4372, 4373, 4374, 4375, 4376, 4377, 4378, 4379, 4380,
           4381, 4382, 4383, 4384, 4385, 4386, 4387, 4388, 4389, 4390, 4391,
           4392, 4393, 4394, 4395, 4396, 4397, 4398, 4399, 4400, 4401, 4402,
           4403, 4404, 4405, 4406, 4407, 4408, 4409, 1890, 1891, 1892, 1893,
           1894, 1895, 1896, 1897, 1898, 1899, 1900, 1901, 1902, 1903, 1904,
           1905, 1906, 1907, 1908, 1909, 1910, 1911, 1912, 1913, 1914, 1915,
           1916, 1917, 1918, 1919, 1920, 1921, 1922, 1923, 1924, 1925, 1926,
           1927, 1928, 1929, 1930, 1931, 1932, 1933, 1934, 1935, 1936, 1937,
           1938, 1939, 1940, 1941, 1942, 1943, 1944, 1945, 1946, 1947, 1948,
           1949, 1950, 1951, 1952, 1953, 1954, 1955, 1956, 1957, 1958, 1959,
           1960, 1961, 1962, 1963, 1964, 1965, 1966, 1967, 1968, 1969, 1970,
           1971, 1972, 1973, 1974, 1975, 1976, 1977, 1978, 1979, 1980, 1981,
           1982, 1983, 1984, 1985, 1986, 1987, 1988, 1989, 1990, 1991, 1992,
           1993, 1994, 1995, 1996, 1997, 1998, 1999, 2000, 2001, 2002, 2003,
           2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014,
           2015, 2016, 2017, 2018, 2019, 2020, 2021, 2022, 2023, 2024, 2025,
           2026, 2027, 2028, 2029, 2030, 2031, 2032, 2033, 2034, 2035, 2036,
           2037, 2038, 2039, 2040, 2041, 2042, 2043, 2044, 2045, 2046, 2047,
           2048, 2049, 2050, 2051, 2052, 2053, 2054, 2055, 2056, 2057, 2058,
           2059, 2060, 2061, 2062, 2063, 2064, 2065, 2066, 2067, 2068, 2069,
           2070, 2071, 2072, 2073, 2074, 2075, 2076, 2077, 2078, 2079, 2080,
           2081, 2082, 2083, 2084, 2085, 2086, 2087, 2088, 2089, 2090, 2091,
           2092, 2093, 2094, 2095, 2096, 2097, 2098, 2099, 4410, 4411, 4412,
           4413, 4414, 4415, 4416, 4417, 4418, 4419, 4420, 4421, 4422, 4423,
           4424, 4425, 4426, 4427, 4428, 4429, 4430, 4431, 4432, 4433, 4434,
           4435, 4436, 4437, 4438, 4439, 4440, 4441, 4442, 4443, 4444, 4445,
           4446, 4447, 4448, 4449, 4450, 4451, 4452, 4453, 4454, 4455, 4456,
           4457, 4458, 4459, 4460, 4461, 4462, 4463, 4464, 4465, 4466, 4467,
           4468, 4469, 4470, 4471, 4472, 4473, 4474, 4475, 4476, 4477, 4478,
           4479, 4480, 4481, 4482, 4483, 4484, 4485, 4486, 4487, 4488, 4489,
           4490, 4491, 4492, 4493, 4494, 4495, 4496, 4497, 4498, 4499, 4500,
           4501, 4502, 4503, 4504, 4505, 4506, 4507, 4508, 4509, 4510, 4511,
           4512, 4513, 4514, 4515, 4516, 4517, 4518, 4519, 4520, 4521, 4522,
           4523, 4524, 4525, 4526, 4527, 4528, 4529, 4530, 4531, 4532, 4533,
           4534, 4535, 4536, 4537, 4538, 4539, 4540, 4541, 4542, 4543, 4544,
           4545, 4546, 4547, 4548, 4549, 4550, 4551, 4552, 4553, 4554, 4555,
           4556, 4557, 4558, 4559, 4560, 4561, 4562, 4563, 4564, 4565, 4566,
           4567, 4568, 4569, 4570, 4571, 4572, 4573, 4574, 4575, 4576, 4577,
           4578, 4579, 4580, 4581, 4582, 4583, 4584, 4585, 4586, 4587, 4588,
           4589, 4590, 4591, 4592, 4593, 4594, 4595, 4596, 4597, 4598, 4599,
           4600, 4601, 4602, 4603, 4604, 4605, 4606, 4607, 4608, 4609, 4610,
           4611, 4612, 4613, 4614, 4615, 4616, 4617, 4618, 4619, 2100, 2101,
           2102, 2103, 2104, 2105, 2106, 2107, 2108, 2109, 2110, 2111, 2112,
           2113, 2114, 2115, 2116, 2117, 2118, 2119, 2120, 2121, 2122, 2123,
           2124, 2125, 2126, 2127, 2128, 2129, 2130, 2131, 2132, 2133, 2134,
           2135, 2136, 2137, 2138, 2139, 2140, 2141, 2142, 2143, 2144, 2145,
           2146, 2147, 2148, 2149, 2150, 2151, 2152, 2153, 2154, 2155, 2156,
           2157, 2158, 2159, 2160, 2161, 2162, 2163, 2164, 2165, 2166, 2167,
           2168, 2169, 2170, 2171, 2172, 2173, 2174, 2175, 2176, 2177, 2178,
           2179, 2180, 2181, 2182, 2183, 2184, 2185, 2186, 2187, 2188, 2189,
           2190, 2191, 2192, 2193, 2194, 2195, 2196, 2197, 2198, 2199, 2200,
           2201, 2202, 2203, 2204, 2205, 2206, 2207, 2208, 2209, 2210, 2211,
           2212, 2213, 2214, 2215, 2216, 2217, 2218, 2219, 2220, 2221, 2222,
           2223, 2224, 2225, 2226, 2227, 2228, 2229, 2230, 2231, 2232, 2233,
           2234, 2235, 2236, 2237, 2238, 2239, 2240, 2241, 2242, 2243, 2244,
           2245, 2246, 2247, 2248, 2249, 2250, 2251, 2252, 2253, 2254, 2255,
           2256, 2257, 2258, 2259, 2260, 2261, 2262, 2263, 2264, 2265, 2266,
           2267, 2268, 2269, 2270, 2271, 2272, 2273, 2274, 2275, 2276, 2277,
           2278, 2279, 2280, 2281, 2282, 2283, 2284, 2285, 2286, 2287, 2288,
           2289, 2290, 2291, 2292, 2293, 2294, 2295, 2296, 2297, 2298, 2299,
           2300, 2301, 2302, 2303, 2304, 2305, 2306, 2307, 2308, 2309, 4620,
           4621, 4622, 4623, 4624, 4625, 4626, 4627, 4628, 4629, 4630, 4631,
           4632, 4633, 4634, 4635, 4636, 4637, 4638, 4639, 4640, 4641, 4642,
           4643, 4644, 4645, 4646, 4647, 4648, 4649, 4650, 4651, 4652, 4653,
           4654, 4655, 4656, 4657, 4658, 4659, 4660, 4661, 4662, 4663, 4664,
           4665, 4666, 4667, 4668, 4669, 4670, 4671, 4672, 4673, 4674, 4675,
           4676, 4677, 4678, 4679, 4680, 4681, 4682, 4683, 4684, 4685, 4686,
           4687, 4688, 4689, 4690, 4691, 4692, 4693, 4694, 4695, 4696, 4697,
           4698, 4699, 4700, 4701, 4702, 4703, 4704, 4705, 4706, 4707, 4708,
           4709, 4710, 4711, 4712, 4713, 4714, 4715, 4716, 4717, 4718, 4719,
           4720, 4721, 4722, 4723, 4724, 4725, 4726, 4727, 4728, 4729, 4730,
           4731, 4732, 4733, 4734, 4735, 4736, 4737, 4738, 4739, 4740, 4741,
           4742, 4743, 4744, 4745, 4746, 4747, 4748, 4749, 4750, 4751, 4752,
           4753, 4754, 4755, 4756, 4757, 4758, 4759, 4760, 4761, 4762, 4763,
           4764, 4765, 4766, 4767, 4768, 4769, 4770, 4771, 4772, 4773, 4774,
           4775, 4776, 4777, 4778, 4779, 4780, 4781, 4782, 4783, 4784, 4785,
           4786, 4787, 4788, 4789, 4790, 4791, 4792, 4793, 4794, 4795, 4796,
           4797, 4798, 4799, 4800, 4801, 4802, 4803, 4804, 4805, 4806, 4807,
           4808, 4809, 4810, 4811, 4812, 4813, 4814, 4815, 4816, 4817, 4818,
           4819, 4820, 4821, 4822, 4823, 4824, 4825, 4826, 4827, 4828, 4829,
           2310, 2311, 2312, 2313, 2314, 2315, 2316, 2317, 2318, 2319, 2320,
           2321, 2322, 2323, 2324, 2325, 2326, 2327, 2328, 2329, 2330, 2331,
           2332, 2333, 2334, 2335, 2336, 2337, 2338, 2339, 2340, 2341, 2342,
           2343, 2344, 2345, 2346, 2347, 2348, 2349, 2350, 2351, 2352, 2353,
           2354, 2355, 2356, 2357, 2358, 2359, 2360, 2361, 2362, 2363, 2364,
           2365, 2366, 2367, 2368, 2369, 2370, 2371, 2372, 2373, 2374, 2375,
           2376, 2377, 2378, 2379, 2380, 2381, 2382, 2383, 2384, 2385, 2386,
           2387, 2388, 2389, 2390, 2391, 2392, 2393, 2394, 2395, 2396, 2397,
           2398, 2399, 2400, 2401, 2402, 2403, 2404, 2405, 2406, 2407, 2408,
           2409, 2410, 2411, 2412, 2413, 2414, 2415, 2416, 2417, 2418, 2419,
           2420, 2421, 2422, 2423, 2424, 2425, 2426, 2427, 2428, 2429, 2430,
           2431, 2432, 2433, 2434, 2435, 2436, 2437, 2438, 2439, 2440, 2441,
           2442, 2443, 2444, 2445, 2446, 2447, 2448, 2449, 2450, 2451, 2452,
           2453, 2454, 2455, 2456, 2457, 2458, 2459, 2460, 2461, 2462, 2463,
           2464, 2465, 2466, 2467, 2468, 2469, 2470, 2471, 2472, 2473, 2474,
           2475, 2476, 2477, 2478, 2479, 2480, 2481, 2482, 2483, 2484, 2485,
           2486, 2487, 2488, 2489, 2490, 2491, 2492, 2493, 2494, 2495, 2496,
           2497, 2498, 2499, 2500, 2501, 2502, 2503, 2504, 2505, 2506, 2507,
           2508, 2509, 2510, 2511, 2512, 2513, 2514, 2515, 2516, 2517, 2518,
           2519, 4830, 4831, 4832, 4833, 4834, 4835, 4836, 4837, 4838, 4839,
           4840, 4841, 4842, 4843, 4844, 4845, 4846, 4847, 4848, 4849, 4850,
           4851, 4852, 4853, 4854, 4855, 4856, 4857, 4858, 4859, 4860, 4861,
           4862, 4863, 4864, 4865, 4866, 4867, 4868, 4869, 4870, 4871, 4872,
           4873, 4874, 4875, 4876, 4877, 4878, 4879, 4880, 4881, 4882, 4883,
           4884, 4885, 4886, 4887, 4888, 4889, 4890, 4891, 4892, 4893, 4894,
           4895, 4896, 4897, 4898, 4899, 4900, 4901, 4902, 4903, 4904, 4905,
           4906, 4907, 4908, 4909, 4910, 4911, 4912, 4913, 4914, 4915, 4916,
           4917, 4918, 4919, 4920, 4921, 4922, 4923, 4924, 4925, 4926, 4927,
           4928, 4929, 4930, 4931, 4932, 4933, 4934, 4935, 4936, 4937, 4938,
           4939, 4940, 4941, 4942, 4943, 4944, 4945, 4946, 4947, 4948, 4949,
           4950, 4951, 4952, 4953, 4954, 4955, 4956, 4957, 4958, 4959, 4960,
           4961, 4962, 4963, 4964, 4965, 4966, 4967, 4968, 4969, 4970, 4971,
           4972, 4973, 4974, 4975, 4976, 4977, 4978, 4979, 4980, 4981, 4982,
           4983, 4984, 4985, 4986, 4987, 4988, 4989, 4990, 4991, 4992, 4993,
           4994, 4995, 4996, 4997, 4998, 4999, 5000, 5001, 5002, 5003, 5004,
           5005, 5006, 5007, 5008, 5009, 5010, 5011, 5012, 5013, 5014, 5015,
           5016, 5017, 5018, 5019, 5020, 5021, 5022, 5023, 5024, 5025, 5026,
           5027, 5028, 5029, 5030, 5031, 5032, 5033, 5034, 5035, 5036, 5037,
           5038, 5039}));
}

TEST(TransposeTest, TestRefOps6D4) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5, 6, 7},
                                /*perms=*/{1, 2, 3, 0, 4, 5}),
      ElementsAreArray(
          {0,    1,    2,    3,    4,    5,    6,    7,    8,    9,    10,
           11,   12,   13,   14,   15,   16,   17,   18,   19,   20,   21,
           22,   23,   24,   25,   26,   27,   28,   29,   30,   31,   32,
           33,   34,   35,   36,   37,   38,   39,   40,   41,   2520, 2521,
           2522, 2523, 2524, 2525, 2526, 2527, 2528, 2529, 2530, 2531, 2532,
           2533, 2534, 2535, 2536, 2537, 2538, 2539, 2540, 2541, 2542, 2543,
           2544, 2545, 2546, 2547, 2548, 2549, 2550, 2551, 2552, 2553, 2554,
           2555, 2556, 2557, 2558, 2559, 2560, 2561, 42,   43,   44,   45,
           46,   47,   48,   49,   50,   51,   52,   53,   54,   55,   56,
           57,   58,   59,   60,   61,   62,   63,   64,   65,   66,   67,
           68,   69,   70,   71,   72,   73,   74,   75,   76,   77,   78,
           79,   80,   81,   82,   83,   2562, 2563, 2564, 2565, 2566, 2567,
           2568, 2569, 2570, 2571, 2572, 2573, 2574, 2575, 2576, 2577, 2578,
           2579, 2580, 2581, 2582, 2583, 2584, 2585, 2586, 2587, 2588, 2589,
           2590, 2591, 2592, 2593, 2594, 2595, 2596, 2597, 2598, 2599, 2600,
           2601, 2602, 2603, 84,   85,   86,   87,   88,   89,   90,   91,
           92,   93,   94,   95,   96,   97,   98,   99,   100,  101,  102,
           103,  104,  105,  106,  107,  108,  109,  110,  111,  112,  113,
           114,  115,  116,  117,  118,  119,  120,  121,  122,  123,  124,
           125,  2604, 2605, 2606, 2607, 2608, 2609, 2610, 2611, 2612, 2613,
           2614, 2615, 2616, 2617, 2618, 2619, 2620, 2621, 2622, 2623, 2624,
           2625, 2626, 2627, 2628, 2629, 2630, 2631, 2632, 2633, 2634, 2635,
           2636, 2637, 2638, 2639, 2640, 2641, 2642, 2643, 2644, 2645, 126,
           127,  128,  129,  130,  131,  132,  133,  134,  135,  136,  137,
           138,  139,  140,  141,  142,  143,  144,  145,  146,  147,  148,
           149,  150,  151,  152,  153,  154,  155,  156,  157,  158,  159,
           160,  161,  162,  163,  164,  165,  166,  167,  2646, 2647, 2648,
           2649, 2650, 2651, 2652, 2653, 2654, 2655, 2656, 2657, 2658, 2659,
           2660, 2661, 2662, 2663, 2664, 2665, 2666, 2667, 2668, 2669, 2670,
           2671, 2672, 2673, 2674, 2675, 2676, 2677, 2678, 2679, 2680, 2681,
           2682, 2683, 2684, 2685, 2686, 2687, 168,  169,  170,  171,  172,
           173,  174,  175,  176,  177,  178,  179,  180,  181,  182,  183,
           184,  185,  186,  187,  188,  189,  190,  191,  192,  193,  194,
           195,  196,  197,  198,  199,  200,  201,  202,  203,  204,  205,
           206,  207,  208,  209,  2688, 2689, 2690, 2691, 2692, 2693, 2694,
           2695, 2696, 2697, 2698, 2699, 2700, 2701, 2702, 2703, 2704, 2705,
           2706, 2707, 2708, 2709, 2710, 2711, 2712, 2713, 2714, 2715, 2716,
           2717, 2718, 2719, 2720, 2721, 2722, 2723, 2724, 2725, 2726, 2727,
           2728, 2729, 210,  211,  212,  213,  214,  215,  216,  217,  218,
           219,  220,  221,  222,  223,  224,  225,  226,  227,  228,  229,
           230,  231,  232,  233,  234,  235,  236,  237,  238,  239,  240,
           241,  242,  243,  244,  245,  246,  247,  248,  249,  250,  251,
           2730, 2731, 2732, 2733, 2734, 2735, 2736, 2737, 2738, 2739, 2740,
           2741, 2742, 2743, 2744, 2745, 2746, 2747, 2748, 2749, 2750, 2751,
           2752, 2753, 2754, 2755, 2756, 2757, 2758, 2759, 2760, 2761, 2762,
           2763, 2764, 2765, 2766, 2767, 2768, 2769, 2770, 2771, 252,  253,
           254,  255,  256,  257,  258,  259,  260,  261,  262,  263,  264,
           265,  266,  267,  268,  269,  270,  271,  272,  273,  274,  275,
           276,  277,  278,  279,  280,  281,  282,  283,  284,  285,  286,
           287,  288,  289,  290,  291,  292,  293,  2772, 2773, 2774, 2775,
           2776, 2777, 2778, 2779, 2780, 2781, 2782, 2783, 2784, 2785, 2786,
           2787, 2788, 2789, 2790, 2791, 2792, 2793, 2794, 2795, 2796, 2797,
           2798, 2799, 2800, 2801, 2802, 2803, 2804, 2805, 2806, 2807, 2808,
           2809, 2810, 2811, 2812, 2813, 294,  295,  296,  297,  298,  299,
           300,  301,  302,  303,  304,  305,  306,  307,  308,  309,  310,
           311,  312,  313,  314,  315,  316,  317,  318,  319,  320,  321,
           322,  323,  324,  325,  326,  327,  328,  329,  330,  331,  332,
           333,  334,  335,  2814, 2815, 2816, 2817, 2818, 2819, 2820, 2821,
           2822, 2823, 2824, 2825, 2826, 2827, 2828, 2829, 2830, 2831, 2832,
           2833, 2834, 2835, 2836, 2837, 2838, 2839, 2840, 2841, 2842, 2843,
           2844, 2845, 2846, 2847, 2848, 2849, 2850, 2851, 2852, 2853, 2854,
           2855, 336,  337,  338,  339,  340,  341,  342,  343,  344,  345,
           346,  347,  348,  349,  350,  351,  352,  353,  354,  355,  356,
           357,  358,  359,  360,  361,  362,  363,  364,  365,  366,  367,
           368,  369,  370,  371,  372,  373,  374,  375,  376,  377,  2856,
           2857, 2858, 2859, 2860, 2861, 2862, 2863, 2864, 2865, 2866, 2867,
           2868, 2869, 2870, 2871, 2872, 2873, 2874, 2875, 2876, 2877, 2878,
           2879, 2880, 2881, 2882, 2883, 2884, 2885, 2886, 2887, 2888, 2889,
           2890, 2891, 2892, 2893, 2894, 2895, 2896, 2897, 378,  379,  380,
           381,  382,  383,  384,  385,  386,  387,  388,  389,  390,  391,
           392,  393,  394,  395,  396,  397,  398,  399,  400,  401,  402,
           403,  404,  405,  406,  407,  408,  409,  410,  411,  412,  413,
           414,  415,  416,  417,  418,  419,  2898, 2899, 2900, 2901, 2902,
           2903, 2904, 2905, 2906, 2907, 2908, 2909, 2910, 2911, 2912, 2913,
           2914, 2915, 2916, 2917, 2918, 2919, 2920, 2921, 2922, 2923, 2924,
           2925, 2926, 2927, 2928, 2929, 2930, 2931, 2932, 2933, 2934, 2935,
           2936, 2937, 2938, 2939, 420,  421,  422,  423,  424,  425,  426,
           427,  428,  429,  430,  431,  432,  433,  434,  435,  436,  437,
           438,  439,  440,  441,  442,  443,  444,  445,  446,  447,  448,
           449,  450,  451,  452,  453,  454,  455,  456,  457,  458,  459,
           460,  461,  2940, 2941, 2942, 2943, 2944, 2945, 2946, 2947, 2948,
           2949, 2950, 2951, 2952, 2953, 2954, 2955, 2956, 2957, 2958, 2959,
           2960, 2961, 2962, 2963, 2964, 2965, 2966, 2967, 2968, 2969, 2970,
           2971, 2972, 2973, 2974, 2975, 2976, 2977, 2978, 2979, 2980, 2981,
           462,  463,  464,  465,  466,  467,  468,  469,  470,  471,  472,
           473,  474,  475,  476,  477,  478,  479,  480,  481,  482,  483,
           484,  485,  486,  487,  488,  489,  490,  491,  492,  493,  494,
           495,  496,  497,  498,  499,  500,  501,  502,  503,  2982, 2983,
           2984, 2985, 2986, 2987, 2988, 2989, 2990, 2991, 2992, 2993, 2994,
           2995, 2996, 2997, 2998, 2999, 3000, 3001, 3002, 3003, 3004, 3005,
           3006, 3007, 3008, 3009, 3010, 3011, 3012, 3013, 3014, 3015, 3016,
           3017, 3018, 3019, 3020, 3021, 3022, 3023, 504,  505,  506,  507,
           508,  509,  510,  511,  512,  513,  514,  515,  516,  517,  518,
           519,  520,  521,  522,  523,  524,  525,  526,  527,  528,  529,
           530,  531,  532,  533,  534,  535,  536,  537,  538,  539,  540,
           541,  542,  543,  544,  545,  3024, 3025, 3026, 3027, 3028, 3029,
           3030, 3031, 3032, 3033, 3034, 3035, 3036, 3037, 3038, 3039, 3040,
           3041, 3042, 3043, 3044, 3045, 3046, 3047, 3048, 3049, 3050, 3051,
           3052, 3053, 3054, 3055, 3056, 3057, 3058, 3059, 3060, 3061, 3062,
           3063, 3064, 3065, 546,  547,  548,  549,  550,  551,  552,  553,
           554,  555,  556,  557,  558,  559,  560,  561,  562,  563,  564,
           565,  566,  567,  568,  569,  570,  571,  572,  573,  574,  575,
           576,  577,  578,  579,  580,  581,  582,  583,  584,  585,  586,
           587,  3066, 3067, 3068, 3069, 3070, 3071, 3072, 3073, 3074, 3075,
           3076, 3077, 3078, 3079, 3080, 3081, 3082, 3083, 3084, 3085, 3086,
           3087, 3088, 3089, 3090, 3091, 3092, 3093, 3094, 3095, 3096, 3097,
           3098, 3099, 3100, 3101, 3102, 3103, 3104, 3105, 3106, 3107, 588,
           589,  590,  591,  592,  593,  594,  595,  596,  597,  598,  599,
           600,  601,  602,  603,  604,  605,  606,  607,  608,  609,  610,
           611,  612,  613,  614,  615,  616,  617,  618,  619,  620,  621,
           622,  623,  624,  625,  626,  627,  628,  629,  3108, 3109, 3110,
           3111, 3112, 3113, 3114, 3115, 3116, 3117, 3118, 3119, 3120, 3121,
           3122, 3123, 3124, 3125, 3126, 3127, 3128, 3129, 3130, 3131, 3132,
           3133, 3134, 3135, 3136, 3137, 3138, 3139, 3140, 3141, 3142, 3143,
           3144, 3145, 3146, 3147, 3148, 3149, 630,  631,  632,  633,  634,
           635,  636,  637,  638,  639,  640,  641,  642,  643,  644,  645,
           646,  647,  648,  649,  650,  651,  652,  653,  654,  655,  656,
           657,  658,  659,  660,  661,  662,  663,  664,  665,  666,  667,
           668,  669,  670,  671,  3150, 3151, 3152, 3153, 3154, 3155, 3156,
           3157, 3158, 3159, 3160, 3161, 3162, 3163, 3164, 3165, 3166, 3167,
           3168, 3169, 3170, 3171, 3172, 3173, 3174, 3175, 3176, 3177, 3178,
           3179, 3180, 3181, 3182, 3183, 3184, 3185, 3186, 3187, 3188, 3189,
           3190, 3191, 672,  673,  674,  675,  676,  677,  678,  679,  680,
           681,  682,  683,  684,  685,  686,  687,  688,  689,  690,  691,
           692,  693,  694,  695,  696,  697,  698,  699,  700,  701,  702,
           703,  704,  705,  706,  707,  708,  709,  710,  711,  712,  713,
           3192, 3193, 3194, 3195, 3196, 3197, 3198, 3199, 3200, 3201, 3202,
           3203, 3204, 3205, 3206, 3207, 3208, 3209, 3210, 3211, 3212, 3213,
           3214, 3215, 3216, 3217, 3218, 3219, 3220, 3221, 3222, 3223, 3224,
           3225, 3226, 3227, 3228, 3229, 3230, 3231, 3232, 3233, 714,  715,
           716,  717,  718,  719,  720,  721,  722,  723,  724,  725,  726,
           727,  728,  729,  730,  731,  732,  733,  734,  735,  736,  737,
           738,  739,  740,  741,  742,  743,  744,  745,  746,  747,  748,
           749,  750,  751,  752,  753,  754,  755,  3234, 3235, 3236, 3237,
           3238, 3239, 3240, 3241, 3242, 3243, 3244, 3245, 3246, 3247, 3248,
           3249, 3250, 3251, 3252, 3253, 3254, 3255, 3256, 3257, 3258, 3259,
           3260, 3261, 3262, 3263, 3264, 3265, 3266, 3267, 3268, 3269, 3270,
           3271, 3272, 3273, 3274, 3275, 756,  757,  758,  759,  760,  761,
           762,  763,  764,  765,  766,  767,  768,  769,  770,  771,  772,
           773,  774,  775,  776,  777,  778,  779,  780,  781,  782,  783,
           784,  785,  786,  787,  788,  789,  790,  791,  792,  793,  794,
           795,  796,  797,  3276, 3277, 3278, 3279, 3280, 3281, 3282, 3283,
           3284, 3285, 3286, 3287, 3288, 3289, 3290, 3291, 3292, 3293, 3294,
           3295, 3296, 3297, 3298, 3299, 3300, 3301, 3302, 3303, 3304, 3305,
           3306, 3307, 3308, 3309, 3310, 3311, 3312, 3313, 3314, 3315, 3316,
           3317, 798,  799,  800,  801,  802,  803,  804,  805,  806,  807,
           808,  809,  810,  811,  812,  813,  814,  815,  816,  817,  818,
           819,  820,  821,  822,  823,  824,  825,  826,  827,  828,  829,
           830,  831,  832,  833,  834,  835,  836,  837,  838,  839,  3318,
           3319, 3320, 3321, 3322, 3323, 3324, 3325, 3326, 3327, 3328, 3329,
           3330, 3331, 3332, 3333, 3334, 3335, 3336, 3337, 3338, 3339, 3340,
           3341, 3342, 3343, 3344, 3345, 3346, 3347, 3348, 3349, 3350, 3351,
           3352, 3353, 3354, 3355, 3356, 3357, 3358, 3359, 840,  841,  842,
           843,  844,  845,  846,  847,  848,  849,  850,  851,  852,  853,
           854,  855,  856,  857,  858,  859,  860,  861,  862,  863,  864,
           865,  866,  867,  868,  869,  870,  871,  872,  873,  874,  875,
           876,  877,  878,  879,  880,  881,  3360, 3361, 3362, 3363, 3364,
           3365, 3366, 3367, 3368, 3369, 3370, 3371, 3372, 3373, 3374, 3375,
           3376, 3377, 3378, 3379, 3380, 3381, 3382, 3383, 3384, 3385, 3386,
           3387, 3388, 3389, 3390, 3391, 3392, 3393, 3394, 3395, 3396, 3397,
           3398, 3399, 3400, 3401, 882,  883,  884,  885,  886,  887,  888,
           889,  890,  891,  892,  893,  894,  895,  896,  897,  898,  899,
           900,  901,  902,  903,  904,  905,  906,  907,  908,  909,  910,
           911,  912,  913,  914,  915,  916,  917,  918,  919,  920,  921,
           922,  923,  3402, 3403, 3404, 3405, 3406, 3407, 3408, 3409, 3410,
           3411, 3412, 3413, 3414, 3415, 3416, 3417, 3418, 3419, 3420, 3421,
           3422, 3423, 3424, 3425, 3426, 3427, 3428, 3429, 3430, 3431, 3432,
           3433, 3434, 3435, 3436, 3437, 3438, 3439, 3440, 3441, 3442, 3443,
           924,  925,  926,  927,  928,  929,  930,  931,  932,  933,  934,
           935,  936,  937,  938,  939,  940,  941,  942,  943,  944,  945,
           946,  947,  948,  949,  950,  951,  952,  953,  954,  955,  956,
           957,  958,  959,  960,  961,  962,  963,  964,  965,  3444, 3445,
           3446, 3447, 3448, 3449, 3450, 3451, 3452, 3453, 3454, 3455, 3456,
           3457, 3458, 3459, 3460, 3461, 3462, 3463, 3464, 3465, 3466, 3467,
           3468, 3469, 3470, 3471, 3472, 3473, 3474, 3475, 3476, 3477, 3478,
           3479, 3480, 3481, 3482, 3483, 3484, 3485, 966,  967,  968,  969,
           970,  971,  972,  973,  974,  975,  976,  977,  978,  979,  980,
           981,  982,  983,  984,  985,  986,  987,  988,  989,  990,  991,
           992,  993,  994,  995,  996,  997,  998,  999,  1000, 1001, 1002,
           1003, 1004, 1005, 1006, 1007, 3486, 3487, 3488, 3489, 3490, 3491,
           3492, 3493, 3494, 3495, 3496, 3497, 3498, 3499, 3500, 3501, 3502,
           3503, 3504, 3505, 3506, 3507, 3508, 3509, 3510, 3511, 3512, 3513,
           3514, 3515, 3516, 3517, 3518, 3519, 3520, 3521, 3522, 3523, 3524,
           3525, 3526, 3527, 1008, 1009, 1010, 1011, 1012, 1013, 1014, 1015,
           1016, 1017, 1018, 1019, 1020, 1021, 1022, 1023, 1024, 1025, 1026,
           1027, 1028, 1029, 1030, 1031, 1032, 1033, 1034, 1035, 1036, 1037,
           1038, 1039, 1040, 1041, 1042, 1043, 1044, 1045, 1046, 1047, 1048,
           1049, 3528, 3529, 3530, 3531, 3532, 3533, 3534, 3535, 3536, 3537,
           3538, 3539, 3540, 3541, 3542, 3543, 3544, 3545, 3546, 3547, 3548,
           3549, 3550, 3551, 3552, 3553, 3554, 3555, 3556, 3557, 3558, 3559,
           3560, 3561, 3562, 3563, 3564, 3565, 3566, 3567, 3568, 3569, 1050,
           1051, 1052, 1053, 1054, 1055, 1056, 1057, 1058, 1059, 1060, 1061,
           1062, 1063, 1064, 1065, 1066, 1067, 1068, 1069, 1070, 1071, 1072,
           1073, 1074, 1075, 1076, 1077, 1078, 1079, 1080, 1081, 1082, 1083,
           1084, 1085, 1086, 1087, 1088, 1089, 1090, 1091, 3570, 3571, 3572,
           3573, 3574, 3575, 3576, 3577, 3578, 3579, 3580, 3581, 3582, 3583,
           3584, 3585, 3586, 3587, 3588, 3589, 3590, 3591, 3592, 3593, 3594,
           3595, 3596, 3597, 3598, 3599, 3600, 3601, 3602, 3603, 3604, 3605,
           3606, 3607, 3608, 3609, 3610, 3611, 1092, 1093, 1094, 1095, 1096,
           1097, 1098, 1099, 1100, 1101, 1102, 1103, 1104, 1105, 1106, 1107,
           1108, 1109, 1110, 1111, 1112, 1113, 1114, 1115, 1116, 1117, 1118,
           1119, 1120, 1121, 1122, 1123, 1124, 1125, 1126, 1127, 1128, 1129,
           1130, 1131, 1132, 1133, 3612, 3613, 3614, 3615, 3616, 3617, 3618,
           3619, 3620, 3621, 3622, 3623, 3624, 3625, 3626, 3627, 3628, 3629,
           3630, 3631, 3632, 3633, 3634, 3635, 3636, 3637, 3638, 3639, 3640,
           3641, 3642, 3643, 3644, 3645, 3646, 3647, 3648, 3649, 3650, 3651,
           3652, 3653, 1134, 1135, 1136, 1137, 1138, 1139, 1140, 1141, 1142,
           1143, 1144, 1145, 1146, 1147, 1148, 1149, 1150, 1151, 1152, 1153,
           1154, 1155, 1156, 1157, 1158, 1159, 1160, 1161, 1162, 1163, 1164,
           1165, 1166, 1167, 1168, 1169, 1170, 1171, 1172, 1173, 1174, 1175,
           3654, 3655, 3656, 3657, 3658, 3659, 3660, 3661, 3662, 3663, 3664,
           3665, 3666, 3667, 3668, 3669, 3670, 3671, 3672, 3673, 3674, 3675,
           3676, 3677, 3678, 3679, 3680, 3681, 3682, 3683, 3684, 3685, 3686,
           3687, 3688, 3689, 3690, 3691, 3692, 3693, 3694, 3695, 1176, 1177,
           1178, 1179, 1180, 1181, 1182, 1183, 1184, 1185, 1186, 1187, 1188,
           1189, 1190, 1191, 1192, 1193, 1194, 1195, 1196, 1197, 1198, 1199,
           1200, 1201, 1202, 1203, 1204, 1205, 1206, 1207, 1208, 1209, 1210,
           1211, 1212, 1213, 1214, 1215, 1216, 1217, 3696, 3697, 3698, 3699,
           3700, 3701, 3702, 3703, 3704, 3705, 3706, 3707, 3708, 3709, 3710,
           3711, 3712, 3713, 3714, 3715, 3716, 3717, 3718, 3719, 3720, 3721,
           3722, 3723, 3724, 3725, 3726, 3727, 3728, 3729, 3730, 3731, 3732,
           3733, 3734, 3735, 3736, 3737, 1218, 1219, 1220, 1221, 1222, 1223,
           1224, 1225, 1226, 1227, 1228, 1229, 1230, 1231, 1232, 1233, 1234,
           1235, 1236, 1237, 1238, 1239, 1240, 1241, 1242, 1243, 1244, 1245,
           1246, 1247, 1248, 1249, 1250, 1251, 1252, 1253, 1254, 1255, 1256,
           1257, 1258, 1259, 3738, 3739, 3740, 3741, 3742, 3743, 3744, 3745,
           3746, 3747, 3748, 3749, 3750, 3751, 3752, 3753, 3754, 3755, 3756,
           3757, 3758, 3759, 3760, 3761, 3762, 3763, 3764, 3765, 3766, 3767,
           3768, 3769, 3770, 3771, 3772, 3773, 3774, 3775, 3776, 3777, 3778,
           3779, 1260, 1261, 1262, 1263, 1264, 1265, 1266, 1267, 1268, 1269,
           1270, 1271, 1272, 1273, 1274, 1275, 1276, 1277, 1278, 1279, 1280,
           1281, 1282, 1283, 1284, 1285, 1286, 1287, 1288, 1289, 1290, 1291,
           1292, 1293, 1294, 1295, 1296, 1297, 1298, 1299, 1300, 1301, 3780,
           3781, 3782, 3783, 3784, 3785, 3786, 3787, 3788, 3789, 3790, 3791,
           3792, 3793, 3794, 3795, 3796, 3797, 3798, 3799, 3800, 3801, 3802,
           3803, 3804, 3805, 3806, 3807, 3808, 3809, 3810, 3811, 3812, 3813,
           3814, 3815, 3816, 3817, 3818, 3819, 3820, 3821, 1302, 1303, 1304,
           1305, 1306, 1307, 1308, 1309, 1310, 1311, 1312, 1313, 1314, 1315,
           1316, 1317, 1318, 1319, 1320, 1321, 1322, 1323, 1324, 1325, 1326,
           1327, 1328, 1329, 1330, 1331, 1332, 1333, 1334, 1335, 1336, 1337,
           1338, 1339, 1340, 1341, 1342, 1343, 3822, 3823, 3824, 3825, 3826,
           3827, 3828, 3829, 3830, 3831, 3832, 3833, 3834, 3835, 3836, 3837,
           3838, 3839, 3840, 3841, 3842, 3843, 3844, 3845, 3846, 3847, 3848,
           3849, 3850, 3851, 3852, 3853, 3854, 3855, 3856, 3857, 3858, 3859,
           3860, 3861, 3862, 3863, 1344, 1345, 1346, 1347, 1348, 1349, 1350,
           1351, 1352, 1353, 1354, 1355, 1356, 1357, 1358, 1359, 1360, 1361,
           1362, 1363, 1364, 1365, 1366, 1367, 1368, 1369, 1370, 1371, 1372,
           1373, 1374, 1375, 1376, 1377, 1378, 1379, 1380, 1381, 1382, 1383,
           1384, 1385, 3864, 3865, 3866, 3867, 3868, 3869, 3870, 3871, 3872,
           3873, 3874, 3875, 3876, 3877, 3878, 3879, 3880, 3881, 3882, 3883,
           3884, 3885, 3886, 3887, 3888, 3889, 3890, 3891, 3892, 3893, 3894,
           3895, 3896, 3897, 3898, 3899, 3900, 3901, 3902, 3903, 3904, 3905,
           1386, 1387, 1388, 1389, 1390, 1391, 1392, 1393, 1394, 1395, 1396,
           1397, 1398, 1399, 1400, 1401, 1402, 1403, 1404, 1405, 1406, 1407,
           1408, 1409, 1410, 1411, 1412, 1413, 1414, 1415, 1416, 1417, 1418,
           1419, 1420, 1421, 1422, 1423, 1424, 1425, 1426, 1427, 3906, 3907,
           3908, 3909, 3910, 3911, 3912, 3913, 3914, 3915, 3916, 3917, 3918,
           3919, 3920, 3921, 3922, 3923, 3924, 3925, 3926, 3927, 3928, 3929,
           3930, 3931, 3932, 3933, 3934, 3935, 3936, 3937, 3938, 3939, 3940,
           3941, 3942, 3943, 3944, 3945, 3946, 3947, 1428, 1429, 1430, 1431,
           1432, 1433, 1434, 1435, 1436, 1437, 1438, 1439, 1440, 1441, 1442,
           1443, 1444, 1445, 1446, 1447, 1448, 1449, 1450, 1451, 1452, 1453,
           1454, 1455, 1456, 1457, 1458, 1459, 1460, 1461, 1462, 1463, 1464,
           1465, 1466, 1467, 1468, 1469, 3948, 3949, 3950, 3951, 3952, 3953,
           3954, 3955, 3956, 3957, 3958, 3959, 3960, 3961, 3962, 3963, 3964,
           3965, 3966, 3967, 3968, 3969, 3970, 3971, 3972, 3973, 3974, 3975,
           3976, 3977, 3978, 3979, 3980, 3981, 3982, 3983, 3984, 3985, 3986,
           3987, 3988, 3989, 1470, 1471, 1472, 1473, 1474, 1475, 1476, 1477,
           1478, 1479, 1480, 1481, 1482, 1483, 1484, 1485, 1486, 1487, 1488,
           1489, 1490, 1491, 1492, 1493, 1494, 1495, 1496, 1497, 1498, 1499,
           1500, 1501, 1502, 1503, 1504, 1505, 1506, 1507, 1508, 1509, 1510,
           1511, 3990, 3991, 3992, 3993, 3994, 3995, 3996, 3997, 3998, 3999,
           4000, 4001, 4002, 4003, 4004, 4005, 4006, 4007, 4008, 4009, 4010,
           4011, 4012, 4013, 4014, 4015, 4016, 4017, 4018, 4019, 4020, 4021,
           4022, 4023, 4024, 4025, 4026, 4027, 4028, 4029, 4030, 4031, 1512,
           1513, 1514, 1515, 1516, 1517, 1518, 1519, 1520, 1521, 1522, 1523,
           1524, 1525, 1526, 1527, 1528, 1529, 1530, 1531, 1532, 1533, 1534,
           1535, 1536, 1537, 1538, 1539, 1540, 1541, 1542, 1543, 1544, 1545,
           1546, 1547, 1548, 1549, 1550, 1551, 1552, 1553, 4032, 4033, 4034,
           4035, 4036, 4037, 4038, 4039, 4040, 4041, 4042, 4043, 4044, 4045,
           4046, 4047, 4048, 4049, 4050, 4051, 4052, 4053, 4054, 4055, 4056,
           4057, 4058, 4059, 4060, 4061, 4062, 4063, 4064, 4065, 4066, 4067,
           4068, 4069, 4070, 4071, 4072, 4073, 1554, 1555, 1556, 1557, 1558,
           1559, 1560, 1561, 1562, 1563, 1564, 1565, 1566, 1567, 1568, 1569,
           1570, 1571, 1572, 1573, 1574, 1575, 1576, 1577, 1578, 1579, 1580,
           1581, 1582, 1583, 1584, 1585, 1586, 1587, 1588, 1589, 1590, 1591,
           1592, 1593, 1594, 1595, 4074, 4075, 4076, 4077, 4078, 4079, 4080,
           4081, 4082, 4083, 4084, 4085, 4086, 4087, 4088, 4089, 4090, 4091,
           4092, 4093, 4094, 4095, 4096, 4097, 4098, 4099, 4100, 4101, 4102,
           4103, 4104, 4105, 4106, 4107, 4108, 4109, 4110, 4111, 4112, 4113,
           4114, 4115, 1596, 1597, 1598, 1599, 1600, 1601, 1602, 1603, 1604,
           1605, 1606, 1607, 1608, 1609, 1610, 1611, 1612, 1613, 1614, 1615,
           1616, 1617, 1618, 1619, 1620, 1621, 1622, 1623, 1624, 1625, 1626,
           1627, 1628, 1629, 1630, 1631, 1632, 1633, 1634, 1635, 1636, 1637,
           4116, 4117, 4118, 4119, 4120, 4121, 4122, 4123, 4124, 4125, 4126,
           4127, 4128, 4129, 4130, 4131, 4132, 4133, 4134, 4135, 4136, 4137,
           4138, 4139, 4140, 4141, 4142, 4143, 4144, 4145, 4146, 4147, 4148,
           4149, 4150, 4151, 4152, 4153, 4154, 4155, 4156, 4157, 1638, 1639,
           1640, 1641, 1642, 1643, 1644, 1645, 1646, 1647, 1648, 1649, 1650,
           1651, 1652, 1653, 1654, 1655, 1656, 1657, 1658, 1659, 1660, 1661,
           1662, 1663, 1664, 1665, 1666, 1667, 1668, 1669, 1670, 1671, 1672,
           1673, 1674, 1675, 1676, 1677, 1678, 1679, 4158, 4159, 4160, 4161,
           4162, 4163, 4164, 4165, 4166, 4167, 4168, 4169, 4170, 4171, 4172,
           4173, 4174, 4175, 4176, 4177, 4178, 4179, 4180, 4181, 4182, 4183,
           4184, 4185, 4186, 4187, 4188, 4189, 4190, 4191, 4192, 4193, 4194,
           4195, 4196, 4197, 4198, 4199, 1680, 1681, 1682, 1683, 1684, 1685,
           1686, 1687, 1688, 1689, 1690, 1691, 1692, 1693, 1694, 1695, 1696,
           1697, 1698, 1699, 1700, 1701, 1702, 1703, 1704, 1705, 1706, 1707,
           1708, 1709, 1710, 1711, 1712, 1713, 1714, 1715, 1716, 1717, 1718,
           1719, 1720, 1721, 4200, 4201, 4202, 4203, 4204, 4205, 4206, 4207,
           4208, 4209, 4210, 4211, 4212, 4213, 4214, 4215, 4216, 4217, 4218,
           4219, 4220, 4221, 4222, 4223, 4224, 4225, 4226, 4227, 4228, 4229,
           4230, 4231, 4232, 4233, 4234, 4235, 4236, 4237, 4238, 4239, 4240,
           4241, 1722, 1723, 1724, 1725, 1726, 1727, 1728, 1729, 1730, 1731,
           1732, 1733, 1734, 1735, 1736, 1737, 1738, 1739, 1740, 1741, 1742,
           1743, 1744, 1745, 1746, 1747, 1748, 1749, 1750, 1751, 1752, 1753,
           1754, 1755, 1756, 1757, 1758, 1759, 1760, 1761, 1762, 1763, 4242,
           4243, 4244, 4245, 4246, 4247, 4248, 4249, 4250, 4251, 4252, 4253,
           4254, 4255, 4256, 4257, 4258, 4259, 4260, 4261, 4262, 4263, 4264,
           4265, 4266, 4267, 4268, 4269, 4270, 4271, 4272, 4273, 4274, 4275,
           4276, 4277, 4278, 4279, 4280, 4281, 4282, 4283, 1764, 1765, 1766,
           1767, 1768, 1769, 1770, 1771, 1772, 1773, 1774, 1775, 1776, 1777,
           1778, 1779, 1780, 1781, 1782, 1783, 1784, 1785, 1786, 1787, 1788,
           1789, 1790, 1791, 1792, 1793, 1794, 1795, 1796, 1797, 1798, 1799,
           1800, 1801, 1802, 1803, 1804, 1805, 4284, 4285, 4286, 4287, 4288,
           4289, 4290, 4291, 4292, 4293, 4294, 4295, 4296, 4297, 4298, 4299,
           4300, 4301, 4302, 4303, 4304, 4305, 4306, 4307, 4308, 4309, 4310,
           4311, 4312, 4313, 4314, 4315, 4316, 4317, 4318, 4319, 4320, 4321,
           4322, 4323, 4324, 4325, 1806, 1807, 1808, 1809, 1810, 1811, 1812,
           1813, 1814, 1815, 1816, 1817, 1818, 1819, 1820, 1821, 1822, 1823,
           1824, 1825, 1826, 1827, 1828, 1829, 1830, 1831, 1832, 1833, 1834,
           1835, 1836, 1837, 1838, 1839, 1840, 1841, 1842, 1843, 1844, 1845,
           1846, 1847, 4326, 4327, 4328, 4329, 4330, 4331, 4332, 4333, 4334,
           4335, 4336, 4337, 4338, 4339, 4340, 4341, 4342, 4343, 4344, 4345,
           4346, 4347, 4348, 4349, 4350, 4351, 4352, 4353, 4354, 4355, 4356,
           4357, 4358, 4359, 4360, 4361, 4362, 4363, 4364, 4365, 4366, 4367,
           1848, 1849, 1850, 1851, 1852, 1853, 1854, 1855, 1856, 1857, 1858,
           1859, 1860, 1861, 1862, 1863, 1864, 1865, 1866, 1867, 1868, 1869,
           1870, 1871, 1872, 1873, 1874, 1875, 1876, 1877, 1878, 1879, 1880,
           1881, 1882, 1883, 1884, 1885, 1886, 1887, 1888, 1889, 4368, 4369,
           4370, 4371, 4372, 4373, 4374, 4375, 4376, 4377, 4378, 4379, 4380,
           4381, 4382, 4383, 4384, 4385, 4386, 4387, 4388, 4389, 4390, 4391,
           4392, 4393, 4394, 4395, 4396, 4397, 4398, 4399, 4400, 4401, 4402,
           4403, 4404, 4405, 4406, 4407, 4408, 4409, 1890, 1891, 1892, 1893,
           1894, 1895, 1896, 1897, 1898, 1899, 1900, 1901, 1902, 1903, 1904,
           1905, 1906, 1907, 1908, 1909, 1910, 1911, 1912, 1913, 1914, 1915,
           1916, 1917, 1918, 1919, 1920, 1921, 1922, 1923, 1924, 1925, 1926,
           1927, 1928, 1929, 1930, 1931, 4410, 4411, 4412, 4413, 4414, 4415,
           4416, 4417, 4418, 4419, 4420, 4421, 4422, 4423, 4424, 4425, 4426,
           4427, 4428, 4429, 4430, 4431, 4432, 4433, 4434, 4435, 4436, 4437,
           4438, 4439, 4440, 4441, 4442, 4443, 4444, 4445, 4446, 4447, 4448,
           4449, 4450, 4451, 1932, 1933, 1934, 1935, 1936, 1937, 1938, 1939,
           1940, 1941, 1942, 1943, 1944, 1945, 1946, 1947, 1948, 1949, 1950,
           1951, 1952, 1953, 1954, 1955, 1956, 1957, 1958, 1959, 1960, 1961,
           1962, 1963, 1964, 1965, 1966, 1967, 1968, 1969, 1970, 1971, 1972,
           1973, 4452, 4453, 4454, 4455, 4456, 4457, 4458, 4459, 4460, 4461,
           4462, 4463, 4464, 4465, 4466, 4467, 4468, 4469, 4470, 4471, 4472,
           4473, 4474, 4475, 4476, 4477, 4478, 4479, 4480, 4481, 4482, 4483,
           4484, 4485, 4486, 4487, 4488, 4489, 4490, 4491, 4492, 4493, 1974,
           1975, 1976, 1977, 1978, 1979, 1980, 1981, 1982, 1983, 1984, 1985,
           1986, 1987, 1988, 1989, 1990, 1991, 1992, 1993, 1994, 1995, 1996,
           1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006, 2007,
           2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 4494, 4495, 4496,
           4497, 4498, 4499, 4500, 4501, 4502, 4503, 4504, 4505, 4506, 4507,
           4508, 4509, 4510, 4511, 4512, 4513, 4514, 4515, 4516, 4517, 4518,
           4519, 4520, 4521, 4522, 4523, 4524, 4525, 4526, 4527, 4528, 4529,
           4530, 4531, 4532, 4533, 4534, 4535, 2016, 2017, 2018, 2019, 2020,
           2021, 2022, 2023, 2024, 2025, 2026, 2027, 2028, 2029, 2030, 2031,
           2032, 2033, 2034, 2035, 2036, 2037, 2038, 2039, 2040, 2041, 2042,
           2043, 2044, 2045, 2046, 2047, 2048, 2049, 2050, 2051, 2052, 2053,
           2054, 2055, 2056, 2057, 4536, 4537, 4538, 4539, 4540, 4541, 4542,
           4543, 4544, 4545, 4546, 4547, 4548, 4549, 4550, 4551, 4552, 4553,
           4554, 4555, 4556, 4557, 4558, 4559, 4560, 4561, 4562, 4563, 4564,
           4565, 4566, 4567, 4568, 4569, 4570, 4571, 4572, 4573, 4574, 4575,
           4576, 4577, 2058, 2059, 2060, 2061, 2062, 2063, 2064, 2065, 2066,
           2067, 2068, 2069, 2070, 2071, 2072, 2073, 2074, 2075, 2076, 2077,
           2078, 2079, 2080, 2081, 2082, 2083, 2084, 2085, 2086, 2087, 2088,
           2089, 2090, 2091, 2092, 2093, 2094, 2095, 2096, 2097, 2098, 2099,
           4578, 4579, 4580, 4581, 4582, 4583, 4584, 4585, 4586, 4587, 4588,
           4589, 4590, 4591, 4592, 4593, 4594, 4595, 4596, 4597, 4598, 4599,
           4600, 4601, 4602, 4603, 4604, 4605, 4606, 4607, 4608, 4609, 4610,
           4611, 4612, 4613, 4614, 4615, 4616, 4617, 4618, 4619, 2100, 2101,
           2102, 2103, 2104, 2105, 2106, 2107, 2108, 2109, 2110, 2111, 2112,
           2113, 2114, 2115, 2116, 2117, 2118, 2119, 2120, 2121, 2122, 2123,
           2124, 2125, 2126, 2127, 2128, 2129, 2130, 2131, 2132, 2133, 2134,
           2135, 2136, 2137, 2138, 2139, 2140, 2141, 4620, 4621, 4622, 4623,
           4624, 4625, 4626, 4627, 4628, 4629, 4630, 4631, 4632, 4633, 4634,
           4635, 4636, 4637, 4638, 4639, 4640, 4641, 4642, 4643, 4644, 4645,
           4646, 4647, 4648, 4649, 4650, 4651, 4652, 4653, 4654, 4655, 4656,
           4657, 4658, 4659, 4660, 4661, 2142, 2143, 2144, 2145, 2146, 2147,
           2148, 2149, 2150, 2151, 2152, 2153, 2154, 2155, 2156, 2157, 2158,
           2159, 2160, 2161, 2162, 2163, 2164, 2165, 2166, 2167, 2168, 2169,
           2170, 2171, 2172, 2173, 2174, 2175, 2176, 2177, 2178, 2179, 2180,
           2181, 2182, 2183, 4662, 4663, 4664, 4665, 4666, 4667, 4668, 4669,
           4670, 4671, 4672, 4673, 4674, 4675, 4676, 4677, 4678, 4679, 4680,
           4681, 4682, 4683, 4684, 4685, 4686, 4687, 4688, 4689, 4690, 4691,
           4692, 4693, 4694, 4695, 4696, 4697, 4698, 4699, 4700, 4701, 4702,
           4703, 2184, 2185, 2186, 2187, 2188, 2189, 2190, 2191, 2192, 2193,
           2194, 2195, 2196, 2197, 2198, 2199, 2200, 2201, 2202, 2203, 2204,
           2205, 2206, 2207, 2208, 2209, 2210, 2211, 2212, 2213, 2214, 2215,
           2216, 2217, 2218, 2219, 2220, 2221, 2222, 2223, 2224, 2225, 4704,
           4705, 4706, 4707, 4708, 4709, 4710, 4711, 4712, 4713, 4714, 4715,
           4716, 4717, 4718, 4719, 4720, 4721, 4722, 4723, 4724, 4725, 4726,
           4727, 4728, 4729, 4730, 4731, 4732, 4733, 4734, 4735, 4736, 4737,
           4738, 4739, 4740, 4741, 4742, 4743, 4744, 4745, 2226, 2227, 2228,
           2229, 2230, 2231, 2232, 2233, 2234, 2235, 2236, 2237, 2238, 2239,
           2240, 2241, 2242, 2243, 2244, 2245, 2246, 2247, 2248, 2249, 2250,
           2251, 2252, 2253, 2254, 2255, 2256, 2257, 2258, 2259, 2260, 2261,
           2262, 2263, 2264, 2265, 2266, 2267, 4746, 4747, 4748, 4749, 4750,
           4751, 4752, 4753, 4754, 4755, 4756, 4757, 4758, 4759, 4760, 4761,
           4762, 4763, 4764, 4765, 4766, 4767, 4768, 4769, 4770, 4771, 4772,
           4773, 4774, 4775, 4776, 4777, 4778, 4779, 4780, 4781, 4782, 4783,
           4784, 4785, 4786, 4787, 2268, 2269, 2270, 2271, 2272, 2273, 2274,
           2275, 2276, 2277, 2278, 2279, 2280, 2281, 2282, 2283, 2284, 2285,
           2286, 2287, 2288, 2289, 2290, 2291, 2292, 2293, 2294, 2295, 2296,
           2297, 2298, 2299, 2300, 2301, 2302, 2303, 2304, 2305, 2306, 2307,
           2308, 2309, 4788, 4789, 4790, 4791, 4792, 4793, 4794, 4795, 4796,
           4797, 4798, 4799, 4800, 4801, 4802, 4803, 4804, 4805, 4806, 4807,
           4808, 4809, 4810, 4811, 4812, 4813, 4814, 4815, 4816, 4817, 4818,
           4819, 4820, 4821, 4822, 4823, 4824, 4825, 4826, 4827, 4828, 4829,
           2310, 2311, 2312, 2313, 2314, 2315, 2316, 2317, 2318, 2319, 2320,
           2321, 2322, 2323, 2324, 2325, 2326, 2327, 2328, 2329, 2330, 2331,
           2332, 2333, 2334, 2335, 2336, 2337, 2338, 2339, 2340, 2341, 2342,
           2343, 2344, 2345, 2346, 2347, 2348, 2349, 2350, 2351, 4830, 4831,
           4832, 4833, 4834, 4835, 4836, 4837, 4838, 4839, 4840, 4841, 4842,
           4843, 4844, 4845, 4846, 4847, 4848, 4849, 4850, 4851, 4852, 4853,
           4854, 4855, 4856, 4857, 4858, 4859, 4860, 4861, 4862, 4863, 4864,
           4865, 4866, 4867, 4868, 4869, 4870, 4871, 2352, 2353, 2354, 2355,
           2356, 2357, 2358, 2359, 2360, 2361, 2362, 2363, 2364, 2365, 2366,
           2367, 2368, 2369, 2370, 2371, 2372, 2373, 2374, 2375, 2376, 2377,
           2378, 2379, 2380, 2381, 2382, 2383, 2384, 2385, 2386, 2387, 2388,
           2389, 2390, 2391, 2392, 2393, 4872, 4873, 4874, 4875, 4876, 4877,
           4878, 4879, 4880, 4881, 4882, 4883, 4884, 4885, 4886, 4887, 4888,
           4889, 4890, 4891, 4892, 4893, 4894, 4895, 4896, 4897, 4898, 4899,
           4900, 4901, 4902, 4903, 4904, 4905, 4906, 4907, 4908, 4909, 4910,
           4911, 4912, 4913, 2394, 2395, 2396, 2397, 2398, 2399, 2400, 2401,
           2402, 2403, 2404, 2405, 2406, 2407, 2408, 2409, 2410, 2411, 2412,
           2413, 2414, 2415, 2416, 2417, 2418, 2419, 2420, 2421, 2422, 2423,
           2424, 2425, 2426, 2427, 2428, 2429, 2430, 2431, 2432, 2433, 2434,
           2435, 4914, 4915, 4916, 4917, 4918, 4919, 4920, 4921, 4922, 4923,
           4924, 4925, 4926, 4927, 4928, 4929, 4930, 4931, 4932, 4933, 4934,
           4935, 4936, 4937, 4938, 4939, 4940, 4941, 4942, 4943, 4944, 4945,
           4946, 4947, 4948, 4949, 4950, 4951, 4952, 4953, 4954, 4955, 2436,
           2437, 2438, 2439, 2440, 2441, 2442, 2443, 2444, 2445, 2446, 2447,
           2448, 2449, 2450, 2451, 2452, 2453, 2454, 2455, 2456, 2457, 2458,
           2459, 2460, 2461, 2462, 2463, 2464, 2465, 2466, 2467, 2468, 2469,
           2470, 2471, 2472, 2473, 2474, 2475, 2476, 2477, 4956, 4957, 4958,
           4959, 4960, 4961, 4962, 4963, 4964, 4965, 4966, 4967, 4968, 4969,
           4970, 4971, 4972, 4973, 4974, 4975, 4976, 4977, 4978, 4979, 4980,
           4981, 4982, 4983, 4984, 4985, 4986, 4987, 4988, 4989, 4990, 4991,
           4992, 4993, 4994, 4995, 4996, 4997, 2478, 2479, 2480, 2481, 2482,
           2483, 2484, 2485, 2486, 2487, 2488, 2489, 2490, 2491, 2492, 2493,
           2494, 2495, 2496, 2497, 2498, 2499, 2500, 2501, 2502, 2503, 2504,
           2505, 2506, 2507, 2508, 2509, 2510, 2511, 2512, 2513, 2514, 2515,
           2516, 2517, 2518, 2519, 4998, 4999, 5000, 5001, 5002, 5003, 5004,
           5005, 5006, 5007, 5008, 5009, 5010, 5011, 5012, 5013, 5014, 5015,
           5016, 5017, 5018, 5019, 5020, 5021, 5022, 5023, 5024, 5025, 5026,
           5027, 5028, 5029, 5030, 5031, 5032, 5033, 5034, 5035, 5036, 5037,
           5038, 5039}));
}

TEST(TransposeTest, TestRefOps6D5) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5, 6, 7},
                                /*perms=*/{1, 2, 3, 4, 0, 5}),
      ElementsAreArray(
          {0,    1,    2,    3,    4,    5,    6,    2520, 2521, 2522, 2523,
           2524, 2525, 2526, 7,    8,    9,    10,   11,   12,   13,   2527,
           2528, 2529, 2530, 2531, 2532, 2533, 14,   15,   16,   17,   18,
           19,   20,   2534, 2535, 2536, 2537, 2538, 2539, 2540, 21,   22,
           23,   24,   25,   26,   27,   2541, 2542, 2543, 2544, 2545, 2546,
           2547, 28,   29,   30,   31,   32,   33,   34,   2548, 2549, 2550,
           2551, 2552, 2553, 2554, 35,   36,   37,   38,   39,   40,   41,
           2555, 2556, 2557, 2558, 2559, 2560, 2561, 42,   43,   44,   45,
           46,   47,   48,   2562, 2563, 2564, 2565, 2566, 2567, 2568, 49,
           50,   51,   52,   53,   54,   55,   2569, 2570, 2571, 2572, 2573,
           2574, 2575, 56,   57,   58,   59,   60,   61,   62,   2576, 2577,
           2578, 2579, 2580, 2581, 2582, 63,   64,   65,   66,   67,   68,
           69,   2583, 2584, 2585, 2586, 2587, 2588, 2589, 70,   71,   72,
           73,   74,   75,   76,   2590, 2591, 2592, 2593, 2594, 2595, 2596,
           77,   78,   79,   80,   81,   82,   83,   2597, 2598, 2599, 2600,
           2601, 2602, 2603, 84,   85,   86,   87,   88,   89,   90,   2604,
           2605, 2606, 2607, 2608, 2609, 2610, 91,   92,   93,   94,   95,
           96,   97,   2611, 2612, 2613, 2614, 2615, 2616, 2617, 98,   99,
           100,  101,  102,  103,  104,  2618, 2619, 2620, 2621, 2622, 2623,
           2624, 105,  106,  107,  108,  109,  110,  111,  2625, 2626, 2627,
           2628, 2629, 2630, 2631, 112,  113,  114,  115,  116,  117,  118,
           2632, 2633, 2634, 2635, 2636, 2637, 2638, 119,  120,  121,  122,
           123,  124,  125,  2639, 2640, 2641, 2642, 2643, 2644, 2645, 126,
           127,  128,  129,  130,  131,  132,  2646, 2647, 2648, 2649, 2650,
           2651, 2652, 133,  134,  135,  136,  137,  138,  139,  2653, 2654,
           2655, 2656, 2657, 2658, 2659, 140,  141,  142,  143,  144,  145,
           146,  2660, 2661, 2662, 2663, 2664, 2665, 2666, 147,  148,  149,
           150,  151,  152,  153,  2667, 2668, 2669, 2670, 2671, 2672, 2673,
           154,  155,  156,  157,  158,  159,  160,  2674, 2675, 2676, 2677,
           2678, 2679, 2680, 161,  162,  163,  164,  165,  166,  167,  2681,
           2682, 2683, 2684, 2685, 2686, 2687, 168,  169,  170,  171,  172,
           173,  174,  2688, 2689, 2690, 2691, 2692, 2693, 2694, 175,  176,
           177,  178,  179,  180,  181,  2695, 2696, 2697, 2698, 2699, 2700,
           2701, 182,  183,  184,  185,  186,  187,  188,  2702, 2703, 2704,
           2705, 2706, 2707, 2708, 189,  190,  191,  192,  193,  194,  195,
           2709, 2710, 2711, 2712, 2713, 2714, 2715, 196,  197,  198,  199,
           200,  201,  202,  2716, 2717, 2718, 2719, 2720, 2721, 2722, 203,
           204,  205,  206,  207,  208,  209,  2723, 2724, 2725, 2726, 2727,
           2728, 2729, 210,  211,  212,  213,  214,  215,  216,  2730, 2731,
           2732, 2733, 2734, 2735, 2736, 217,  218,  219,  220,  221,  222,
           223,  2737, 2738, 2739, 2740, 2741, 2742, 2743, 224,  225,  226,
           227,  228,  229,  230,  2744, 2745, 2746, 2747, 2748, 2749, 2750,
           231,  232,  233,  234,  235,  236,  237,  2751, 2752, 2753, 2754,
           2755, 2756, 2757, 238,  239,  240,  241,  242,  243,  244,  2758,
           2759, 2760, 2761, 2762, 2763, 2764, 245,  246,  247,  248,  249,
           250,  251,  2765, 2766, 2767, 2768, 2769, 2770, 2771, 252,  253,
           254,  255,  256,  257,  258,  2772, 2773, 2774, 2775, 2776, 2777,
           2778, 259,  260,  261,  262,  263,  264,  265,  2779, 2780, 2781,
           2782, 2783, 2784, 2785, 266,  267,  268,  269,  270,  271,  272,
           2786, 2787, 2788, 2789, 2790, 2791, 2792, 273,  274,  275,  276,
           277,  278,  279,  2793, 2794, 2795, 2796, 2797, 2798, 2799, 280,
           281,  282,  283,  284,  285,  286,  2800, 2801, 2802, 2803, 2804,
           2805, 2806, 287,  288,  289,  290,  291,  292,  293,  2807, 2808,
           2809, 2810, 2811, 2812, 2813, 294,  295,  296,  297,  298,  299,
           300,  2814, 2815, 2816, 2817, 2818, 2819, 2820, 301,  302,  303,
           304,  305,  306,  307,  2821, 2822, 2823, 2824, 2825, 2826, 2827,
           308,  309,  310,  311,  312,  313,  314,  2828, 2829, 2830, 2831,
           2832, 2833, 2834, 315,  316,  317,  318,  319,  320,  321,  2835,
           2836, 2837, 2838, 2839, 2840, 2841, 322,  323,  324,  325,  326,
           327,  328,  2842, 2843, 2844, 2845, 2846, 2847, 2848, 329,  330,
           331,  332,  333,  334,  335,  2849, 2850, 2851, 2852, 2853, 2854,
           2855, 336,  337,  338,  339,  340,  341,  342,  2856, 2857, 2858,
           2859, 2860, 2861, 2862, 343,  344,  345,  346,  347,  348,  349,
           2863, 2864, 2865, 2866, 2867, 2868, 2869, 350,  351,  352,  353,
           354,  355,  356,  2870, 2871, 2872, 2873, 2874, 2875, 2876, 357,
           358,  359,  360,  361,  362,  363,  2877, 2878, 2879, 2880, 2881,
           2882, 2883, 364,  365,  366,  367,  368,  369,  370,  2884, 2885,
           2886, 2887, 2888, 2889, 2890, 371,  372,  373,  374,  375,  376,
           377,  2891, 2892, 2893, 2894, 2895, 2896, 2897, 378,  379,  380,
           381,  382,  383,  384,  2898, 2899, 2900, 2901, 2902, 2903, 2904,
           385,  386,  387,  388,  389,  390,  391,  2905, 2906, 2907, 2908,
           2909, 2910, 2911, 392,  393,  394,  395,  396,  397,  398,  2912,
           2913, 2914, 2915, 2916, 2917, 2918, 399,  400,  401,  402,  403,
           404,  405,  2919, 2920, 2921, 2922, 2923, 2924, 2925, 406,  407,
           408,  409,  410,  411,  412,  2926, 2927, 2928, 2929, 2930, 2931,
           2932, 413,  414,  415,  416,  417,  418,  419,  2933, 2934, 2935,
           2936, 2937, 2938, 2939, 420,  421,  422,  423,  424,  425,  426,
           2940, 2941, 2942, 2943, 2944, 2945, 2946, 427,  428,  429,  430,
           431,  432,  433,  2947, 2948, 2949, 2950, 2951, 2952, 2953, 434,
           435,  436,  437,  438,  439,  440,  2954, 2955, 2956, 2957, 2958,
           2959, 2960, 441,  442,  443,  444,  445,  446,  447,  2961, 2962,
           2963, 2964, 2965, 2966, 2967, 448,  449,  450,  451,  452,  453,
           454,  2968, 2969, 2970, 2971, 2972, 2973, 2974, 455,  456,  457,
           458,  459,  460,  461,  2975, 2976, 2977, 2978, 2979, 2980, 2981,
           462,  463,  464,  465,  466,  467,  468,  2982, 2983, 2984, 2985,
           2986, 2987, 2988, 469,  470,  471,  472,  473,  474,  475,  2989,
           2990, 2991, 2992, 2993, 2994, 2995, 476,  477,  478,  479,  480,
           481,  482,  2996, 2997, 2998, 2999, 3000, 3001, 3002, 483,  484,
           485,  486,  487,  488,  489,  3003, 3004, 3005, 3006, 3007, 3008,
           3009, 490,  491,  492,  493,  494,  495,  496,  3010, 3011, 3012,
           3013, 3014, 3015, 3016, 497,  498,  499,  500,  501,  502,  503,
           3017, 3018, 3019, 3020, 3021, 3022, 3023, 504,  505,  506,  507,
           508,  509,  510,  3024, 3025, 3026, 3027, 3028, 3029, 3030, 511,
           512,  513,  514,  515,  516,  517,  3031, 3032, 3033, 3034, 3035,
           3036, 3037, 518,  519,  520,  521,  522,  523,  524,  3038, 3039,
           3040, 3041, 3042, 3043, 3044, 525,  526,  527,  528,  529,  530,
           531,  3045, 3046, 3047, 3048, 3049, 3050, 3051, 532,  533,  534,
           535,  536,  537,  538,  3052, 3053, 3054, 3055, 3056, 3057, 3058,
           539,  540,  541,  542,  543,  544,  545,  3059, 3060, 3061, 3062,
           3063, 3064, 3065, 546,  547,  548,  549,  550,  551,  552,  3066,
           3067, 3068, 3069, 3070, 3071, 3072, 553,  554,  555,  556,  557,
           558,  559,  3073, 3074, 3075, 3076, 3077, 3078, 3079, 560,  561,
           562,  563,  564,  565,  566,  3080, 3081, 3082, 3083, 3084, 3085,
           3086, 567,  568,  569,  570,  571,  572,  573,  3087, 3088, 3089,
           3090, 3091, 3092, 3093, 574,  575,  576,  577,  578,  579,  580,
           3094, 3095, 3096, 3097, 3098, 3099, 3100, 581,  582,  583,  584,
           585,  586,  587,  3101, 3102, 3103, 3104, 3105, 3106, 3107, 588,
           589,  590,  591,  592,  593,  594,  3108, 3109, 3110, 3111, 3112,
           3113, 3114, 595,  596,  597,  598,  599,  600,  601,  3115, 3116,
           3117, 3118, 3119, 3120, 3121, 602,  603,  604,  605,  606,  607,
           608,  3122, 3123, 3124, 3125, 3126, 3127, 3128, 609,  610,  611,
           612,  613,  614,  615,  3129, 3130, 3131, 3132, 3133, 3134, 3135,
           616,  617,  618,  619,  620,  621,  622,  3136, 3137, 3138, 3139,
           3140, 3141, 3142, 623,  624,  625,  626,  627,  628,  629,  3143,
           3144, 3145, 3146, 3147, 3148, 3149, 630,  631,  632,  633,  634,
           635,  636,  3150, 3151, 3152, 3153, 3154, 3155, 3156, 637,  638,
           639,  640,  641,  642,  643,  3157, 3158, 3159, 3160, 3161, 3162,
           3163, 644,  645,  646,  647,  648,  649,  650,  3164, 3165, 3166,
           3167, 3168, 3169, 3170, 651,  652,  653,  654,  655,  656,  657,
           3171, 3172, 3173, 3174, 3175, 3176, 3177, 658,  659,  660,  661,
           662,  663,  664,  3178, 3179, 3180, 3181, 3182, 3183, 3184, 665,
           666,  667,  668,  669,  670,  671,  3185, 3186, 3187, 3188, 3189,
           3190, 3191, 672,  673,  674,  675,  676,  677,  678,  3192, 3193,
           3194, 3195, 3196, 3197, 3198, 679,  680,  681,  682,  683,  684,
           685,  3199, 3200, 3201, 3202, 3203, 3204, 3205, 686,  687,  688,
           689,  690,  691,  692,  3206, 3207, 3208, 3209, 3210, 3211, 3212,
           693,  694,  695,  696,  697,  698,  699,  3213, 3214, 3215, 3216,
           3217, 3218, 3219, 700,  701,  702,  703,  704,  705,  706,  3220,
           3221, 3222, 3223, 3224, 3225, 3226, 707,  708,  709,  710,  711,
           712,  713,  3227, 3228, 3229, 3230, 3231, 3232, 3233, 714,  715,
           716,  717,  718,  719,  720,  3234, 3235, 3236, 3237, 3238, 3239,
           3240, 721,  722,  723,  724,  725,  726,  727,  3241, 3242, 3243,
           3244, 3245, 3246, 3247, 728,  729,  730,  731,  732,  733,  734,
           3248, 3249, 3250, 3251, 3252, 3253, 3254, 735,  736,  737,  738,
           739,  740,  741,  3255, 3256, 3257, 3258, 3259, 3260, 3261, 742,
           743,  744,  745,  746,  747,  748,  3262, 3263, 3264, 3265, 3266,
           3267, 3268, 749,  750,  751,  752,  753,  754,  755,  3269, 3270,
           3271, 3272, 3273, 3274, 3275, 756,  757,  758,  759,  760,  761,
           762,  3276, 3277, 3278, 3279, 3280, 3281, 3282, 763,  764,  765,
           766,  767,  768,  769,  3283, 3284, 3285, 3286, 3287, 3288, 3289,
           770,  771,  772,  773,  774,  775,  776,  3290, 3291, 3292, 3293,
           3294, 3295, 3296, 777,  778,  779,  780,  781,  782,  783,  3297,
           3298, 3299, 3300, 3301, 3302, 3303, 784,  785,  786,  787,  788,
           789,  790,  3304, 3305, 3306, 3307, 3308, 3309, 3310, 791,  792,
           793,  794,  795,  796,  797,  3311, 3312, 3313, 3314, 3315, 3316,
           3317, 798,  799,  800,  801,  802,  803,  804,  3318, 3319, 3320,
           3321, 3322, 3323, 3324, 805,  806,  807,  808,  809,  810,  811,
           3325, 3326, 3327, 3328, 3329, 3330, 3331, 812,  813,  814,  815,
           816,  817,  818,  3332, 3333, 3334, 3335, 3336, 3337, 3338, 819,
           820,  821,  822,  823,  824,  825,  3339, 3340, 3341, 3342, 3343,
           3344, 3345, 826,  827,  828,  829,  830,  831,  832,  3346, 3347,
           3348, 3349, 3350, 3351, 3352, 833,  834,  835,  836,  837,  838,
           839,  3353, 3354, 3355, 3356, 3357, 3358, 3359, 840,  841,  842,
           843,  844,  845,  846,  3360, 3361, 3362, 3363, 3364, 3365, 3366,
           847,  848,  849,  850,  851,  852,  853,  3367, 3368, 3369, 3370,
           3371, 3372, 3373, 854,  855,  856,  857,  858,  859,  860,  3374,
           3375, 3376, 3377, 3378, 3379, 3380, 861,  862,  863,  864,  865,
           866,  867,  3381, 3382, 3383, 3384, 3385, 3386, 3387, 868,  869,
           870,  871,  872,  873,  874,  3388, 3389, 3390, 3391, 3392, 3393,
           3394, 875,  876,  877,  878,  879,  880,  881,  3395, 3396, 3397,
           3398, 3399, 3400, 3401, 882,  883,  884,  885,  886,  887,  888,
           3402, 3403, 3404, 3405, 3406, 3407, 3408, 889,  890,  891,  892,
           893,  894,  895,  3409, 3410, 3411, 3412, 3413, 3414, 3415, 896,
           897,  898,  899,  900,  901,  902,  3416, 3417, 3418, 3419, 3420,
           3421, 3422, 903,  904,  905,  906,  907,  908,  909,  3423, 3424,
           3425, 3426, 3427, 3428, 3429, 910,  911,  912,  913,  914,  915,
           916,  3430, 3431, 3432, 3433, 3434, 3435, 3436, 917,  918,  919,
           920,  921,  922,  923,  3437, 3438, 3439, 3440, 3441, 3442, 3443,
           924,  925,  926,  927,  928,  929,  930,  3444, 3445, 3446, 3447,
           3448, 3449, 3450, 931,  932,  933,  934,  935,  936,  937,  3451,
           3452, 3453, 3454, 3455, 3456, 3457, 938,  939,  940,  941,  942,
           943,  944,  3458, 3459, 3460, 3461, 3462, 3463, 3464, 945,  946,
           947,  948,  949,  950,  951,  3465, 3466, 3467, 3468, 3469, 3470,
           3471, 952,  953,  954,  955,  956,  957,  958,  3472, 3473, 3474,
           3475, 3476, 3477, 3478, 959,  960,  961,  962,  963,  964,  965,
           3479, 3480, 3481, 3482, 3483, 3484, 3485, 966,  967,  968,  969,
           970,  971,  972,  3486, 3487, 3488, 3489, 3490, 3491, 3492, 973,
           974,  975,  976,  977,  978,  979,  3493, 3494, 3495, 3496, 3497,
           3498, 3499, 980,  981,  982,  983,  984,  985,  986,  3500, 3501,
           3502, 3503, 3504, 3505, 3506, 987,  988,  989,  990,  991,  992,
           993,  3507, 3508, 3509, 3510, 3511, 3512, 3513, 994,  995,  996,
           997,  998,  999,  1000, 3514, 3515, 3516, 3517, 3518, 3519, 3520,
           1001, 1002, 1003, 1004, 1005, 1006, 1007, 3521, 3522, 3523, 3524,
           3525, 3526, 3527, 1008, 1009, 1010, 1011, 1012, 1013, 1014, 3528,
           3529, 3530, 3531, 3532, 3533, 3534, 1015, 1016, 1017, 1018, 1019,
           1020, 1021, 3535, 3536, 3537, 3538, 3539, 3540, 3541, 1022, 1023,
           1024, 1025, 1026, 1027, 1028, 3542, 3543, 3544, 3545, 3546, 3547,
           3548, 1029, 1030, 1031, 1032, 1033, 1034, 1035, 3549, 3550, 3551,
           3552, 3553, 3554, 3555, 1036, 1037, 1038, 1039, 1040, 1041, 1042,
           3556, 3557, 3558, 3559, 3560, 3561, 3562, 1043, 1044, 1045, 1046,
           1047, 1048, 1049, 3563, 3564, 3565, 3566, 3567, 3568, 3569, 1050,
           1051, 1052, 1053, 1054, 1055, 1056, 3570, 3571, 3572, 3573, 3574,
           3575, 3576, 1057, 1058, 1059, 1060, 1061, 1062, 1063, 3577, 3578,
           3579, 3580, 3581, 3582, 3583, 1064, 1065, 1066, 1067, 1068, 1069,
           1070, 3584, 3585, 3586, 3587, 3588, 3589, 3590, 1071, 1072, 1073,
           1074, 1075, 1076, 1077, 3591, 3592, 3593, 3594, 3595, 3596, 3597,
           1078, 1079, 1080, 1081, 1082, 1083, 1084, 3598, 3599, 3600, 3601,
           3602, 3603, 3604, 1085, 1086, 1087, 1088, 1089, 1090, 1091, 3605,
           3606, 3607, 3608, 3609, 3610, 3611, 1092, 1093, 1094, 1095, 1096,
           1097, 1098, 3612, 3613, 3614, 3615, 3616, 3617, 3618, 1099, 1100,
           1101, 1102, 1103, 1104, 1105, 3619, 3620, 3621, 3622, 3623, 3624,
           3625, 1106, 1107, 1108, 1109, 1110, 1111, 1112, 3626, 3627, 3628,
           3629, 3630, 3631, 3632, 1113, 1114, 1115, 1116, 1117, 1118, 1119,
           3633, 3634, 3635, 3636, 3637, 3638, 3639, 1120, 1121, 1122, 1123,
           1124, 1125, 1126, 3640, 3641, 3642, 3643, 3644, 3645, 3646, 1127,
           1128, 1129, 1130, 1131, 1132, 1133, 3647, 3648, 3649, 3650, 3651,
           3652, 3653, 1134, 1135, 1136, 1137, 1138, 1139, 1140, 3654, 3655,
           3656, 3657, 3658, 3659, 3660, 1141, 1142, 1143, 1144, 1145, 1146,
           1147, 3661, 3662, 3663, 3664, 3665, 3666, 3667, 1148, 1149, 1150,
           1151, 1152, 1153, 1154, 3668, 3669, 3670, 3671, 3672, 3673, 3674,
           1155, 1156, 1157, 1158, 1159, 1160, 1161, 3675, 3676, 3677, 3678,
           3679, 3680, 3681, 1162, 1163, 1164, 1165, 1166, 1167, 1168, 3682,
           3683, 3684, 3685, 3686, 3687, 3688, 1169, 1170, 1171, 1172, 1173,
           1174, 1175, 3689, 3690, 3691, 3692, 3693, 3694, 3695, 1176, 1177,
           1178, 1179, 1180, 1181, 1182, 3696, 3697, 3698, 3699, 3700, 3701,
           3702, 1183, 1184, 1185, 1186, 1187, 1188, 1189, 3703, 3704, 3705,
           3706, 3707, 3708, 3709, 1190, 1191, 1192, 1193, 1194, 1195, 1196,
           3710, 3711, 3712, 3713, 3714, 3715, 3716, 1197, 1198, 1199, 1200,
           1201, 1202, 1203, 3717, 3718, 3719, 3720, 3721, 3722, 3723, 1204,
           1205, 1206, 1207, 1208, 1209, 1210, 3724, 3725, 3726, 3727, 3728,
           3729, 3730, 1211, 1212, 1213, 1214, 1215, 1216, 1217, 3731, 3732,
           3733, 3734, 3735, 3736, 3737, 1218, 1219, 1220, 1221, 1222, 1223,
           1224, 3738, 3739, 3740, 3741, 3742, 3743, 3744, 1225, 1226, 1227,
           1228, 1229, 1230, 1231, 3745, 3746, 3747, 3748, 3749, 3750, 3751,
           1232, 1233, 1234, 1235, 1236, 1237, 1238, 3752, 3753, 3754, 3755,
           3756, 3757, 3758, 1239, 1240, 1241, 1242, 1243, 1244, 1245, 3759,
           3760, 3761, 3762, 3763, 3764, 3765, 1246, 1247, 1248, 1249, 1250,
           1251, 1252, 3766, 3767, 3768, 3769, 3770, 3771, 3772, 1253, 1254,
           1255, 1256, 1257, 1258, 1259, 3773, 3774, 3775, 3776, 3777, 3778,
           3779, 1260, 1261, 1262, 1263, 1264, 1265, 1266, 3780, 3781, 3782,
           3783, 3784, 3785, 3786, 1267, 1268, 1269, 1270, 1271, 1272, 1273,
           3787, 3788, 3789, 3790, 3791, 3792, 3793, 1274, 1275, 1276, 1277,
           1278, 1279, 1280, 3794, 3795, 3796, 3797, 3798, 3799, 3800, 1281,
           1282, 1283, 1284, 1285, 1286, 1287, 3801, 3802, 3803, 3804, 3805,
           3806, 3807, 1288, 1289, 1290, 1291, 1292, 1293, 1294, 3808, 3809,
           3810, 3811, 3812, 3813, 3814, 1295, 1296, 1297, 1298, 1299, 1300,
           1301, 3815, 3816, 3817, 3818, 3819, 3820, 3821, 1302, 1303, 1304,
           1305, 1306, 1307, 1308, 3822, 3823, 3824, 3825, 3826, 3827, 3828,
           1309, 1310, 1311, 1312, 1313, 1314, 1315, 3829, 3830, 3831, 3832,
           3833, 3834, 3835, 1316, 1317, 1318, 1319, 1320, 1321, 1322, 3836,
           3837, 3838, 3839, 3840, 3841, 3842, 1323, 1324, 1325, 1326, 1327,
           1328, 1329, 3843, 3844, 3845, 3846, 3847, 3848, 3849, 1330, 1331,
           1332, 1333, 1334, 1335, 1336, 3850, 3851, 3852, 3853, 3854, 3855,
           3856, 1337, 1338, 1339, 1340, 1341, 1342, 1343, 3857, 3858, 3859,
           3860, 3861, 3862, 3863, 1344, 1345, 1346, 1347, 1348, 1349, 1350,
           3864, 3865, 3866, 3867, 3868, 3869, 3870, 1351, 1352, 1353, 1354,
           1355, 1356, 1357, 3871, 3872, 3873, 3874, 3875, 3876, 3877, 1358,
           1359, 1360, 1361, 1362, 1363, 1364, 3878, 3879, 3880, 3881, 3882,
           3883, 3884, 1365, 1366, 1367, 1368, 1369, 1370, 1371, 3885, 3886,
           3887, 3888, 3889, 3890, 3891, 1372, 1373, 1374, 1375, 1376, 1377,
           1378, 3892, 3893, 3894, 3895, 3896, 3897, 3898, 1379, 1380, 1381,
           1382, 1383, 1384, 1385, 3899, 3900, 3901, 3902, 3903, 3904, 3905,
           1386, 1387, 1388, 1389, 1390, 1391, 1392, 3906, 3907, 3908, 3909,
           3910, 3911, 3912, 1393, 1394, 1395, 1396, 1397, 1398, 1399, 3913,
           3914, 3915, 3916, 3917, 3918, 3919, 1400, 1401, 1402, 1403, 1404,
           1405, 1406, 3920, 3921, 3922, 3923, 3924, 3925, 3926, 1407, 1408,
           1409, 1410, 1411, 1412, 1413, 3927, 3928, 3929, 3930, 3931, 3932,
           3933, 1414, 1415, 1416, 1417, 1418, 1419, 1420, 3934, 3935, 3936,
           3937, 3938, 3939, 3940, 1421, 1422, 1423, 1424, 1425, 1426, 1427,
           3941, 3942, 3943, 3944, 3945, 3946, 3947, 1428, 1429, 1430, 1431,
           1432, 1433, 1434, 3948, 3949, 3950, 3951, 3952, 3953, 3954, 1435,
           1436, 1437, 1438, 1439, 1440, 1441, 3955, 3956, 3957, 3958, 3959,
           3960, 3961, 1442, 1443, 1444, 1445, 1446, 1447, 1448, 3962, 3963,
           3964, 3965, 3966, 3967, 3968, 1449, 1450, 1451, 1452, 1453, 1454,
           1455, 3969, 3970, 3971, 3972, 3973, 3974, 3975, 1456, 1457, 1458,
           1459, 1460, 1461, 1462, 3976, 3977, 3978, 3979, 3980, 3981, 3982,
           1463, 1464, 1465, 1466, 1467, 1468, 1469, 3983, 3984, 3985, 3986,
           3987, 3988, 3989, 1470, 1471, 1472, 1473, 1474, 1475, 1476, 3990,
           3991, 3992, 3993, 3994, 3995, 3996, 1477, 1478, 1479, 1480, 1481,
           1482, 1483, 3997, 3998, 3999, 4000, 4001, 4002, 4003, 1484, 1485,
           1486, 1487, 1488, 1489, 1490, 4004, 4005, 4006, 4007, 4008, 4009,
           4010, 1491, 1492, 1493, 1494, 1495, 1496, 1497, 4011, 4012, 4013,
           4014, 4015, 4016, 4017, 1498, 1499, 1500, 1501, 1502, 1503, 1504,
           4018, 4019, 4020, 4021, 4022, 4023, 4024, 1505, 1506, 1507, 1508,
           1509, 1510, 1511, 4025, 4026, 4027, 4028, 4029, 4030, 4031, 1512,
           1513, 1514, 1515, 1516, 1517, 1518, 4032, 4033, 4034, 4035, 4036,
           4037, 4038, 1519, 1520, 1521, 1522, 1523, 1524, 1525, 4039, 4040,
           4041, 4042, 4043, 4044, 4045, 1526, 1527, 1528, 1529, 1530, 1531,
           1532, 4046, 4047, 4048, 4049, 4050, 4051, 4052, 1533, 1534, 1535,
           1536, 1537, 1538, 1539, 4053, 4054, 4055, 4056, 4057, 4058, 4059,
           1540, 1541, 1542, 1543, 1544, 1545, 1546, 4060, 4061, 4062, 4063,
           4064, 4065, 4066, 1547, 1548, 1549, 1550, 1551, 1552, 1553, 4067,
           4068, 4069, 4070, 4071, 4072, 4073, 1554, 1555, 1556, 1557, 1558,
           1559, 1560, 4074, 4075, 4076, 4077, 4078, 4079, 4080, 1561, 1562,
           1563, 1564, 1565, 1566, 1567, 4081, 4082, 4083, 4084, 4085, 4086,
           4087, 1568, 1569, 1570, 1571, 1572, 1573, 1574, 4088, 4089, 4090,
           4091, 4092, 4093, 4094, 1575, 1576, 1577, 1578, 1579, 1580, 1581,
           4095, 4096, 4097, 4098, 4099, 4100, 4101, 1582, 1583, 1584, 1585,
           1586, 1587, 1588, 4102, 4103, 4104, 4105, 4106, 4107, 4108, 1589,
           1590, 1591, 1592, 1593, 1594, 1595, 4109, 4110, 4111, 4112, 4113,
           4114, 4115, 1596, 1597, 1598, 1599, 1600, 1601, 1602, 4116, 4117,
           4118, 4119, 4120, 4121, 4122, 1603, 1604, 1605, 1606, 1607, 1608,
           1609, 4123, 4124, 4125, 4126, 4127, 4128, 4129, 1610, 1611, 1612,
           1613, 1614, 1615, 1616, 4130, 4131, 4132, 4133, 4134, 4135, 4136,
           1617, 1618, 1619, 1620, 1621, 1622, 1623, 4137, 4138, 4139, 4140,
           4141, 4142, 4143, 1624, 1625, 1626, 1627, 1628, 1629, 1630, 4144,
           4145, 4146, 4147, 4148, 4149, 4150, 1631, 1632, 1633, 1634, 1635,
           1636, 1637, 4151, 4152, 4153, 4154, 4155, 4156, 4157, 1638, 1639,
           1640, 1641, 1642, 1643, 1644, 4158, 4159, 4160, 4161, 4162, 4163,
           4164, 1645, 1646, 1647, 1648, 1649, 1650, 1651, 4165, 4166, 4167,
           4168, 4169, 4170, 4171, 1652, 1653, 1654, 1655, 1656, 1657, 1658,
           4172, 4173, 4174, 4175, 4176, 4177, 4178, 1659, 1660, 1661, 1662,
           1663, 1664, 1665, 4179, 4180, 4181, 4182, 4183, 4184, 4185, 1666,
           1667, 1668, 1669, 1670, 1671, 1672, 4186, 4187, 4188, 4189, 4190,
           4191, 4192, 1673, 1674, 1675, 1676, 1677, 1678, 1679, 4193, 4194,
           4195, 4196, 4197, 4198, 4199, 1680, 1681, 1682, 1683, 1684, 1685,
           1686, 4200, 4201, 4202, 4203, 4204, 4205, 4206, 1687, 1688, 1689,
           1690, 1691, 1692, 1693, 4207, 4208, 4209, 4210, 4211, 4212, 4213,
           1694, 1695, 1696, 1697, 1698, 1699, 1700, 4214, 4215, 4216, 4217,
           4218, 4219, 4220, 1701, 1702, 1703, 1704, 1705, 1706, 1707, 4221,
           4222, 4223, 4224, 4225, 4226, 4227, 1708, 1709, 1710, 1711, 1712,
           1713, 1714, 4228, 4229, 4230, 4231, 4232, 4233, 4234, 1715, 1716,
           1717, 1718, 1719, 1720, 1721, 4235, 4236, 4237, 4238, 4239, 4240,
           4241, 1722, 1723, 1724, 1725, 1726, 1727, 1728, 4242, 4243, 4244,
           4245, 4246, 4247, 4248, 1729, 1730, 1731, 1732, 1733, 1734, 1735,
           4249, 4250, 4251, 4252, 4253, 4254, 4255, 1736, 1737, 1738, 1739,
           1740, 1741, 1742, 4256, 4257, 4258, 4259, 4260, 4261, 4262, 1743,
           1744, 1745, 1746, 1747, 1748, 1749, 4263, 4264, 4265, 4266, 4267,
           4268, 4269, 1750, 1751, 1752, 1753, 1754, 1755, 1756, 4270, 4271,
           4272, 4273, 4274, 4275, 4276, 1757, 1758, 1759, 1760, 1761, 1762,
           1763, 4277, 4278, 4279, 4280, 4281, 4282, 4283, 1764, 1765, 1766,
           1767, 1768, 1769, 1770, 4284, 4285, 4286, 4287, 4288, 4289, 4290,
           1771, 1772, 1773, 1774, 1775, 1776, 1777, 4291, 4292, 4293, 4294,
           4295, 4296, 4297, 1778, 1779, 1780, 1781, 1782, 1783, 1784, 4298,
           4299, 4300, 4301, 4302, 4303, 4304, 1785, 1786, 1787, 1788, 1789,
           1790, 1791, 4305, 4306, 4307, 4308, 4309, 4310, 4311, 1792, 1793,
           1794, 1795, 1796, 1797, 1798, 4312, 4313, 4314, 4315, 4316, 4317,
           4318, 1799, 1800, 1801, 1802, 1803, 1804, 1805, 4319, 4320, 4321,
           4322, 4323, 4324, 4325, 1806, 1807, 1808, 1809, 1810, 1811, 1812,
           4326, 4327, 4328, 4329, 4330, 4331, 4332, 1813, 1814, 1815, 1816,
           1817, 1818, 1819, 4333, 4334, 4335, 4336, 4337, 4338, 4339, 1820,
           1821, 1822, 1823, 1824, 1825, 1826, 4340, 4341, 4342, 4343, 4344,
           4345, 4346, 1827, 1828, 1829, 1830, 1831, 1832, 1833, 4347, 4348,
           4349, 4350, 4351, 4352, 4353, 1834, 1835, 1836, 1837, 1838, 1839,
           1840, 4354, 4355, 4356, 4357, 4358, 4359, 4360, 1841, 1842, 1843,
           1844, 1845, 1846, 1847, 4361, 4362, 4363, 4364, 4365, 4366, 4367,
           1848, 1849, 1850, 1851, 1852, 1853, 1854, 4368, 4369, 4370, 4371,
           4372, 4373, 4374, 1855, 1856, 1857, 1858, 1859, 1860, 1861, 4375,
           4376, 4377, 4378, 4379, 4380, 4381, 1862, 1863, 1864, 1865, 1866,
           1867, 1868, 4382, 4383, 4384, 4385, 4386, 4387, 4388, 1869, 1870,
           1871, 1872, 1873, 1874, 1875, 4389, 4390, 4391, 4392, 4393, 4394,
           4395, 1876, 1877, 1878, 1879, 1880, 1881, 1882, 4396, 4397, 4398,
           4399, 4400, 4401, 4402, 1883, 1884, 1885, 1886, 1887, 1888, 1889,
           4403, 4404, 4405, 4406, 4407, 4408, 4409, 1890, 1891, 1892, 1893,
           1894, 1895, 1896, 4410, 4411, 4412, 4413, 4414, 4415, 4416, 1897,
           1898, 1899, 1900, 1901, 1902, 1903, 4417, 4418, 4419, 4420, 4421,
           4422, 4423, 1904, 1905, 1906, 1907, 1908, 1909, 1910, 4424, 4425,
           4426, 4427, 4428, 4429, 4430, 1911, 1912, 1913, 1914, 1915, 1916,
           1917, 4431, 4432, 4433, 4434, 4435, 4436, 4437, 1918, 1919, 1920,
           1921, 1922, 1923, 1924, 4438, 4439, 4440, 4441, 4442, 4443, 4444,
           1925, 1926, 1927, 1928, 1929, 1930, 1931, 4445, 4446, 4447, 4448,
           4449, 4450, 4451, 1932, 1933, 1934, 1935, 1936, 1937, 1938, 4452,
           4453, 4454, 4455, 4456, 4457, 4458, 1939, 1940, 1941, 1942, 1943,
           1944, 1945, 4459, 4460, 4461, 4462, 4463, 4464, 4465, 1946, 1947,
           1948, 1949, 1950, 1951, 1952, 4466, 4467, 4468, 4469, 4470, 4471,
           4472, 1953, 1954, 1955, 1956, 1957, 1958, 1959, 4473, 4474, 4475,
           4476, 4477, 4478, 4479, 1960, 1961, 1962, 1963, 1964, 1965, 1966,
           4480, 4481, 4482, 4483, 4484, 4485, 4486, 1967, 1968, 1969, 1970,
           1971, 1972, 1973, 4487, 4488, 4489, 4490, 4491, 4492, 4493, 1974,
           1975, 1976, 1977, 1978, 1979, 1980, 4494, 4495, 4496, 4497, 4498,
           4499, 4500, 1981, 1982, 1983, 1984, 1985, 1986, 1987, 4501, 4502,
           4503, 4504, 4505, 4506, 4507, 1988, 1989, 1990, 1991, 1992, 1993,
           1994, 4508, 4509, 4510, 4511, 4512, 4513, 4514, 1995, 1996, 1997,
           1998, 1999, 2000, 2001, 4515, 4516, 4517, 4518, 4519, 4520, 4521,
           2002, 2003, 2004, 2005, 2006, 2007, 2008, 4522, 4523, 4524, 4525,
           4526, 4527, 4528, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 4529,
           4530, 4531, 4532, 4533, 4534, 4535, 2016, 2017, 2018, 2019, 2020,
           2021, 2022, 4536, 4537, 4538, 4539, 4540, 4541, 4542, 2023, 2024,
           2025, 2026, 2027, 2028, 2029, 4543, 4544, 4545, 4546, 4547, 4548,
           4549, 2030, 2031, 2032, 2033, 2034, 2035, 2036, 4550, 4551, 4552,
           4553, 4554, 4555, 4556, 2037, 2038, 2039, 2040, 2041, 2042, 2043,
           4557, 4558, 4559, 4560, 4561, 4562, 4563, 2044, 2045, 2046, 2047,
           2048, 2049, 2050, 4564, 4565, 4566, 4567, 4568, 4569, 4570, 2051,
           2052, 2053, 2054, 2055, 2056, 2057, 4571, 4572, 4573, 4574, 4575,
           4576, 4577, 2058, 2059, 2060, 2061, 2062, 2063, 2064, 4578, 4579,
           4580, 4581, 4582, 4583, 4584, 2065, 2066, 2067, 2068, 2069, 2070,
           2071, 4585, 4586, 4587, 4588, 4589, 4590, 4591, 2072, 2073, 2074,
           2075, 2076, 2077, 2078, 4592, 4593, 4594, 4595, 4596, 4597, 4598,
           2079, 2080, 2081, 2082, 2083, 2084, 2085, 4599, 4600, 4601, 4602,
           4603, 4604, 4605, 2086, 2087, 2088, 2089, 2090, 2091, 2092, 4606,
           4607, 4608, 4609, 4610, 4611, 4612, 2093, 2094, 2095, 2096, 2097,
           2098, 2099, 4613, 4614, 4615, 4616, 4617, 4618, 4619, 2100, 2101,
           2102, 2103, 2104, 2105, 2106, 4620, 4621, 4622, 4623, 4624, 4625,
           4626, 2107, 2108, 2109, 2110, 2111, 2112, 2113, 4627, 4628, 4629,
           4630, 4631, 4632, 4633, 2114, 2115, 2116, 2117, 2118, 2119, 2120,
           4634, 4635, 4636, 4637, 4638, 4639, 4640, 2121, 2122, 2123, 2124,
           2125, 2126, 2127, 4641, 4642, 4643, 4644, 4645, 4646, 4647, 2128,
           2129, 2130, 2131, 2132, 2133, 2134, 4648, 4649, 4650, 4651, 4652,
           4653, 4654, 2135, 2136, 2137, 2138, 2139, 2140, 2141, 4655, 4656,
           4657, 4658, 4659, 4660, 4661, 2142, 2143, 2144, 2145, 2146, 2147,
           2148, 4662, 4663, 4664, 4665, 4666, 4667, 4668, 2149, 2150, 2151,
           2152, 2153, 2154, 2155, 4669, 4670, 4671, 4672, 4673, 4674, 4675,
           2156, 2157, 2158, 2159, 2160, 2161, 2162, 4676, 4677, 4678, 4679,
           4680, 4681, 4682, 2163, 2164, 2165, 2166, 2167, 2168, 2169, 4683,
           4684, 4685, 4686, 4687, 4688, 4689, 2170, 2171, 2172, 2173, 2174,
           2175, 2176, 4690, 4691, 4692, 4693, 4694, 4695, 4696, 2177, 2178,
           2179, 2180, 2181, 2182, 2183, 4697, 4698, 4699, 4700, 4701, 4702,
           4703, 2184, 2185, 2186, 2187, 2188, 2189, 2190, 4704, 4705, 4706,
           4707, 4708, 4709, 4710, 2191, 2192, 2193, 2194, 2195, 2196, 2197,
           4711, 4712, 4713, 4714, 4715, 4716, 4717, 2198, 2199, 2200, 2201,
           2202, 2203, 2204, 4718, 4719, 4720, 4721, 4722, 4723, 4724, 2205,
           2206, 2207, 2208, 2209, 2210, 2211, 4725, 4726, 4727, 4728, 4729,
           4730, 4731, 2212, 2213, 2214, 2215, 2216, 2217, 2218, 4732, 4733,
           4734, 4735, 4736, 4737, 4738, 2219, 2220, 2221, 2222, 2223, 2224,
           2225, 4739, 4740, 4741, 4742, 4743, 4744, 4745, 2226, 2227, 2228,
           2229, 2230, 2231, 2232, 4746, 4747, 4748, 4749, 4750, 4751, 4752,
           2233, 2234, 2235, 2236, 2237, 2238, 2239, 4753, 4754, 4755, 4756,
           4757, 4758, 4759, 2240, 2241, 2242, 2243, 2244, 2245, 2246, 4760,
           4761, 4762, 4763, 4764, 4765, 4766, 2247, 2248, 2249, 2250, 2251,
           2252, 2253, 4767, 4768, 4769, 4770, 4771, 4772, 4773, 2254, 2255,
           2256, 2257, 2258, 2259, 2260, 4774, 4775, 4776, 4777, 4778, 4779,
           4780, 2261, 2262, 2263, 2264, 2265, 2266, 2267, 4781, 4782, 4783,
           4784, 4785, 4786, 4787, 2268, 2269, 2270, 2271, 2272, 2273, 2274,
           4788, 4789, 4790, 4791, 4792, 4793, 4794, 2275, 2276, 2277, 2278,
           2279, 2280, 2281, 4795, 4796, 4797, 4798, 4799, 4800, 4801, 2282,
           2283, 2284, 2285, 2286, 2287, 2288, 4802, 4803, 4804, 4805, 4806,
           4807, 4808, 2289, 2290, 2291, 2292, 2293, 2294, 2295, 4809, 4810,
           4811, 4812, 4813, 4814, 4815, 2296, 2297, 2298, 2299, 2300, 2301,
           2302, 4816, 4817, 4818, 4819, 4820, 4821, 4822, 2303, 2304, 2305,
           2306, 2307, 2308, 2309, 4823, 4824, 4825, 4826, 4827, 4828, 4829,
           2310, 2311, 2312, 2313, 2314, 2315, 2316, 4830, 4831, 4832, 4833,
           4834, 4835, 4836, 2317, 2318, 2319, 2320, 2321, 2322, 2323, 4837,
           4838, 4839, 4840, 4841, 4842, 4843, 2324, 2325, 2326, 2327, 2328,
           2329, 2330, 4844, 4845, 4846, 4847, 4848, 4849, 4850, 2331, 2332,
           2333, 2334, 2335, 2336, 2337, 4851, 4852, 4853, 4854, 4855, 4856,
           4857, 2338, 2339, 2340, 2341, 2342, 2343, 2344, 4858, 4859, 4860,
           4861, 4862, 4863, 4864, 2345, 2346, 2347, 2348, 2349, 2350, 2351,
           4865, 4866, 4867, 4868, 4869, 4870, 4871, 2352, 2353, 2354, 2355,
           2356, 2357, 2358, 4872, 4873, 4874, 4875, 4876, 4877, 4878, 2359,
           2360, 2361, 2362, 2363, 2364, 2365, 4879, 4880, 4881, 4882, 4883,
           4884, 4885, 2366, 2367, 2368, 2369, 2370, 2371, 2372, 4886, 4887,
           4888, 4889, 4890, 4891, 4892, 2373, 2374, 2375, 2376, 2377, 2378,
           2379, 4893, 4894, 4895, 4896, 4897, 4898, 4899, 2380, 2381, 2382,
           2383, 2384, 2385, 2386, 4900, 4901, 4902, 4903, 4904, 4905, 4906,
           2387, 2388, 2389, 2390, 2391, 2392, 2393, 4907, 4908, 4909, 4910,
           4911, 4912, 4913, 2394, 2395, 2396, 2397, 2398, 2399, 2400, 4914,
           4915, 4916, 4917, 4918, 4919, 4920, 2401, 2402, 2403, 2404, 2405,
           2406, 2407, 4921, 4922, 4923, 4924, 4925, 4926, 4927, 2408, 2409,
           2410, 2411, 2412, 2413, 2414, 4928, 4929, 4930, 4931, 4932, 4933,
           4934, 2415, 2416, 2417, 2418, 2419, 2420, 2421, 4935, 4936, 4937,
           4938, 4939, 4940, 4941, 2422, 2423, 2424, 2425, 2426, 2427, 2428,
           4942, 4943, 4944, 4945, 4946, 4947, 4948, 2429, 2430, 2431, 2432,
           2433, 2434, 2435, 4949, 4950, 4951, 4952, 4953, 4954, 4955, 2436,
           2437, 2438, 2439, 2440, 2441, 2442, 4956, 4957, 4958, 4959, 4960,
           4961, 4962, 2443, 2444, 2445, 2446, 2447, 2448, 2449, 4963, 4964,
           4965, 4966, 4967, 4968, 4969, 2450, 2451, 2452, 2453, 2454, 2455,
           2456, 4970, 4971, 4972, 4973, 4974, 4975, 4976, 2457, 2458, 2459,
           2460, 2461, 2462, 2463, 4977, 4978, 4979, 4980, 4981, 4982, 4983,
           2464, 2465, 2466, 2467, 2468, 2469, 2470, 4984, 4985, 4986, 4987,
           4988, 4989, 4990, 2471, 2472, 2473, 2474, 2475, 2476, 2477, 4991,
           4992, 4993, 4994, 4995, 4996, 4997, 2478, 2479, 2480, 2481, 2482,
           2483, 2484, 4998, 4999, 5000, 5001, 5002, 5003, 5004, 2485, 2486,
           2487, 2488, 2489, 2490, 2491, 5005, 5006, 5007, 5008, 5009, 5010,
           5011, 2492, 2493, 2494, 2495, 2496, 2497, 2498, 5012, 5013, 5014,
           5015, 5016, 5017, 5018, 2499, 2500, 2501, 2502, 2503, 2504, 2505,
           5019, 5020, 5021, 5022, 5023, 5024, 5025, 2506, 2507, 2508, 2509,
           2510, 2511, 2512, 5026, 5027, 5028, 5029, 5030, 5031, 5032, 2513,
           2514, 2515, 2516, 2517, 2518, 2519, 5033, 5034, 5035, 5036, 5037,
           5038, 5039}));
}

TEST(TransposeTest, TestRefOps6D6) {
  EXPECT_THAT(
      RunTestPermutation<float>(/*shape=*/{2, 3, 4, 5, 6, 7},
                                /*perms=*/{1, 2, 3, 4, 5, 0}),
      ElementsAreArray(
          {0,    2520, 1,    2521, 2,    2522, 3,    2523, 4,    2524, 5,
           2525, 6,    2526, 7,    2527, 8,    2528, 9,    2529, 10,   2530,
           11,   2531, 12,   2532, 13,   2533, 14,   2534, 15,   2535, 16,
           2536, 17,   2537, 18,   2538, 19,   2539, 20,   2540, 21,   2541,
           22,   2542, 23,   2543, 24,   2544, 25,   2545, 26,   2546, 27,
           2547, 28,   2548, 29,   2549, 30,   2550, 31,   2551, 32,   2552,
           33,   2553, 34,   2554, 35,   2555, 36,   2556, 37,   2557, 38,
           2558, 39,   2559, 40,   2560, 41,   2561, 42,   2562, 43,   2563,
           44,   2564, 45,   2565, 46,   2566, 47,   2567, 48,   2568, 49,
           2569, 50,   2570, 51,   2571, 52,   2572, 53,   2573, 54,   2574,
           55,   2575, 56,   2576, 57,   2577, 58,   2578, 59,   2579, 60,
           2580, 61,   2581, 62,   2582, 63,   2583, 64,   2584, 65,   2585,
           66,   2586, 67,   2587, 68,   2588, 69,   2589, 70,   2590, 71,
           2591, 72,   2592, 73,   2593, 74,   2594, 75,   2595, 76,   2596,
           77,   2597, 78,   2598, 79,   2599, 80,   2600, 81,   2601, 82,
           2602, 83,   2603, 84,   2604, 85,   2605, 86,   2606, 87,   2607,
           88,   2608, 89,   2609, 90,   2610, 91,   2611, 92,   2612, 93,
           2613, 94,   2614, 95,   2615, 96,   2616, 97,   2617, 98,   2618,
           99,   2619, 100,  2620, 101,  2621, 102,  2622, 103,  2623, 104,
           2624, 105,  2625, 106,  2626, 107,  2627, 108,  2628, 109,  2629,
           110,  2630, 111,  2631, 112,  2632, 113,  2633, 114,  2634, 115,
           2635, 116,  2636, 117,  2637, 118,  2638, 119,  2639, 120,  2640,
           121,  2641, 122,  2642, 123,  2643, 124,  2644, 125,  2645, 126,
           2646, 127,  2647, 128,  2648, 129,  2649, 130,  2650, 131,  2651,
           132,  2652, 133,  2653, 134,  2654, 135,  2655, 136,  2656, 137,
           2657, 138,  2658, 139,  2659, 140,  2660, 141,  2661, 142,  2662,
           143,  2663, 144,  2664, 145,  2665, 146,  2666, 147,  2667, 148,
           2668, 149,  2669, 150,  2670, 151,  2671, 152,  2672, 153,  2673,
           154,  2674, 155,  2675, 156,  2676, 157,  2677, 158,  2678, 159,
           2679, 160,  2680, 161,  2681, 162,  2682, 163,  2683, 164,  2684,
           165,  2685, 166,  2686, 167,  2687, 168,  2688, 169,  2689, 170,
           2690, 171,  2691, 172,  2692, 173,  2693, 174,  2694, 175,  2695,
           176,  2696, 177,  2697, 178,  2698, 179,  2699, 180,  2700, 181,
           2701, 182,  2702, 183,  2703, 184,  2704, 185,  2705, 186,  2706,
           187,  2707, 188,  2708, 189,  2709, 190,  2710, 191,  2711, 192,
           2712, 193,  2713, 194,  2714, 195,  2715, 196,  2716, 197,  2717,
           198,  2718, 199,  2719, 200,  2720, 201,  2721, 202,  2722, 203,
           2723, 204,  2724, 205,  2725, 206,  2726, 207,  2727, 208,  2728,
           209,  2729, 210,  2730, 211,  2731, 212,  2732, 213,  2733, 214,
           2734, 215,  2735, 216,  2736, 217,  2737, 218,  2738, 219,  2739,
           220,  2740, 221,  2741, 222,  2742, 223,  2743, 224,  2744, 225,
           2745, 226,  2746, 227,  2747, 228,  2748, 229,  2749, 230,  2750,
           231,  2751, 232,  2752, 233,  2753, 234,  2754, 235,  2755, 236,
           2756, 237,  2757, 238,  2758, 239,  2759, 240,  2760, 241,  2761,
           242,  2762, 243,  2763, 244,  2764, 245,  2765, 246,  2766, 247,
           2767, 248,  2768, 249,  2769, 250,  2770, 251,  2771, 252,  2772,
           253,  2773, 254,  2774, 255,  2775, 256,  2776, 257,  2777, 258,
           2778, 259,  2779, 260,  2780, 261,  2781, 262,  2782, 263,  2783,
           264,  2784, 265,  2785, 266,  2786, 267,  2787, 268,  2788, 269,
           2789, 270,  2790, 271,  2791, 272,  2792, 273,  2793, 274,  2794,
           275,  2795, 276,  2796, 277,  2797, 278,  2798, 279,  2799, 280,
           2800, 281,  2801, 282,  2802, 283,  2803, 284,  2804, 285,  2805,
           286,  2806, 287,  2807, 288,  2808, 289,  2809, 290,  2810, 291,
           2811, 292,  2812, 293,  2813, 294,  2814, 295,  2815, 296,  2816,
           297,  2817, 298,  2818, 299,  2819, 300,  2820, 301,  2821, 302,
           2822, 303,  2823, 304,  2824, 305,  2825, 306,  2826, 307,  2827,
           308,  2828, 309,  2829, 310,  2830, 311,  2831, 312,  2832, 313,
           2833, 314,  2834, 315,  2835, 316,  2836, 317,  2837, 318,  2838,
           319,  2839, 320,  2840, 321,  2841, 322,  2842, 323,  2843, 324,
           2844, 325,  2845, 326,  2846, 327,  2847, 328,  2848, 329,  2849,
           330,  2850, 331,  2851, 332,  2852, 333,  2853, 334,  2854, 335,
           2855, 336,  2856, 337,  2857, 338,  2858, 339,  2859, 340,  2860,
           341,  2861, 342,  2862, 343,  2863, 344,  2864, 345,  2865, 346,
           2866, 347,  2867, 348,  2868, 349,  2869, 350,  2870, 351,  2871,
           352,  2872, 353,  2873, 354,  2874, 355,  2875, 356,  2876, 357,
           2877, 358,  2878, 359,  2879, 360,  2880, 361,  2881, 362,  2882,
           363,  2883, 364,  2884, 365,  2885, 366,  2886, 367,  2887, 368,
           2888, 369,  2889, 370,  2890, 371,  2891, 372,  2892, 373,  2893,
           374,  2894, 375,  2895, 376,  2896, 377,  2897, 378,  2898, 379,
           2899, 380,  2900, 381,  2901, 382,  2902, 383,  2903, 384,  2904,
           385,  2905, 386,  2906, 387,  2907, 388,  2908, 389,  2909, 390,
           2910, 391,  2911, 392,  2912, 393,  2913, 394,  2914, 395,  2915,
           396,  2916, 397,  2917, 398,  2918, 399,  2919, 400,  2920, 401,
           2921, 402,  2922, 403,  2923, 404,  2924, 405,  2925, 406,  2926,
           407,  2927, 408,  2928, 409,  2929, 410,  2930, 411,  2931, 412,
           2932, 413,  2933, 414,  2934, 415,  2935, 416,  2936, 417,  2937,
           418,  2938, 419,  2939, 420,  2940, 421,  2941, 422,  2942, 423,
           2943, 424,  2944, 425,  2945, 426,  2946, 427,  2947, 428,  2948,
           429,  2949, 430,  2950, 431,  2951, 432,  2952, 433,  2953, 434,
           2954, 435,  2955, 436,  2956, 437,  2957, 438,  2958, 439,  2959,
           440,  2960, 441,  2961, 442,  2962, 443,  2963, 444,  2964, 445,
           2965, 446,  2966, 447,  2967, 448,  2968, 449,  2969, 450,  2970,
           451,  2971, 452,  2972, 453,  2973, 454,  2974, 455,  2975, 456,
           2976, 457,  2977, 458,  2978, 459,  2979, 460,  2980, 461,  2981,
           462,  2982, 463,  2983, 464,  2984, 465,  2985, 466,  2986, 467,
           2987, 468,  2988, 469,  2989, 470,  2990, 471,  2991, 472,  2992,
           473,  2993, 474,  2994, 475,  2995, 476,  2996, 477,  2997, 478,
           2998, 479,  2999, 480,  3000, 481,  3001, 482,  3002, 483,  3003,
           484,  3004, 485,  3005, 486,  3006, 487,  3007, 488,  3008, 489,
           3009, 490,  3010, 491,  3011, 492,  3012, 493,  3013, 494,  3014,
           495,  3015, 496,  3016, 497,  3017, 498,  3018, 499,  3019, 500,
           3020, 501,  3021, 502,  3022, 503,  3023, 504,  3024, 505,  3025,
           506,  3026, 507,  3027, 508,  3028, 509,  3029, 510,  3030, 511,
           3031, 512,  3032, 513,  3033, 514,  3034, 515,  3035, 516,  3036,
           517,  3037, 518,  3038, 519,  3039, 520,  3040, 521,  3041, 522,
           3042, 523,  3043, 524,  3044, 525,  3045, 526,  3046, 527,  3047,
           528,  3048, 529,  3049, 530,  3050, 531,  3051, 532,  3052, 533,
           3053, 534,  3054, 535,  3055, 536,  3056, 537,  3057, 538,  3058,
           539,  3059, 540,  3060, 541,  3061, 542,  3062, 543,  3063, 544,
           3064, 545,  3065, 546,  3066, 547,  3067, 548,  3068, 549,  3069,
           550,  3070, 551,  3071, 552,  3072, 553,  3073, 554,  3074, 555,
           3075, 556,  3076, 557,  3077, 558,  3078, 559,  3079, 560,  3080,
           561,  3081, 562,  3082, 563,  3083, 564,  3084, 565,  3085, 566,
           3086, 567,  3087, 568,  3088, 569,  3089, 570,  3090, 571,  3091,
           572,  3092, 573,  3093, 574,  3094, 575,  3095, 576,  3096, 577,
           3097, 578,  3098, 579,  3099, 580,  3100, 581,  3101, 582,  3102,
           583,  3103, 584,  3104, 585,  3105, 586,  3106, 587,  3107, 588,
           3108, 589,  3109, 590,  3110, 591,  3111, 592,  3112, 593,  3113,
           594,  3114, 595,  3115, 596,  3116, 597,  3117, 598,  3118, 599,
           3119, 600,  3120, 601,  3121, 602,  3122, 603,  3123, 604,  3124,
           605,  3125, 606,  3126, 607,  3127, 608,  3128, 609,  3129, 610,
           3130, 611,  3131, 612,  3132, 613,  3133, 614,  3134, 615,  3135,
           616,  3136, 617,  3137, 618,  3138, 619,  3139, 620,  3140, 621,
           3141, 622,  3142, 623,  3143, 624,  3144, 625,  3145, 626,  3146,
           627,  3147, 628,  3148, 629,  3149, 630,  3150, 631,  3151, 632,
           3152, 633,  3153, 634,  3154, 635,  3155, 636,  3156, 637,  3157,
           638,  3158, 639,  3159, 640,  3160, 641,  3161, 642,  3162, 643,
           3163, 644,  3164, 645,  3165, 646,  3166, 647,  3167, 648,  3168,
           649,  3169, 650,  3170, 651,  3171, 652,  3172, 653,  3173, 654,
           3174, 655,  3175, 656,  3176, 657,  3177, 658,  3178, 659,  3179,
           660,  3180, 661,  3181, 662,  3182, 663,  3183, 664,  3184, 665,
           3185, 666,  3186, 667,  3187, 668,  3188, 669,  3189, 670,  3190,
           671,  3191, 672,  3192, 673,  3193, 674,  3194, 675,  3195, 676,
           3196, 677,  3197, 678,  3198, 679,  3199, 680,  3200, 681,  3201,
           682,  3202, 683,  3203, 684,  3204, 685,  3205, 686,  3206, 687,
           3207, 688,  3208, 689,  3209, 690,  3210, 691,  3211, 692,  3212,
           693,  3213, 694,  3214, 695,  3215, 696,  3216, 697,  3217, 698,
           3218, 699,  3219, 700,  3220, 701,  3221, 702,  3222, 703,  3223,
           704,  3224, 705,  3225, 706,  3226, 707,  3227, 708,  3228, 709,
           3229, 710,  3230, 711,  3231, 712,  3232, 713,  3233, 714,  3234,
           715,  3235, 716,  3236, 717,  3237, 718,  3238, 719,  3239, 720,
           3240, 721,  3241, 722,  3242, 723,  3243, 724,  3244, 725,  3245,
           726,  3246, 727,  3247, 728,  3248, 729,  3249, 730,  3250, 731,
           3251, 732,  3252, 733,  3253, 734,  3254, 735,  3255, 736,  3256,
           737,  3257, 738,  3258, 739,  3259, 740,  3260, 741,  3261, 742,
           3262, 743,  3263, 744,  3264, 745,  3265, 746,  3266, 747,  3267,
           748,  3268, 749,  3269, 750,  3270, 751,  3271, 752,  3272, 753,
           3273, 754,  3274, 755,  3275, 756,  3276, 757,  3277, 758,  3278,
           759,  3279, 760,  3280, 761,  3281, 762,  3282, 763,  3283, 764,
           3284, 765,  3285, 766,  3286, 767,  3287, 768,  3288, 769,  3289,
           770,  3290, 771,  3291, 772,  3292, 773,  3293, 774,  3294, 775,
           3295, 776,  3296, 777,  3297, 778,  3298, 779,  3299, 780,  3300,
           781,  3301, 782,  3302, 783,  3303, 784,  3304, 785,  3305, 786,
           3306, 787,  3307, 788,  3308, 789,  3309, 790,  3310, 791,  3311,
           792,  3312, 793,  3313, 794,  3314, 795,  3315, 796,  3316, 797,
           3317, 798,  3318, 799,  3319, 800,  3320, 801,  3321, 802,  3322,
           803,  3323, 804,  3324, 805,  3325, 806,  3326, 807,  3327, 808,
           3328, 809,  3329, 810,  3330, 811,  3331, 812,  3332, 813,  3333,
           814,  3334, 815,  3335, 816,  3336, 817,  3337, 818,  3338, 819,
           3339, 820,  3340, 821,  3341, 822,  3342, 823,  3343, 824,  3344,
           825,  3345, 826,  3346, 827,  3347, 828,  3348, 829,  3349, 830,
           3350, 831,  3351, 832,  3352, 833,  3353, 834,  3354, 835,  3355,
           836,  3356, 837,  3357, 838,  3358, 839,  3359, 840,  3360, 841,
           3361, 842,  3362, 843,  3363, 844,  3364, 845,  3365, 846,  3366,
           847,  3367, 848,  3368, 849,  3369, 850,  3370, 851,  3371, 852,
           3372, 853,  3373, 854,  3374, 855,  3375, 856,  3376, 857,  3377,
           858,  3378, 859,  3379, 860,  3380, 861,  3381, 862,  3382, 863,
           3383, 864,  3384, 865,  3385, 866,  3386, 867,  3387, 868,  3388,
           869,  3389, 870,  3390, 871,  3391, 872,  3392, 873,  3393, 874,
           3394, 875,  3395, 876,  3396, 877,  3397, 878,  3398, 879,  3399,
           880,  3400, 881,  3401, 882,  3402, 883,  3403, 884,  3404, 885,
           3405, 886,  3406, 887,  3407, 888,  3408, 889,  3409, 890,  3410,
           891,  3411, 892,  3412, 893,  3413, 894,  3414, 895,  3415, 896,
           3416, 897,  3417, 898,  3418, 899,  3419, 900,  3420, 901,  3421,
           902,  3422, 903,  3423, 904,  3424, 905,  3425, 906,  3426, 907,
           3427, 908,  3428, 909,  3429, 910,  3430, 911,  3431, 912,  3432,
           913,  3433, 914,  3434, 915,  3435, 916,  3436, 917,  3437, 918,
           3438, 919,  3439, 920,  3440, 921,  3441, 922,  3442, 923,  3443,
           924,  3444, 925,  3445, 926,  3446, 927,  3447, 928,  3448, 929,
           3449, 930,  3450, 931,  3451, 932,  3452, 933,  3453, 934,  3454,
           935,  3455, 936,  3456, 937,  3457, 938,  3458, 939,  3459, 940,
           3460, 941,  3461, 942,  3462, 943,  3463, 944,  3464, 945,  3465,
           946,  3466, 947,  3467, 948,  3468, 949,  3469, 950,  3470, 951,
           3471, 952,  3472, 953,  3473, 954,  3474, 955,  3475, 956,  3476,
           957,  3477, 958,  3478, 959,  3479, 960,  3480, 961,  3481, 962,
           3482, 963,  3483, 964,  3484, 965,  3485, 966,  3486, 967,  3487,
           968,  3488, 969,  3489, 970,  3490, 971,  3491, 972,  3492, 973,
           3493, 974,  3494, 975,  3495, 976,  3496, 977,  3497, 978,  3498,
           979,  3499, 980,  3500, 981,  3501, 982,  3502, 983,  3503, 984,
           3504, 985,  3505, 986,  3506, 987,  3507, 988,  3508, 989,  3509,
           990,  3510, 991,  3511, 992,  3512, 993,  3513, 994,  3514, 995,
           3515, 996,  3516, 997,  3517, 998,  3518, 999,  3519, 1000, 3520,
           1001, 3521, 1002, 3522, 1003, 3523, 1004, 3524, 1005, 3525, 1006,
           3526, 1007, 3527, 1008, 3528, 1009, 3529, 1010, 3530, 1011, 3531,
           1012, 3532, 1013, 3533, 1014, 3534, 1015, 3535, 1016, 3536, 1017,
           3537, 1018, 3538, 1019, 3539, 1020, 3540, 1021, 3541, 1022, 3542,
           1023, 3543, 1024, 3544, 1025, 3545, 1026, 3546, 1027, 3547, 1028,
           3548, 1029, 3549, 1030, 3550, 1031, 3551, 1032, 3552, 1033, 3553,
           1034, 3554, 1035, 3555, 1036, 3556, 1037, 3557, 1038, 3558, 1039,
           3559, 1040, 3560, 1041, 3561, 1042, 3562, 1043, 3563, 1044, 3564,
           1045, 3565, 1046, 3566, 1047, 3567, 1048, 3568, 1049, 3569, 1050,
           3570, 1051, 3571, 1052, 3572, 1053, 3573, 1054, 3574, 1055, 3575,
           1056, 3576, 1057, 3577, 1058, 3578, 1059, 3579, 1060, 3580, 1061,
           3581, 1062, 3582, 1063, 3583, 1064, 3584, 1065, 3585, 1066, 3586,
           1067, 3587, 1068, 3588, 1069, 3589, 1070, 3590, 1071, 3591, 1072,
           3592, 1073, 3593, 1074, 3594, 1075, 3595, 1076, 3596, 1077, 3597,
           1078, 3598, 1079, 3599, 1080, 3600, 1081, 3601, 1082, 3602, 1083,
           3603, 1084, 3604, 1085, 3605, 1086, 3606, 1087, 3607, 1088, 3608,
           1089, 3609, 1090, 3610, 1091, 3611, 1092, 3612, 1093, 3613, 1094,
           3614, 1095, 3615, 1096, 3616, 1097, 3617, 1098, 3618, 1099, 3619,
           1100, 3620, 1101, 3621, 1102, 3622, 1103, 3623, 1104, 3624, 1105,
           3625, 1106, 3626, 1107, 3627, 1108, 3628, 1109, 3629, 1110, 3630,
           1111, 3631, 1112, 3632, 1113, 3633, 1114, 3634, 1115, 3635, 1116,
           3636, 1117, 3637, 1118, 3638, 1119, 3639, 1120, 3640, 1121, 3641,
           1122, 3642, 1123, 3643, 1124, 3644, 1125, 3645, 1126, 3646, 1127,
           3647, 1128, 3648, 1129, 3649, 1130, 3650, 1131, 3651, 1132, 3652,
           1133, 3653, 1134, 3654, 1135, 3655, 1136, 3656, 1137, 3657, 1138,
           3658, 1139, 3659, 1140, 3660, 1141, 3661, 1142, 3662, 1143, 3663,
           1144, 3664, 1145, 3665, 1146, 3666, 1147, 3667, 1148, 3668, 1149,
           3669, 1150, 3670, 1151, 3671, 1152, 3672, 1153, 3673, 1154, 3674,
           1155, 3675, 1156, 3676, 1157, 3677, 1158, 3678, 1159, 3679, 1160,
           3680, 1161, 3681, 1162, 3682, 1163, 3683, 1164, 3684, 1165, 3685,
           1166, 3686, 1167, 3687, 1168, 3688, 1169, 3689, 1170, 3690, 1171,
           3691, 1172, 3692, 1173, 3693, 1174, 3694, 1175, 3695, 1176, 3696,
           1177, 3697, 1178, 3698, 1179, 3699, 1180, 3700, 1181, 3701, 1182,
           3702, 1183, 3703, 1184, 3704, 1185, 3705, 1186, 3706, 1187, 3707,
           1188, 3708, 1189, 3709, 1190, 3710, 1191, 3711, 1192, 3712, 1193,
           3713, 1194, 3714, 1195, 3715, 1196, 3716, 1197, 3717, 1198, 3718,
           1199, 3719, 1200, 3720, 1201, 3721, 1202, 3722, 1203, 3723, 1204,
           3724, 1205, 3725, 1206, 3726, 1207, 3727, 1208, 3728, 1209, 3729,
           1210, 3730, 1211, 3731, 1212, 3732, 1213, 3733, 1214, 3734, 1215,
           3735, 1216, 3736, 1217, 3737, 1218, 3738, 1219, 3739, 1220, 3740,
           1221, 3741, 1222, 3742, 1223, 3743, 1224, 3744, 1225, 3745, 1226,
           3746, 1227, 3747, 1228, 3748, 1229, 3749, 1230, 3750, 1231, 3751,
           1232, 3752, 1233, 3753, 1234, 3754, 1235, 3755, 1236, 3756, 1237,
           3757, 1238, 3758, 1239, 3759, 1240, 3760, 1241, 3761, 1242, 3762,
           1243, 3763, 1244, 3764, 1245, 3765, 1246, 3766, 1247, 3767, 1248,
           3768, 1249, 3769, 1250, 3770, 1251, 3771, 1252, 3772, 1253, 3773,
           1254, 3774, 1255, 3775, 1256, 3776, 1257, 3777, 1258, 3778, 1259,
           3779, 1260, 3780, 1261, 3781, 1262, 3782, 1263, 3783, 1264, 3784,
           1265, 3785, 1266, 3786, 1267, 3787, 1268, 3788, 1269, 3789, 1270,
           3790, 1271, 3791, 1272, 3792, 1273, 3793, 1274, 3794, 1275, 3795,
           1276, 3796, 1277, 3797, 1278, 3798, 1279, 3799, 1280, 3800, 1281,
           3801, 1282, 3802, 1283, 3803, 1284, 3804, 1285, 3805, 1286, 3806,
           1287, 3807, 1288, 3808, 1289, 3809, 1290, 3810, 1291, 3811, 1292,
           3812, 1293, 3813, 1294, 3814, 1295, 3815, 1296, 3816, 1297, 3817,
           1298, 3818, 1299, 3819, 1300, 3820, 1301, 3821, 1302, 3822, 1303,
           3823, 1304, 3824, 1305, 3825, 1306, 3826, 1307, 3827, 1308, 3828,
           1309, 3829, 1310, 3830, 1311, 3831, 1312, 3832, 1313, 3833, 1314,
           3834, 1315, 3835, 1316, 3836, 1317, 3837, 1318, 3838, 1319, 3839,
           1320, 3840, 1321, 3841, 1322, 3842, 1323, 3843, 1324, 3844, 1325,
           3845, 1326, 3846, 1327, 3847, 1328, 3848, 1329, 3849, 1330, 3850,
           1331, 3851, 1332, 3852, 1333, 3853, 1334, 3854, 1335, 3855, 1336,
           3856, 1337, 3857, 1338, 3858, 1339, 3859, 1340, 3860, 1341, 3861,
           1342, 3862, 1343, 3863, 1344, 3864, 1345, 3865, 1346, 3866, 1347,
           3867, 1348, 3868, 1349, 3869, 1350, 3870, 1351, 3871, 1352, 3872,
           1353, 3873, 1354, 3874, 1355, 3875, 1356, 3876, 1357, 3877, 1358,
           3878, 1359, 3879, 1360, 3880, 1361, 3881, 1362, 3882, 1363, 3883,
           1364, 3884, 1365, 3885, 1366, 3886, 1367, 3887, 1368, 3888, 1369,
           3889, 1370, 3890, 1371, 3891, 1372, 3892, 1373, 3893, 1374, 3894,
           1375, 3895, 1376, 3896, 1377, 3897, 1378, 3898, 1379, 3899, 1380,
           3900, 1381, 3901, 1382, 3902, 1383, 3903, 1384, 3904, 1385, 3905,
           1386, 3906, 1387, 3907, 1388, 3908, 1389, 3909, 1390, 3910, 1391,
           3911, 1392, 3912, 1393, 3913, 1394, 3914, 1395, 3915, 1396, 3916,
           1397, 3917, 1398, 3918, 1399, 3919, 1400, 3920, 1401, 3921, 1402,
           3922, 1403, 3923, 1404, 3924, 1405, 3925, 1406, 3926, 1407, 3927,
           1408, 3928, 1409, 3929, 1410, 3930, 1411, 3931, 1412, 3932, 1413,
           3933, 1414, 3934, 1415, 3935, 1416, 3936, 1417, 3937, 1418, 3938,
           1419, 3939, 1420, 3940, 1421, 3941, 1422, 3942, 1423, 3943, 1424,
           3944, 1425, 3945, 1426, 3946, 1427, 3947, 1428, 3948, 1429, 3949,
           1430, 3950, 1431, 3951, 1432, 3952, 1433, 3953, 1434, 3954, 1435,
           3955, 1436, 3956, 1437, 3957, 1438, 3958, 1439, 3959, 1440, 3960,
           1441, 3961, 1442, 3962, 1443, 3963, 1444, 3964, 1445, 3965, 1446,
           3966, 1447, 3967, 1448, 3968, 1449, 3969, 1450, 3970, 1451, 3971,
           1452, 3972, 1453, 3973, 1454, 3974, 1455, 3975, 1456, 3976, 1457,
           3977, 1458, 3978, 1459, 3979, 1460, 3980, 1461, 3981, 1462, 3982,
           1463, 3983, 1464, 3984, 1465, 3985, 1466, 3986, 1467, 3987, 1468,
           3988, 1469, 3989, 1470, 3990, 1471, 3991, 1472, 3992, 1473, 3993,
           1474, 3994, 1475, 3995, 1476, 3996, 1477, 3997, 1478, 3998, 1479,
           3999, 1480, 4000, 1481, 4001, 1482, 4002, 1483, 4003, 1484, 4004,
           1485, 4005, 1486, 4006, 1487, 4007, 1488, 4008, 1489, 4009, 1490,
           4010, 1491, 4011, 1492, 4012, 1493, 4013, 1494, 4014, 1495, 4015,
           1496, 4016, 1497, 4017, 1498, 4018, 1499, 4019, 1500, 4020, 1501,
           4021, 1502, 4022, 1503, 4023, 1504, 4024, 1505, 4025, 1506, 4026,
           1507, 4027, 1508, 4028, 1509, 4029, 1510, 4030, 1511, 4031, 1512,
           4032, 1513, 4033, 1514, 4034, 1515, 4035, 1516, 4036, 1517, 4037,
           1518, 4038, 1519, 4039, 1520, 4040, 1521, 4041, 1522, 4042, 1523,
           4043, 1524, 4044, 1525, 4045, 1526, 4046, 1527, 4047, 1528, 4048,
           1529, 4049, 1530, 4050, 1531, 4051, 1532, 4052, 1533, 4053, 1534,
           4054, 1535, 4055, 1536, 4056, 1537, 4057, 1538, 4058, 1539, 4059,
           1540, 4060, 1541, 4061, 1542, 4062, 1543, 4063, 1544, 4064, 1545,
           4065, 1546, 4066, 1547, 4067, 1548, 4068, 1549, 4069, 1550, 4070,
           1551, 4071, 1552, 4072, 1553, 4073, 1554, 4074, 1555, 4075, 1556,
           4076, 1557, 4077, 1558, 4078, 1559, 4079, 1560, 4080, 1561, 4081,
           1562, 4082, 1563, 4083, 1564, 4084, 1565, 4085, 1566, 4086, 1567,
           4087, 1568, 4088, 1569, 4089, 1570, 4090, 1571, 4091, 1572, 4092,
           1573, 4093, 1574, 4094, 1575, 4095, 1576, 4096, 1577, 4097, 1578,
           4098, 1579, 4099, 1580, 4100, 1581, 4101, 1582, 4102, 1583, 4103,
           1584, 4104, 1585, 4105, 1586, 4106, 1587, 4107, 1588, 4108, 1589,
           4109, 1590, 4110, 1591, 4111, 1592, 4112, 1593, 4113, 1594, 4114,
           1595, 4115, 1596, 4116, 1597, 4117, 1598, 4118, 1599, 4119, 1600,
           4120, 1601, 4121, 1602, 4122, 1603, 4123, 1604, 4124, 1605, 4125,
           1606, 4126, 1607, 4127, 1608, 4128, 1609, 4129, 1610, 4130, 1611,
           4131, 1612, 4132, 1613, 4133, 1614, 4134, 1615, 4135, 1616, 4136,
           1617, 4137, 1618, 4138, 1619, 4139, 1620, 4140, 1621, 4141, 1622,
           4142, 1623, 4143, 1624, 4144, 1625, 4145, 1626, 4146, 1627, 4147,
           1628, 4148, 1629, 4149, 1630, 4150, 1631, 4151, 1632, 4152, 1633,
           4153, 1634, 4154, 1635, 4155, 1636, 4156, 1637, 4157, 1638, 4158,
           1639, 4159, 1640, 4160, 1641, 4161, 1642, 4162, 1643, 4163, 1644,
           4164, 1645, 4165, 1646, 4166, 1647, 4167, 1648, 4168, 1649, 4169,
           1650, 4170, 1651, 4171, 1652, 4172, 1653, 4173, 1654, 4174, 1655,
           4175, 1656, 4176, 1657, 4177, 1658, 4178, 1659, 4179, 1660, 4180,
           1661, 4181, 1662, 4182, 1663, 4183, 1664, 4184, 1665, 4185, 1666,
           4186, 1667, 4187, 1668, 4188, 1669, 4189, 1670, 4190, 1671, 4191,
           1672, 4192, 1673, 4193, 1674, 4194, 1675, 4195, 1676, 4196, 1677,
           4197, 1678, 4198, 1679, 4199, 1680, 4200, 1681, 4201, 1682, 4202,
           1683, 4203, 1684, 4204, 1685, 4205, 1686, 4206, 1687, 4207, 1688,
           4208, 1689, 4209, 1690, 4210, 1691, 4211, 1692, 4212, 1693, 4213,
           1694, 4214, 1695, 4215, 1696, 4216, 1697, 4217, 1698, 4218, 1699,
           4219, 1700, 4220, 1701, 4221, 1702, 4222, 1703, 4223, 1704, 4224,
           1705, 4225, 1706, 4226, 1707, 4227, 1708, 4228, 1709, 4229, 1710,
           4230, 1711, 4231, 1712, 4232, 1713, 4233, 1714, 4234, 1715, 4235,
           1716, 4236, 1717, 4237, 1718, 4238, 1719, 4239, 1720, 4240, 1721,
           4241, 1722, 4242, 1723, 4243, 1724, 4244, 1725, 4245, 1726, 4246,
           1727, 4247, 1728, 4248, 1729, 4249, 1730, 4250, 1731, 4251, 1732,
           4252, 1733, 4253, 1734, 4254, 1735, 4255, 1736, 4256, 1737, 4257,
           1738, 4258, 1739, 4259, 1740, 4260, 1741, 4261, 1742, 4262, 1743,
           4263, 1744, 4264, 1745, 4265, 1746, 4266, 1747, 4267, 1748, 4268,
           1749, 4269, 1750, 4270, 1751, 4271, 1752, 4272, 1753, 4273, 1754,
           4274, 1755, 4275, 1756, 4276, 1757, 4277, 1758, 4278, 1759, 4279,
           1760, 4280, 1761, 4281, 1762, 4282, 1763, 4283, 1764, 4284, 1765,
           4285, 1766, 4286, 1767, 4287, 1768, 4288, 1769, 4289, 1770, 4290,
           1771, 4291, 1772, 4292, 1773, 4293, 1774, 4294, 1775, 4295, 1776,
           4296, 1777, 4297, 1778, 4298, 1779, 4299, 1780, 4300, 1781, 4301,
           1782, 4302, 1783, 4303, 1784, 4304, 1785, 4305, 1786, 4306, 1787,
           4307, 1788, 4308, 1789, 4309, 1790, 4310, 1791, 4311, 1792, 4312,
           1793, 4313, 1794, 4314, 1795, 4315, 1796, 4316, 1797, 4317, 1798,
           4318, 1799, 4319, 1800, 4320, 1801, 4321, 1802, 4322, 1803, 4323,
           1804, 4324, 1805, 4325, 1806, 4326, 1807, 4327, 1808, 4328, 1809,
           4329, 1810, 4330, 1811, 4331, 1812, 4332, 1813, 4333, 1814, 4334,
           1815, 4335, 1816, 4336, 1817, 4337, 1818, 4338, 1819, 4339, 1820,
           4340, 1821, 4341, 1822, 4342, 1823, 4343, 1824, 4344, 1825, 4345,
           1826, 4346, 1827, 4347, 1828, 4348, 1829, 4349, 1830, 4350, 1831,
           4351, 1832, 4352, 1833, 4353, 1834, 4354, 1835, 4355, 1836, 4356,
           1837, 4357, 1838, 4358, 1839, 4359, 1840, 4360, 1841, 4361, 1842,
           4362, 1843, 4363, 1844, 4364, 1845, 4365, 1846, 4366, 1847, 4367,
           1848, 4368, 1849, 4369, 1850, 4370, 1851, 4371, 1852, 4372, 1853,
           4373, 1854, 4374, 1855, 4375, 1856, 4376, 1857, 4377, 1858, 4378,
           1859, 4379, 1860, 4380, 1861, 4381, 1862, 4382, 1863, 4383, 1864,
           4384, 1865, 4385, 1866, 4386, 1867, 4387, 1868, 4388, 1869, 4389,
           1870, 4390, 1871, 4391, 1872, 4392, 1873, 4393, 1874, 4394, 1875,
           4395, 1876, 4396, 1877, 4397, 1878, 4398, 1879, 4399, 1880, 4400,
           1881, 4401, 1882, 4402, 1883, 4403, 1884, 4404, 1885, 4405, 1886,
           4406, 1887, 4407, 1888, 4408, 1889, 4409, 1890, 4410, 1891, 4411,
           1892, 4412, 1893, 4413, 1894, 4414, 1895, 4415, 1896, 4416, 1897,
           4417, 1898, 4418, 1899, 4419, 1900, 4420, 1901, 4421, 1902, 4422,
           1903, 4423, 1904, 4424, 1905, 4425, 1906, 4426, 1907, 4427, 1908,
           4428, 1909, 4429, 1910, 4430, 1911, 4431, 1912, 4432, 1913, 4433,
           1914, 4434, 1915, 4435, 1916, 4436, 1917, 4437, 1918, 4438, 1919,
           4439, 1920, 4440, 1921, 4441, 1922, 4442, 1923, 4443, 1924, 4444,
           1925, 4445, 1926, 4446, 1927, 4447, 1928, 4448, 1929, 4449, 1930,
           4450, 1931, 4451, 1932, 4452, 1933, 4453, 1934, 4454, 1935, 4455,
           1936, 4456, 1937, 4457, 1938, 4458, 1939, 4459, 1940, 4460, 1941,
           4461, 1942, 4462, 1943, 4463, 1944, 4464, 1945, 4465, 1946, 4466,
           1947, 4467, 1948, 4468, 1949, 4469, 1950, 4470, 1951, 4471, 1952,
           4472, 1953, 4473, 1954, 4474, 1955, 4475, 1956, 4476, 1957, 4477,
           1958, 4478, 1959, 4479, 1960, 4480, 1961, 4481, 1962, 4482, 1963,
           4483, 1964, 4484, 1965, 4485, 1966, 4486, 1967, 4487, 1968, 4488,
           1969, 4489, 1970, 4490, 1971, 4491, 1972, 4492, 1973, 4493, 1974,
           4494, 1975, 4495, 1976, 4496, 1977, 4497, 1978, 4498, 1979, 4499,
           1980, 4500, 1981, 4501, 1982, 4502, 1983, 4503, 1984, 4504, 1985,
           4505, 1986, 4506, 1987, 4507, 1988, 4508, 1989, 4509, 1990, 4510,
           1991, 4511, 1992, 4512, 1993, 4513, 1994, 4514, 1995, 4515, 1996,
           4516, 1997, 4517, 1998, 4518, 1999, 4519, 2000, 4520, 2001, 4521,
           2002, 4522, 2003, 4523, 2004, 4524, 2005, 4525, 2006, 4526, 2007,
           4527, 2008, 4528, 2009, 4529, 2010, 4530, 2011, 4531, 2012, 4532,
           2013, 4533, 2014, 4534, 2015, 4535, 2016, 4536, 2017, 4537, 2018,
           4538, 2019, 4539, 2020, 4540, 2021, 4541, 2022, 4542, 2023, 4543,
           2024, 4544, 2025, 4545, 2026, 4546, 2027, 4547, 2028, 4548, 2029,
           4549, 2030, 4550, 2031, 4551, 2032, 4552, 2033, 4553, 2034, 4554,
           2035, 4555, 2036, 4556, 2037, 4557, 2038, 4558, 2039, 4559, 2040,
           4560, 2041, 4561, 2042, 4562, 2043, 4563, 2044, 4564, 2045, 4565,
           2046, 4566, 2047, 4567, 2048, 4568, 2049, 4569, 2050, 4570, 2051,
           4571, 2052, 4572, 2053, 4573, 2054, 4574, 2055, 4575, 2056, 4576,
           2057, 4577, 2058, 4578, 2059, 4579, 2060, 4580, 2061, 4581, 2062,
           4582, 2063, 4583, 2064, 4584, 2065, 4585, 2066, 4586, 2067, 4587,
           2068, 4588, 2069, 4589, 2070, 4590, 2071, 4591, 2072, 4592, 2073,
           4593, 2074, 4594, 2075, 4595, 2076, 4596, 2077, 4597, 2078, 4598,
           2079, 4599, 2080, 4600, 2081, 4601, 2082, 4602, 2083, 4603, 2084,
           4604, 2085, 4605, 2086, 4606, 2087, 4607, 2088, 4608, 2089, 4609,
           2090, 4610, 2091, 4611, 2092, 4612, 2093, 4613, 2094, 4614, 2095,
           4615, 2096, 4616, 2097, 4617, 2098, 4618, 2099, 4619, 2100, 4620,
           2101, 4621, 2102, 4622, 2103, 4623, 2104, 4624, 2105, 4625, 2106,
           4626, 2107, 4627, 2108, 4628, 2109, 4629, 2110, 4630, 2111, 4631,
           2112, 4632, 2113, 4633, 2114, 4634, 2115, 4635, 2116, 4636, 2117,
           4637, 2118, 4638, 2119, 4639, 2120, 4640, 2121, 4641, 2122, 4642,
           2123, 4643, 2124, 4644, 2125, 4645, 2126, 4646, 2127, 4647, 2128,
           4648, 2129, 4649, 2130, 4650, 2131, 4651, 2132, 4652, 2133, 4653,
           2134, 4654, 2135, 4655, 2136, 4656, 2137, 4657, 2138, 4658, 2139,
           4659, 2140, 4660, 2141, 4661, 2142, 4662, 2143, 4663, 2144, 4664,
           2145, 4665, 2146, 4666, 2147, 4667, 2148, 4668, 2149, 4669, 2150,
           4670, 2151, 4671, 2152, 4672, 2153, 4673, 2154, 4674, 2155, 4675,
           2156, 4676, 2157, 4677, 2158, 4678, 2159, 4679, 2160, 4680, 2161,
           4681, 2162, 4682, 2163, 4683, 2164, 4684, 2165, 4685, 2166, 4686,
           2167, 4687, 2168, 4688, 2169, 4689, 2170, 4690, 2171, 4691, 2172,
           4692, 2173, 4693, 2174, 4694, 2175, 4695, 2176, 4696, 2177, 4697,
           2178, 4698, 2179, 4699, 2180, 4700, 2181, 4701, 2182, 4702, 2183,
           4703, 2184, 4704, 2185, 4705, 2186, 4706, 2187, 4707, 2188, 4708,
           2189, 4709, 2190, 4710, 2191, 4711, 2192, 4712, 2193, 4713, 2194,
           4714, 2195, 4715, 2196, 4716, 2197, 4717, 2198, 4718, 2199, 4719,
           2200, 4720, 2201, 4721, 2202, 4722, 2203, 4723, 2204, 4724, 2205,
           4725, 2206, 4726, 2207, 4727, 2208, 4728, 2209, 4729, 2210, 4730,
           2211, 4731, 2212, 4732, 2213, 4733, 2214, 4734, 2215, 4735, 2216,
           4736, 2217, 4737, 2218, 4738, 2219, 4739, 2220, 4740, 2221, 4741,
           2222, 4742, 2223, 4743, 2224, 4744, 2225, 4745, 2226, 4746, 2227,
           4747, 2228, 4748, 2229, 4749, 2230, 4750, 2231, 4751, 2232, 4752,
           2233, 4753, 2234, 4754, 2235, 4755, 2236, 4756, 2237, 4757, 2238,
           4758, 2239, 4759, 2240, 4760, 2241, 4761, 2242, 4762, 2243, 4763,
           2244, 4764, 2245, 4765, 2246, 4766, 2247, 4767, 2248, 4768, 2249,
           4769, 2250, 4770, 2251, 4771, 2252, 4772, 2253, 4773, 2254, 4774,
           2255, 4775, 2256, 4776, 2257, 4777, 2258, 4778, 2259, 4779, 2260,
           4780, 2261, 4781, 2262, 4782, 2263, 4783, 2264, 4784, 2265, 4785,
           2266, 4786, 2267, 4787, 2268, 4788, 2269, 4789, 2270, 4790, 2271,
           4791, 2272, 4792, 2273, 4793, 2274, 4794, 2275, 4795, 2276, 4796,
           2277, 4797, 2278, 4798, 2279, 4799, 2280, 4800, 2281, 4801, 2282,
           4802, 2283, 4803, 2284, 4804, 2285, 4805, 2286, 4806, 2287, 4807,
           2288, 4808, 2289, 4809, 2290, 4810, 2291, 4811, 2292, 4812, 2293,
           4813, 2294, 4814, 2295, 4815, 2296, 4816, 2297, 4817, 2298, 4818,
           2299, 4819, 2300, 4820, 2301, 4821, 2302, 4822, 2303, 4823, 2304,
           4824, 2305, 4825, 2306, 4826, 2307, 4827, 2308, 4828, 2309, 4829,
           2310, 4830, 2311, 4831, 2312, 4832, 2313, 4833, 2314, 4834, 2315,
           4835, 2316, 4836, 2317, 4837, 2318, 4838, 2319, 4839, 2320, 4840,
           2321, 4841, 2322, 4842, 2323, 4843, 2324, 4844, 2325, 4845, 2326,
           4846, 2327, 4847, 2328, 4848, 2329, 4849, 2330, 4850, 2331, 4851,
           2332, 4852, 2333, 4853, 2334, 4854, 2335, 4855, 2336, 4856, 2337,
           4857, 2338, 4858, 2339, 4859, 2340, 4860, 2341, 4861, 2342, 4862,
           2343, 4863, 2344, 4864, 2345, 4865, 2346, 4866, 2347, 4867, 2348,
           4868, 2349, 4869, 2350, 4870, 2351, 4871, 2352, 4872, 2353, 4873,
           2354, 4874, 2355, 4875, 2356, 4876, 2357, 4877, 2358, 4878, 2359,
           4879, 2360, 4880, 2361, 4881, 2362, 4882, 2363, 4883, 2364, 4884,
           2365, 4885, 2366, 4886, 2367, 4887, 2368, 4888, 2369, 4889, 2370,
           4890, 2371, 4891, 2372, 4892, 2373, 4893, 2374, 4894, 2375, 4895,
           2376, 4896, 2377, 4897, 2378, 4898, 2379, 4899, 2380, 4900, 2381,
           4901, 2382, 4902, 2383, 4903, 2384, 4904, 2385, 4905, 2386, 4906,
           2387, 4907, 2388, 4908, 2389, 4909, 2390, 4910, 2391, 4911, 2392,
           4912, 2393, 4913, 2394, 4914, 2395, 4915, 2396, 4916, 2397, 4917,
           2398, 4918, 2399, 4919, 2400, 4920, 2401, 4921, 2402, 4922, 2403,
           4923, 2404, 4924, 2405, 4925, 2406, 4926, 2407, 4927, 2408, 4928,
           2409, 4929, 2410, 4930, 2411, 4931, 2412, 4932, 2413, 4933, 2414,
           4934, 2415, 4935, 2416, 4936, 2417, 4937, 2418, 4938, 2419, 4939,
           2420, 4940, 2421, 4941, 2422, 4942, 2423, 4943, 2424, 4944, 2425,
           4945, 2426, 4946, 2427, 4947, 2428, 4948, 2429, 4949, 2430, 4950,
           2431, 4951, 2432, 4952, 2433, 4953, 2434, 4954, 2435, 4955, 2436,
           4956, 2437, 4957, 2438, 4958, 2439, 4959, 2440, 4960, 2441, 4961,
           2442, 4962, 2443, 4963, 2444, 4964, 2445, 4965, 2446, 4966, 2447,
           4967, 2448, 4968, 2449, 4969, 2450, 4970, 2451, 4971, 2452, 4972,
           2453, 4973, 2454, 4974, 2455, 4975, 2456, 4976, 2457, 4977, 2458,
           4978, 2459, 4979, 2460, 4980, 2461, 4981, 2462, 4982, 2463, 4983,
           2464, 4984, 2465, 4985, 2466, 4986, 2467, 4987, 2468, 4988, 2469,
           4989, 2470, 4990, 2471, 4991, 2472, 4992, 2473, 4993, 2474, 4994,
           2475, 4995, 2476, 4996, 2477, 4997, 2478, 4998, 2479, 4999, 2480,
           5000, 2481, 5001, 2482, 5002, 2483, 5003, 2484, 5004, 2485, 5005,
           2486, 5006, 2487, 5007, 2488, 5008, 2489, 5009, 2490, 5010, 2491,
           5011, 2492, 5012, 2493, 5013, 2494, 5014, 2495, 5015, 2496, 5016,
           2497, 5017, 2498, 5018, 2499, 5019, 2500, 5020, 2501, 5021, 2502,
           5022, 2503, 5023, 2504, 5024, 2505, 5025, 2506, 5026, 2507, 5027,
           2508, 5028, 2509, 5029, 2510, 5030, 2511, 5031, 2512, 5032, 2513,
           5033, 2514, 5034, 2515, 5035, 2516, 5036, 2517, 5037, 2518, 5038,
           2519, 5039}));
}

}  // namespace
}  // namespace tflite
