#!/usr/bin/env python
# vim: expandtab sw=4 ts=4 sts=4:
#
# Copyright © 2003 - 2018 Michal Čihař <michal@cihar.com>
#
# This file is part of python-gammu <https://wammu.eu/python-gammu/>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#
"""Service numbers dialogue example."""

import sys

import gammu


class ServiceHandler:
    received_reply = False

    def __init__(self) -> None:
        self.state_machine = self.get_state_machine()

    def callback(self, _state_machine, callback_type, data) -> None:
        """Callback on USSD data."""
        if callback_type != "USSD":
            print(f"Unexpected event type: {callback_type}")
            sys.exit(1)

        self.received_reply = True

        print("Network reply:")
        print(f"Status: {data['Status']}")
        print(data["Text"])

        if data["Status"] == "ActionNeeded":
            self.do_service()

    def get_state_machine(self):
        """Initializes gammu and callbacks."""
        state_machine = gammu.StateMachine()
        if len(sys.argv) >= 2:
            state_machine.ReadConfig(Filename=sys.argv[1])
        else:
            state_machine.ReadConfig()
        state_machine.Init()
        state_machine.SetIncomingCallback(self.callback)
        try:
            state_machine.SetIncomingUSSD()
        except gammu.ERR_NOTSUPPORTED:
            print("Incoming USSD notification is not supported.")
            sys.exit(1)
        return state_machine

    def do_service(self) -> None:
        """Main code to talk with worker."""
        if len(sys.argv) >= 3:
            code = sys.argv[2]
            del sys.argv[2]
        else:
            prompt = "Enter code (empty string to end): "
            try:
                code = raw_input(prompt)
            except NameError:
                code = input(prompt)
        if code:
            print("Talking to network...")
            self.received_reply = False
            self.state_machine.DialService(code)
            loops = 0
            while not self.received_reply and loops < 10:
                self.state_machine.ReadDevice()
                loops += 1


def main() -> None:
    print("This example shows interaction with network using service codes")
    service = ServiceHandler()
    service.do_service()


if __name__ == "__main__":
    main()
