from __future__ import annotations

import sys
from textwrap import dedent
from typing import TYPE_CHECKING

import pytest

from pipdeptree._render.graphviz import dump_graphviz, print_graphviz

if TYPE_CHECKING:
    from pathlib import Path

    from pytest_mock import MockerFixture

    from pipdeptree._models import PackageDAG


def test_render_dot(
    capsys: pytest.CaptureFixture[str],
    example_dag: PackageDAG,
    randomized_example_dag: PackageDAG,
) -> None:
    # Check both the sorted and randomized package tree produces the same sorted graphviz output.
    for package_tree in (example_dag, randomized_example_dag):
        output = dump_graphviz(package_tree, output_format="dot")
        print_graphviz(output)
        out, _ = capsys.readouterr()
        assert out == dedent(
            """\
            digraph {
            \ta -> b [label=">=2.0.0"]
            \ta -> c [label=">=5.7.1"]
            \ta [label="a\\n3.4.0"]
            \tb -> d [label=">=2.30,<2.42"]
            \tb [label="b\\n2.3.1"]
            \tc -> d [label=">=2.30"]
            \tc -> e [label=">=0.12.1"]
            \tc [label="c\\n5.10.0"]
            \td -> e [label=">=0.9.0"]
            \td [label="d\\n2.35"]
            \te [label="e\\n0.12.1"]
            \tf -> b [label=">=2.1.0"]
            \tf [label="f\\n3.1"]
            \tg -> e [label=">=0.9.0"]
            \tg -> f [label=">=3.0.0"]
            \tg [label="g\\n6.8.3rc1"]
            }

            """,
        )


def test_render_pdf(tmp_path: Path, mocker: MockerFixture, example_dag: PackageDAG) -> None:
    output = dump_graphviz(example_dag, output_format="pdf")
    res = tmp_path / "file"
    with pytest.raises(OSError, match="Bad file"):  # noqa: PT012, SIM117 # because we reopen the file
        with res.open("wb") as buf:
            mocker.patch.object(sys, "stdout", buf)
            print_graphviz(output)
    assert res.read_bytes()[:4] == b"%PDF"


def test_render_svg(capsys: pytest.CaptureFixture[str], example_dag: PackageDAG) -> None:
    output = dump_graphviz(example_dag, output_format="svg")
    print_graphviz(output)
    out, _ = capsys.readouterr()
    assert out.startswith("<?xml")
    assert "<svg" in out
    assert out.strip().endswith("</svg>")


def test_render_dot_with_depth(example_dag: PackageDAG) -> None:
    output = dump_graphviz(example_dag, output_format="dot", max_depth=1)
    assert isinstance(output, str)
    # Roots are a and g (depth 0), their direct deps are at depth 1
    # Deeper deps (d) should not appear
    assert "a ->" in output
    assert "g ->" in output
    assert 'd [label="d' not in output  # d is only reachable at depth 2+
    # Direct deps of roots should appear
    assert 'b [label="b' in output
    assert 'c [label="c' in output
    assert 'e [label="e' in output
    assert 'f [label="f' in output


def test_render_dot_reverse_with_depth(example_dag: PackageDAG) -> None:
    reversed_dag = example_dag.reverse()
    output = dump_graphviz(reversed_dag, output_format="dot", is_reverse=True, max_depth=1)
    assert isinstance(output, str)
    # In reverse, root is e (only leaf in forward tree that's not a parent in reverse)
    # At depth 0: e, at depth 1: c, d, g (e's parents)
    assert 'e [label="e' in output
    assert 'c [label="c' in output
    assert 'd [label="d' in output
    assert 'g [label="g' in output
    # Deeper nodes should not appear
    assert 'a [label="a' not in output
    assert 'b [label="b' not in output
    assert 'f [label="f' not in output
    # Edges from e to its parents should exist
    assert "e -> c" in output
    assert "e -> d" in output
    assert "e -> g" in output


def test_render_dot_reverse_infinite_depth(example_dag: PackageDAG) -> None:
    reversed_dag = example_dag.reverse()
    output = dump_graphviz(reversed_dag, output_format="dot", is_reverse=True)
    assert isinstance(output, str)
    # All nodes should appear with no depth limit
    for node in ("a", "b", "c", "d", "e", "f", "g"):
        assert f'{node} [label="' in output
    # Edges should include reverse relationships
    assert "e -> c" in output
    assert "e -> d" in output
    assert "b -> a" in output
    assert "f -> g" in output


def test_render_dot_with_depth_zero(example_dag: PackageDAG) -> None:
    output = dump_graphviz(example_dag, output_format="dot", max_depth=0)
    assert isinstance(output, str)
    # Depth 0 means only root nodes, no edges
    assert "a [label=" in output
    assert "g [label=" in output
    assert "->" not in output
