/*
 * Copyright (c) 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package jdk.internal.lang;

import java.util.Arrays;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import static java.util.Map.entry;

/**
 * Utility class that handles Unicode case folding properties defined in
 * CasingFolding.txt, including 1:M full case folding.
 */
public final class CaseFolding {

    private CaseFolding()  {}

   /**
    * Tests whether the specified code point has a folding mapping entry defined.
    *
    * @param  cp
    *         the Unicode code point to test
    * @return {@code true} if the given code point has a case folding mapping entry
    *         defined in (@code caseFoldingMap}, {@code false} otherwise
    */
    public static boolean isDefined(int cp) {
         return getDefined(cp) != -1;
    }

   /**
    * Returns the case-folded form of the specified code point according
    * to the Unicode case folding mappings.
    * <p>
    * If the code point has no case folding mapping defined, this method returns
    * the original code point.
    *
    * Possible combinations of the returning case-folding form as a long value
    *
    *  +---+---------+--------+---------+--------+--------+
    *  | 1:1 mapping |  0000  |   0000  |  000x  |  xxxx  |  0041 => 0061 or 1E921 => 1E943
    *  +---+---------+--------+---------+--------+--------+
    *  | 1:2 mapping |  0002  |   0000  |  xxxx  |  xxxx  |  FB02 => 0066 006C
    *  +---+---------+--------+---------+--------+--------+
    *  | 1:3 mapping |  0003  |   xxxx  |  xxxx  |  xxxx  |  FB03 => 0066 0066 0069
    *  +---+---------+--------+---------+--------+--------+
    *
    * @param  cp
    *         the Unicode code point to fold
    * @return a long value representing the case-folded form of the input
    *         code point, encoded as TBD
    */
    public static long fold(int cp) {
        var fold = getDefined(cp);
        return fold == -1 ? cp : fold;
    }

    public static boolean isSingleCodePoint(long fold) {
        return (fold >> 48) == 0;
    }

   /**
    * Returns an expansion set to "close" a given regex Unicode character class range for case-sensitive
    * matching, according to the
    * <a href="https://www.unicode.org/reports/tr18/#Simple_Loose_Matches">Simple Loose Matches</a>
    * rule defined in Unicode Technical Standard #18: Unicode Regular Expressions.
    * <p>
    * To conform with Level 1 of UTS #18, specifically RL1.5: Simple Loose Matches, simple case folding must
    * be applied to literals and (optionally) to character classes. When applied to character classes, each
    * character class is expected to be closed under simple case folding. See the standard for the
    * detailed explanation and example of "closed".
    * <p>
    * RL1.5 states: To meet this requirement, an implementation that supports case-sensitive matching should
    * <ol>
    * <li>Provide at least the simple, default Unicode case-insensitive matching, and</li>
    * <li>Specify which character properties or constructs are closed under the matching.</li>
    * </ol>
    * <p>
    * In the {@code  Pattern} implementation, 5 types of constructs maybe case-sensitive when matching:
    * back-refs, string slice (sequences), single, family(char-property) and class range. Single and
    * family may appears independently or within a class.
    * <p>
    * For loose/case-insensitive matching, the back-refs, slices and singles apply {@code toUpperCase} and
    * {@code toLowerCase} to both the pattern and the input string. This effectively 'close' the class for
    * matching.
    * <p>
    * The family/char-properties are not "closed" and should remain unchanged. This is acceptable per RL1.5,
    * if their behavior is clearly specified.
    * <p>
    * This method addresses that requirement for the "range" construct within in character class by computing
    * the additional characters that should be included to close the range under simple case folding:
    * <p>
    * For each character in the input range {@code [start, end]} (inclusive), if the character has a simple
    * case folding mapping in Unicode's CaseFolding.txt, the mapping is not a round-trip map, and the mapped
    * character is not already in the range, then that mapped character (typically lowercase) is added to
    * the expansion set.
    * <p>
    * This allows regex character class "range" implementation to use the returned expansion set to support
    * additional case-insensitive matching, without duplicating characters already covered by the existing
    * regex range implementation. The expectation is the matching is done using both the uppercase and
    * lowercase forms of the input character, for example
    *
    * <pre>{@code
    *
    *     ch -> inRange(lower, Character.toUpperCase(ch), upper) ||
    *           inRange(lower, Character.toLower(ch), upper) ||
    *           additionalClosingCharacters.contains(Character.toUpperCase(ch)) ||
    *           additionalClosingCharacters.contains(Character.toUpperCase(ch))
    * }</pre>
    *
    * @param start the starting code point of the character range
    * @param end the ending code point of the character range
    * @return a {@code int[]} containing the all simple case equivalents of characters in the range, excluding
    *         those already in the range
    * @spec https://www.unicode.org/reports/tr18/#Simple_Loose_Matches
    */
    public static int[] getClassRangeClosingCharacters(int start, int end) {
        int[] expanded = new int[expanded_case_cps.length];
        int off = 0;
        for (int cp : expanded_case_cps) {
            if (cp >= start && cp <= end) {
                int folding = expanded_case_map.get(cp);
                if (folding < start || folding > end) {
                    expanded[off++] = folding;
                }
            }
        }
        return Arrays.copyOf(expanded, off);
    }

    private static final Map<Integer, Integer> expanded_case_map = Map.ofEntries(
        entry(0x0131, 0x0049),
        entry(0x00B5, 0x03BC),
        entry(0x0130, 0x0069),
        entry(0x017F, 0x0073),
        entry(0x01C5, 0x01C6),
        entry(0x01C8, 0x01C9),
        entry(0x01CB, 0x01CC),
        entry(0x01F2, 0x01F3),
        entry(0x0345, 0x03B9),
        entry(0x03C2, 0x03C3),
        entry(0x03D0, 0x03B2),
        entry(0x03D1, 0x03B8),
        entry(0x03D5, 0x03C6),
        entry(0x03D6, 0x03C0),
        entry(0x03F0, 0x03BA),
        entry(0x03F1, 0x03C1),
        entry(0x03F4, 0x03B8),
        entry(0x03F5, 0x03B5),
        entry(0x1C80, 0x0432),
        entry(0x1C81, 0x0434),
        entry(0x1C82, 0x043E),
        entry(0x1C83, 0x0441),
        entry(0x1C84, 0x0442),
        entry(0x1C85, 0x0442),
        entry(0x1C86, 0x044A),
        entry(0x1C87, 0x0463),
        entry(0x1C88, 0xA64B),
        entry(0x1E9B, 0x1E61),
        entry(0x1E9E, 0x00DF),
        entry(0x1FBE, 0x03B9),
        entry(0x1FD3, 0x0390),
        entry(0x1FE3, 0x03B0),
        entry(0x2126, 0x03C9),
        entry(0x212A, 0x006B),
        entry(0x212B, 0x00E5),
        entry(0xFB05, 0xFB06)
    );

    private static final int[] expanded_case_cps = expanded_case_map.keySet()
      .stream()
      .mapToInt(Integer::intValue)
      .toArray();

    private static final int HASH_CP = 0;
    private static final int HASH_INDEX = 1;
    private static final int HASH_NEXT = 2;

    private static int[][] hashKeys(int[] keys) {
        var hashes = new int[keys.length << 1][3];  // cp + hash + next
        var off = keys.length;
        for (int i = 0; i < keys.length; i++) {
            var cp = keys[i];
            var hash = cp % keys.length;
            while (hashes[hash][HASH_CP] != 0) {
                var next = hashes[hash][HASH_NEXT];
                if (next == 0) {
                    hashes[hash][HASH_NEXT] = off;
                    hash = off++;
                    break;
                } else {
                    hash = next;
                }
            }
            hashes[hash][HASH_CP] = cp;
            hashes[hash][HASH_INDEX] = i;
        }
        return Arrays.copyOf(hashes, off);
    }

    private static long getDefined(int cp) {
        var hashes = CASE_FOLDING_HASHES;
        var length = CASE_FOLDING_CPS.length;  // hashed based on total defined.
        var hash = cp % length;
        while (hashes[hash][HASH_CP] != cp) {
            var next = hashes[hash][HASH_NEXT];
            if (next == 0) {
                return -1;   // hash miss
            }
            hash = next;
        }
        var index = hashes[hash][HASH_INDEX];
        return CASE_FOLDING_VALUES[index];
    }

    private static final int[] CASE_FOLDING_CPS = {
        0X0041, 0X0042, 0X0043, 0X0044, 0X0045, 0X0046, 0X0047, 0X0048, 0X0049, 0X004A, 
        0X004B, 0X004C, 0X004D, 0X004E, 0X004F, 0X0050, 0X0051, 0X0052, 0X0053, 0X0054, 
        0X0055, 0X0056, 0X0057, 0X0058, 0X0059, 0X005A, 0X00B5, 0X00C0, 0X00C1, 0X00C2, 
        0X00C3, 0X00C4, 0X00C5, 0X00C6, 0X00C7, 0X00C8, 0X00C9, 0X00CA, 0X00CB, 0X00CC, 
        0X00CD, 0X00CE, 0X00CF, 0X00D0, 0X00D1, 0X00D2, 0X00D3, 0X00D4, 0X00D5, 0X00D6, 
        0X00D8, 0X00D9, 0X00DA, 0X00DB, 0X00DC, 0X00DD, 0X00DE, 0X00DF, 0X0100, 0X0102, 
        0X0104, 0X0106, 0X0108, 0X010A, 0X010C, 0X010E, 0X0110, 0X0112, 0X0114, 0X0116, 
        0X0118, 0X011A, 0X011C, 0X011E, 0X0120, 0X0122, 0X0124, 0X0126, 0X0128, 0X012A, 
        0X012C, 0X012E, 0X0130, 0X0132, 0X0134, 0X0136, 0X0139, 0X013B, 0X013D, 0X013F, 
        0X0141, 0X0143, 0X0145, 0X0147, 0X0149, 0X014A, 0X014C, 0X014E, 0X0150, 0X0152, 
        0X0154, 0X0156, 0X0158, 0X015A, 0X015C, 0X015E, 0X0160, 0X0162, 0X0164, 0X0166, 
        0X0168, 0X016A, 0X016C, 0X016E, 0X0170, 0X0172, 0X0174, 0X0176, 0X0178, 0X0179, 
        0X017B, 0X017D, 0X017F, 0X0181, 0X0182, 0X0184, 0X0186, 0X0187, 0X0189, 0X018A, 
        0X018B, 0X018E, 0X018F, 0X0190, 0X0191, 0X0193, 0X0194, 0X0196, 0X0197, 0X0198, 
        0X019C, 0X019D, 0X019F, 0X01A0, 0X01A2, 0X01A4, 0X01A6, 0X01A7, 0X01A9, 0X01AC, 
        0X01AE, 0X01AF, 0X01B1, 0X01B2, 0X01B3, 0X01B5, 0X01B7, 0X01B8, 0X01BC, 0X01C4, 
        0X01C5, 0X01C7, 0X01C8, 0X01CA, 0X01CB, 0X01CD, 0X01CF, 0X01D1, 0X01D3, 0X01D5, 
        0X01D7, 0X01D9, 0X01DB, 0X01DE, 0X01E0, 0X01E2, 0X01E4, 0X01E6, 0X01E8, 0X01EA, 
        0X01EC, 0X01EE, 0X01F0, 0X01F1, 0X01F2, 0X01F4, 0X01F6, 0X01F7, 0X01F8, 0X01FA, 
        0X01FC, 0X01FE, 0X0200, 0X0202, 0X0204, 0X0206, 0X0208, 0X020A, 0X020C, 0X020E, 
        0X0210, 0X0212, 0X0214, 0X0216, 0X0218, 0X021A, 0X021C, 0X021E, 0X0220, 0X0222, 
        0X0224, 0X0226, 0X0228, 0X022A, 0X022C, 0X022E, 0X0230, 0X0232, 0X023A, 0X023B, 
        0X023D, 0X023E, 0X0241, 0X0243, 0X0244, 0X0245, 0X0246, 0X0248, 0X024A, 0X024C, 
        0X024E, 0X0345, 0X0370, 0X0372, 0X0376, 0X037F, 0X0386, 0X0388, 0X0389, 0X038A, 
        0X038C, 0X038E, 0X038F, 0X0390, 0X0391, 0X0392, 0X0393, 0X0394, 0X0395, 0X0396, 
        0X0397, 0X0398, 0X0399, 0X039A, 0X039B, 0X039C, 0X039D, 0X039E, 0X039F, 0X03A0, 
        0X03A1, 0X03A3, 0X03A4, 0X03A5, 0X03A6, 0X03A7, 0X03A8, 0X03A9, 0X03AA, 0X03AB, 
        0X03B0, 0X03C2, 0X03CF, 0X03D0, 0X03D1, 0X03D5, 0X03D6, 0X03D8, 0X03DA, 0X03DC, 
        0X03DE, 0X03E0, 0X03E2, 0X03E4, 0X03E6, 0X03E8, 0X03EA, 0X03EC, 0X03EE, 0X03F0, 
        0X03F1, 0X03F4, 0X03F5, 0X03F7, 0X03F9, 0X03FA, 0X03FD, 0X03FE, 0X03FF, 0X0400, 
        0X0401, 0X0402, 0X0403, 0X0404, 0X0405, 0X0406, 0X0407, 0X0408, 0X0409, 0X040A, 
        0X040B, 0X040C, 0X040D, 0X040E, 0X040F, 0X0410, 0X0411, 0X0412, 0X0413, 0X0414, 
        0X0415, 0X0416, 0X0417, 0X0418, 0X0419, 0X041A, 0X041B, 0X041C, 0X041D, 0X041E, 
        0X041F, 0X0420, 0X0421, 0X0422, 0X0423, 0X0424, 0X0425, 0X0426, 0X0427, 0X0428, 
        0X0429, 0X042A, 0X042B, 0X042C, 0X042D, 0X042E, 0X042F, 0X0460, 0X0462, 0X0464, 
        0X0466, 0X0468, 0X046A, 0X046C, 0X046E, 0X0470, 0X0472, 0X0474, 0X0476, 0X0478, 
        0X047A, 0X047C, 0X047E, 0X0480, 0X048A, 0X048C, 0X048E, 0X0490, 0X0492, 0X0494, 
        0X0496, 0X0498, 0X049A, 0X049C, 0X049E, 0X04A0, 0X04A2, 0X04A4, 0X04A6, 0X04A8, 
        0X04AA, 0X04AC, 0X04AE, 0X04B0, 0X04B2, 0X04B4, 0X04B6, 0X04B8, 0X04BA, 0X04BC, 
        0X04BE, 0X04C0, 0X04C1, 0X04C3, 0X04C5, 0X04C7, 0X04C9, 0X04CB, 0X04CD, 0X04D0, 
        0X04D2, 0X04D4, 0X04D6, 0X04D8, 0X04DA, 0X04DC, 0X04DE, 0X04E0, 0X04E2, 0X04E4, 
        0X04E6, 0X04E8, 0X04EA, 0X04EC, 0X04EE, 0X04F0, 0X04F2, 0X04F4, 0X04F6, 0X04F8, 
        0X04FA, 0X04FC, 0X04FE, 0X0500, 0X0502, 0X0504, 0X0506, 0X0508, 0X050A, 0X050C, 
        0X050E, 0X0510, 0X0512, 0X0514, 0X0516, 0X0518, 0X051A, 0X051C, 0X051E, 0X0520, 
        0X0522, 0X0524, 0X0526, 0X0528, 0X052A, 0X052C, 0X052E, 0X0531, 0X0532, 0X0533, 
        0X0534, 0X0535, 0X0536, 0X0537, 0X0538, 0X0539, 0X053A, 0X053B, 0X053C, 0X053D, 
        0X053E, 0X053F, 0X0540, 0X0541, 0X0542, 0X0543, 0X0544, 0X0545, 0X0546, 0X0547, 
        0X0548, 0X0549, 0X054A, 0X054B, 0X054C, 0X054D, 0X054E, 0X054F, 0X0550, 0X0551, 
        0X0552, 0X0553, 0X0554, 0X0555, 0X0556, 0X0587, 0X10A0, 0X10A1, 0X10A2, 0X10A3, 
        0X10A4, 0X10A5, 0X10A6, 0X10A7, 0X10A8, 0X10A9, 0X10AA, 0X10AB, 0X10AC, 0X10AD, 
        0X10AE, 0X10AF, 0X10B0, 0X10B1, 0X10B2, 0X10B3, 0X10B4, 0X10B5, 0X10B6, 0X10B7, 
        0X10B8, 0X10B9, 0X10BA, 0X10BB, 0X10BC, 0X10BD, 0X10BE, 0X10BF, 0X10C0, 0X10C1, 
        0X10C2, 0X10C3, 0X10C4, 0X10C5, 0X10C7, 0X10CD, 0X13F8, 0X13F9, 0X13FA, 0X13FB, 
        0X13FC, 0X13FD, 0X1C80, 0X1C81, 0X1C82, 0X1C83, 0X1C84, 0X1C85, 0X1C86, 0X1C87, 
        0X1C88, 0X1C89, 0X1C90, 0X1C91, 0X1C92, 0X1C93, 0X1C94, 0X1C95, 0X1C96, 0X1C97, 
        0X1C98, 0X1C99, 0X1C9A, 0X1C9B, 0X1C9C, 0X1C9D, 0X1C9E, 0X1C9F, 0X1CA0, 0X1CA1, 
        0X1CA2, 0X1CA3, 0X1CA4, 0X1CA5, 0X1CA6, 0X1CA7, 0X1CA8, 0X1CA9, 0X1CAA, 0X1CAB, 
        0X1CAC, 0X1CAD, 0X1CAE, 0X1CAF, 0X1CB0, 0X1CB1, 0X1CB2, 0X1CB3, 0X1CB4, 0X1CB5, 
        0X1CB6, 0X1CB7, 0X1CB8, 0X1CB9, 0X1CBA, 0X1CBD, 0X1CBE, 0X1CBF, 0X1E00, 0X1E02, 
        0X1E04, 0X1E06, 0X1E08, 0X1E0A, 0X1E0C, 0X1E0E, 0X1E10, 0X1E12, 0X1E14, 0X1E16, 
        0X1E18, 0X1E1A, 0X1E1C, 0X1E1E, 0X1E20, 0X1E22, 0X1E24, 0X1E26, 0X1E28, 0X1E2A, 
        0X1E2C, 0X1E2E, 0X1E30, 0X1E32, 0X1E34, 0X1E36, 0X1E38, 0X1E3A, 0X1E3C, 0X1E3E, 
        0X1E40, 0X1E42, 0X1E44, 0X1E46, 0X1E48, 0X1E4A, 0X1E4C, 0X1E4E, 0X1E50, 0X1E52, 
        0X1E54, 0X1E56, 0X1E58, 0X1E5A, 0X1E5C, 0X1E5E, 0X1E60, 0X1E62, 0X1E64, 0X1E66, 
        0X1E68, 0X1E6A, 0X1E6C, 0X1E6E, 0X1E70, 0X1E72, 0X1E74, 0X1E76, 0X1E78, 0X1E7A, 
        0X1E7C, 0X1E7E, 0X1E80, 0X1E82, 0X1E84, 0X1E86, 0X1E88, 0X1E8A, 0X1E8C, 0X1E8E, 
        0X1E90, 0X1E92, 0X1E94, 0X1E96, 0X1E97, 0X1E98, 0X1E99, 0X1E9A, 0X1E9B, 0X1E9E, 
        0X1EA0, 0X1EA2, 0X1EA4, 0X1EA6, 0X1EA8, 0X1EAA, 0X1EAC, 0X1EAE, 0X1EB0, 0X1EB2, 
        0X1EB4, 0X1EB6, 0X1EB8, 0X1EBA, 0X1EBC, 0X1EBE, 0X1EC0, 0X1EC2, 0X1EC4, 0X1EC6, 
        0X1EC8, 0X1ECA, 0X1ECC, 0X1ECE, 0X1ED0, 0X1ED2, 0X1ED4, 0X1ED6, 0X1ED8, 0X1EDA, 
        0X1EDC, 0X1EDE, 0X1EE0, 0X1EE2, 0X1EE4, 0X1EE6, 0X1EE8, 0X1EEA, 0X1EEC, 0X1EEE, 
        0X1EF0, 0X1EF2, 0X1EF4, 0X1EF6, 0X1EF8, 0X1EFA, 0X1EFC, 0X1EFE, 0X1F08, 0X1F09, 
        0X1F0A, 0X1F0B, 0X1F0C, 0X1F0D, 0X1F0E, 0X1F0F, 0X1F18, 0X1F19, 0X1F1A, 0X1F1B, 
        0X1F1C, 0X1F1D, 0X1F28, 0X1F29, 0X1F2A, 0X1F2B, 0X1F2C, 0X1F2D, 0X1F2E, 0X1F2F, 
        0X1F38, 0X1F39, 0X1F3A, 0X1F3B, 0X1F3C, 0X1F3D, 0X1F3E, 0X1F3F, 0X1F48, 0X1F49, 
        0X1F4A, 0X1F4B, 0X1F4C, 0X1F4D, 0X1F50, 0X1F52, 0X1F54, 0X1F56, 0X1F59, 0X1F5B, 
        0X1F5D, 0X1F5F, 0X1F68, 0X1F69, 0X1F6A, 0X1F6B, 0X1F6C, 0X1F6D, 0X1F6E, 0X1F6F, 
        0X1F80, 0X1F81, 0X1F82, 0X1F83, 0X1F84, 0X1F85, 0X1F86, 0X1F87, 0X1F88, 0X1F89, 
        0X1F8A, 0X1F8B, 0X1F8C, 0X1F8D, 0X1F8E, 0X1F8F, 0X1F90, 0X1F91, 0X1F92, 0X1F93, 
        0X1F94, 0X1F95, 0X1F96, 0X1F97, 0X1F98, 0X1F99, 0X1F9A, 0X1F9B, 0X1F9C, 0X1F9D, 
        0X1F9E, 0X1F9F, 0X1FA0, 0X1FA1, 0X1FA2, 0X1FA3, 0X1FA4, 0X1FA5, 0X1FA6, 0X1FA7, 
        0X1FA8, 0X1FA9, 0X1FAA, 0X1FAB, 0X1FAC, 0X1FAD, 0X1FAE, 0X1FAF, 0X1FB2, 0X1FB3, 
        0X1FB4, 0X1FB6, 0X1FB7, 0X1FB8, 0X1FB9, 0X1FBA, 0X1FBB, 0X1FBC, 0X1FBE, 0X1FC2, 
        0X1FC3, 0X1FC4, 0X1FC6, 0X1FC7, 0X1FC8, 0X1FC9, 0X1FCA, 0X1FCB, 0X1FCC, 0X1FD2, 
        0X1FD3, 0X1FD6, 0X1FD7, 0X1FD8, 0X1FD9, 0X1FDA, 0X1FDB, 0X1FE2, 0X1FE3, 0X1FE4, 
        0X1FE6, 0X1FE7, 0X1FE8, 0X1FE9, 0X1FEA, 0X1FEB, 0X1FEC, 0X1FF2, 0X1FF3, 0X1FF4, 
        0X1FF6, 0X1FF7, 0X1FF8, 0X1FF9, 0X1FFA, 0X1FFB, 0X1FFC, 0X2126, 0X212A, 0X212B, 
        0X2132, 0X2160, 0X2161, 0X2162, 0X2163, 0X2164, 0X2165, 0X2166, 0X2167, 0X2168, 
        0X2169, 0X216A, 0X216B, 0X216C, 0X216D, 0X216E, 0X216F, 0X2183, 0X24B6, 0X24B7, 
        0X24B8, 0X24B9, 0X24BA, 0X24BB, 0X24BC, 0X24BD, 0X24BE, 0X24BF, 0X24C0, 0X24C1, 
        0X24C2, 0X24C3, 0X24C4, 0X24C5, 0X24C6, 0X24C7, 0X24C8, 0X24C9, 0X24CA, 0X24CB, 
        0X24CC, 0X24CD, 0X24CE, 0X24CF, 0X2C00, 0X2C01, 0X2C02, 0X2C03, 0X2C04, 0X2C05, 
        0X2C06, 0X2C07, 0X2C08, 0X2C09, 0X2C0A, 0X2C0B, 0X2C0C, 0X2C0D, 0X2C0E, 0X2C0F, 
        0X2C10, 0X2C11, 0X2C12, 0X2C13, 0X2C14, 0X2C15, 0X2C16, 0X2C17, 0X2C18, 0X2C19, 
        0X2C1A, 0X2C1B, 0X2C1C, 0X2C1D, 0X2C1E, 0X2C1F, 0X2C20, 0X2C21, 0X2C22, 0X2C23, 
        0X2C24, 0X2C25, 0X2C26, 0X2C27, 0X2C28, 0X2C29, 0X2C2A, 0X2C2B, 0X2C2C, 0X2C2D, 
        0X2C2E, 0X2C2F, 0X2C60, 0X2C62, 0X2C63, 0X2C64, 0X2C67, 0X2C69, 0X2C6B, 0X2C6D, 
        0X2C6E, 0X2C6F, 0X2C70, 0X2C72, 0X2C75, 0X2C7E, 0X2C7F, 0X2C80, 0X2C82, 0X2C84, 
        0X2C86, 0X2C88, 0X2C8A, 0X2C8C, 0X2C8E, 0X2C90, 0X2C92, 0X2C94, 0X2C96, 0X2C98, 
        0X2C9A, 0X2C9C, 0X2C9E, 0X2CA0, 0X2CA2, 0X2CA4, 0X2CA6, 0X2CA8, 0X2CAA, 0X2CAC, 
        0X2CAE, 0X2CB0, 0X2CB2, 0X2CB4, 0X2CB6, 0X2CB8, 0X2CBA, 0X2CBC, 0X2CBE, 0X2CC0, 
        0X2CC2, 0X2CC4, 0X2CC6, 0X2CC8, 0X2CCA, 0X2CCC, 0X2CCE, 0X2CD0, 0X2CD2, 0X2CD4, 
        0X2CD6, 0X2CD8, 0X2CDA, 0X2CDC, 0X2CDE, 0X2CE0, 0X2CE2, 0X2CEB, 0X2CED, 0X2CF2, 
        0XA640, 0XA642, 0XA644, 0XA646, 0XA648, 0XA64A, 0XA64C, 0XA64E, 0XA650, 0XA652, 
        0XA654, 0XA656, 0XA658, 0XA65A, 0XA65C, 0XA65E, 0XA660, 0XA662, 0XA664, 0XA666, 
        0XA668, 0XA66A, 0XA66C, 0XA680, 0XA682, 0XA684, 0XA686, 0XA688, 0XA68A, 0XA68C, 
        0XA68E, 0XA690, 0XA692, 0XA694, 0XA696, 0XA698, 0XA69A, 0XA722, 0XA724, 0XA726, 
        0XA728, 0XA72A, 0XA72C, 0XA72E, 0XA732, 0XA734, 0XA736, 0XA738, 0XA73A, 0XA73C, 
        0XA73E, 0XA740, 0XA742, 0XA744, 0XA746, 0XA748, 0XA74A, 0XA74C, 0XA74E, 0XA750, 
        0XA752, 0XA754, 0XA756, 0XA758, 0XA75A, 0XA75C, 0XA75E, 0XA760, 0XA762, 0XA764, 
        0XA766, 0XA768, 0XA76A, 0XA76C, 0XA76E, 0XA779, 0XA77B, 0XA77D, 0XA77E, 0XA780, 
        0XA782, 0XA784, 0XA786, 0XA78B, 0XA78D, 0XA790, 0XA792, 0XA796, 0XA798, 0XA79A, 
        0XA79C, 0XA79E, 0XA7A0, 0XA7A2, 0XA7A4, 0XA7A6, 0XA7A8, 0XA7AA, 0XA7AB, 0XA7AC, 
        0XA7AD, 0XA7AE, 0XA7B0, 0XA7B1, 0XA7B2, 0XA7B3, 0XA7B4, 0XA7B6, 0XA7B8, 0XA7BA, 
        0XA7BC, 0XA7BE, 0XA7C0, 0XA7C2, 0XA7C4, 0XA7C5, 0XA7C6, 0XA7C7, 0XA7C9, 0XA7CB, 
        0XA7CC, 0XA7CE, 0XA7D0, 0XA7D2, 0XA7D4, 0XA7D6, 0XA7D8, 0XA7DA, 0XA7DC, 0XA7F5, 
        0XAB70, 0XAB71, 0XAB72, 0XAB73, 0XAB74, 0XAB75, 0XAB76, 0XAB77, 0XAB78, 0XAB79, 
        0XAB7A, 0XAB7B, 0XAB7C, 0XAB7D, 0XAB7E, 0XAB7F, 0XAB80, 0XAB81, 0XAB82, 0XAB83, 
        0XAB84, 0XAB85, 0XAB86, 0XAB87, 0XAB88, 0XAB89, 0XAB8A, 0XAB8B, 0XAB8C, 0XAB8D, 
        0XAB8E, 0XAB8F, 0XAB90, 0XAB91, 0XAB92, 0XAB93, 0XAB94, 0XAB95, 0XAB96, 0XAB97, 
        0XAB98, 0XAB99, 0XAB9A, 0XAB9B, 0XAB9C, 0XAB9D, 0XAB9E, 0XAB9F, 0XABA0, 0XABA1, 
        0XABA2, 0XABA3, 0XABA4, 0XABA5, 0XABA6, 0XABA7, 0XABA8, 0XABA9, 0XABAA, 0XABAB, 
        0XABAC, 0XABAD, 0XABAE, 0XABAF, 0XABB0, 0XABB1, 0XABB2, 0XABB3, 0XABB4, 0XABB5, 
        0XABB6, 0XABB7, 0XABB8, 0XABB9, 0XABBA, 0XABBB, 0XABBC, 0XABBD, 0XABBE, 0XABBF, 
        0XFB00, 0XFB01, 0XFB02, 0XFB03, 0XFB04, 0XFB05, 0XFB06, 0XFB13, 0XFB14, 0XFB15, 
        0XFB16, 0XFB17, 0XFF21, 0XFF22, 0XFF23, 0XFF24, 0XFF25, 0XFF26, 0XFF27, 0XFF28, 
        0XFF29, 0XFF2A, 0XFF2B, 0XFF2C, 0XFF2D, 0XFF2E, 0XFF2F, 0XFF30, 0XFF31, 0XFF32, 
        0XFF33, 0XFF34, 0XFF35, 0XFF36, 0XFF37, 0XFF38, 0XFF39, 0XFF3A, 0X10400, 0X10401, 
        0X10402, 0X10403, 0X10404, 0X10405, 0X10406, 0X10407, 0X10408, 0X10409, 0X1040A, 0X1040B, 
        0X1040C, 0X1040D, 0X1040E, 0X1040F, 0X10410, 0X10411, 0X10412, 0X10413, 0X10414, 0X10415, 
        0X10416, 0X10417, 0X10418, 0X10419, 0X1041A, 0X1041B, 0X1041C, 0X1041D, 0X1041E, 0X1041F, 
        0X10420, 0X10421, 0X10422, 0X10423, 0X10424, 0X10425, 0X10426, 0X10427, 0X104B0, 0X104B1, 
        0X104B2, 0X104B3, 0X104B4, 0X104B5, 0X104B6, 0X104B7, 0X104B8, 0X104B9, 0X104BA, 0X104BB, 
        0X104BC, 0X104BD, 0X104BE, 0X104BF, 0X104C0, 0X104C1, 0X104C2, 0X104C3, 0X104C4, 0X104C5, 
        0X104C6, 0X104C7, 0X104C8, 0X104C9, 0X104CA, 0X104CB, 0X104CC, 0X104CD, 0X104CE, 0X104CF, 
        0X104D0, 0X104D1, 0X104D2, 0X104D3, 0X10570, 0X10571, 0X10572, 0X10573, 0X10574, 0X10575, 
        0X10576, 0X10577, 0X10578, 0X10579, 0X1057A, 0X1057C, 0X1057D, 0X1057E, 0X1057F, 0X10580, 
        0X10581, 0X10582, 0X10583, 0X10584, 0X10585, 0X10586, 0X10587, 0X10588, 0X10589, 0X1058A, 
        0X1058C, 0X1058D, 0X1058E, 0X1058F, 0X10590, 0X10591, 0X10592, 0X10594, 0X10595, 0X10C80, 
        0X10C81, 0X10C82, 0X10C83, 0X10C84, 0X10C85, 0X10C86, 0X10C87, 0X10C88, 0X10C89, 0X10C8A, 
        0X10C8B, 0X10C8C, 0X10C8D, 0X10C8E, 0X10C8F, 0X10C90, 0X10C91, 0X10C92, 0X10C93, 0X10C94, 
        0X10C95, 0X10C96, 0X10C97, 0X10C98, 0X10C99, 0X10C9A, 0X10C9B, 0X10C9C, 0X10C9D, 0X10C9E, 
        0X10C9F, 0X10CA0, 0X10CA1, 0X10CA2, 0X10CA3, 0X10CA4, 0X10CA5, 0X10CA6, 0X10CA7, 0X10CA8, 
        0X10CA9, 0X10CAA, 0X10CAB, 0X10CAC, 0X10CAD, 0X10CAE, 0X10CAF, 0X10CB0, 0X10CB1, 0X10CB2, 
        0X10D50, 0X10D51, 0X10D52, 0X10D53, 0X10D54, 0X10D55, 0X10D56, 0X10D57, 0X10D58, 0X10D59, 
        0X10D5A, 0X10D5B, 0X10D5C, 0X10D5D, 0X10D5E, 0X10D5F, 0X10D60, 0X10D61, 0X10D62, 0X10D63, 
        0X10D64, 0X10D65, 0X118A0, 0X118A1, 0X118A2, 0X118A3, 0X118A4, 0X118A5, 0X118A6, 0X118A7, 
        0X118A8, 0X118A9, 0X118AA, 0X118AB, 0X118AC, 0X118AD, 0X118AE, 0X118AF, 0X118B0, 0X118B1, 
        0X118B2, 0X118B3, 0X118B4, 0X118B5, 0X118B6, 0X118B7, 0X118B8, 0X118B9, 0X118BA, 0X118BB, 
        0X118BC, 0X118BD, 0X118BE, 0X118BF, 0X16E40, 0X16E41, 0X16E42, 0X16E43, 0X16E44, 0X16E45, 
        0X16E46, 0X16E47, 0X16E48, 0X16E49, 0X16E4A, 0X16E4B, 0X16E4C, 0X16E4D, 0X16E4E, 0X16E4F, 
        0X16E50, 0X16E51, 0X16E52, 0X16E53, 0X16E54, 0X16E55, 0X16E56, 0X16E57, 0X16E58, 0X16E59, 
        0X16E5A, 0X16E5B, 0X16E5C, 0X16E5D, 0X16E5E, 0X16E5F, 0X16EA0, 0X16EA1, 0X16EA2, 0X16EA3, 
        0X16EA4, 0X16EA5, 0X16EA6, 0X16EA7, 0X16EA8, 0X16EA9, 0X16EAA, 0X16EAB, 0X16EAC, 0X16EAD, 
        0X16EAE, 0X16EAF, 0X16EB0, 0X16EB1, 0X16EB2, 0X16EB3, 0X16EB4, 0X16EB5, 0X16EB6, 0X16EB7, 
        0X16EB8, 0X1E900, 0X1E901, 0X1E902, 0X1E903, 0X1E904, 0X1E905, 0X1E906, 0X1E907, 0X1E908, 
        0X1E909, 0X1E90A, 0X1E90B, 0X1E90C, 0X1E90D, 0X1E90E, 0X1E90F, 0X1E910, 0X1E911, 0X1E912, 
        0X1E913, 0X1E914, 0X1E915, 0X1E916, 0X1E917, 0X1E918, 0X1E919, 0X1E91A, 0X1E91B, 0X1E91C, 
        0X1E91D, 0X1E91E, 0X1E91F, 0X1E920, 0X1E921
    };

    private static final long[] CASE_FOLDING_VALUES = {
        0x0000000000061L, 0x0000000000062L, 0x0000000000063L, 0x0000000000064L, 0x0000000000065L, 0x0000000000066L, 
        0x0000000000067L, 0x0000000000068L, 0x0000000000069L, 0x000000000006aL, 0x000000000006bL, 0x000000000006cL, 
        0x000000000006dL, 0x000000000006eL, 0x000000000006fL, 0x0000000000070L, 0x0000000000071L, 0x0000000000072L, 
        0x0000000000073L, 0x0000000000074L, 0x0000000000075L, 0x0000000000076L, 0x0000000000077L, 0x0000000000078L, 
        0x0000000000079L, 0x000000000007aL, 0x00000000003bcL, 0x00000000000e0L, 0x00000000000e1L, 0x00000000000e2L, 
        0x00000000000e3L, 0x00000000000e4L, 0x00000000000e5L, 0x00000000000e6L, 0x00000000000e7L, 0x00000000000e8L, 
        0x00000000000e9L, 0x00000000000eaL, 0x00000000000ebL, 0x00000000000ecL, 0x00000000000edL, 0x00000000000eeL, 
        0x00000000000efL, 0x00000000000f0L, 0x00000000000f1L, 0x00000000000f2L, 0x00000000000f3L, 0x00000000000f4L, 
        0x00000000000f5L, 0x00000000000f6L, 0x00000000000f8L, 0x00000000000f9L, 0x00000000000faL, 0x00000000000fbL, 
        0x00000000000fcL, 0x00000000000fdL, 0x00000000000feL, 0x2000000730073L, 0x0000000000101L, 0x0000000000103L, 
        0x0000000000105L, 0x0000000000107L, 0x0000000000109L, 0x000000000010bL, 0x000000000010dL, 0x000000000010fL, 
        0x0000000000111L, 0x0000000000113L, 0x0000000000115L, 0x0000000000117L, 0x0000000000119L, 0x000000000011bL, 
        0x000000000011dL, 0x000000000011fL, 0x0000000000121L, 0x0000000000123L, 0x0000000000125L, 0x0000000000127L, 
        0x0000000000129L, 0x000000000012bL, 0x000000000012dL, 0x000000000012fL, 0x2000003070069L, 0x0000000000133L, 
        0x0000000000135L, 0x0000000000137L, 0x000000000013aL, 0x000000000013cL, 0x000000000013eL, 0x0000000000140L, 
        0x0000000000142L, 0x0000000000144L, 0x0000000000146L, 0x0000000000148L, 0x20000006e02bcL, 0x000000000014bL, 
        0x000000000014dL, 0x000000000014fL, 0x0000000000151L, 0x0000000000153L, 0x0000000000155L, 0x0000000000157L, 
        0x0000000000159L, 0x000000000015bL, 0x000000000015dL, 0x000000000015fL, 0x0000000000161L, 0x0000000000163L, 
        0x0000000000165L, 0x0000000000167L, 0x0000000000169L, 0x000000000016bL, 0x000000000016dL, 0x000000000016fL, 
        0x0000000000171L, 0x0000000000173L, 0x0000000000175L, 0x0000000000177L, 0x00000000000ffL, 0x000000000017aL, 
        0x000000000017cL, 0x000000000017eL, 0x0000000000073L, 0x0000000000253L, 0x0000000000183L, 0x0000000000185L, 
        0x0000000000254L, 0x0000000000188L, 0x0000000000256L, 0x0000000000257L, 0x000000000018cL, 0x00000000001ddL, 
        0x0000000000259L, 0x000000000025bL, 0x0000000000192L, 0x0000000000260L, 0x0000000000263L, 0x0000000000269L, 
        0x0000000000268L, 0x0000000000199L, 0x000000000026fL, 0x0000000000272L, 0x0000000000275L, 0x00000000001a1L, 
        0x00000000001a3L, 0x00000000001a5L, 0x0000000000280L, 0x00000000001a8L, 0x0000000000283L, 0x00000000001adL, 
        0x0000000000288L, 0x00000000001b0L, 0x000000000028aL, 0x000000000028bL, 0x00000000001b4L, 0x00000000001b6L, 
        0x0000000000292L, 0x00000000001b9L, 0x00000000001bdL, 0x00000000001c6L, 0x00000000001c6L, 0x00000000001c9L, 
        0x00000000001c9L, 0x00000000001ccL, 0x00000000001ccL, 0x00000000001ceL, 0x00000000001d0L, 0x00000000001d2L, 
        0x00000000001d4L, 0x00000000001d6L, 0x00000000001d8L, 0x00000000001daL, 0x00000000001dcL, 0x00000000001dfL, 
        0x00000000001e1L, 0x00000000001e3L, 0x00000000001e5L, 0x00000000001e7L, 0x00000000001e9L, 0x00000000001ebL, 
        0x00000000001edL, 0x00000000001efL, 0x20000030c006aL, 0x00000000001f3L, 0x00000000001f3L, 0x00000000001f5L, 
        0x0000000000195L, 0x00000000001bfL, 0x00000000001f9L, 0x00000000001fbL, 0x00000000001fdL, 0x00000000001ffL, 
        0x0000000000201L, 0x0000000000203L, 0x0000000000205L, 0x0000000000207L, 0x0000000000209L, 0x000000000020bL, 
        0x000000000020dL, 0x000000000020fL, 0x0000000000211L, 0x0000000000213L, 0x0000000000215L, 0x0000000000217L, 
        0x0000000000219L, 0x000000000021bL, 0x000000000021dL, 0x000000000021fL, 0x000000000019eL, 0x0000000000223L, 
        0x0000000000225L, 0x0000000000227L, 0x0000000000229L, 0x000000000022bL, 0x000000000022dL, 0x000000000022fL, 
        0x0000000000231L, 0x0000000000233L, 0x0000000002c65L, 0x000000000023cL, 0x000000000019aL, 0x0000000002c66L, 
        0x0000000000242L, 0x0000000000180L, 0x0000000000289L, 0x000000000028cL, 0x0000000000247L, 0x0000000000249L, 
        0x000000000024bL, 0x000000000024dL, 0x000000000024fL, 0x00000000003b9L, 0x0000000000371L, 0x0000000000373L, 
        0x0000000000377L, 0x00000000003f3L, 0x00000000003acL, 0x00000000003adL, 0x00000000003aeL, 0x00000000003afL, 
        0x00000000003ccL, 0x00000000003cdL, 0x00000000003ceL, 0x30301030803b9L, 0x00000000003b1L, 0x00000000003b2L, 
        0x00000000003b3L, 0x00000000003b4L, 0x00000000003b5L, 0x00000000003b6L, 0x00000000003b7L, 0x00000000003b8L, 
        0x00000000003b9L, 0x00000000003baL, 0x00000000003bbL, 0x00000000003bcL, 0x00000000003bdL, 0x00000000003beL, 
        0x00000000003bfL, 0x00000000003c0L, 0x00000000003c1L, 0x00000000003c3L, 0x00000000003c4L, 0x00000000003c5L, 
        0x00000000003c6L, 0x00000000003c7L, 0x00000000003c8L, 0x00000000003c9L, 0x00000000003caL, 0x00000000003cbL, 
        0x30301030803c5L, 0x00000000003c3L, 0x00000000003d7L, 0x00000000003b2L, 0x00000000003b8L, 0x00000000003c6L, 
        0x00000000003c0L, 0x00000000003d9L, 0x00000000003dbL, 0x00000000003ddL, 0x00000000003dfL, 0x00000000003e1L, 
        0x00000000003e3L, 0x00000000003e5L, 0x00000000003e7L, 0x00000000003e9L, 0x00000000003ebL, 0x00000000003edL, 
        0x00000000003efL, 0x00000000003baL, 0x00000000003c1L, 0x00000000003b8L, 0x00000000003b5L, 0x00000000003f8L, 
        0x00000000003f2L, 0x00000000003fbL, 0x000000000037bL, 0x000000000037cL, 0x000000000037dL, 0x0000000000450L, 
        0x0000000000451L, 0x0000000000452L, 0x0000000000453L, 0x0000000000454L, 0x0000000000455L, 0x0000000000456L, 
        0x0000000000457L, 0x0000000000458L, 0x0000000000459L, 0x000000000045aL, 0x000000000045bL, 0x000000000045cL, 
        0x000000000045dL, 0x000000000045eL, 0x000000000045fL, 0x0000000000430L, 0x0000000000431L, 0x0000000000432L, 
        0x0000000000433L, 0x0000000000434L, 0x0000000000435L, 0x0000000000436L, 0x0000000000437L, 0x0000000000438L, 
        0x0000000000439L, 0x000000000043aL, 0x000000000043bL, 0x000000000043cL, 0x000000000043dL, 0x000000000043eL, 
        0x000000000043fL, 0x0000000000440L, 0x0000000000441L, 0x0000000000442L, 0x0000000000443L, 0x0000000000444L, 
        0x0000000000445L, 0x0000000000446L, 0x0000000000447L, 0x0000000000448L, 0x0000000000449L, 0x000000000044aL, 
        0x000000000044bL, 0x000000000044cL, 0x000000000044dL, 0x000000000044eL, 0x000000000044fL, 0x0000000000461L, 
        0x0000000000463L, 0x0000000000465L, 0x0000000000467L, 0x0000000000469L, 0x000000000046bL, 0x000000000046dL, 
        0x000000000046fL, 0x0000000000471L, 0x0000000000473L, 0x0000000000475L, 0x0000000000477L, 0x0000000000479L, 
        0x000000000047bL, 0x000000000047dL, 0x000000000047fL, 0x0000000000481L, 0x000000000048bL, 0x000000000048dL, 
        0x000000000048fL, 0x0000000000491L, 0x0000000000493L, 0x0000000000495L, 0x0000000000497L, 0x0000000000499L, 
        0x000000000049bL, 0x000000000049dL, 0x000000000049fL, 0x00000000004a1L, 0x00000000004a3L, 0x00000000004a5L, 
        0x00000000004a7L, 0x00000000004a9L, 0x00000000004abL, 0x00000000004adL, 0x00000000004afL, 0x00000000004b1L, 
        0x00000000004b3L, 0x00000000004b5L, 0x00000000004b7L, 0x00000000004b9L, 0x00000000004bbL, 0x00000000004bdL, 
        0x00000000004bfL, 0x00000000004cfL, 0x00000000004c2L, 0x00000000004c4L, 0x00000000004c6L, 0x00000000004c8L, 
        0x00000000004caL, 0x00000000004ccL, 0x00000000004ceL, 0x00000000004d1L, 0x00000000004d3L, 0x00000000004d5L, 
        0x00000000004d7L, 0x00000000004d9L, 0x00000000004dbL, 0x00000000004ddL, 0x00000000004dfL, 0x00000000004e1L, 
        0x00000000004e3L, 0x00000000004e5L, 0x00000000004e7L, 0x00000000004e9L, 0x00000000004ebL, 0x00000000004edL, 
        0x00000000004efL, 0x00000000004f1L, 0x00000000004f3L, 0x00000000004f5L, 0x00000000004f7L, 0x00000000004f9L, 
        0x00000000004fbL, 0x00000000004fdL, 0x00000000004ffL, 0x0000000000501L, 0x0000000000503L, 0x0000000000505L, 
        0x0000000000507L, 0x0000000000509L, 0x000000000050bL, 0x000000000050dL, 0x000000000050fL, 0x0000000000511L, 
        0x0000000000513L, 0x0000000000515L, 0x0000000000517L, 0x0000000000519L, 0x000000000051bL, 0x000000000051dL, 
        0x000000000051fL, 0x0000000000521L, 0x0000000000523L, 0x0000000000525L, 0x0000000000527L, 0x0000000000529L, 
        0x000000000052bL, 0x000000000052dL, 0x000000000052fL, 0x0000000000561L, 0x0000000000562L, 0x0000000000563L, 
        0x0000000000564L, 0x0000000000565L, 0x0000000000566L, 0x0000000000567L, 0x0000000000568L, 0x0000000000569L, 
        0x000000000056aL, 0x000000000056bL, 0x000000000056cL, 0x000000000056dL, 0x000000000056eL, 0x000000000056fL, 
        0x0000000000570L, 0x0000000000571L, 0x0000000000572L, 0x0000000000573L, 0x0000000000574L, 0x0000000000575L, 
        0x0000000000576L, 0x0000000000577L, 0x0000000000578L, 0x0000000000579L, 0x000000000057aL, 0x000000000057bL, 
        0x000000000057cL, 0x000000000057dL, 0x000000000057eL, 0x000000000057fL, 0x0000000000580L, 0x0000000000581L, 
        0x0000000000582L, 0x0000000000583L, 0x0000000000584L, 0x0000000000585L, 0x0000000000586L, 0x2000005820565L, 
        0x0000000002d00L, 0x0000000002d01L, 0x0000000002d02L, 0x0000000002d03L, 0x0000000002d04L, 0x0000000002d05L, 
        0x0000000002d06L, 0x0000000002d07L, 0x0000000002d08L, 0x0000000002d09L, 0x0000000002d0aL, 0x0000000002d0bL, 
        0x0000000002d0cL, 0x0000000002d0dL, 0x0000000002d0eL, 0x0000000002d0fL, 0x0000000002d10L, 0x0000000002d11L, 
        0x0000000002d12L, 0x0000000002d13L, 0x0000000002d14L, 0x0000000002d15L, 0x0000000002d16L, 0x0000000002d17L, 
        0x0000000002d18L, 0x0000000002d19L, 0x0000000002d1aL, 0x0000000002d1bL, 0x0000000002d1cL, 0x0000000002d1dL, 
        0x0000000002d1eL, 0x0000000002d1fL, 0x0000000002d20L, 0x0000000002d21L, 0x0000000002d22L, 0x0000000002d23L, 
        0x0000000002d24L, 0x0000000002d25L, 0x0000000002d27L, 0x0000000002d2dL, 0x00000000013f0L, 0x00000000013f1L, 
        0x00000000013f2L, 0x00000000013f3L, 0x00000000013f4L, 0x00000000013f5L, 0x0000000000432L, 0x0000000000434L, 
        0x000000000043eL, 0x0000000000441L, 0x0000000000442L, 0x0000000000442L, 0x000000000044aL, 0x0000000000463L, 
        0x000000000a64bL, 0x0000000001c8aL, 0x00000000010d0L, 0x00000000010d1L, 0x00000000010d2L, 0x00000000010d3L, 
        0x00000000010d4L, 0x00000000010d5L, 0x00000000010d6L, 0x00000000010d7L, 0x00000000010d8L, 0x00000000010d9L, 
        0x00000000010daL, 0x00000000010dbL, 0x00000000010dcL, 0x00000000010ddL, 0x00000000010deL, 0x00000000010dfL, 
        0x00000000010e0L, 0x00000000010e1L, 0x00000000010e2L, 0x00000000010e3L, 0x00000000010e4L, 0x00000000010e5L, 
        0x00000000010e6L, 0x00000000010e7L, 0x00000000010e8L, 0x00000000010e9L, 0x00000000010eaL, 0x00000000010ebL, 
        0x00000000010ecL, 0x00000000010edL, 0x00000000010eeL, 0x00000000010efL, 0x00000000010f0L, 0x00000000010f1L, 
        0x00000000010f2L, 0x00000000010f3L, 0x00000000010f4L, 0x00000000010f5L, 0x00000000010f6L, 0x00000000010f7L, 
        0x00000000010f8L, 0x00000000010f9L, 0x00000000010faL, 0x00000000010fdL, 0x00000000010feL, 0x00000000010ffL, 
        0x0000000001e01L, 0x0000000001e03L, 0x0000000001e05L, 0x0000000001e07L, 0x0000000001e09L, 0x0000000001e0bL, 
        0x0000000001e0dL, 0x0000000001e0fL, 0x0000000001e11L, 0x0000000001e13L, 0x0000000001e15L, 0x0000000001e17L, 
        0x0000000001e19L, 0x0000000001e1bL, 0x0000000001e1dL, 0x0000000001e1fL, 0x0000000001e21L, 0x0000000001e23L, 
        0x0000000001e25L, 0x0000000001e27L, 0x0000000001e29L, 0x0000000001e2bL, 0x0000000001e2dL, 0x0000000001e2fL, 
        0x0000000001e31L, 0x0000000001e33L, 0x0000000001e35L, 0x0000000001e37L, 0x0000000001e39L, 0x0000000001e3bL, 
        0x0000000001e3dL, 0x0000000001e3fL, 0x0000000001e41L, 0x0000000001e43L, 0x0000000001e45L, 0x0000000001e47L, 
        0x0000000001e49L, 0x0000000001e4bL, 0x0000000001e4dL, 0x0000000001e4fL, 0x0000000001e51L, 0x0000000001e53L, 
        0x0000000001e55L, 0x0000000001e57L, 0x0000000001e59L, 0x0000000001e5bL, 0x0000000001e5dL, 0x0000000001e5fL, 
        0x0000000001e61L, 0x0000000001e63L, 0x0000000001e65L, 0x0000000001e67L, 0x0000000001e69L, 0x0000000001e6bL, 
        0x0000000001e6dL, 0x0000000001e6fL, 0x0000000001e71L, 0x0000000001e73L, 0x0000000001e75L, 0x0000000001e77L, 
        0x0000000001e79L, 0x0000000001e7bL, 0x0000000001e7dL, 0x0000000001e7fL, 0x0000000001e81L, 0x0000000001e83L, 
        0x0000000001e85L, 0x0000000001e87L, 0x0000000001e89L, 0x0000000001e8bL, 0x0000000001e8dL, 0x0000000001e8fL, 
        0x0000000001e91L, 0x0000000001e93L, 0x0000000001e95L, 0x2000003310068L, 0x2000003080074L, 0x20000030a0077L, 
        0x20000030a0079L, 0x2000002be0061L, 0x0000000001e61L, 0x2000000730073L, 0x0000000001ea1L, 0x0000000001ea3L, 
        0x0000000001ea5L, 0x0000000001ea7L, 0x0000000001ea9L, 0x0000000001eabL, 0x0000000001eadL, 0x0000000001eafL, 
        0x0000000001eb1L, 0x0000000001eb3L, 0x0000000001eb5L, 0x0000000001eb7L, 0x0000000001eb9L, 0x0000000001ebbL, 
        0x0000000001ebdL, 0x0000000001ebfL, 0x0000000001ec1L, 0x0000000001ec3L, 0x0000000001ec5L, 0x0000000001ec7L, 
        0x0000000001ec9L, 0x0000000001ecbL, 0x0000000001ecdL, 0x0000000001ecfL, 0x0000000001ed1L, 0x0000000001ed3L, 
        0x0000000001ed5L, 0x0000000001ed7L, 0x0000000001ed9L, 0x0000000001edbL, 0x0000000001eddL, 0x0000000001edfL, 
        0x0000000001ee1L, 0x0000000001ee3L, 0x0000000001ee5L, 0x0000000001ee7L, 0x0000000001ee9L, 0x0000000001eebL, 
        0x0000000001eedL, 0x0000000001eefL, 0x0000000001ef1L, 0x0000000001ef3L, 0x0000000001ef5L, 0x0000000001ef7L, 
        0x0000000001ef9L, 0x0000000001efbL, 0x0000000001efdL, 0x0000000001effL, 0x0000000001f00L, 0x0000000001f01L, 
        0x0000000001f02L, 0x0000000001f03L, 0x0000000001f04L, 0x0000000001f05L, 0x0000000001f06L, 0x0000000001f07L, 
        0x0000000001f10L, 0x0000000001f11L, 0x0000000001f12L, 0x0000000001f13L, 0x0000000001f14L, 0x0000000001f15L, 
        0x0000000001f20L, 0x0000000001f21L, 0x0000000001f22L, 0x0000000001f23L, 0x0000000001f24L, 0x0000000001f25L, 
        0x0000000001f26L, 0x0000000001f27L, 0x0000000001f30L, 0x0000000001f31L, 0x0000000001f32L, 0x0000000001f33L, 
        0x0000000001f34L, 0x0000000001f35L, 0x0000000001f36L, 0x0000000001f37L, 0x0000000001f40L, 0x0000000001f41L, 
        0x0000000001f42L, 0x0000000001f43L, 0x0000000001f44L, 0x0000000001f45L, 0x20000031303c5L, 0x30300031303c5L, 
        0x30301031303c5L, 0x30342031303c5L, 0x0000000001f51L, 0x0000000001f53L, 0x0000000001f55L, 0x0000000001f57L, 
        0x0000000001f60L, 0x0000000001f61L, 0x0000000001f62L, 0x0000000001f63L, 0x0000000001f64L, 0x0000000001f65L, 
        0x0000000001f66L, 0x0000000001f67L, 0x2000003b91f00L, 0x2000003b91f01L, 0x2000003b91f02L, 0x2000003b91f03L, 
        0x2000003b91f04L, 0x2000003b91f05L, 0x2000003b91f06L, 0x2000003b91f07L, 0x2000003b91f00L, 0x2000003b91f01L, 
        0x2000003b91f02L, 0x2000003b91f03L, 0x2000003b91f04L, 0x2000003b91f05L, 0x2000003b91f06L, 0x2000003b91f07L, 
        0x2000003b91f20L, 0x2000003b91f21L, 0x2000003b91f22L, 0x2000003b91f23L, 0x2000003b91f24L, 0x2000003b91f25L, 
        0x2000003b91f26L, 0x2000003b91f27L, 0x2000003b91f20L, 0x2000003b91f21L, 0x2000003b91f22L, 0x2000003b91f23L, 
        0x2000003b91f24L, 0x2000003b91f25L, 0x2000003b91f26L, 0x2000003b91f27L, 0x2000003b91f60L, 0x2000003b91f61L, 
        0x2000003b91f62L, 0x2000003b91f63L, 0x2000003b91f64L, 0x2000003b91f65L, 0x2000003b91f66L, 0x2000003b91f67L, 
        0x2000003b91f60L, 0x2000003b91f61L, 0x2000003b91f62L, 0x2000003b91f63L, 0x2000003b91f64L, 0x2000003b91f65L, 
        0x2000003b91f66L, 0x2000003b91f67L, 0x2000003b91f70L, 0x2000003b903b1L, 0x2000003b903acL, 0x20000034203b1L, 
        0x303b9034203b1L, 0x0000000001fb0L, 0x0000000001fb1L, 0x0000000001f70L, 0x0000000001f71L, 0x2000003b903b1L, 
        0x00000000003b9L, 0x2000003b91f74L, 0x2000003b903b7L, 0x2000003b903aeL, 0x20000034203b7L, 0x303b9034203b7L, 
        0x0000000001f72L, 0x0000000001f73L, 0x0000000001f74L, 0x0000000001f75L, 0x2000003b903b7L, 0x30300030803b9L, 
        0x30301030803b9L, 0x20000034203b9L, 0x30342030803b9L, 0x0000000001fd0L, 0x0000000001fd1L, 0x0000000001f76L, 
        0x0000000001f77L, 0x30300030803c5L, 0x30301030803c5L, 0x20000031303c1L, 0x20000034203c5L, 0x30342030803c5L, 
        0x0000000001fe0L, 0x0000000001fe1L, 0x0000000001f7aL, 0x0000000001f7bL, 0x0000000001fe5L, 0x2000003b91f7cL, 
        0x2000003b903c9L, 0x2000003b903ceL, 0x20000034203c9L, 0x303b9034203c9L, 0x0000000001f78L, 0x0000000001f79L, 
        0x0000000001f7cL, 0x0000000001f7dL, 0x2000003b903c9L, 0x00000000003c9L, 0x000000000006bL, 0x00000000000e5L, 
        0x000000000214eL, 0x0000000002170L, 0x0000000002171L, 0x0000000002172L, 0x0000000002173L, 0x0000000002174L, 
        0x0000000002175L, 0x0000000002176L, 0x0000000002177L, 0x0000000002178L, 0x0000000002179L, 0x000000000217aL, 
        0x000000000217bL, 0x000000000217cL, 0x000000000217dL, 0x000000000217eL, 0x000000000217fL, 0x0000000002184L, 
        0x00000000024d0L, 0x00000000024d1L, 0x00000000024d2L, 0x00000000024d3L, 0x00000000024d4L, 0x00000000024d5L, 
        0x00000000024d6L, 0x00000000024d7L, 0x00000000024d8L, 0x00000000024d9L, 0x00000000024daL, 0x00000000024dbL, 
        0x00000000024dcL, 0x00000000024ddL, 0x00000000024deL, 0x00000000024dfL, 0x00000000024e0L, 0x00000000024e1L, 
        0x00000000024e2L, 0x00000000024e3L, 0x00000000024e4L, 0x00000000024e5L, 0x00000000024e6L, 0x00000000024e7L, 
        0x00000000024e8L, 0x00000000024e9L, 0x0000000002c30L, 0x0000000002c31L, 0x0000000002c32L, 0x0000000002c33L, 
        0x0000000002c34L, 0x0000000002c35L, 0x0000000002c36L, 0x0000000002c37L, 0x0000000002c38L, 0x0000000002c39L, 
        0x0000000002c3aL, 0x0000000002c3bL, 0x0000000002c3cL, 0x0000000002c3dL, 0x0000000002c3eL, 0x0000000002c3fL, 
        0x0000000002c40L, 0x0000000002c41L, 0x0000000002c42L, 0x0000000002c43L, 0x0000000002c44L, 0x0000000002c45L, 
        0x0000000002c46L, 0x0000000002c47L, 0x0000000002c48L, 0x0000000002c49L, 0x0000000002c4aL, 0x0000000002c4bL, 
        0x0000000002c4cL, 0x0000000002c4dL, 0x0000000002c4eL, 0x0000000002c4fL, 0x0000000002c50L, 0x0000000002c51L, 
        0x0000000002c52L, 0x0000000002c53L, 0x0000000002c54L, 0x0000000002c55L, 0x0000000002c56L, 0x0000000002c57L, 
        0x0000000002c58L, 0x0000000002c59L, 0x0000000002c5aL, 0x0000000002c5bL, 0x0000000002c5cL, 0x0000000002c5dL, 
        0x0000000002c5eL, 0x0000000002c5fL, 0x0000000002c61L, 0x000000000026bL, 0x0000000001d7dL, 0x000000000027dL, 
        0x0000000002c68L, 0x0000000002c6aL, 0x0000000002c6cL, 0x0000000000251L, 0x0000000000271L, 0x0000000000250L, 
        0x0000000000252L, 0x0000000002c73L, 0x0000000002c76L, 0x000000000023fL, 0x0000000000240L, 0x0000000002c81L, 
        0x0000000002c83L, 0x0000000002c85L, 0x0000000002c87L, 0x0000000002c89L, 0x0000000002c8bL, 0x0000000002c8dL, 
        0x0000000002c8fL, 0x0000000002c91L, 0x0000000002c93L, 0x0000000002c95L, 0x0000000002c97L, 0x0000000002c99L, 
        0x0000000002c9bL, 0x0000000002c9dL, 0x0000000002c9fL, 0x0000000002ca1L, 0x0000000002ca3L, 0x0000000002ca5L, 
        0x0000000002ca7L, 0x0000000002ca9L, 0x0000000002cabL, 0x0000000002cadL, 0x0000000002cafL, 0x0000000002cb1L, 
        0x0000000002cb3L, 0x0000000002cb5L, 0x0000000002cb7L, 0x0000000002cb9L, 0x0000000002cbbL, 0x0000000002cbdL, 
        0x0000000002cbfL, 0x0000000002cc1L, 0x0000000002cc3L, 0x0000000002cc5L, 0x0000000002cc7L, 0x0000000002cc9L, 
        0x0000000002ccbL, 0x0000000002ccdL, 0x0000000002ccfL, 0x0000000002cd1L, 0x0000000002cd3L, 0x0000000002cd5L, 
        0x0000000002cd7L, 0x0000000002cd9L, 0x0000000002cdbL, 0x0000000002cddL, 0x0000000002cdfL, 0x0000000002ce1L, 
        0x0000000002ce3L, 0x0000000002cecL, 0x0000000002ceeL, 0x0000000002cf3L, 0x000000000a641L, 0x000000000a643L, 
        0x000000000a645L, 0x000000000a647L, 0x000000000a649L, 0x000000000a64bL, 0x000000000a64dL, 0x000000000a64fL, 
        0x000000000a651L, 0x000000000a653L, 0x000000000a655L, 0x000000000a657L, 0x000000000a659L, 0x000000000a65bL, 
        0x000000000a65dL, 0x000000000a65fL, 0x000000000a661L, 0x000000000a663L, 0x000000000a665L, 0x000000000a667L, 
        0x000000000a669L, 0x000000000a66bL, 0x000000000a66dL, 0x000000000a681L, 0x000000000a683L, 0x000000000a685L, 
        0x000000000a687L, 0x000000000a689L, 0x000000000a68bL, 0x000000000a68dL, 0x000000000a68fL, 0x000000000a691L, 
        0x000000000a693L, 0x000000000a695L, 0x000000000a697L, 0x000000000a699L, 0x000000000a69bL, 0x000000000a723L, 
        0x000000000a725L, 0x000000000a727L, 0x000000000a729L, 0x000000000a72bL, 0x000000000a72dL, 0x000000000a72fL, 
        0x000000000a733L, 0x000000000a735L, 0x000000000a737L, 0x000000000a739L, 0x000000000a73bL, 0x000000000a73dL, 
        0x000000000a73fL, 0x000000000a741L, 0x000000000a743L, 0x000000000a745L, 0x000000000a747L, 0x000000000a749L, 
        0x000000000a74bL, 0x000000000a74dL, 0x000000000a74fL, 0x000000000a751L, 0x000000000a753L, 0x000000000a755L, 
        0x000000000a757L, 0x000000000a759L, 0x000000000a75bL, 0x000000000a75dL, 0x000000000a75fL, 0x000000000a761L, 
        0x000000000a763L, 0x000000000a765L, 0x000000000a767L, 0x000000000a769L, 0x000000000a76bL, 0x000000000a76dL, 
        0x000000000a76fL, 0x000000000a77aL, 0x000000000a77cL, 0x0000000001d79L, 0x000000000a77fL, 0x000000000a781L, 
        0x000000000a783L, 0x000000000a785L, 0x000000000a787L, 0x000000000a78cL, 0x0000000000265L, 0x000000000a791L, 
        0x000000000a793L, 0x000000000a797L, 0x000000000a799L, 0x000000000a79bL, 0x000000000a79dL, 0x000000000a79fL, 
        0x000000000a7a1L, 0x000000000a7a3L, 0x000000000a7a5L, 0x000000000a7a7L, 0x000000000a7a9L, 0x0000000000266L, 
        0x000000000025cL, 0x0000000000261L, 0x000000000026cL, 0x000000000026aL, 0x000000000029eL, 0x0000000000287L, 
        0x000000000029dL, 0x000000000ab53L, 0x000000000a7b5L, 0x000000000a7b7L, 0x000000000a7b9L, 0x000000000a7bbL, 
        0x000000000a7bdL, 0x000000000a7bfL, 0x000000000a7c1L, 0x000000000a7c3L, 0x000000000a794L, 0x0000000000282L, 
        0x0000000001d8eL, 0x000000000a7c8L, 0x000000000a7caL, 0x0000000000264L, 0x000000000a7cdL, 0x000000000a7cfL, 
        0x000000000a7d1L, 0x000000000a7d3L, 0x000000000a7d5L, 0x000000000a7d7L, 0x000000000a7d9L, 0x000000000a7dbL, 
        0x000000000019bL, 0x000000000a7f6L, 0x00000000013a0L, 0x00000000013a1L, 0x00000000013a2L, 0x00000000013a3L, 
        0x00000000013a4L, 0x00000000013a5L, 0x00000000013a6L, 0x00000000013a7L, 0x00000000013a8L, 0x00000000013a9L, 
        0x00000000013aaL, 0x00000000013abL, 0x00000000013acL, 0x00000000013adL, 0x00000000013aeL, 0x00000000013afL, 
        0x00000000013b0L, 0x00000000013b1L, 0x00000000013b2L, 0x00000000013b3L, 0x00000000013b4L, 0x00000000013b5L, 
        0x00000000013b6L, 0x00000000013b7L, 0x00000000013b8L, 0x00000000013b9L, 0x00000000013baL, 0x00000000013bbL, 
        0x00000000013bcL, 0x00000000013bdL, 0x00000000013beL, 0x00000000013bfL, 0x00000000013c0L, 0x00000000013c1L, 
        0x00000000013c2L, 0x00000000013c3L, 0x00000000013c4L, 0x00000000013c5L, 0x00000000013c6L, 0x00000000013c7L, 
        0x00000000013c8L, 0x00000000013c9L, 0x00000000013caL, 0x00000000013cbL, 0x00000000013ccL, 0x00000000013cdL, 
        0x00000000013ceL, 0x00000000013cfL, 0x00000000013d0L, 0x00000000013d1L, 0x00000000013d2L, 0x00000000013d3L, 
        0x00000000013d4L, 0x00000000013d5L, 0x00000000013d6L, 0x00000000013d7L, 0x00000000013d8L, 0x00000000013d9L, 
        0x00000000013daL, 0x00000000013dbL, 0x00000000013dcL, 0x00000000013ddL, 0x00000000013deL, 0x00000000013dfL, 
        0x00000000013e0L, 0x00000000013e1L, 0x00000000013e2L, 0x00000000013e3L, 0x00000000013e4L, 0x00000000013e5L, 
        0x00000000013e6L, 0x00000000013e7L, 0x00000000013e8L, 0x00000000013e9L, 0x00000000013eaL, 0x00000000013ebL, 
        0x00000000013ecL, 0x00000000013edL, 0x00000000013eeL, 0x00000000013efL, 0x2000000660066L, 0x2000000690066L, 
        0x20000006c0066L, 0x3006900660066L, 0x3006c00660066L, 0x2000000740073L, 0x2000000740073L, 0x2000005760574L, 
        0x2000005650574L, 0x20000056b0574L, 0x200000576057eL, 0x20000056d0574L, 0x000000000ff41L, 0x000000000ff42L, 
        0x000000000ff43L, 0x000000000ff44L, 0x000000000ff45L, 0x000000000ff46L, 0x000000000ff47L, 0x000000000ff48L, 
        0x000000000ff49L, 0x000000000ff4aL, 0x000000000ff4bL, 0x000000000ff4cL, 0x000000000ff4dL, 0x000000000ff4eL, 
        0x000000000ff4fL, 0x000000000ff50L, 0x000000000ff51L, 0x000000000ff52L, 0x000000000ff53L, 0x000000000ff54L, 
        0x000000000ff55L, 0x000000000ff56L, 0x000000000ff57L, 0x000000000ff58L, 0x000000000ff59L, 0x000000000ff5aL, 
        0x0000000010428L, 0x0000000010429L, 0x000000001042aL, 0x000000001042bL, 0x000000001042cL, 0x000000001042dL, 
        0x000000001042eL, 0x000000001042fL, 0x0000000010430L, 0x0000000010431L, 0x0000000010432L, 0x0000000010433L, 
        0x0000000010434L, 0x0000000010435L, 0x0000000010436L, 0x0000000010437L, 0x0000000010438L, 0x0000000010439L, 
        0x000000001043aL, 0x000000001043bL, 0x000000001043cL, 0x000000001043dL, 0x000000001043eL, 0x000000001043fL, 
        0x0000000010440L, 0x0000000010441L, 0x0000000010442L, 0x0000000010443L, 0x0000000010444L, 0x0000000010445L, 
        0x0000000010446L, 0x0000000010447L, 0x0000000010448L, 0x0000000010449L, 0x000000001044aL, 0x000000001044bL, 
        0x000000001044cL, 0x000000001044dL, 0x000000001044eL, 0x000000001044fL, 0x00000000104d8L, 0x00000000104d9L, 
        0x00000000104daL, 0x00000000104dbL, 0x00000000104dcL, 0x00000000104ddL, 0x00000000104deL, 0x00000000104dfL, 
        0x00000000104e0L, 0x00000000104e1L, 0x00000000104e2L, 0x00000000104e3L, 0x00000000104e4L, 0x00000000104e5L, 
        0x00000000104e6L, 0x00000000104e7L, 0x00000000104e8L, 0x00000000104e9L, 0x00000000104eaL, 0x00000000104ebL, 
        0x00000000104ecL, 0x00000000104edL, 0x00000000104eeL, 0x00000000104efL, 0x00000000104f0L, 0x00000000104f1L, 
        0x00000000104f2L, 0x00000000104f3L, 0x00000000104f4L, 0x00000000104f5L, 0x00000000104f6L, 0x00000000104f7L, 
        0x00000000104f8L, 0x00000000104f9L, 0x00000000104faL, 0x00000000104fbL, 0x0000000010597L, 0x0000000010598L, 
        0x0000000010599L, 0x000000001059aL, 0x000000001059bL, 0x000000001059cL, 0x000000001059dL, 0x000000001059eL, 
        0x000000001059fL, 0x00000000105a0L, 0x00000000105a1L, 0x00000000105a3L, 0x00000000105a4L, 0x00000000105a5L, 
        0x00000000105a6L, 0x00000000105a7L, 0x00000000105a8L, 0x00000000105a9L, 0x00000000105aaL, 0x00000000105abL, 
        0x00000000105acL, 0x00000000105adL, 0x00000000105aeL, 0x00000000105afL, 0x00000000105b0L, 0x00000000105b1L, 
        0x00000000105b3L, 0x00000000105b4L, 0x00000000105b5L, 0x00000000105b6L, 0x00000000105b7L, 0x00000000105b8L, 
        0x00000000105b9L, 0x00000000105bbL, 0x00000000105bcL, 0x0000000010cc0L, 0x0000000010cc1L, 0x0000000010cc2L, 
        0x0000000010cc3L, 0x0000000010cc4L, 0x0000000010cc5L, 0x0000000010cc6L, 0x0000000010cc7L, 0x0000000010cc8L, 
        0x0000000010cc9L, 0x0000000010ccaL, 0x0000000010ccbL, 0x0000000010cccL, 0x0000000010ccdL, 0x0000000010cceL, 
        0x0000000010ccfL, 0x0000000010cd0L, 0x0000000010cd1L, 0x0000000010cd2L, 0x0000000010cd3L, 0x0000000010cd4L, 
        0x0000000010cd5L, 0x0000000010cd6L, 0x0000000010cd7L, 0x0000000010cd8L, 0x0000000010cd9L, 0x0000000010cdaL, 
        0x0000000010cdbL, 0x0000000010cdcL, 0x0000000010cddL, 0x0000000010cdeL, 0x0000000010cdfL, 0x0000000010ce0L, 
        0x0000000010ce1L, 0x0000000010ce2L, 0x0000000010ce3L, 0x0000000010ce4L, 0x0000000010ce5L, 0x0000000010ce6L, 
        0x0000000010ce7L, 0x0000000010ce8L, 0x0000000010ce9L, 0x0000000010ceaL, 0x0000000010cebL, 0x0000000010cecL, 
        0x0000000010cedL, 0x0000000010ceeL, 0x0000000010cefL, 0x0000000010cf0L, 0x0000000010cf1L, 0x0000000010cf2L, 
        0x0000000010d70L, 0x0000000010d71L, 0x0000000010d72L, 0x0000000010d73L, 0x0000000010d74L, 0x0000000010d75L, 
        0x0000000010d76L, 0x0000000010d77L, 0x0000000010d78L, 0x0000000010d79L, 0x0000000010d7aL, 0x0000000010d7bL, 
        0x0000000010d7cL, 0x0000000010d7dL, 0x0000000010d7eL, 0x0000000010d7fL, 0x0000000010d80L, 0x0000000010d81L, 
        0x0000000010d82L, 0x0000000010d83L, 0x0000000010d84L, 0x0000000010d85L, 0x00000000118c0L, 0x00000000118c1L, 
        0x00000000118c2L, 0x00000000118c3L, 0x00000000118c4L, 0x00000000118c5L, 0x00000000118c6L, 0x00000000118c7L, 
        0x00000000118c8L, 0x00000000118c9L, 0x00000000118caL, 0x00000000118cbL, 0x00000000118ccL, 0x00000000118cdL, 
        0x00000000118ceL, 0x00000000118cfL, 0x00000000118d0L, 0x00000000118d1L, 0x00000000118d2L, 0x00000000118d3L, 
        0x00000000118d4L, 0x00000000118d5L, 0x00000000118d6L, 0x00000000118d7L, 0x00000000118d8L, 0x00000000118d9L, 
        0x00000000118daL, 0x00000000118dbL, 0x00000000118dcL, 0x00000000118ddL, 0x00000000118deL, 0x00000000118dfL, 
        0x0000000016e60L, 0x0000000016e61L, 0x0000000016e62L, 0x0000000016e63L, 0x0000000016e64L, 0x0000000016e65L, 
        0x0000000016e66L, 0x0000000016e67L, 0x0000000016e68L, 0x0000000016e69L, 0x0000000016e6aL, 0x0000000016e6bL, 
        0x0000000016e6cL, 0x0000000016e6dL, 0x0000000016e6eL, 0x0000000016e6fL, 0x0000000016e70L, 0x0000000016e71L, 
        0x0000000016e72L, 0x0000000016e73L, 0x0000000016e74L, 0x0000000016e75L, 0x0000000016e76L, 0x0000000016e77L, 
        0x0000000016e78L, 0x0000000016e79L, 0x0000000016e7aL, 0x0000000016e7bL, 0x0000000016e7cL, 0x0000000016e7dL, 
        0x0000000016e7eL, 0x0000000016e7fL, 0x0000000016ebbL, 0x0000000016ebcL, 0x0000000016ebdL, 0x0000000016ebeL, 
        0x0000000016ebfL, 0x0000000016ec0L, 0x0000000016ec1L, 0x0000000016ec2L, 0x0000000016ec3L, 0x0000000016ec4L, 
        0x0000000016ec5L, 0x0000000016ec6L, 0x0000000016ec7L, 0x0000000016ec8L, 0x0000000016ec9L, 0x0000000016ecaL, 
        0x0000000016ecbL, 0x0000000016eccL, 0x0000000016ecdL, 0x0000000016eceL, 0x0000000016ecfL, 0x0000000016ed0L, 
        0x0000000016ed1L, 0x0000000016ed2L, 0x0000000016ed3L, 0x000000001e922L, 0x000000001e923L, 0x000000001e924L, 
        0x000000001e925L, 0x000000001e926L, 0x000000001e927L, 0x000000001e928L, 0x000000001e929L, 0x000000001e92aL, 
        0x000000001e92bL, 0x000000001e92cL, 0x000000001e92dL, 0x000000001e92eL, 0x000000001e92fL, 0x000000001e930L, 
        0x000000001e931L, 0x000000001e932L, 0x000000001e933L, 0x000000001e934L, 0x000000001e935L, 0x000000001e936L, 
        0x000000001e937L, 0x000000001e938L, 0x000000001e939L, 0x000000001e93aL, 0x000000001e93bL, 0x000000001e93cL, 
        0x000000001e93dL, 0x000000001e93eL, 0x000000001e93fL, 0x000000001e940L, 0x000000001e941L, 0x000000001e942L, 
        0x000000001e943L
    };


    private static final int[][] CASE_FOLDING_HASHES = hashKeys(CASE_FOLDING_CPS);
}
